#
# Copyright 2024 Centreon (http://www.centreon.com/)
#
# Centreon is a full-fledged industry-strength solution that meets
# the needs in IT infrastructure and application monitoring for
# service performance.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

package apps::backup::quadstor::local::mode::vtljobstatus;

use base qw(centreon::plugins::templates::counter);

use strict;
use warnings;
use centreon::plugins::misc;
use Digest::MD5 qw(md5_hex);
use centreon::plugins::templates::catalog_functions qw(catalog_status_threshold_ng);

sub custom_status_output {
    my ($self, %options) = @_;

    return 'status : ' . $self->{result_values}->{status};
}

sub custom_long_output {
    my ($self, %options) = @_;

    return 'elapsed time : ' . centreon::plugins::misc::change_seconds(value => $self->{result_values}->{elapsed});
}

sub custom_frozen_threshold {
    my ($self, %options) = @_;

    my $status = catalog_status_threshold_ng($self, %options);
    $self->{instance_mode}->{last_status_frozen} = $status;
    return $status;
}

sub custom_frozen_output {
    my ($self, %options) = @_;
    my $msg = 'frozen : no';

    if (!$self->{output}->is_status(value => $self->{instance_mode}->{last_status_frozen}, compare => 'ok', litteral => 1)) {
        $msg = 'frozen: yes';
    }    
    return $msg;
}

sub custom_frozen_calc {
    my ($self, %options) = @_;
    
    $self->{result_values}->{status} = $options{new_datas}->{$self->{instance} . '_status'};
    $self->{result_values}->{display} = $options{new_datas}->{$self->{instance} . '_display'};
    $self->{result_values}->{elapsed} = $options{new_datas}->{$self->{instance} . '_elapsed'};
    $self->{result_values}->{kb} = $options{new_datas}->{$self->{instance} . '_kb'} - $options{old_datas}->{$self->{instance} . '_kb'};

    return 0;
}

sub prefix_job_output {
    my ($self, %options) = @_;
    
    return "job '" . $options{instance_value}->{display} . "' ";
}

sub set_counters {
    my ($self, %options) = @_;
    
    $self->{maps_counters_type} = [
       { name => 'jobs', type => 2, message_multiple => '0 problem(s) detected', display_counter_problem => { label => 'alerts', min => 0 },
          group => [ { name => 'job', cb_prefix_output => 'prefix_job_output', skipped_code => { -11 => 1 } } ] 
        }
    ];
    
    $self->{maps_counters}->{job} = [
        { label => 'status', type => 2, critical_default => '%{status} =~ /error/i', set => {
                key_values => [ { name => 'status' }, { name => 'display' } ],
                closure_custom_output => $self->can('custom_status_output'),
                closure_custom_perfdata => sub { return 0; },
                closure_custom_threshold_check => \&catalog_status_threshold_ng
            }
        },
        { label => 'long', threshold => 2, set => {
                key_values => [ { name => 'status' }, { name => 'display' }, { name => 'elapsed' } ],
                closure_custom_output => $self->can('custom_long_output'),
                closure_custom_perfdata => sub { return 0; },
                closure_custom_threshold_check => \&catalog_status_threshold_ng
            }
        },
        { label => 'frozen', threshold => 0, set => {
                key_values => [ { name => 'kb', diff => 1 }, { name => 'status' }, { name => 'display' }, { name => 'elapsed' } ],
                closure_custom_calc => $self->can('custom_frozen_calc'),
                closure_custom_output => $self->can('custom_frozen_output'),
                closure_custom_perfdata => sub { return 0; },
                closure_custom_threshold_check => $self->can('custom_frozen_threshold')
            }
        }
    ];
}

sub new {
    my ($class, %options) = @_;
    my $self = $class->SUPER::new(package => __PACKAGE__, %options, statefile => 1);
    bless $self, $class;
    
    $options{options}->add_options(arguments => {});

    return $self;
}

sub manage_selection {
    my ($self, %options) = @_;

    $self->{cache_name} = 'quadstor_' . $self->{mode} . '_' . $options{custom}->get_identifier() . '_' .
        (defined($self->{option_results}->{filter_counters}) ? md5_hex($self->{option_results}->{filter_counters}) : md5_hex('all'));

    my ($stdout) = $options{custom}->execute_command(
        command => 'impexp',
        command_options => '-l',
        command_path => '/quadstorvtl/bin'
    );

    $self->{jobs}->{global} = { job => {} };
    #JobID  Type     Source           State        Transfer       Elapsed
    #252    Import   701831L2         Error        36.00 GB       572
    #253    Export   701849L2         Completed    19.43 GB       262
    #254    Export   701850L2         Completed    16.05 GB       1072
    #255    Export   701854L2         Completed    6.31 GB        142
    my $current_time = time();
    my @lines = split(/\n/, $stdout);
    shift @lines;
    foreach (@lines) {
        next if (! /^(\d+)\s+\S+\s+(\S+)\s+(\S+)\s+([0-9\.]+)\s+\S+\s+(\d+)/);

        my ($job_id, $job_source, $job_state, $job_kb, $job_elapsed) = 
            ($1, $2, $3, $4, $5);

        my $name = $job_source . '.' . $job_id;
        $self->{jobs}->{global}->{job}->{$name} = { 
            display => $name,
            status => $job_state,
            kb => $job_kb * 1024,
            elapsed => $job_elapsed
        };
    }

    if (scalar(keys %{$self->{jobs}->{global}}) <= 0) {
        $self->{output}->add_option_msg(short_msg => "No job found.");
        $self->{output}->option_exit();
    }
}

1;

__END__

=head1 MODE

Check job status.

Command used: '/quadstorvtl/bin/impexp -l'

=over 8

=item B<--warning-status>

Define the conditions to match for the status to be WARNING (default: none)
You can use the following variables: %{display}, %{status}

=item B<--critical-status>

Define the conditions to match for the status to be CRITICAL (default: '%{status} =~ /error/i').
You can use the following variables: %{display}, %{status}

=item B<--warning-long>

Set warning threshold for long jobs (default: none)
You can use the following variables: %{display}, %{status}, %{elapsed}

=item B<--critical-long>

Set critical threshold for long jobs (default: none).
You can use the following variables: %{display}, %{status}, %{elapsed}

=item B<--warning-frozen>

Set warning threshold for frozen jobs (default: none)
You can use the following variables: %{display}, %{status}, %{elapsed}, %{kb}

=item B<--critical-frozen>

Set critical threshold for frozen jobs (default: none).
You can use the following variables: %{display}, %{status}, %{elapsed}, %{kb}

=back

=cut
