"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BinaryView = exports.Read = void 0;
const errors_1 = require("@ot-builder/errors");
const primitive_types_1 = require("./primitive-types");
function Read(proc) {
    return { read: proc };
}
exports.Read = Read;
class BinaryView {
    constructor(buffer, displace = 0, cursor = displace) {
        this.buffer = buffer;
        this.displace = displace;
        this.cursor = cursor;
    }
    get sourceBufferSize() {
        return this.buffer.byteLength;
    }
    next(inst, ...args) {
        return inst.read(this, ...args);
    }
    array(count, inst, ...args) {
        const arr = [];
        for (let mu = 0; mu < count; mu++) {
            arr[mu] = inst.read(this, ...args);
        }
        return arr;
    }
    /** Create a new BinaryView using an offset relative to current view's displacement. */
    lift(offset) {
        return new BinaryView(this.buffer, this.displace + offset);
    }
    /** Create a new BinaryView using an offset relative to the current cursor. */
    liftRelative(offset) {
        return new BinaryView(this.buffer, this.cursor + offset);
    }
    // Primitive methods
    uint8() {
        const n = this.buffer.readUInt8(this.cursor);
        this.cursor += primitive_types_1.SizeofUInt8;
        return n;
    }
    uint16() {
        const n = this.buffer.readUInt16BE(this.cursor);
        this.cursor += primitive_types_1.SizeofUInt16;
        return n;
    }
    uint32() {
        const n = this.buffer.readUInt32BE(this.cursor);
        this.cursor += primitive_types_1.SizeofUInt32;
        return n;
    }
    int8() {
        const n = this.buffer.readInt8(this.cursor);
        this.cursor += primitive_types_1.SizeofUInt8;
        return n;
    }
    int16() {
        const n = this.buffer.readInt16BE(this.cursor);
        this.cursor += primitive_types_1.SizeofUInt16;
        return n;
    }
    int32() {
        const n = this.buffer.readInt32BE(this.cursor);
        this.cursor += primitive_types_1.SizeofUInt32;
        return n;
    }
    ptr16() {
        const v = this.uint16();
        if (v)
            return this.lift(v);
        else
            throw errors_1.Errors.NullPtr();
    }
    ptr16Nullable() {
        const v = this.uint16();
        if (v)
            return this.lift(v);
        else
            return null;
    }
    ptr32() {
        const v = this.uint32();
        if (v)
            return this.lift(v);
        else
            throw errors_1.Errors.NullPtr();
    }
    ptr32Nullable() {
        const v = this.uint32();
        if (v)
            return this.lift(v);
        else
            return null;
    }
    bytes(length) {
        const buf = this.buffer.slice(this.cursor, this.cursor + length);
        this.cursor += length;
        return buf;
    }
    *repeat(count) {
        for (let mu = 0; mu < count; mu++) {
            yield [this, mu];
        }
    }
}
exports.BinaryView = BinaryView;
//# sourceMappingURL=binary-view.js.map