/* -*- c++ -*- ----------------------------------------------------------
   LAMMPS - Large-scale Atomic/Molecular Massively Parallel Simulator
   https://www.lammps.org/, Sandia National Laboratories
   LAMMPS development team: developers@lammps.org

   Copyright (2003) Sandia Corporation.  Under the terms of Contract
   DE-AC04-94AL85000 with Sandia Corporation, the U.S. Government retains
   certain rights in this software.  This software is distributed under
   the GNU General Public License.

   See the README file in the top-level LAMMPS directory.
  -------------------------------------------------------------------------
   Contributing authors:
      Sonia Salomoni^1,^2
      Arthur France-Lanord^1

      ^1: IMPMC, CNRS, Sorbonne Universite, Paris, France
      ^2: SCAI, Sorbonne Universite, Paris, France
------------------------------------------------------------------------- */
#ifndef LMP_D3_PARAMETERS_H
#define LMP_D3_PARAMETERS_H

// clang-format off

/* scale r4/r2 values of the atoms by sqrt(Z)
   * sqrt is also globally close to optimum
   * together with the factor 1/2 this yield reasonable
   * c8 for he, ne and ar. for larger Z, C8 becomes too large
   * which effectively mimics higher R^n terms neglected due
   * to stability reasons.
   * r2r4 =sqrt(0.5*r2r4(i)*ddouble(i)**0.5 ) with i=elementnumber
   * the large number of digits is just to keep the results consistent
   * with older versions. They should not imply any higher accuracy than
   * the old values */

static constexpr double r2r4_ref[NUM_ELEMENTS + 1] = {
    0,           2.00734898,  1.56637132,  5.01986934,  3.85379032,  3.64446594,  3.10492822,
    2.71175247,  2.59361680,  2.38825250,  2.21522516,  6.58585536,  5.46295967,  5.65216669,
    4.88284902,  4.29727576,  4.04108902,  3.72932356,  3.44677275,  7.97762753,  7.07623947,
    6.60844053,  6.28791364,  6.07728703,  5.54643096,  5.80491167,  5.58415602,  5.41374528,
    5.28497229,  5.22592821,  5.09817141,  6.12149689,  5.54083734,  5.06696878,  4.87005108,
    4.59089647,  4.31176304,  9.55461698,  8.67396077,  7.97210197,  7.43439917,  6.58711862,
    6.19536215,  6.01517290,  5.81623410,  5.65710424,  5.52640661,  5.44263305,  5.58285373,
    7.02081898,  6.46815523,  5.98089120,  5.81686657,  5.53321815,  5.25477007, 11.02204549,
   10.15679528,  9.35167836,  9.06926079,  8.97241155,  8.90092807,  8.85984840,  8.81736827,
    8.79317710,  7.89969626,  8.80588454,  8.42439218,  8.54289262,  8.47583370,  8.45090888,
    8.47339339,  7.83525634,  8.20702843,  7.70559063,  7.32755997,  7.03887381,  6.68978720,
    6.05450052,  5.88752022,  5.70661499,  5.78450695,  7.79780729,  7.26443867,  6.78151984,
    6.67883169,  6.39024318,  6.09527958, 11.79156076, 11.10997644,  9.51377795,  8.67197068,
    8.77140725,  8.65402716,  8.53923501,  8.85024712};    // atomic <r^2>/<r^4> values

static constexpr double rcov_ref[NUM_ELEMENTS + 1] = {
    0,           0.80628308,  1.15903197,  3.02356173,  2.36845659,  1.94011865,  1.88972601,
    1.78894056,  1.58736983,  1.61256616,  1.68815527,  3.52748848,  3.14954334,  2.84718717,
    2.62041997,  2.77159820,  2.57002732,  2.49443835,  2.41884923,  4.43455700,  3.88023730,
    3.35111422,  3.07395437,  3.04875805,  2.77159820,  2.69600923,  2.62041997,  2.51963467,
    2.49443835,  2.54483100,  2.74640188,  2.82199085,  2.74640188,  2.89757982,  2.77159820,
    2.87238349,  2.94797246,  4.76210950,  4.20778980,  3.70386304,  3.50229216,  3.32591790,
    3.12434702,  2.89757982,  2.84718717,  2.84718717,  2.72120556,  2.89757982,  3.09915070,
    3.22513231,  3.17473967,  3.17473967,  3.09915070,  3.32591790,  3.30072128,  5.26603625,
    4.43455700,  4.08180818,  3.70386304,  3.98102289,  3.95582657,  3.93062995,  3.90543362,
    3.80464833,  3.82984466,  3.80464833,  3.77945201,  3.75425569,  3.75425569,  3.72905937,
    3.85504098,  3.67866672,  3.45189952,  3.30072128,  3.09915070,  2.97316878,  2.92277614,
    2.79679452,  2.82199085,  2.84718717,  3.32591790,  3.27552496,  3.27552496,  3.42670319,
    3.30072128,  3.47709584,  3.57788113,  5.06446567,  4.56053862,  4.20778980,  3.98102289,
    3.82984466,  3.85504098,  3.88023730,  3.90543362};    // covalent radii

static constexpr double r0ab_table[NUM_ELEMENTS][NUM_ELEMENTS] = {
    {2.18230000E+00, 1.85470000E+00, 2.90860000E+00, 2.35500000E+00, 2.51410000E+00, 2.44920000E+00,
     2.36670000E+00, 2.17680000E+00, 2.06460000E+00, 1.98920000E+00, 2.98980000E+00, 2.51290000E+00,
     2.87660000E+00, 2.88840000E+00, 2.83040000E+00, 2.61900000E+00, 2.47570000E+00, 2.37250000E+00,
     3.21470000E+00, 2.75970000E+00, 2.73610000E+00, 2.68320000E+00, 2.63790000E+00, 2.92300000E+00,
     2.54900000E+00, 2.49260000E+00, 2.46670000E+00, 2.45150000E+00, 2.44130000E+00, 2.43180000E+00,
     2.85020000E+00, 2.92260000E+00, 2.91400000E+00, 2.73000000E+00, 2.60260000E+00, 2.50870000E+00,
     3.28930000E+00, 2.87860000E+00, 2.89970000E+00, 2.85350000E+00, 3.09050000E+00, 3.03010000E+00,
     2.91890000E+00, 2.85780000E+00, 2.82400000E+00, 2.80040000E+00, 2.78460000E+00, 2.54240000E+00,
     2.97430000E+00, 3.07750000E+00, 3.08930000E+00, 2.90910000E+00, 2.78060000E+00, 2.68480000E+00,
     3.39680000E+00, 3.06300000E+00, 3.06320000E+00, 2.94550000E+00, 2.92440000E+00, 2.90460000E+00,
     2.88780000E+00, 2.87090000E+00, 2.85680000E+00, 2.67980000E+00, 2.51890000E+00, 3.04810000E+00,
     2.81360000E+00, 2.80320000E+00, 2.79670000E+00, 2.78970000E+00, 2.83950000E+00, 2.90070000E+00,
     2.85590000E+00, 2.81100000E+00, 2.76760000E+00, 2.69120000E+00, 2.78950000E+00, 2.75220000E+00,
     2.72020000E+00, 2.53150000E+00, 2.98520000E+00, 3.11600000E+00, 3.14160000E+00, 2.96640000E+00,
     2.83880000E+00, 2.74120000E+00, 3.41550000E+00, 3.05480000E+00, 3.38500000E+00, 3.09420000E+00,
     3.01870000E+00, 2.98750000E+00, 2.96590000E+00, 2.87630000E+00},
    {1.85470000E+00, 1.73470000E+00, 2.57320000E+00, 2.50950000E+00, 2.39170000E+00, 2.25270000E+00,
     2.13280000E+00, 2.06250000E+00, 1.98910000E+00, 1.92510000E+00, 2.63970000E+00, 2.73210000E+00,
     2.74270000E+00, 2.66050000E+00, 2.55380000E+00, 2.47970000E+00, 2.40190000E+00, 2.32890000E+00,
     2.83150000E+00, 3.05880000E+00, 2.97650000E+00, 2.90600000E+00, 2.84940000E+00, 2.62230000E+00,
     2.76230000E+00, 2.71360000E+00, 2.67090000E+00, 2.63380000E+00, 2.60060000E+00, 2.57290000E+00,
     2.69340000E+00, 2.68380000E+00, 2.62710000E+00, 2.58700000E+00, 2.53550000E+00, 2.48140000E+00,
     2.89910000E+00, 3.16580000E+00, 3.09530000E+00, 3.01670000E+00, 2.79980000E+00, 2.73120000E+00,
     2.67770000E+00, 2.63340000E+00, 2.59600000E+00, 2.56210000E+00, 2.53240000E+00, 2.66010000E+00,
     2.80580000E+00, 2.82750000E+00, 2.79170000E+00, 2.76380000E+00, 2.72090000E+00, 2.67400000E+00,
     3.01220000E+00, 3.32710000E+00, 3.26620000E+00, 3.24700000E+00, 3.22410000E+00, 3.20440000E+00,
     3.18900000E+00, 3.17370000E+00, 3.16080000E+00, 3.01510000E+00, 2.97030000E+00, 2.95840000E+00,
     3.09510000E+00, 3.08050000E+00, 3.06890000E+00, 3.06360000E+00, 3.00810000E+00, 2.96740000E+00,
     2.90500000E+00, 2.85200000E+00, 2.80780000E+00, 2.76040000E+00, 2.58490000E+00, 2.55240000E+00,
     2.51830000E+00, 2.61040000E+00, 2.78810000E+00, 2.84820000E+00, 2.83250000E+00, 2.82090000E+00,
     2.78900000E+00, 2.74890000E+00, 3.05220000E+00, 3.32060000E+00, 3.42080000E+00, 3.20750000E+00,
     3.19180000E+00, 3.16740000E+00, 3.14420000E+00, 3.12940000E+00},
    {2.90860000E+00, 2.57320000E+00, 3.49560000E+00, 2.98020000E+00, 2.99770000E+00, 3.19330000E+00,
     2.87840000E+00, 2.63950000E+00, 2.50860000E+00, 2.41900000E+00, 3.60310000E+00, 3.10520000E+00,
     3.27760000E+00, 3.75130000E+00, 3.39320000E+00, 3.13310000E+00, 2.97890000E+00, 2.87600000E+00,
     3.87240000E+00, 3.34740000E+00, 3.24750000E+00, 3.26130000E+00, 3.18400000E+00, 3.41610000E+00,
     3.09910000E+00, 3.06930000E+00, 3.05750000E+00, 3.05110000E+00, 3.05400000E+00, 3.04160000E+00,
     3.24670000E+00, 3.78670000E+00, 3.47710000E+00, 3.24710000E+00, 3.11290000E+00, 3.02390000E+00,
     4.03880000E+00, 3.58230000E+00, 3.45240000E+00, 3.39780000E+00, 3.64480000E+00, 3.58210000E+00,
     3.49250000E+00, 3.46730000E+00, 3.46350000E+00, 3.46170000E+00, 3.46520000E+00, 3.25550000E+00,
     3.48300000E+00, 4.01810000E+00, 3.72620000E+00, 3.50660000E+00, 3.38120000E+00, 3.29250000E+00,
     4.22410000E+00, 4.04320000E+00, 3.64320000E+00, 3.61080000E+00, 3.58480000E+00, 3.56210000E+00,
     3.54120000E+00, 3.51990000E+00, 3.50500000E+00, 3.25860000E+00, 3.30630000E+00, 3.69190000E+00,
     3.46350000E+00, 3.45190000E+00, 3.44450000E+00, 3.43440000E+00, 3.31710000E+00, 3.81740000E+00,
     3.83250000E+00, 3.74800000E+00, 3.67250000E+00, 3.60370000E+00, 3.64840000E+00, 3.57420000E+00,
     3.49470000E+00, 3.27340000E+00, 3.52840000E+00, 4.07390000E+00, 3.77350000E+00, 3.55470000E+00,
     3.43290000E+00, 3.33740000E+00, 4.25410000E+00, 3.81320000E+00, 3.70040000E+00, 3.51520000E+00,
     3.51270000E+00, 3.49710000E+00, 3.47950000E+00, 3.55980000E+00},
    {2.35500000E+00, 2.50950000E+00, 2.98020000E+00, 3.09820000E+00, 2.94840000E+00, 3.02140000E+00,
     2.76600000E+00, 2.66480000E+00, 2.69080000E+00, 2.54730000E+00, 3.12190000E+00, 3.29620000E+00,
     3.23630000E+00, 3.16130000E+00, 3.11930000E+00, 3.05400000E+00, 3.14680000E+00, 2.98040000E+00,
     3.46210000E+00, 3.62140000E+00, 3.50040000E+00, 3.43590000E+00, 3.42250000E+00, 2.89990000E+00,
     3.32520000E+00, 3.26990000E+00, 3.23570000E+00, 3.22290000E+00, 3.19870000E+00, 3.16390000E+00,
     3.19210000E+00, 3.17320000E+00, 3.17740000E+00, 3.14870000E+00, 3.28630000E+00, 3.13120000E+00,
     3.61000000E+00, 3.76890000E+00, 3.61070000E+00, 3.53630000E+00, 3.07390000E+00, 3.04730000E+00,
     2.96480000E+00, 2.92450000E+00, 2.90320000E+00, 2.91540000E+00, 2.87590000E+00, 3.28070000E+00,
     3.34740000E+00, 3.33850000E+00, 3.35180000E+00, 3.33770000E+00, 3.49580000E+00, 3.33860000E+00,
     3.78870000E+00, 4.00460000E+00, 3.80880000E+00, 3.85220000E+00, 3.82930000E+00, 3.80780000E+00,
     3.79080000E+00, 3.77340000E+00, 3.75980000E+00, 3.52920000E+00, 3.62350000E+00, 3.59900000E+00,
     3.68750000E+00, 3.67000000E+00, 3.65810000E+00, 3.64800000E+00, 3.48780000E+00, 3.58560000E+00,
     3.54420000E+00, 3.51050000E+00, 3.48040000E+00, 3.41940000E+00, 3.01620000E+00, 2.95080000E+00,
     2.87310000E+00, 3.22990000E+00, 3.34930000E+00, 3.35990000E+00, 3.37870000E+00, 3.37960000E+00,
     3.55930000E+00, 3.39500000E+00, 3.82180000E+00, 3.97200000E+00, 3.88000000E+00, 3.66590000E+00,
     3.68750000E+00, 3.67150000E+00, 3.65030000E+00, 3.74650000E+00},
    {2.51410000E+00, 2.39170000E+00, 2.99770000E+00, 2.94840000E+00, 3.21600000E+00, 2.95310000E+00,
     2.77760000E+00, 2.64820000E+00, 2.62330000E+00, 2.49940000E+00, 3.76200000E+00, 3.13310000E+00,
     3.59290000E+00, 3.36050000E+00, 3.18660000E+00, 3.06510000E+00, 2.97680000E+00, 2.90930000E+00,
     3.88230000E+00, 3.43530000E+00, 3.41850000E+00, 3.35380000E+00, 3.27710000E+00, 3.05570000E+00,
     3.18360000E+00, 3.12720000E+00, 3.09080000E+00, 3.06300000E+00, 3.04900000E+00, 3.01960000E+00,
     3.56630000E+00, 3.38720000E+00, 3.25600000E+00, 3.16670000E+00, 3.10290000E+00, 3.05350000E+00,
     3.93880000E+00, 3.57620000E+00, 3.60620000E+00, 3.53930000E+00, 3.29960000E+00, 3.21370000E+00,
     3.12160000E+00, 3.07320000E+00, 3.04310000E+00, 3.02030000E+00, 3.00510000E+00, 3.13840000E+00,
     3.68630000E+00, 3.53790000E+00, 3.42410000E+00, 3.34810000E+00, 3.29130000E+00, 3.24730000E+00,
     4.00490000E+00, 4.15550000E+00, 3.79100000E+00, 3.66250000E+00, 3.63950000E+00, 3.61850000E+00,
     3.59740000E+00, 3.58020000E+00, 3.56650000E+00, 3.53910000E+00, 3.45170000E+00, 3.88810000E+00,
     3.49870000E+00, 3.48270000E+00, 3.47170000E+00, 3.46260000E+00, 3.53600000E+00, 3.64860000E+00,
     3.50770000E+00, 3.43460000E+00, 3.37750000E+00, 3.30940000E+00, 3.12670000E+00, 3.07510000E+00,
     3.01980000E+00, 3.10900000E+00, 3.69580000E+00, 3.56980000E+00, 3.46320000E+00, 3.39850000E+00,
     3.34880000E+00, 3.30760000E+00, 4.04380000E+00, 3.76750000E+00, 3.98860000E+00, 3.82890000E+00,
     3.74040000E+00, 3.71140000E+00, 3.68140000E+00, 3.56590000E+00},
    {2.44920000E+00, 2.25270000E+00, 3.19330000E+00, 3.02140000E+00, 2.95310000E+00, 2.91030000E+00,
     2.70630000E+00, 2.56970000E+00, 2.47700000E+00, 2.40910000E+00, 3.24850000E+00, 3.20000000E+00,
     3.28260000E+00, 3.33250000E+00, 3.12450000E+00, 2.98790000E+00, 2.88480000E+00, 2.80400000E+00,
     3.37600000E+00, 3.47290000E+00, 3.43780000E+00, 3.38600000E+00, 3.34010000E+00, 3.33080000E+00,
     3.24280000E+00, 3.18930000E+00, 3.15370000E+00, 3.12650000E+00, 3.10580000E+00, 3.08430000E+00,
     3.25410000E+00, 3.36430000E+00, 3.19700000E+00, 3.09140000E+00, 3.01080000E+00, 2.94570000E+00,
     3.44750000E+00, 3.57890000E+00, 3.57830000E+00, 3.53010000E+00, 3.52620000E+00, 3.46790000E+00,
     3.29400000E+00, 3.24350000E+00, 3.21150000E+00, 3.18750000E+00, 3.16920000E+00, 3.17370000E+00,
     3.36170000E+00, 3.50360000E+00, 3.54330000E+00, 3.26330000E+00, 3.18880000E+00, 3.12840000E+00,
     3.53840000E+00, 3.74260000E+00, 3.73810000E+00, 3.65980000E+00, 3.64000000E+00, 3.62200000E+00,
     3.60780000E+00, 3.59340000E+00, 3.58030000E+00, 3.49020000E+00, 3.39890000E+00, 3.42090000E+00,
     3.51830000E+00, 3.50500000E+00, 3.49510000E+00, 3.48920000E+00, 3.51450000E+00, 3.53390000E+00,
     3.49050000E+00, 3.34610000E+00, 3.24110000E+00, 3.17100000E+00, 3.21250000E+00, 3.01580000E+00,
     3.14570000E+00, 2.99420000E+00, 3.36420000E+00, 3.53660000E+00, 3.59230000E+00, 3.31640000E+00,
     3.24860000E+00, 3.19100000E+00, 3.58750000E+00, 3.73510000E+00, 3.90400000E+00, 3.74590000E+00,
     3.69430000E+00, 3.66920000E+00, 3.64760000E+00, 3.58160000E+00},
    {2.36670000E+00, 2.13280000E+00, 2.87840000E+00, 2.76600000E+00, 2.77760000E+00, 2.70630000E+00,
     2.62250000E+00, 2.48460000E+00, 2.38850000E+00, 2.31760000E+00, 2.93570000E+00, 2.95860000E+00,
     3.09110000E+00, 3.09910000E+00, 3.04650000E+00, 2.90540000E+00, 2.79520000E+00, 2.70710000E+00,
     3.07460000E+00, 3.24870000E+00, 3.20840000E+00, 3.15500000E+00, 3.10720000E+00, 3.05550000E+00,
     3.02590000E+00, 2.96580000E+00, 2.92350000E+00, 2.89090000E+00, 2.86430000E+00, 2.84130000E+00,
     3.05710000E+00, 3.12670000E+00, 3.12070000E+00, 3.01070000E+00, 2.92270000E+00, 2.84960000E+00,
     3.15900000E+00, 3.35520000E+00, 3.34630000E+00, 3.29600000E+00, 3.25590000E+00, 3.19750000E+00,
     3.09750000E+00, 3.03380000E+00, 2.98920000E+00, 2.95480000E+00, 2.92730000E+00, 2.93970000E+00,
     3.16080000E+00, 3.25890000E+00, 3.27730000E+00, 3.18100000E+00, 3.09900000E+00, 3.03010000E+00,
     3.26980000E+00, 3.51300000E+00, 3.50930000E+00, 3.44110000E+00, 3.42040000E+00, 3.40190000E+00,
     3.38710000E+00, 3.37240000E+00, 3.36010000E+00, 3.28870000E+00, 3.21070000E+00, 3.16060000E+00,
     3.29370000E+00, 3.27990000E+00, 3.26940000E+00, 3.26260000E+00, 3.28090000E+00, 3.28320000E+00,
     3.23960000E+00, 3.19710000E+00, 3.15810000E+00, 3.08620000E+00, 3.00430000E+00, 2.96440000E+00,
     2.92760000E+00, 2.91590000E+00, 3.15680000E+00, 3.28540000E+00, 3.32140000E+00, 3.23640000E+00,
     3.16150000E+00, 3.09610000E+00, 3.32860000E+00, 3.51670000E+00, 3.67190000E+00, 3.51560000E+00,
     3.47020000E+00, 3.44460000E+00, 3.42220000E+00, 3.35990000E+00},
    {2.17680000E+00, 2.06250000E+00, 2.63950000E+00, 2.66480000E+00, 2.64820000E+00, 2.56970000E+00,
     2.48460000E+00, 2.48170000E+00, 2.35110000E+00, 2.25710000E+00, 2.70930000E+00, 3.08220000E+00,
     2.93690000E+00, 2.92970000E+00, 2.87270000E+00, 2.88050000E+00, 2.74570000E+00, 2.63860000E+00,
     2.88170000E+00, 3.32000000E+00, 3.27870000E+00, 3.23000000E+00, 3.18850000E+00, 2.85080000E+00,
     3.12250000E+00, 3.09720000E+00, 3.06690000E+00, 3.02530000E+00, 2.99480000E+00, 2.74360000E+00,
     2.90480000E+00, 2.95410000E+00, 2.94060000E+00, 2.98120000E+00, 2.86940000E+00, 2.77800000E+00,
     2.98120000E+00, 3.40040000E+00, 3.38550000E+00, 3.33520000E+00, 3.05180000E+00, 2.99690000E+00,
     2.97570000E+00, 2.94620000E+00, 2.91480000E+00, 2.80380000E+00, 2.76150000E+00, 2.84290000E+00,
     3.00690000E+00, 3.08040000E+00, 3.08900000E+00, 3.14280000E+00, 3.03940000E+00, 2.95310000E+00,
     3.10830000E+00, 3.51740000E+00, 3.51550000E+00, 3.46600000E+00, 3.44990000E+00, 3.43590000E+00,
     3.42430000E+00, 3.41280000E+00, 3.40310000E+00, 3.33220000E+00, 3.24340000E+00, 3.19380000E+00,
     3.35800000E+00, 3.34820000E+00, 3.33970000E+00, 3.33440000E+00, 3.33070000E+00, 3.31540000E+00,
     3.27200000E+00, 3.23260000E+00, 3.19830000E+00, 3.17890000E+00, 2.95720000E+00, 2.83380000E+00,
     2.78260000E+00, 2.83240000E+00, 3.00550000E+00, 3.10390000E+00, 3.12850000E+00, 3.19560000E+00,
     3.10000000E+00, 3.01750000E+00, 3.16820000E+00, 3.52740000E+00, 3.65470000E+00, 3.51970000E+00,
     3.48880000E+00, 3.46760000E+00, 3.44910000E+00, 3.40240000E+00},
    {2.06460000E+00, 1.98910000E+00, 2.50860000E+00, 2.69080000E+00, 2.62330000E+00, 2.47700000E+00,
     2.38850000E+00, 2.35110000E+00, 2.29960000E+00, 2.19460000E+00, 2.57810000E+00, 2.85820000E+00,
     2.90300000E+00, 2.86740000E+00, 2.76640000E+00, 2.73300000E+00, 2.68810000E+00, 2.57200000E+00,
     2.75520000E+00, 3.09140000E+00, 3.06040000E+00, 3.01330000E+00, 2.97140000E+00, 2.73850000E+00,
     2.90320000E+00, 2.87780000E+00, 2.84760000E+00, 2.81840000E+00, 2.79080000E+00, 2.76080000E+00,
     2.86570000E+00, 2.85050000E+00, 2.83220000E+00, 2.83000000E+00, 2.81090000E+00, 2.78280000E+00,
     2.85860000E+00, 3.17220000E+00, 3.17460000E+00, 3.12870000E+00, 2.93940000E+00, 2.88470000E+00,
     2.84930000E+00, 2.81430000E+00, 2.78010000E+00, 2.74720000E+00, 2.71640000E+00, 2.84920000E+00,
     2.96400000E+00, 3.00940000E+00, 2.97750000E+00, 2.98720000E+00, 2.97890000E+00, 2.96020000E+00,
     2.99170000E+00, 3.28840000E+00, 3.30470000E+00, 3.24150000E+00, 3.22530000E+00, 3.21100000E+00,
     3.19920000E+00, 3.18770000E+00, 3.17790000E+00, 3.12280000E+00, 3.00940000E+00, 2.99750000E+00,
     3.13250000E+00, 3.12330000E+00, 3.11470000E+00, 3.10880000E+00, 3.12600000E+00, 3.11440000E+00,
     3.07260000E+00, 3.03290000E+00, 2.99730000E+00, 2.97380000E+00, 2.81970000E+00, 2.78910000E+00,
     2.75740000E+00, 2.83500000E+00, 2.95580000E+00, 2.99530000E+00, 3.01470000E+00, 3.03700000E+00,
     3.03940000E+00, 3.02800000E+00, 3.05660000E+00, 3.30850000E+00, 3.46250000E+00, 3.32900000E+00,
     3.29140000E+00, 3.26850000E+00, 3.24940000E+00, 3.18770000E+00},
    {1.98920000E+00, 1.92510000E+00, 2.41900000E+00, 2.54730000E+00, 2.49940000E+00, 2.40910000E+00,
     2.31760000E+00, 2.25710000E+00, 2.19460000E+00, 2.13740000E+00, 2.48390000E+00, 2.71200000E+00,
     2.77890000E+00, 2.75710000E+00, 2.69260000E+00, 2.63310000E+00, 2.57280000E+00, 2.51390000E+00,
     2.66050000E+00, 2.94030000E+00, 2.91870000E+00, 2.87360000E+00, 2.83190000E+00, 2.66400000E+00,
     2.76210000E+00, 2.73580000E+00, 2.70540000E+00, 2.67640000E+00, 2.64910000E+00, 2.62710000E+00,
     2.74380000E+00, 2.77810000E+00, 2.75710000E+00, 2.72840000E+00, 2.69290000E+00, 2.65320000E+00,
     2.76830000E+00, 3.02120000E+00, 3.03810000E+00, 2.99670000E+00, 2.86580000E+00, 2.81100000E+00,
     2.76380000E+00, 2.72400000E+00, 2.68730000E+00, 2.65300000E+00, 2.62120000E+00, 2.72250000E+00,
     2.84270000E+00, 2.90030000E+00, 2.90100000E+00, 2.88370000E+00, 2.85820000E+00, 2.82720000E+00,
     2.90570000E+00, 3.13780000E+00, 3.16810000E+00, 3.09440000E+00, 3.07790000E+00, 3.06350000E+00,
     3.05130000E+00, 3.03960000E+00, 3.02960000E+00, 2.98880000E+00, 2.85800000E+00, 2.86460000E+00,
     2.98320000E+00, 2.97470000E+00, 2.96610000E+00, 2.95970000E+00, 2.99400000E+00, 2.98660000E+00,
     2.94670000E+00, 2.90700000E+00, 2.87050000E+00, 2.84270000E+00, 2.72610000E+00, 2.69330000E+00,
     2.66060000E+00, 2.72160000E+00, 2.83930000E+00, 2.91920000E+00, 2.93660000E+00, 2.93130000E+00,
     2.91650000E+00, 2.89290000E+00, 2.97460000E+00, 3.16530000E+00, 3.33700000E+00, 3.20690000E+00,
     3.16430000E+00, 3.14050000E+00, 3.12090000E+00, 3.04840000E+00},
    {2.98980000E+00, 2.63970000E+00, 3.60310000E+00, 3.12190000E+00, 3.76200000E+00, 3.24850000E+00,
     2.93570000E+00, 2.70930000E+00, 2.57810000E+00, 2.48390000E+00, 3.70820000E+00, 3.25700000E+00,
     3.39210000E+00, 3.81290000E+00, 3.46080000E+00, 3.20960000E+00, 3.05740000E+00, 2.95170000E+00,
     3.97400000E+00, 3.49720000E+00, 3.40370000E+00, 3.40240000E+00, 3.33150000E+00, 3.52630000E+00,
     3.24900000E+00, 3.22060000E+00, 3.20640000E+00, 3.19680000E+00, 3.19500000E+00, 3.18110000E+00,
     3.35470000E+00, 3.84750000E+00, 3.54550000E+00, 3.32590000E+00, 3.19580000E+00, 3.10630000E+00,
     4.14280000E+00, 3.72410000E+00, 3.60190000E+00, 3.66590000E+00, 3.75140000E+00, 3.69310000E+00,
     3.60850000E+00, 3.58320000E+00, 3.57760000E+00, 3.57360000E+00, 3.57440000E+00, 3.38750000E+00,
     3.58850000E+00, 4.08690000E+00, 3.80480000E+00, 3.59290000E+00, 3.47160000E+00, 3.38300000E+00,
     4.33400000E+00, 4.18940000E+00, 3.78710000E+00, 3.75140000E+00, 3.72570000E+00, 3.70370000E+00,
     3.68310000E+00, 3.66240000E+00, 3.64790000E+00, 3.40120000E+00, 3.42530000E+00, 3.81380000E+00,
     3.60780000E+00, 3.59710000E+00, 3.58980000E+00, 3.58040000E+00, 3.47410000E+00, 3.96180000E+00,
     3.96440000E+00, 3.88230000E+00, 3.80700000E+00, 3.73780000E+00, 3.77010000E+00, 3.69260000E+00,
     3.60900000E+00, 3.39940000E+00, 3.62870000E+00, 4.14320000E+00, 3.85270000E+00, 3.64250000E+00,
     3.52670000E+00, 3.43280000E+00, 4.36270000E+00, 3.95000000E+00, 3.83940000E+00, 3.67020000E+00,
     3.66690000E+00, 3.65460000E+00, 3.63240000E+00, 3.70090000E+00},
    {2.51290000E+00, 2.73210000E+00, 3.10520000E+00, 3.29620000E+00, 3.13310000E+00, 3.20000000E+00,
     2.95860000E+00, 3.08220000E+00, 2.85820000E+00, 2.71200000E+00, 3.25700000E+00, 3.48390000E+00,
     3.39700000E+00, 3.32660000E+00, 3.29840000E+00, 3.56680000E+00, 3.32640000E+00, 3.16060000E+00,
     3.61920000E+00, 3.79930000E+00, 3.67590000E+00, 3.61420000E+00, 3.59790000E+00, 3.02770000E+00,
     3.51100000E+00, 3.45660000E+00, 3.45190000E+00, 3.41140000E+00, 3.39220000E+00, 3.35910000E+00,
     3.35100000E+00, 3.33360000E+00, 3.35140000E+00, 3.31820000E+00, 3.46700000E+00, 3.31430000E+00,
     3.79110000E+00, 3.96040000E+00, 3.79380000E+00, 3.72390000E+00, 3.22950000E+00, 3.20760000E+00,
     3.12140000E+00, 3.07890000E+00, 3.05680000E+00, 3.05840000E+00, 3.02750000E+00, 3.49100000E+00,
     3.52190000E+00, 3.50880000E+00, 3.53620000E+00, 3.51830000E+00, 3.68830000E+00, 3.53580000E+00,
     3.99000000E+00, 4.23210000E+00, 3.99240000E+00, 4.03970000E+00, 4.01700000E+00, 3.99580000E+00,
     3.97840000E+00, 3.96080000E+00, 3.94710000E+00, 3.71450000E+00, 3.81570000E+00, 3.79350000E+00,
     3.87570000E+00, 3.85860000E+00, 3.84680000E+00, 3.83590000E+00, 3.66750000E+00, 3.84300000E+00,
     3.80500000E+00, 3.79280000E+00, 3.73920000E+00, 3.67420000E+00, 3.24460000E+00, 3.18140000E+00,
     3.05810000E+00, 3.44750000E+00, 3.52830000E+00, 3.53200000E+00, 3.56020000E+00, 3.55650000E+00,
     3.74790000E+00, 3.58830000E+00, 4.02490000E+00, 4.17300000E+00, 4.03350000E+00, 3.84480000E+00,
     3.87240000E+00, 3.85790000E+00, 3.83750000E+00, 3.94510000E+00},
    {2.87660000E+00, 2.74270000E+00, 3.27760000E+00, 3.23630000E+00, 3.59290000E+00, 3.28260000E+00,
     3.09110000E+00, 2.93690000E+00, 2.90300000E+00, 2.77890000E+00, 3.39210000E+00, 3.39700000E+00,
     4.01060000E+00, 3.71050000E+00, 3.51420000E+00, 3.36840000E+00, 3.35620000E+00, 3.20850000E+00,
     3.65690000E+00, 3.67730000E+00, 3.65860000E+00, 3.59790000E+00, 3.52560000E+00, 3.29900000E+00,
     3.44290000E+00, 3.38960000E+00, 3.35930000E+00, 3.34920000E+00, 3.33160000E+00, 3.30450000E+00,
     3.98370000E+00, 3.73590000E+00, 3.58370000E+00, 3.47070000E+00, 3.40180000E+00, 3.35490000E+00,
     3.82250000E+00, 3.85000000E+00, 3.86970000E+00, 3.80700000E+00, 3.56430000E+00, 3.49430000E+00,
     3.40060000E+00, 3.36170000E+00, 3.34330000E+00, 3.33040000E+00, 3.32490000E+00, 3.45200000E+00,
     4.13140000E+00, 3.91050000E+00, 3.77460000E+00, 3.67290000E+00, 3.61050000E+00, 3.56720000E+00,
     4.65880000E+00, 4.17250000E+00, 4.06370000E+00, 3.92310000E+00, 3.90030000E+00, 3.87920000E+00,
     3.85790000E+00, 3.83970000E+00, 3.82620000E+00, 3.78300000E+00, 3.72580000E+00, 3.70810000E+00,
     3.76160000E+00, 3.74610000E+00, 3.73580000E+00, 3.72510000E+00, 3.78320000E+00, 3.99800000E+00,
     3.89810000E+00, 3.82640000E+00, 3.76680000E+00, 3.69280000E+00, 3.52390000E+00, 3.45280000E+00,
     3.37470000E+00, 3.43540000E+00, 4.15110000E+00, 3.94780000E+00, 3.81310000E+00, 3.72090000E+00,
     3.66500000E+00, 3.62270000E+00, 4.69470000E+00, 4.06130000E+00, 4.23730000E+00, 4.03400000E+00,
     3.99400000E+00, 3.96370000E+00, 3.93970000E+00, 3.84650000E+00},
    {2.88840000E+00, 2.66050000E+00, 3.75130000E+00, 3.16130000E+00, 3.36050000E+00, 3.33250000E+00,
     3.09910000E+00, 2.92970000E+00, 2.86740000E+00, 2.75710000E+00, 3.81290000E+00, 3.32660000E+00,
     3.71050000E+00, 3.79170000E+00, 3.54180000E+00, 3.36860000E+00, 3.25290000E+00, 3.16920000E+00,
     3.95860000E+00, 3.86780000E+00, 3.83270000E+00, 3.52950000E+00, 3.49800000E+00, 3.77210000E+00,
     3.38450000E+00, 3.32570000E+00, 3.29210000E+00, 3.26910000E+00, 3.25850000E+00, 3.23490000E+00,
     3.68710000E+00, 3.82660000E+00, 3.61770000E+00, 3.47480000E+00, 3.38050000E+00, 3.31200000E+00,
     4.03720000E+00, 3.98440000E+00, 3.97810000E+00, 3.71650000E+00, 3.78080000E+00, 3.59560000E+00,
     3.47930000E+00, 3.42460000E+00, 3.39490000E+00, 3.37480000E+00, 3.36270000E+00, 3.36080000E+00,
     3.81200000E+00, 3.98330000E+00, 3.79110000E+00, 3.65960000E+00, 3.57040000E+00, 3.50490000E+00,
     4.12780000E+00, 4.18330000E+00, 4.13820000E+00, 4.05610000E+00, 4.03720000E+00, 4.01940000E+00,
     4.00490000E+00, 3.98930000E+00, 3.97730000E+00, 3.66650000E+00, 3.61320000E+00, 3.91550000E+00,
     3.92220000E+00, 3.91000000E+00, 3.90140000E+00, 3.89400000E+00, 3.67870000E+00, 3.81340000E+00,
     3.77620000E+00, 3.70060000E+00, 3.62630000E+00, 3.55120000E+00, 3.46960000E+00, 3.41860000E+00,
     3.36770000E+00, 3.34380000E+00, 3.82590000E+00, 4.02310000E+00, 3.83490000E+00, 3.71080000E+00,
     3.62630000E+00, 3.56160000E+00, 4.17180000E+00, 4.14930000E+00, 4.30230000E+00, 3.95090000E+00,
     4.08160000E+00, 4.05810000E+00, 3.83110000E+00, 3.98730000E+00},
    {2.83040000E+00, 2.55380000E+00, 3.39320000E+00, 3.11930000E+00, 3.18660000E+00, 3.12450000E+00,
     3.04650000E+00, 2.87270000E+00, 2.76640000E+00, 2.69260000E+00, 3.46080000E+00, 3.29840000E+00,
     3.51420000E+00, 3.54180000E+00, 3.50170000E+00, 3.31800000E+00, 3.19160000E+00, 3.09820000E+00,
     3.61880000E+00, 3.58080000E+00, 3.53720000E+00, 3.48340000E+00, 3.43760000E+00, 3.50170000E+00,
     3.35740000E+00, 3.29460000E+00, 3.25770000E+00, 3.23200000E+00, 3.21360000E+00, 3.19420000E+00,
     3.48620000E+00, 3.57330000E+00, 3.58160000E+00, 3.42790000E+00, 3.32180000E+00, 3.24210000E+00,
     3.70590000E+00, 3.70350000E+00, 3.68770000E+00, 3.63680000E+00, 3.69310000E+00, 3.63790000E+00,
     3.51470000E+00, 3.45050000E+00, 3.41320000E+00, 3.38710000E+00, 3.36860000E+00, 3.30360000E+00,
     3.60150000E+00, 3.71760000E+00, 3.75110000E+00, 3.60820000E+00, 3.50590000E+00, 3.42840000E+00,
     3.81250000E+00, 3.89290000E+00, 3.85910000E+00, 3.78600000E+00, 3.76530000E+00, 3.74600000E+00,
     3.73040000E+00, 3.71450000E+00, 3.70150000E+00, 3.58980000E+00, 3.52970000E+00, 3.59100000E+00,
     3.63700000E+00, 3.62280000E+00, 3.61290000E+00, 3.60470000E+00, 3.61560000E+00, 3.66520000E+00,
     3.62160000E+00, 3.57970000E+00, 3.54020000E+00, 3.46140000E+00, 3.42610000E+00, 3.38360000E+00,
     3.34500000E+00, 3.28070000E+00, 3.60660000E+00, 3.75090000E+00, 3.79950000E+00, 3.66390000E+00,
     3.56580000E+00, 3.48940000E+00, 3.86390000E+00, 3.88230000E+00, 4.03060000E+00, 3.85850000E+00,
     3.80540000E+00, 3.77960000E+00, 3.75810000E+00, 3.70790000E+00},
    {2.61900000E+00, 2.47970000E+00, 3.13310000E+00, 3.05400000E+00, 3.06510000E+00, 2.98790000E+00,
     2.90540000E+00, 2.88050000E+00, 2.73300000E+00, 2.63310000E+00, 3.20960000E+00, 3.56680000E+00,
     3.36840000E+00, 3.36860000E+00, 3.31800000E+00, 3.31070000E+00, 3.15230000E+00, 3.03520000E+00,
     3.39170000E+00, 3.82430000E+00, 3.76650000E+00, 3.71400000E+00, 3.67140000E+00, 3.27510000E+00,
     3.60450000E+00, 3.56930000E+00, 3.21610000E+00, 3.17860000E+00, 3.15160000E+00, 3.12910000E+00,
     3.33890000E+00, 3.39590000E+00, 3.39020000E+00, 3.41820000E+00, 3.28150000E+00, 3.17870000E+00,
     3.49350000E+00, 3.91610000E+00, 3.87360000E+00, 3.81620000E+00, 3.47230000E+00, 3.41900000E+00,
     3.38060000E+00, 3.34430000E+00, 3.31160000E+00, 3.20280000E+00, 3.16690000E+00, 3.23450000E+00,
     3.45020000E+00, 3.53230000E+00, 3.54950000E+00, 3.59270000E+00, 3.46190000E+00, 3.36210000E+00,
     3.61890000E+00, 4.05440000E+00, 4.01640000E+00, 3.98450000E+00, 3.96720000E+00, 3.95170000E+00,
     3.93920000E+00, 3.92660000E+00, 3.91620000E+00, 3.80770000E+00, 3.75660000E+00, 3.48080000E+00,
     3.86470000E+00, 3.85350000E+00, 3.84430000E+00, 3.83750000E+00, 3.80410000E+00, 3.79850000E+00,
     3.75310000E+00, 3.71410000E+00, 3.68070000E+00, 3.40870000E+00, 3.35080000E+00, 3.22130000E+00,
     3.16510000E+00, 3.21690000E+00, 3.45270000E+00, 3.56040000E+00, 3.59190000E+00, 3.64840000E+00,
     3.52240000E+00, 3.42410000E+00, 3.67350000E+00, 4.05370000E+00, 4.14080000E+00, 3.98940000E+00,
     3.96610000E+00, 3.94630000E+00, 3.92740000E+00, 3.90830000E+00},
    {2.47570000E+00, 2.40190000E+00, 2.97890000E+00, 3.14680000E+00, 2.97680000E+00, 2.88480000E+00,
     2.79520000E+00, 2.74570000E+00, 2.68810000E+00, 2.57280000E+00, 3.05740000E+00, 3.32640000E+00,
     3.35620000E+00, 3.25290000E+00, 3.19160000E+00, 3.15230000E+00, 3.10460000E+00, 2.97300000E+00,
     3.24790000E+00, 3.58260000E+00, 3.53100000E+00, 3.47820000E+00, 3.43460000E+00, 3.13680000E+00,
     3.36580000E+00, 3.33120000E+00, 3.29820000E+00, 3.26800000E+00, 3.23640000E+00, 3.21110000E+00,
     3.24130000E+00, 3.27750000E+00, 3.26040000E+00, 3.25470000E+00, 3.23460000E+00, 3.11760000E+00,
     3.35290000E+00, 3.67510000E+00, 3.64510000E+00, 3.58850000E+00, 3.33570000E+00, 3.28080000E+00,
     3.23560000E+00, 3.19640000E+00, 3.16160000E+00, 3.12960000E+00, 3.05840000E+00, 3.29990000E+00,
     3.34980000E+00, 3.41020000E+00, 3.41490000E+00, 3.42240000E+00, 3.41380000E+00, 3.30010000E+00,
     3.48510000E+00, 3.81180000E+00, 3.78780000E+00, 3.74540000E+00, 3.72830000E+00, 3.71280000E+00,
     3.70020000E+00, 3.68770000E+00, 3.67710000E+00, 3.58100000E+00, 3.50950000E+00, 3.48860000E+00,
     3.62560000E+00, 3.61470000E+00, 3.60540000E+00, 3.59900000E+00, 3.58130000E+00, 3.57560000E+00,
     3.52970000E+00, 3.48940000E+00, 3.45450000E+00, 3.42010000E+00, 3.19680000E+00, 3.16260000E+00,
     3.12590000E+00, 3.26770000E+00, 3.34800000E+00, 3.43400000E+00, 3.45390000E+00, 3.47450000E+00,
     3.47620000E+00, 3.36410000E+00, 3.54350000E+00, 3.82000000E+00, 3.92970000E+00, 3.77870000E+00,
     3.74920000E+00, 3.72750000E+00, 3.70850000E+00, 3.67560000E+00},
    {2.37250000E+00, 2.32890000E+00, 2.87600000E+00, 2.98040000E+00, 2.90930000E+00, 2.80400000E+00,
     2.70710000E+00, 2.63860000E+00, 2.57200000E+00, 2.51390000E+00, 2.95170000E+00, 3.16060000E+00,
     3.20850000E+00, 3.16920000E+00, 3.09820000E+00, 3.03520000E+00, 2.97300000E+00, 2.91480000E+00,
     3.14340000E+00, 3.41560000E+00, 3.37000000E+00, 3.31700000E+00, 3.27230000E+00, 3.04350000E+00,
     3.20130000E+00, 3.16700000E+00, 3.13390000E+00, 3.10360000E+00, 3.07520000E+00, 3.05340000E+00,
     3.17080000E+00, 3.19150000E+00, 3.16520000E+00, 3.13530000E+00, 3.09940000E+00, 3.06130000E+00,
     3.24920000E+00, 3.50750000E+00, 3.48900000E+00, 3.43360000E+00, 3.24290000E+00, 3.18600000E+00,
     3.13350000E+00, 3.09130000E+00, 3.05480000E+00, 3.02140000E+00, 2.99150000E+00, 3.14870000E+00,
     3.27770000E+00, 3.32270000E+00, 3.31770000E+00, 3.29970000E+00, 3.27420000E+00, 3.24510000E+00,
     3.38590000E+00, 3.64140000E+00, 3.63160000E+00, 3.58020000E+00, 3.56300000E+00, 3.54740000E+00,
     3.53470000E+00, 3.52220000E+00, 3.51150000E+00, 3.42650000E+00, 3.33680000E+00, 3.33970000E+00,
     3.45950000E+00, 3.44900000E+00, 3.43960000E+00, 3.43290000E+00, 3.43010000E+00, 3.42070000E+00,
     3.37420000E+00, 3.33260000E+00, 3.29620000E+00, 3.26070000E+00, 3.08480000E+00, 3.05070000E+00,
     3.01470000E+00, 3.12960000E+00, 3.27130000E+00, 3.34260000E+00, 3.35400000E+00, 3.34920000E+00,
     3.33420000E+00, 3.31200000E+00, 3.44790000E+00, 3.65820000E+00, 3.78570000E+00, 3.63650000E+00,
     3.60240000E+00, 3.57920000E+00, 3.55980000E+00, 3.51500000E+00},
    {3.21470000E+00, 2.83150000E+00, 3.87240000E+00, 3.46210000E+00, 3.88230000E+00, 3.37600000E+00,
     3.07460000E+00, 2.88170000E+00, 2.75520000E+00, 2.66050000E+00, 3.97400000E+00, 3.61920000E+00,
     3.65690000E+00, 3.95860000E+00, 3.61880000E+00, 3.39170000E+00, 3.24790000E+00, 3.14340000E+00,
     4.24110000E+00, 3.87650000E+00, 3.77880000E+00, 3.74340000E+00, 3.68590000E+00, 3.78730000E+00,
     3.61100000E+00, 3.58050000E+00, 3.56060000E+00, 3.54530000E+00, 3.53680000E+00, 3.51890000E+00,
     3.60960000E+00, 3.98780000E+00, 3.70370000E+00, 3.51160000E+00, 3.39370000E+00, 3.30820000E+00,
     4.43520000E+00, 4.11510000E+00, 3.98580000E+00, 3.98290000E+00, 4.02800000E+00, 3.98500000E+00,
     3.91440000E+00, 3.89210000E+00, 3.88590000E+00, 3.87960000E+00, 3.87730000E+00, 3.74090000E+00,
     3.86350000E+00, 4.27020000E+00, 4.01290000E+00, 3.81900000E+00, 3.70800000E+00, 3.62090000E+00,
     4.65310000E+00, 4.63730000E+00, 4.17410000E+00, 4.13660000E+00, 4.10920000E+00, 4.08720000E+00,
     4.06570000E+00, 4.04480000E+00, 4.03060000E+00, 3.77260000E+00, 3.78900000E+00, 4.13360000E+00,
     3.98740000E+00, 3.97640000E+00, 3.96830000E+00, 3.95970000E+00, 3.84800000E+00, 4.40610000E+00,
     4.38140000E+00, 4.30480000E+00, 4.22830000E+00, 4.15270000E+00, 4.14960000E+00, 4.05480000E+00,
     3.94980000E+00, 3.74930000E+00, 3.90370000E+00, 4.33280000E+00, 4.06540000E+00, 3.87550000E+00,
     3.77380000E+00, 3.68150000E+00, 4.68060000E+00, 4.34220000E+00, 4.19070000E+00, 4.14250000E+00,
     4.04280000E+00, 4.02950000E+00, 4.00800000E+00, 4.08290000E+00},
    {2.75970000E+00, 3.05880000E+00, 3.34740000E+00, 3.62140000E+00, 3.43530000E+00, 3.47290000E+00,
     3.24870000E+00, 3.32000000E+00, 3.09140000E+00, 2.94030000E+00, 3.49720000E+00, 3.79930000E+00,
     3.67730000E+00, 3.86780000E+00, 3.58080000E+00, 3.82430000E+00, 3.58260000E+00, 3.41560000E+00,
     3.87650000E+00, 4.10350000E+00, 3.98040000E+00, 3.96230000E+00, 3.89850000E+00, 3.28580000E+00,
     3.82410000E+00, 3.77110000E+00, 3.75820000E+00, 3.72590000E+00, 3.71170000E+00, 3.68090000E+00,
     3.62800000E+00, 3.88160000E+00, 3.62830000E+00, 3.94320000E+00, 3.72660000E+00, 3.57550000E+00,
     4.08260000E+00, 4.28770000E+00, 4.11790000E+00, 4.05290000E+00, 3.55890000E+00, 3.51050000E+00,
     3.41830000E+00, 3.37130000E+00, 3.37190000E+00, 3.33370000E+00, 3.30990000E+00, 3.83920000E+00,
     3.82320000E+00, 4.08880000E+00, 3.83700000E+00, 4.18650000E+00, 3.97730000E+00, 3.82990000E+00,
     4.31340000E+00, 4.62680000E+00, 4.31660000E+00, 4.35810000E+00, 4.33470000E+00, 4.31380000E+00,
     4.29550000E+00, 4.27710000E+00, 4.26340000E+00, 4.03070000E+00, 4.12980000E+00, 4.11220000E+00,
     4.19380000E+00, 4.17730000E+00, 4.16560000E+00, 4.15420000E+00, 3.98490000E+00, 4.28170000E+00,
     4.28180000E+00, 4.22170000E+00, 4.16980000E+00, 4.09770000E+00, 3.65980000E+00, 3.53120000E+00,
     3.38570000E+00, 3.80750000E+00, 3.83610000E+00, 3.82880000E+00, 3.86030000E+00, 4.24570000E+00,
     4.03330000E+00, 3.87890000E+00, 4.34850000E+00, 4.51110000E+00, 4.32300000E+00, 4.16180000E+00,
     4.19510000E+00, 4.18240000E+00, 4.16410000E+00, 4.27800000E+00},
    {2.73610000E+00, 2.97650000E+00, 3.24750000E+00, 3.50040000E+00, 3.41850000E+00, 3.43780000E+00,
     3.20840000E+00, 3.27870000E+00, 3.06040000E+00, 2.91870000E+00, 3.40370000E+00, 3.67590000E+00,
     3.65860000E+00, 3.83270000E+00, 3.53720000E+00, 3.76650000E+00, 3.53100000E+00, 3.37000000E+00,
     3.77880000E+00, 3.98040000E+00, 3.89030000E+00, 3.81810000E+00, 3.79180000E+00, 3.21400000E+00,
     3.70900000E+00, 3.65360000E+00, 3.64320000E+00, 3.60900000E+00, 3.59410000E+00, 3.56350000E+00,
     3.68600000E+00, 3.58100000E+00, 3.58580000E+00, 3.88280000E+00, 3.66970000E+00, 3.52220000E+00,
     3.97330000E+00, 4.15790000E+00, 4.04300000E+00, 3.95840000E+00, 3.46360000E+00, 3.43300000E+00,
     3.33690000E+00, 3.28730000E+00, 3.29170000E+00, 3.24920000E+00, 3.22310000E+00, 3.71480000E+00,
     3.84860000E+00, 3.75600000E+00, 3.77390000E+00, 4.11140000E+00, 3.90100000E+00, 3.75430000E+00,
     4.22580000E+00, 4.47500000E+00, 4.23950000E+00, 4.23510000E+00, 4.21170000E+00, 4.19060000E+00,
     4.17050000E+00, 4.15230000E+00, 4.13850000E+00, 3.97630000E+00, 4.01900000E+00, 3.98880000E+00,
     4.06790000E+00, 4.05110000E+00, 4.03940000E+00, 4.02780000E+00, 3.93140000E+00, 4.14770000E+00,
     4.10260000E+00, 4.07860000E+00, 4.02400000E+00, 3.95230000E+00, 3.51110000E+00, 3.42440000E+00,
     3.29170000E+00, 3.68460000E+00, 3.85790000E+00, 3.78220000E+00, 3.79720000E+00, 3.77580000E+00,
     3.95680000E+00, 3.80310000E+00, 4.26680000E+00, 4.37950000E+00, 4.26640000E+00, 4.09400000E+00,
     4.14660000E+00, 4.12470000E+00, 4.10570000E+00, 4.15110000E+00},
    {2.68320000E+00, 2.90600000E+00, 3.26130000E+00, 3.43590000E+00, 3.35380000E+00, 3.38600000E+00,
     3.15500000E+00, 3.23000000E+00, 3.01330000E+00, 2.87360000E+00, 3.40240000E+00, 3.61420000E+00,
     3.59790000E+00, 3.52950000E+00, 3.48340000E+00, 3.71400000E+00, 3.47820000E+00, 3.31700000E+00,
     3.74340000E+00, 3.96230000E+00, 3.81810000E+00, 3.76420000E+00, 3.73720000E+00, 3.17270000E+00,
     3.64960000E+00, 3.59270000E+00, 3.58330000E+00, 3.54730000E+00, 3.53130000E+00, 3.50010000E+00,
     3.55680000E+00, 3.53640000E+00, 3.53300000E+00, 3.83030000E+00, 3.61640000E+00, 3.46780000E+00,
     3.92540000E+00, 4.12470000E+00, 3.95630000E+00, 3.90250000E+00, 3.49940000E+00, 3.37970000E+00,
     3.28030000E+00, 3.22810000E+00, 3.23450000E+00, 3.18830000E+00, 3.16000000E+00, 3.64390000E+00,
     3.72150000E+00, 3.76870000E+00, 3.71250000E+00, 4.05400000E+00, 3.84090000E+00, 3.69200000E+00,
     4.13090000E+00, 4.41340000E+00, 4.18310000E+00, 4.20110000E+00, 4.17930000E+00, 4.15930000E+00,
     4.14240000E+00, 4.12470000E+00, 4.11160000E+00, 3.88900000E+00, 3.95730000E+00, 3.95430000E+00,
     4.04300000E+00, 4.02700000E+00, 4.01580000E+00, 4.00480000E+00, 3.84050000E+00, 4.06160000E+00,
     4.02940000E+00, 3.99000000E+00, 3.93410000E+00, 3.86280000E+00, 3.41990000E+00, 3.34090000E+00,
     3.21540000E+00, 3.61040000E+00, 3.73110000E+00, 3.79090000E+00, 3.73580000E+00, 3.71610000E+00,
     3.89750000E+00, 3.74130000E+00, 4.16900000E+00, 4.33620000E+00, 4.21730000E+00, 4.04660000E+00,
     4.05150000E+00, 4.03570000E+00, 4.01580000E+00, 4.12600000E+00},
    {2.63790000E+00, 2.84940000E+00, 3.18400000E+00, 3.42250000E+00, 3.27710000E+00, 3.34010000E+00,
     3.10720000E+00, 3.18850000E+00, 2.97140000E+00, 2.83190000E+00, 3.33150000E+00, 3.59790000E+00,
     3.52560000E+00, 3.49800000E+00, 3.43760000E+00, 3.67140000E+00, 3.43460000E+00, 3.27230000E+00,
     3.68590000E+00, 3.89850000E+00, 3.79180000E+00, 3.73720000E+00, 3.72110000E+00, 3.21780000E+00,
     3.63330000E+00, 3.57750000E+00, 3.56910000E+00, 3.53270000E+00, 3.51640000E+00, 3.48540000E+00,
     3.48360000E+00, 3.50600000E+00, 3.48840000E+00, 3.78800000E+00, 3.57300000E+00, 3.42310000E+00,
     3.86460000E+00, 4.06170000E+00, 3.91820000E+00, 3.86070000E+00, 3.43090000E+00, 3.41550000E+00,
     3.26790000E+00, 3.21650000E+00, 3.22740000E+00, 3.18020000E+00, 3.15200000E+00, 3.61820000E+00,
     3.64870000E+00, 3.66810000E+00, 3.71520000E+00, 3.63250000E+00, 3.79440000E+00, 3.64360000E+00,
     4.06920000E+00, 4.34580000E+00, 4.11070000E+00, 4.14020000E+00, 4.11790000E+00, 4.09730000E+00,
     4.08000000E+00, 4.06260000E+00, 4.04890000E+00, 3.84890000E+00, 3.92370000E+00, 3.89170000E+00,
     3.97810000E+00, 3.96140000E+00, 3.94980000E+00, 3.93900000E+00, 3.80290000E+00, 3.99790000E+00,
     3.96400000E+00, 3.93570000E+00, 3.87110000E+00, 3.80020000E+00, 3.38090000E+00, 3.28100000E+00,
     3.16040000E+00, 3.55770000E+00, 3.65750000E+00, 3.69070000E+00, 3.73920000E+00, 3.66930000E+00,
     3.85210000E+00, 3.69390000E+00, 4.10610000E+00, 4.27510000E+00, 4.14820000E+00, 3.99410000E+00,
     4.00750000E+00, 3.99260000E+00, 3.97260000E+00, 4.05710000E+00},
    {2.92300000E+00, 2.62230000E+00, 3.41610000E+00, 2.89990000E+00, 3.05570000E+00, 3.33080000E+00,
     3.05550000E+00, 2.85080000E+00, 2.73850000E+00, 2.66400000E+00, 3.52630000E+00, 3.02770000E+00,
     3.29900000E+00, 3.77210000E+00, 3.50170000E+00, 3.27510000E+00, 3.13680000E+00, 3.04350000E+00,
     3.78730000E+00, 3.28580000E+00, 3.21400000E+00, 3.17270000E+00, 3.21780000E+00, 3.44140000E+00,
     3.08960000E+00, 3.04110000E+00, 3.01610000E+00, 3.00180000E+00, 2.99620000E+00, 2.98570000E+00,
     3.28680000E+00, 3.80970000E+00, 3.57890000E+00, 3.37600000E+00, 3.25220000E+00, 3.16840000E+00,
     3.93150000E+00, 3.48740000E+00, 3.40020000E+00, 3.36730000E+00, 3.61770000E+00, 3.60330000E+00,
     3.48710000E+00, 3.43860000E+00, 3.41710000E+00, 3.40500000E+00, 3.40230000E+00, 3.17530000E+00,
     3.47240000E+00, 3.64050000E+00, 3.77010000E+00, 3.56970000E+00, 3.44650000E+00, 3.35980000E+00,
     4.09440000E+00, 3.85820000E+00, 3.58570000E+00, 3.53810000E+00, 3.51390000E+00, 3.49190000E+00,
     3.47170000E+00, 3.45300000E+00, 3.43660000E+00, 3.27060000E+00, 3.26770000E+00, 3.58940000E+00,
     3.38860000E+00, 3.37540000E+00, 3.36770000E+00, 3.35710000E+00, 3.29620000E+00, 3.64920000E+00,
     3.62080000E+00, 3.63310000E+00, 3.54890000E+00, 3.47590000E+00, 3.53820000E+00, 3.47820000E+00,
     3.41740000E+00, 3.20520000E+00, 3.51760000E+00, 3.68640000E+00, 3.81570000E+00, 3.61550000E+00,
     3.49290000E+00, 3.40100000E+00, 4.12450000E+00, 3.71030000E+00, 3.68230000E+00, 3.54260000E+00,
     3.50200000E+00, 3.48270000E+00, 3.46670000E+00, 3.48650000E+00},
    {2.54900000E+00, 2.76230000E+00, 3.09910000E+00, 3.32520000E+00, 3.18360000E+00, 3.24280000E+00,
     3.02590000E+00, 3.12250000E+00, 2.90320000E+00, 2.76210000E+00, 3.24900000E+00, 3.51100000E+00,
     3.44290000E+00, 3.38450000E+00, 3.35740000E+00, 3.60450000E+00, 3.36580000E+00, 3.20130000E+00,
     3.61100000E+00, 3.82410000E+00, 3.70900000E+00, 3.64960000E+00, 3.63330000E+00, 3.08960000E+00,
     3.54620000E+00, 3.48850000E+00, 3.48120000E+00, 3.44130000E+00, 3.42250000E+00, 3.38970000E+00,
     3.39940000E+00, 3.39250000E+00, 3.40940000E+00, 3.72180000E+00, 3.50510000E+00, 3.35280000E+00,
     3.78370000E+00, 3.98480000E+00, 3.83100000E+00, 3.76580000E+00, 3.29460000E+00, 3.27370000E+00,
     3.17140000E+00, 3.11640000E+00, 3.12930000E+00, 3.07560000E+00, 3.04260000E+00, 3.52100000E+00,
     3.56270000E+00, 3.55690000E+00, 3.58130000E+00, 3.55610000E+00, 3.72350000E+00, 3.57010000E+00,
     3.98500000E+00, 4.25830000E+00, 4.02700000E+00, 4.06530000E+00, 4.04260000E+00, 4.02160000E+00,
     4.00430000E+00, 3.98660000E+00, 3.97320000E+00, 3.75950000E+00, 3.84800000E+00, 3.81310000E+00,
     3.90080000E+00, 3.88360000E+00, 3.87180000E+00, 3.86080000E+00, 3.71040000E+00, 3.88330000E+00,
     3.84640000E+00, 3.83330000E+00, 3.77980000E+00, 3.71020000E+00, 3.25720000E+00, 3.19050000E+00,
     3.07350000E+00, 3.48030000E+00, 3.56930000E+00, 3.57930000E+00, 3.60550000E+00, 3.59410000E+00,
     3.78300000E+00, 3.62250000E+00, 4.02060000E+00, 4.19730000E+00, 4.07110000E+00, 3.89520000E+00,
     3.91580000E+00, 3.90070000E+00, 3.88020000E+00, 3.97440000E+00},
    {2.49260000E+00, 2.71360000E+00, 3.06930000E+00, 3.26990000E+00, 3.12720000E+00, 3.18930000E+00,
     2.96580000E+00, 3.09720000E+00, 2.87780000E+00, 2.73580000E+00, 3.22060000E+00, 3.45660000E+00,
     3.38960000E+00, 3.32570000E+00, 3.29460000E+00, 3.56930000E+00, 3.33120000E+00, 3.16700000E+00,
     3.58050000E+00, 3.77110000E+00, 3.65360000E+00, 3.59270000E+00, 3.57750000E+00, 3.04110000E+00,
     3.48850000E+00, 3.44210000E+00, 3.43390000E+00, 3.39070000E+00, 3.36990000E+00, 3.33630000E+00,
     3.34760000E+00, 3.33480000E+00, 3.34730000E+00, 3.34080000E+00, 3.46860000E+00, 3.31620000E+00,
     3.74650000E+00, 3.92800000E+00, 3.77160000E+00, 3.70350000E+00, 3.23760000E+00, 3.21450000E+00,
     3.15210000E+00, 3.12200000E+00, 3.05670000E+00, 3.04780000E+00, 3.00990000E+00, 3.46390000E+00,
     3.50870000E+00, 3.49900000E+00, 3.51870000E+00, 3.52590000E+00, 3.68080000E+00, 3.52660000E+00,
     3.94160000E+00, 4.18980000E+00, 3.96760000E+00, 4.00930000E+00, 3.98670000E+00, 3.96570000E+00,
     3.94850000E+00, 3.93100000E+00, 3.91760000E+00, 3.69840000E+00, 3.81570000E+00, 3.76350000E+00,
     3.84630000E+00, 3.82910000E+00, 3.81640000E+00, 3.80560000E+00, 3.65150000E+00, 3.80270000E+00,
     3.76480000E+00, 3.73170000E+00, 3.70000000E+00, 3.64660000E+00, 3.21000000E+00, 3.14940000E+00,
     3.03420000E+00, 3.42360000E+00, 3.51570000E+00, 3.52210000E+00, 3.54380000E+00, 3.56430000E+00,
     3.74090000E+00, 3.57970000E+00, 3.97650000E+00, 4.13850000E+00, 4.01800000E+00, 3.83270000E+00,
     3.85460000E+00, 3.83920000E+00, 3.81880000E+00, 3.91500000E+00},
    {2.46670000E+00, 2.67090000E+00, 3.05750000E+00, 3.23570000E+00, 3.09080000E+00, 3.15370000E+00,
     2.92350000E+00, 3.06690000E+00, 2.84760000E+00, 2.70540000E+00, 3.20640000E+00, 3.45190000E+00,
     3.35930000E+00, 3.29210000E+00, 3.25770000E+00, 3.21610000E+00, 3.29820000E+00, 3.13390000E+00,
     3.56060000E+00, 3.75820000E+00, 3.64320000E+00, 3.58330000E+00, 3.56910000E+00, 3.01610000E+00,
     3.48120000E+00, 3.43390000E+00, 3.43270000E+00, 3.35930000E+00, 3.33700000E+00, 3.30270000E+00,
     3.31700000E+00, 3.30190000E+00, 3.31180000E+00, 3.30590000E+00, 3.43550000E+00, 3.28270000E+00,
     3.72110000E+00, 3.90790000E+00, 3.75430000E+00, 3.68660000E+00, 3.20500000E+00, 3.18070000E+00,
     3.11010000E+00, 3.07610000E+00, 3.05650000E+00, 3.03220000E+00, 2.99200000E+00, 3.42650000E+00,
     3.47570000E+00, 3.46590000E+00, 3.48350000E+00, 3.49010000E+00, 3.64530000E+00, 3.49040000E+00,
     3.91120000E+00, 4.15620000E+00, 3.94630000E+00, 3.98830000E+00, 3.96610000E+00, 3.94540000E+00,
     3.92860000E+00, 3.91150000E+00, 3.89830000E+00, 3.67720000E+00, 3.76560000E+00, 3.74190000E+00,
     3.82880000E+00, 3.81210000E+00, 3.80050000E+00, 3.78990000E+00, 3.63780000E+00, 3.76600000E+00,
     3.72810000E+00, 3.69570000E+00, 3.66540000E+00, 3.61060000E+00, 3.19170000E+00, 3.12210000E+00,
     3.00960000E+00, 3.38450000E+00, 3.48140000E+00, 3.48830000E+00, 3.50890000E+00, 3.52920000E+00,
     3.70620000E+00, 3.54430000E+00, 3.94580000E+00, 4.11290000E+00, 4.00170000E+00, 3.81260000E+00,
     3.83420000E+00, 3.81910000E+00, 3.79890000E+00, 3.89300000E+00},
    {2.45150000E+00, 2.63380000E+00, 3.05110000E+00, 3.22290000E+00, 3.06300000E+00, 3.12650000E+00,
     2.89090000E+00, 3.02530000E+00, 2.81840000E+00, 2.67640000E+00, 3.19680000E+00, 3.41140000E+00,
     3.34920000E+00, 3.26910000E+00, 3.23200000E+00, 3.17860000E+00, 3.26800000E+00, 3.10360000E+00,
     3.54530000E+00, 3.72590000E+00, 3.60900000E+00, 3.54730000E+00, 3.53270000E+00, 3.00180000E+00,
     3.44130000E+00, 3.39070000E+00, 3.35930000E+00, 3.34620000E+00, 3.32340000E+00, 3.28900000E+00,
     3.29500000E+00, 3.27960000E+00, 3.28760000E+00, 3.26980000E+00, 3.40840000E+00, 3.25270000E+00,
     3.70120000E+00, 3.87510000E+00, 3.72030000E+00, 3.65040000E+00, 3.18470000E+00, 3.15960000E+00,
     3.08430000E+00, 3.04800000E+00, 3.02740000E+00, 3.03230000E+00, 2.98090000E+00, 3.40750000E+00,
     3.45170000E+00, 3.44330000E+00, 3.45950000E+00, 3.45520000E+00, 3.61680000E+00, 3.45900000E+00,
     3.88730000E+00, 4.11910000E+00, 3.91500000E+00, 3.95700000E+00, 3.93450000E+00, 3.91340000E+00,
     3.89650000E+00, 3.87920000E+00, 3.86590000E+00, 3.64280000E+00, 3.73170000E+00, 3.70710000E+00,
     3.79500000E+00, 3.77800000E+00, 3.76620000E+00, 3.75600000E+00, 3.60200000E+00, 3.71830000E+00,
     3.67900000E+00, 3.64600000E+00, 3.61540000E+00, 3.55800000E+00, 3.15190000E+00, 3.11280000E+00,
     3.01360000E+00, 3.36400000E+00, 3.45590000E+00, 3.46490000E+00, 3.48530000E+00, 3.49500000E+00,
     3.67860000E+00, 3.51390000E+00, 3.92170000E+00, 4.08000000E+00, 3.97470000E+00, 3.77960000E+00,
     3.80080000E+00, 3.78510000E+00, 3.76440000E+00, 3.85780000E+00},
    {2.44130000E+00, 2.60060000E+00, 3.05400000E+00, 3.19870000E+00, 3.04900000E+00, 3.10580000E+00,
     2.86430000E+00, 2.99480000E+00, 2.79080000E+00, 2.64910000E+00, 3.19500000E+00, 3.39220000E+00,
     3.33160000E+00, 3.25850000E+00, 3.21360000E+00, 3.15160000E+00, 3.23640000E+00, 3.07520000E+00,
     3.53680000E+00, 3.71170000E+00, 3.59410000E+00, 3.53130000E+00, 3.51640000E+00, 2.99620000E+00,
     3.42250000E+00, 3.36990000E+00, 3.33700000E+00, 3.32340000E+00, 3.30080000E+00, 3.26550000E+00,
     3.28740000E+00, 3.26620000E+00, 3.27070000E+00, 3.24460000E+00, 3.37480000E+00, 3.23080000E+00,
     3.68930000E+00, 3.86040000E+00, 3.70530000E+00, 3.63390000E+00, 3.17150000E+00, 3.14610000E+00,
     3.06700000E+00, 3.02950000E+00, 3.00870000E+00, 3.01630000E+00, 2.98000000E+00, 3.38280000E+00,
     3.44230000E+00, 3.43300000E+00, 3.44390000E+00, 3.43150000E+00, 3.58440000E+00, 3.43640000E+00,
     3.87360000E+00, 4.10690000E+00, 3.90210000E+00, 3.94290000E+00, 3.92000000E+00, 3.89860000E+00,
     3.88150000E+00, 3.86410000E+00, 3.85070000E+00, 3.62430000E+00, 3.71260000E+00, 3.68800000E+00,
     3.77900000E+00, 3.76190000E+00, 3.75000000E+00, 3.74000000E+00, 3.58490000E+00, 3.69540000E+00,
     3.65420000E+00, 3.61970000E+00, 3.58820000E+00, 3.52820000E+00, 3.11980000E+00, 3.08530000E+00,
     2.98550000E+00, 3.33650000E+00, 3.44450000E+00, 3.45140000E+00, 3.46980000E+00, 3.47200000E+00,
     3.64710000E+00, 3.49230000E+00, 3.90750000E+00, 4.06470000E+00, 3.96340000E+00, 3.76350000E+00,
     3.78450000E+00, 3.76870000E+00, 3.74740000E+00, 3.84020000E+00},
    {2.43180000E+00, 2.57290000E+00, 3.04160000E+00, 3.16390000E+00, 3.01960000E+00, 3.08430000E+00,
     2.84130000E+00, 2.74360000E+00, 2.76080000E+00, 2.62710000E+00, 3.18110000E+00, 3.35910000E+00,
     3.30450000E+00, 3.23490000E+00, 3.19420000E+00, 3.12910000E+00, 3.21110000E+00, 3.05340000E+00,
     3.51890000E+00, 3.68090000E+00, 3.56350000E+00, 3.50010000E+00, 3.48540000E+00, 2.98570000E+00,
     3.38970000E+00, 3.33630000E+00, 3.30270000E+00, 3.28900000E+00, 3.26550000E+00, 3.23090000E+00,
     3.26060000E+00, 3.24640000E+00, 3.25210000E+00, 3.22290000E+00, 3.34960000E+00, 3.20290000E+00,
     3.66760000E+00, 3.82770000E+00, 3.67420000E+00, 3.60240000E+00, 3.15990000E+00, 3.13370000E+00,
     3.05390000E+00, 3.01550000E+00, 2.99390000E+00, 3.00190000E+00, 2.96460000E+00, 3.34740000E+00,
     3.41390000E+00, 3.40920000E+00, 3.42420000E+00, 3.40910000E+00, 3.55760000E+00, 3.40770000E+00,
     3.84730000E+00, 4.06390000E+00, 3.87080000E+00, 3.91120000E+00, 3.88830000E+00, 3.86690000E+00,
     3.85000000E+00, 3.83260000E+00, 3.81910000E+00, 3.59510000E+00, 3.68140000E+00, 3.65740000E+00,
     3.74720000E+00, 3.73000000E+00, 3.71810000E+00, 3.70810000E+00, 3.55500000E+00, 3.65250000E+00,
     3.61170000E+00, 3.57790000E+00, 3.54720000E+00, 3.48780000E+00, 3.10050000E+00, 3.03840000E+00,
     2.96800000E+00, 3.30100000E+00, 3.41600000E+00, 3.43010000E+00, 3.45080000E+00, 3.45030000E+00,
     3.62080000E+00, 3.46420000E+00, 3.88130000E+00, 4.03080000E+00, 3.93830000E+00, 3.73560000E+00,
     3.75490000E+00, 3.73870000E+00, 3.71730000E+00, 3.80730000E+00},
    {2.85020000E+00, 2.69340000E+00, 3.24670000E+00, 3.19210000E+00, 3.56630000E+00, 3.25410000E+00,
     3.05710000E+00, 2.90480000E+00, 2.86570000E+00, 2.74380000E+00, 3.35470000E+00, 3.35100000E+00,
     3.98370000E+00, 3.68710000E+00, 3.48620000E+00, 3.33890000E+00, 3.24130000E+00, 3.17080000E+00,
     3.60960000E+00, 3.62800000E+00, 3.68600000E+00, 3.55680000E+00, 3.48360000E+00, 3.28680000E+00,
     3.39940000E+00, 3.34760000E+00, 3.31700000E+00, 3.29500000E+00, 3.28740000E+00, 3.26060000E+00,
     3.95790000E+00, 3.71360000E+00, 3.55700000E+00, 3.44220000E+00, 3.36920000E+00, 3.31730000E+00,
     3.77360000E+00, 3.80020000E+00, 3.83180000E+00, 3.77080000E+00, 3.55550000E+00, 3.48120000E+00,
     3.38900000E+00, 3.34950000E+00, 3.32930000E+00, 3.31450000E+00, 3.30680000E+00, 3.40710000E+00,
     4.10280000E+00, 3.88670000E+00, 3.74760000E+00, 3.64380000E+00, 3.57720000E+00, 3.52870000E+00,
     4.60270000E+00, 4.12570000E+00, 4.02400000E+00, 3.87280000E+00, 3.84980000E+00, 3.82890000E+00,
     3.80730000E+00, 3.78920000E+00, 3.77570000E+00, 3.74970000E+00, 3.67930000E+00, 3.65460000E+00,
     3.71170000E+00, 3.69650000E+00, 3.68630000E+00, 3.67580000E+00, 3.74940000E+00, 3.96690000E+00,
     3.86500000E+00, 3.79090000E+00, 3.72890000E+00, 3.65470000E+00, 3.50710000E+00, 3.43660000E+00,
     3.36040000E+00, 3.39380000E+00, 4.12310000E+00, 3.92560000E+00, 3.78820000E+00, 3.69360000E+00,
     3.63370000E+00, 3.58600000E+00, 3.99470000E+00, 4.00960000E+00, 4.19930000E+00, 4.00470000E+00,
     3.96020000E+00, 3.92900000E+00, 3.90490000E+00, 3.79770000E+00},
    {2.92260000E+00, 2.68380000E+00, 3.78670000E+00, 3.17320000E+00, 3.38720000E+00, 3.36430000E+00,
     3.12670000E+00, 2.95410000E+00, 2.85050000E+00, 2.77810000E+00, 3.84750000E+00, 3.33360000E+00,
     3.73590000E+00, 3.82660000E+00, 3.57330000E+00, 3.39590000E+00, 3.27750000E+00, 3.19150000E+00,
     3.98780000E+00, 3.88160000E+00, 3.58100000E+00, 3.53640000E+00, 3.50600000E+00, 3.80970000E+00,
     3.39250000E+00, 3.33480000E+00, 3.30190000E+00, 3.27960000E+00, 3.26620000E+00, 3.24640000E+00,
     3.71360000E+00, 3.86190000E+00, 3.64960000E+00, 3.50230000E+00, 3.40520000E+00, 3.33430000E+00,
     4.06600000E+00, 3.99810000E+00, 3.76310000E+00, 3.72830000E+00, 3.81110000E+00, 3.62510000E+00,
     3.50860000E+00, 3.45430000E+00, 3.42490000E+00, 3.40500000E+00, 3.39300000E+00, 3.37540000E+00,
     3.83880000E+00, 4.01900000E+00, 3.82390000E+00, 3.68790000E+00, 3.59590000E+00, 3.52800000E+00,
     4.15380000E+00, 4.19740000E+00, 4.15510000E+00, 4.06820000E+00, 4.04960000E+00, 4.03230000E+00,
     4.01800000E+00, 4.00250000E+00, 3.99070000E+00, 3.67750000E+00, 3.62180000E+00, 3.93750000E+00,
     3.93710000E+00, 3.92530000E+00, 3.91700000E+00, 3.90950000E+00, 3.68930000E+00, 3.83710000E+00,
     3.80100000E+00, 3.72570000E+00, 3.65100000E+00, 3.57630000E+00, 3.50860000E+00, 3.45620000E+00,
     3.40370000E+00, 3.36240000E+00, 3.85430000E+00, 4.05960000E+00, 3.86820000E+00, 3.73920000E+00,
     3.65190000E+00, 3.58490000E+00, 4.19890000E+00, 4.16190000E+00, 4.32050000E+00, 3.96550000E+00,
     3.88720000E+00, 3.86060000E+00, 3.84240000E+00, 4.00360000E+00},
    {2.91400000E+00, 2.62710000E+00, 3.47710000E+00, 3.17740000E+00, 3.25600000E+00, 3.19700000E+00,
     3.12070000E+00, 2.94060000E+00, 2.83220000E+00, 2.75710000E+00, 3.54550000E+00, 3.35140000E+00,
     3.58370000E+00, 3.61770000E+00, 3.58160000E+00, 3.39020000E+00, 3.26040000E+00, 3.16520000E+00,
     3.70370000E+00, 3.62830000E+00, 3.58580000E+00, 3.53300000E+00, 3.48840000E+00, 3.57890000E+00,
     3.40940000E+00, 3.34730000E+00, 3.31180000E+00, 3.28760000E+00, 3.27070000E+00, 3.25210000E+00,
     3.55700000E+00, 3.64960000E+00, 3.66250000E+00, 3.50090000E+00, 3.39100000E+00, 3.30920000E+00,
     3.79260000E+00, 3.75440000E+00, 3.73920000E+00, 3.68960000E+00, 3.76930000E+00, 3.71520000E+00,
     3.58950000E+00, 3.52600000E+00, 3.49020000E+00, 3.46560000E+00, 3.44860000E+00, 3.36460000E+00,
     3.67450000E+00, 3.79610000E+00, 3.83460000E+00, 3.68320000E+00, 3.57680000E+00, 3.49690000E+00,
     3.89940000E+00, 3.95320000E+00, 3.91080000E+00, 3.83510000E+00, 3.81450000E+00, 3.79540000E+00,
     3.77960000E+00, 3.76360000E+00, 3.75060000E+00, 3.63640000E+00, 3.57880000E+00, 3.65790000E+00,
     3.68730000E+00, 3.67340000E+00, 3.66370000E+00, 3.65520000E+00, 3.66660000E+00, 3.73250000E+00,
     3.68940000E+00, 3.64760000E+00, 3.60780000E+00, 3.52890000E+00, 3.50730000E+00, 3.46380000E+00,
     3.42430000E+00, 3.34400000E+00, 3.68160000E+00, 3.83070000E+00, 3.88370000E+00, 3.73880000E+00,
     3.63630000E+00, 3.55700000E+00, 3.95070000E+00, 3.93600000E+00, 4.08210000E+00, 3.91160000E+00,
     3.85640000E+00, 3.83060000E+00, 3.80950000E+00, 3.76040000E+00},
    {2.73000000E+00, 2.58700000E+00, 3.24710000E+00, 3.14870000E+00, 3.16670000E+00, 3.09140000E+00,
     3.01070000E+00, 2.98120000E+00, 2.83000000E+00, 2.72840000E+00, 3.32590000E+00, 3.31820000E+00,
     3.47070000E+00, 3.47480000E+00, 3.42790000E+00, 3.41820000E+00, 3.25470000E+00, 3.13530000E+00,
     3.51160000E+00, 3.94320000E+00, 3.88280000E+00, 3.83030000E+00, 3.78800000E+00, 3.37600000E+00,
     3.72180000E+00, 3.34080000E+00, 3.30590000E+00, 3.26980000E+00, 3.24460000E+00, 3.22290000E+00,
     3.44220000E+00, 3.50230000E+00, 3.50090000E+00, 3.52680000E+00, 3.38490000E+00, 3.27950000E+00,
     3.61580000E+00, 4.03710000E+00, 3.98920000E+00, 3.93150000E+00, 3.57180000E+00, 3.52010000E+00,
     3.47830000E+00, 3.44130000E+00, 3.40910000E+00, 3.30210000E+00, 3.26820000E+00, 3.33080000E+00,
     3.55620000E+00, 3.64120000E+00, 3.66270000E+00, 3.70430000E+00, 3.56780000E+00, 3.46500000E+00,
     3.74190000E+00, 4.17940000E+00, 4.13370000E+00, 4.10540000E+00, 4.08810000E+00, 4.07250000E+00,
     4.05980000E+00, 4.04710000E+00, 4.03650000E+00, 3.92030000E+00, 3.87630000E+00, 3.58700000E+00,
     3.98460000E+00, 3.97330000E+00, 3.96410000E+00, 3.95720000E+00, 3.91700000E+00, 3.91600000E+00,
     3.87130000E+00, 3.57290000E+00, 3.53550000E+00, 3.50860000E+00, 3.45090000E+00, 3.32110000E+00,
     3.26330000E+00, 3.31320000E+00, 3.56020000E+00, 3.67020000E+00, 3.70550000E+00, 3.76020000E+00,
     3.62780000E+00, 3.52570000E+00, 3.79600000E+00, 4.17660000E+00, 4.25470000E+00, 4.10100000E+00,
     4.07930000E+00, 4.06010000E+00, 4.04120000E+00, 4.02880000E+00},
    {2.60260000E+00, 2.53550000E+00, 3.11290000E+00, 3.28630000E+00, 3.10290000E+00, 3.01080000E+00,
     2.92270000E+00, 2.86940000E+00, 2.81090000E+00, 2.69290000E+00, 3.19580000E+00, 3.46700000E+00,
     3.40180000E+00, 3.38050000E+00, 3.32180000E+00, 3.28150000E+00, 3.23460000E+00, 3.09940000E+00,
     3.39370000E+00, 3.72660000E+00, 3.66970000E+00, 3.61640000E+00, 3.57300000E+00, 3.25220000E+00,
     3.50510000E+00, 3.46860000E+00, 3.43550000E+00, 3.40840000E+00, 3.37480000E+00, 3.34960000E+00,
     3.36920000E+00, 3.40520000E+00, 3.39100000E+00, 3.38490000E+00, 3.36620000E+00, 3.24520000E+00,
     3.50170000E+00, 3.82250000E+00, 3.78320000E+00, 3.72500000E+00, 3.44980000E+00, 3.39660000E+00,
     3.34840000E+00, 3.30850000E+00, 3.27440000E+00, 3.24330000E+00, 3.17290000E+00, 3.43930000E+00,
     3.48060000E+00, 3.54050000E+00, 3.54790000E+00, 3.55570000E+00, 3.54860000E+00, 3.43040000E+00,
     3.63560000E+00, 3.96600000E+00, 3.92890000E+00, 3.89280000E+00, 3.87560000E+00, 3.85980000E+00,
     3.84700000E+00, 3.83430000E+00, 3.82350000E+00, 3.71570000E+00, 3.65720000E+00, 3.63610000E+00,
     3.77090000E+00, 3.75970000E+00, 3.75030000E+00, 3.74360000E+00, 3.71500000E+00, 3.71560000E+00,
     3.66990000E+00, 3.63040000E+00, 3.59630000E+00, 3.55930000E+00, 3.31200000E+00, 3.27620000E+00,
     3.18100000E+00, 3.40350000E+00, 3.47980000E+00, 3.56510000E+00, 3.58700000E+00, 3.60780000E+00,
     3.61100000E+00, 3.49360000E+00, 3.69250000E+00, 3.97050000E+00, 4.06590000E+00, 3.91020000E+00,
     3.88350000E+00, 3.86250000E+00, 3.84360000E+00, 3.82100000E+00},
    {2.50870000E+00, 2.48140000E+00, 3.02390000E+00, 3.13120000E+00, 3.05350000E+00, 2.94570000E+00,
     2.84960000E+00, 2.77800000E+00, 2.78280000E+00, 2.65320000E+00, 3.10630000E+00, 3.31430000E+00,
     3.35490000E+00, 3.31200000E+00, 3.24210000E+00, 3.17870000E+00, 3.11760000E+00, 3.06130000E+00,
     3.30820000E+00, 3.57550000E+00, 3.52220000E+00, 3.46780000E+00, 3.42310000E+00, 3.16840000E+00,
     3.35280000E+00, 3.31620000E+00, 3.28270000E+00, 3.25270000E+00, 3.23080000E+00, 3.20290000E+00,
     3.31730000E+00, 3.33430000E+00, 3.30920000E+00, 3.27950000E+00, 3.24520000E+00, 3.20960000E+00,
     3.41660000E+00, 3.67180000E+00, 3.64060000E+00, 3.58190000E+00, 3.36620000E+00, 3.31070000E+00,
     3.25590000E+00, 3.21340000E+00, 3.17760000E+00, 3.14530000E+00, 3.11680000E+00, 3.29930000E+00,
     3.42720000E+00, 3.46810000E+00, 3.46390000E+00, 3.44660000E+00, 3.42280000E+00, 3.39630000E+00,
     3.55480000E+00, 3.81300000E+00, 3.78730000E+00, 3.74450000E+00, 3.72710000E+00, 3.71130000E+00,
     3.69830000E+00, 3.68540000E+00, 3.67450000E+00, 3.57460000E+00, 3.50220000E+00, 3.50390000E+00,
     3.62100000E+00, 3.60990000E+00, 3.60040000E+00, 3.59330000E+00, 3.57600000E+00, 3.57140000E+00,
     3.52440000E+00, 3.48340000E+00, 3.44800000E+00, 3.40990000E+00, 3.20820000E+00, 3.17300000E+00,
     3.13510000E+00, 3.27540000E+00, 3.42080000E+00, 3.48840000E+00, 3.50000000E+00, 3.49600000E+00,
     3.48250000E+00, 3.46280000E+00, 3.61500000E+00, 3.82620000E+00, 3.93590000E+00, 3.78000000E+00,
     3.74950000E+00, 3.72690000E+00, 3.70770000E+00, 3.67570000E+00},
    {3.28930000E+00, 2.89910000E+00, 4.03880000E+00, 3.61000000E+00, 3.93880000E+00, 3.44750000E+00,
     3.15900000E+00, 2.98120000E+00, 2.85860000E+00, 2.76830000E+00, 4.14280000E+00, 3.79110000E+00,
     3.82250000E+00, 4.03720000E+00, 3.70590000E+00, 3.49350000E+00, 3.35290000E+00, 3.24920000E+00,
     4.43520000E+00, 4.08260000E+00, 3.97330000E+00, 3.92540000E+00, 3.86460000E+00, 3.93150000E+00,
     3.78370000E+00, 3.74650000E+00, 3.72110000E+00, 3.70120000E+00, 3.68930000E+00, 3.66760000E+00,
     3.77360000E+00, 4.06600000E+00, 3.79260000E+00, 3.61580000E+00, 3.50170000E+00, 3.41660000E+00,
     4.61760000E+00, 4.30920000E+00, 4.17010000E+00, 4.14570000E+00, 4.16080000E+00, 4.11280000E+00,
     4.04220000E+00, 4.01700000E+00, 4.00780000E+00, 3.99910000E+00, 3.99520000E+00, 3.87680000E+00,
     4.01820000E+00, 4.35380000E+00, 4.10260000E+00, 3.92030000E+00, 3.81070000E+00, 3.72290000E+00,
     4.83530000E+00, 4.81600000E+00, 4.36660000E+00, 4.34150000E+00, 4.31280000E+00, 4.28960000E+00,
     4.26780000E+00, 4.24640000E+00, 4.23140000E+00, 3.94940000E+00, 3.97370000E+00, 4.31490000E+00,
     4.18120000E+00, 4.16830000E+00, 4.15900000E+00, 4.15080000E+00, 4.02680000E+00, 4.60360000E+00,
     4.51510000E+00, 4.43680000E+00, 4.35870000E+00, 4.27880000E+00, 4.26110000E+00, 4.16320000E+00,
     4.05570000E+00, 3.87010000E+00, 4.05420000E+00, 4.41820000E+00, 4.15730000E+00, 3.98000000E+00,
     3.87950000E+00, 3.78740000E+00, 4.85350000E+00, 4.53480000E+00, 4.39520000E+00, 4.29640000E+00,
     4.22130000E+00, 4.20620000E+00, 4.18370000E+00, 4.26460000E+00},
    {2.87860000E+00, 3.16580000E+00, 3.58230000E+00, 3.76890000E+00, 3.57620000E+00, 3.57890000E+00,
     3.35520000E+00, 3.40040000E+00, 3.17220000E+00, 3.02120000E+00, 3.72410000E+00, 3.96040000E+00,
     3.85000000E+00, 3.98440000E+00, 3.70350000E+00, 3.91610000E+00, 3.67510000E+00, 3.50750000E+00,
     4.11510000E+00, 4.28770000E+00, 4.15790000E+00, 4.12470000E+00, 4.06170000E+00, 3.48740000E+00,
     3.98480000E+00, 3.92800000E+00, 3.90790000E+00, 3.87510000E+00, 3.86040000E+00, 3.82770000E+00,
     3.80020000E+00, 3.99810000E+00, 3.75440000E+00, 4.03710000E+00, 3.82250000E+00, 3.67180000E+00,
     4.30920000E+00, 4.47640000E+00, 4.30160000E+00, 4.22800000E+00, 3.74170000E+00, 3.68990000E+00,
     3.59670000E+00, 3.54640000E+00, 3.53740000E+00, 3.50170000E+00, 3.47960000E+00, 3.98910000E+00,
     3.99910000E+00, 4.21360000E+00, 3.97330000E+00, 4.29190000E+00, 4.08660000E+00, 3.94020000E+00,
     4.54130000E+00, 4.82720000E+00, 4.50800000E+00, 4.54970000E+00, 4.52420000E+00, 4.50210000E+00,
     4.48300000E+00, 4.46350000E+00, 4.44900000E+00, 4.20760000E+00, 4.32550000E+00, 4.29780000E+00,
     4.37500000E+00, 4.35720000E+00, 4.34480000E+00, 4.33370000E+00, 4.15960000E+00, 4.46200000E+00,
     4.45170000E+00, 4.39210000E+00, 4.33900000E+00, 4.26240000E+00, 3.81170000E+00, 3.68250000E+00,
     3.53680000E+00, 3.95230000E+00, 4.01390000E+00, 4.25160000E+00, 4.00050000E+00, 4.35180000E+00,
     4.14480000E+00, 3.99160000E+00, 4.56420000E+00, 4.70250000E+00, 4.51760000E+00, 4.33300000E+00,
     4.37040000E+00, 4.35660000E+00, 4.33660000E+00, 4.45580000E+00},
    {2.89970000E+00, 3.09530000E+00, 3.45240000E+00, 3.61070000E+00, 3.60620000E+00, 3.57830000E+00,
     3.34630000E+00, 3.38550000E+00, 3.17460000E+00, 3.03810000E+00, 3.60190000E+00, 3.79380000E+00,
     3.86970000E+00, 3.97810000E+00, 3.68770000E+00, 3.87360000E+00, 3.64510000E+00, 3.48900000E+00,
     3.98580000E+00, 4.11790000E+00, 4.04300000E+00, 3.95630000E+00, 3.91820000E+00, 3.40020000E+00,
     3.83100000E+00, 3.77160000E+00, 3.75430000E+00, 3.72030000E+00, 3.70530000E+00, 3.67420000E+00,
     3.83180000E+00, 3.76310000E+00, 3.73920000E+00, 3.98920000E+00, 3.78320000E+00, 3.64060000E+00,
     4.17010000E+00, 4.30160000E+00, 4.21960000E+00, 4.11300000E+00, 3.65360000E+00, 3.60820000E+00,
     3.51130000E+00, 3.45870000E+00, 3.45370000E+00, 3.41410000E+00, 3.39010000E+00, 3.83100000E+00,
     4.00070000E+00, 3.93810000E+00, 3.92920000E+00, 4.21960000E+00, 4.01690000E+00, 3.87530000E+00,
     4.38910000E+00, 4.62940000E+00, 4.42320000E+00, 4.38330000E+00, 4.35780000E+00, 4.33250000E+00,
     4.31320000E+00, 4.29390000E+00, 4.27920000E+00, 4.15630000E+00, 4.18280000E+00, 4.13210000E+00,
     4.20440000E+00, 4.18620000E+00, 4.17390000E+00, 4.16240000E+00, 4.11070000E+00, 4.30920000E+00,
     4.25380000E+00, 4.22070000E+00, 4.16350000E+00, 4.08730000E+00, 3.69880000E+00, 3.58220000E+00,
     3.45260000E+00, 3.80180000E+00, 4.01650000E+00, 3.96870000E+00, 3.95680000E+00, 4.28020000E+00,
     4.07360000E+00, 3.92490000E+00, 4.41340000E+00, 4.52680000E+00, 4.38880000E+00, 4.26220000E+00,
     4.33000000E+00, 4.30220000E+00, 4.28160000E+00, 4.28620000E+00},
    {2.85350000E+00, 3.01670000E+00, 3.39780000E+00, 3.53630000E+00, 3.53930000E+00, 3.53010000E+00,
     3.29600000E+00, 3.33520000E+00, 3.12870000E+00, 2.99670000E+00, 3.66590000E+00, 3.72390000E+00,
     3.80700000E+00, 3.71650000E+00, 3.63680000E+00, 3.81620000E+00, 3.58850000E+00, 3.43360000E+00,
     3.98290000E+00, 4.05290000E+00, 3.95840000E+00, 3.90250000E+00, 3.86070000E+00, 3.36730000E+00,
     3.76580000E+00, 3.70350000E+00, 3.68660000E+00, 3.65040000E+00, 3.63390000E+00, 3.60240000E+00,
     3.77080000E+00, 3.72830000E+00, 3.68960000E+00, 3.93150000E+00, 3.72500000E+00, 3.58190000E+00,
     4.14570000E+00, 4.22800000E+00, 4.11300000E+00, 4.05970000E+00, 3.61540000E+00, 3.56040000E+00,
     3.45760000E+00, 3.40060000E+00, 3.39560000E+00, 3.35200000E+00, 3.32550000E+00, 3.74830000E+00,
     3.92820000E+00, 3.89120000E+00, 3.86670000E+00, 4.15420000E+00, 3.94760000E+00, 3.80350000E+00,
     4.34160000E+00, 4.58400000E+00, 4.31550000E+00, 4.31220000E+00, 4.28700000E+00, 4.26450000E+00,
     4.24440000E+00, 4.22520000E+00, 4.21050000E+00, 4.05080000E+00, 4.11580000E+00, 4.12980000E+00,
     4.13400000E+00, 4.11530000E+00, 4.10290000E+00, 4.09160000E+00, 3.99950000E+00, 4.21220000E+00,
     4.14830000E+00, 4.11330000E+00, 4.05360000E+00, 3.97700000E+00, 3.56460000E+00, 3.48700000E+00,
     3.36990000E+00, 3.71490000E+00, 3.94120000E+00, 3.91860000E+00, 3.89360000E+00, 3.85800000E+00,
     4.00450000E+00, 3.85300000E+00, 4.36880000E+00, 4.50760000E+00, 4.36070000E+00, 4.22540000E+00,
     4.21210000E+00, 4.19290000E+00, 4.16860000E+00, 4.21220000E+00},
    {3.09050000E+00, 2.79980000E+00, 3.64480000E+00, 3.07390000E+00, 3.29960000E+00, 3.52620000E+00,
     3.25590000E+00, 3.05180000E+00, 2.93940000E+00, 2.86580000E+00, 3.75140000E+00, 3.22950000E+00,
     3.56430000E+00, 3.78080000E+00, 3.69310000E+00, 3.47230000E+00, 3.33570000E+00, 3.24290000E+00,
     4.02800000E+00, 3.55890000E+00, 3.46360000E+00, 3.49940000E+00, 3.43090000E+00, 3.61770000E+00,
     3.29460000E+00, 3.23760000E+00, 3.20500000E+00, 3.18470000E+00, 3.17150000E+00, 3.15990000E+00,
     3.55550000E+00, 3.81110000E+00, 3.76930000E+00, 3.57180000E+00, 3.44980000E+00, 3.36620000E+00,
     4.16080000E+00, 3.74170000E+00, 3.65360000E+00, 3.61540000E+00, 3.85960000E+00, 3.78340000E+00,
     3.65940000E+00, 3.60270000E+00, 3.57470000E+00, 3.55830000E+00, 3.55300000E+00, 3.34170000E+00,
     3.72380000E+00, 3.96810000E+00, 3.95130000E+00, 3.75730000E+00, 3.63580000E+00, 3.54990000E+00,
     4.32430000E+00, 4.07700000E+00, 3.84610000E+00, 3.80510000E+00, 3.77960000E+00, 3.75710000E+00,
     3.73700000E+00, 3.71690000E+00, 3.70030000E+00, 3.53290000E+00, 3.50780000E+00, 3.81640000E+00,
     3.64590000E+00, 3.63120000E+00, 3.62240000E+00, 3.61260000E+00, 3.55740000E+00, 3.84780000E+00,
     3.86410000E+00, 3.80670000E+00, 3.71930000E+00, 3.64070000E+00, 3.69250000E+00, 3.63250000E+00,
     3.57070000E+00, 3.36730000E+00, 3.76980000E+00, 3.94850000E+00, 3.99900000E+00, 3.80560000E+00,
     3.68430000E+00, 3.59320000E+00, 4.33960000E+00, 3.95620000E+00, 3.95830000E+00, 3.81950000E+00,
     3.76430000E+00, 3.74010000E+00, 3.72930000E+00, 3.70880000E+00},
    {3.03010000E+00, 2.73120000E+00, 3.58210000E+00, 3.04730000E+00, 3.21370000E+00, 3.46790000E+00,
     3.19750000E+00, 2.99690000E+00, 2.88470000E+00, 2.81100000E+00, 3.69310000E+00, 3.20760000E+00,
     3.49430000E+00, 3.59560000E+00, 3.63790000E+00, 3.41900000E+00, 3.28080000E+00, 3.18600000E+00,
     3.98500000E+00, 3.51050000E+00, 3.43300000E+00, 3.37970000E+00, 3.41550000E+00, 3.60330000E+00,
     3.27370000E+00, 3.21450000E+00, 3.18070000E+00, 3.15960000E+00, 3.14610000E+00, 3.13370000E+00,
     3.48120000E+00, 3.62510000E+00, 3.71520000E+00, 3.52010000E+00, 3.39660000E+00, 3.31070000E+00,
     4.11280000E+00, 3.68990000E+00, 3.60820000E+00, 3.56040000E+00, 3.78340000E+00, 3.75430000E+00,
     3.63130000E+00, 3.57300000E+00, 3.54300000E+00, 3.52510000E+00, 3.51830000E+00, 3.30190000E+00,
     3.64980000E+00, 3.79090000E+00, 3.89590000E+00, 3.70390000E+00, 3.58000000E+00, 3.49130000E+00,
     4.27530000E+00, 4.00880000E+00, 3.80070000E+00, 3.75830000E+00, 3.73180000E+00, 3.70830000E+00,
     3.68760000E+00, 3.66750000E+00, 3.65100000E+00, 3.47800000E+00, 3.45950000E+00, 3.76800000E+00,
     3.59290000E+00, 3.57720000E+00, 3.56770000E+00, 3.55820000E+00, 3.51030000E+00, 3.75720000E+00,
     3.72440000E+00, 3.74210000E+00, 3.65290000E+00, 3.57430000E+00, 3.62950000E+00, 3.57400000E+00,
     3.51840000E+00, 3.31990000E+00, 3.69150000E+00, 3.83700000E+00, 3.94330000E+00, 3.75270000E+00,
     3.62910000E+00, 3.53550000E+00, 4.28790000E+00, 3.90650000E+00, 3.92800000E+00, 3.75600000E+00,
     3.71300000E+00, 3.68880000E+00, 3.67090000E+00, 3.67290000E+00},
    {2.91890000E+00, 2.67770000E+00, 3.49250000E+00, 2.96480000E+00, 3.12160000E+00, 3.29400000E+00,
     3.09750000E+00, 2.97570000E+00, 2.84930000E+00, 2.76380000E+00, 3.60850000E+00, 3.12140000E+00,
     3.40060000E+00, 3.47930000E+00, 3.51470000E+00, 3.38060000E+00, 3.23560000E+00, 3.13350000E+00,
     3.91440000E+00, 3.41830000E+00, 3.33690000E+00, 3.28030000E+00, 3.26790000E+00, 3.48710000E+00,
     3.17140000E+00, 3.15210000E+00, 3.11010000E+00, 3.08430000E+00, 3.06700000E+00, 3.05390000E+00,
     3.38900000E+00, 3.50860000E+00, 3.58950000E+00, 3.47830000E+00, 3.34840000E+00, 3.25590000E+00,
     4.04220000E+00, 3.59670000E+00, 3.51130000E+00, 3.45760000E+00, 3.65940000E+00, 3.63130000E+00,
     3.56900000E+00, 3.49450000E+00, 3.45220000E+00, 3.42430000E+00, 3.40970000E+00, 3.22500000E+00,
     3.56050000E+00, 3.67740000E+00, 3.76980000E+00, 3.65460000E+00, 3.52600000E+00, 3.43190000E+00,
     4.20530000E+00, 3.91030000E+00, 3.69910000E+00, 3.66220000E+00, 3.63640000E+00, 3.61360000E+00,
     3.59350000E+00, 3.57390000E+00, 3.55780000E+00, 3.37310000E+00, 3.36000000E+00, 3.71540000E+00,
     3.50360000E+00, 3.48810000E+00, 3.47850000E+00, 3.46840000E+00, 3.41630000E+00, 3.65970000E+00,
     3.62430000E+00, 3.61400000E+00, 3.55120000E+00, 3.51780000E+00, 3.53830000E+00, 3.47330000E+00,
     3.40850000E+00, 3.24830000E+00, 3.60430000E+00, 3.72550000E+00, 3.81720000E+00, 3.70190000E+00,
     3.57410000E+00, 3.47570000E+00, 4.21660000E+00, 3.81190000E+00, 3.83900000E+00, 3.65130000E+00,
     3.61440000E+00, 3.59000000E+00, 3.57000000E+00, 3.58000000E+00},
    {2.85780000E+00, 2.63340000E+00, 3.46730000E+00, 2.92450000E+00, 3.07320000E+00, 3.24350000E+00,
     3.03380000E+00, 2.94620000E+00, 2.81430000E+00, 2.72400000E+00, 3.58320000E+00, 3.07890000E+00,
     3.36170000E+00, 3.42460000E+00, 3.45050000E+00, 3.34430000E+00, 3.19640000E+00, 3.09130000E+00,
     3.89210000E+00, 3.37130000E+00, 3.28730000E+00, 3.22810000E+00, 3.21650000E+00, 3.43860000E+00,
     3.11640000E+00, 3.12200000E+00, 3.07610000E+00, 3.04800000E+00, 3.02950000E+00, 3.01550000E+00,
     3.34950000E+00, 3.45430000E+00, 3.52600000E+00, 3.44130000E+00, 3.30850000E+00, 3.21340000E+00,
     4.01700000E+00, 3.54640000E+00, 3.45870000E+00, 3.40060000E+00, 3.60270000E+00, 3.57300000E+00,
     3.49450000E+00, 3.46230000E+00, 3.41600000E+00, 3.38510000E+00, 3.36830000E+00, 3.18320000E+00,
     3.52110000E+00, 3.62620000E+00, 3.70890000E+00, 3.61510000E+00, 3.48380000E+00, 3.38730000E+00,
     4.17900000E+00, 3.85360000E+00, 3.64470000E+00, 3.61080000E+00, 3.58540000E+00, 3.56280000E+00,
     3.54280000E+00, 3.52350000E+00, 3.50750000E+00, 3.31260000E+00, 3.30880000E+00, 3.68580000E+00,
     3.45770000E+00, 3.44290000E+00, 3.43140000E+00, 3.42120000E+00, 3.36550000E+00, 3.59690000E+00,
     3.55890000E+00, 3.54910000E+00, 3.48370000E+00, 3.47530000E+00, 3.49100000E+00, 3.42470000E+00,
     3.35950000E+00, 3.20690000E+00, 3.56390000E+00, 3.67440000E+00, 3.75660000E+00, 3.66150000E+00,
     3.53120000E+00, 3.43060000E+00, 4.18880000E+00, 3.76050000E+00, 3.79710000E+00, 3.59410000E+00,
     3.55990000E+00, 3.53500000E+00, 3.51530000E+00, 3.52760000E+00},
    {2.82400000E+00, 2.59600000E+00, 3.46350000E+00, 2.90320000E+00, 3.04310000E+00, 3.21150000E+00,
     2.98920000E+00, 2.91480000E+00, 2.78010000E+00, 2.68730000E+00, 3.57760000E+00, 3.05680000E+00,
     3.34330000E+00, 3.39490000E+00, 3.41320000E+00, 3.31160000E+00, 3.16160000E+00, 3.05480000E+00,
     3.88590000E+00, 3.37190000E+00, 3.29170000E+00, 3.23450000E+00, 3.22740000E+00, 3.41710000E+00,
     3.12930000E+00, 3.05670000E+00, 3.05650000E+00, 3.02740000E+00, 3.00870000E+00, 2.99390000E+00,
     3.32930000E+00, 3.42490000E+00, 3.49020000E+00, 3.40910000E+00, 3.27440000E+00, 3.17760000E+00,
     4.00780000E+00, 3.53740000E+00, 3.45370000E+00, 3.39560000E+00, 3.57470000E+00, 3.54300000E+00,
     3.45220000E+00, 3.41600000E+00, 3.39750000E+00, 3.36620000E+00, 3.34920000E+00, 3.15780000E+00,
     3.50090000E+00, 3.59990000E+00, 3.67650000E+00, 3.52930000E+00, 3.45010000E+00, 3.35200000E+00,
     4.16850000E+00, 3.83240000E+00, 3.63080000E+00, 3.59710000E+00, 3.57260000E+00, 3.55070000E+00,
     3.53140000E+00, 3.51260000E+00, 3.49710000E+00, 3.28460000E+00, 3.25750000E+00, 3.67090000E+00,
     3.45280000E+00, 3.43950000E+00, 3.43130000E+00, 3.42070000E+00, 3.36770000E+00, 3.55750000E+00,
     3.51720000E+00, 3.50770000E+00, 3.44000000E+00, 3.39310000E+00, 3.46250000E+00, 3.39690000E+00,
     3.33330000E+00, 3.17930000E+00, 3.54160000E+00, 3.64760000E+00, 3.72460000E+00, 3.57680000E+00,
     3.49740000E+00, 3.39530000E+00, 4.17680000E+00, 3.74470000E+00, 3.79550000E+00, 3.58100000E+00,
     3.54740000E+00, 3.52260000E+00, 3.50390000E+00, 3.51650000E+00},
    {2.80040000E+00, 2.56210000E+00, 3.46170000E+00, 2.91540000E+00, 3.02030000E+00, 3.18750000E+00,
     2.95480000E+00, 2.80380000E+00, 2.74720000E+00, 2.65300000E+00, 3.57360000E+00, 3.05840000E+00,
     3.33040000E+00, 3.37480000E+00, 3.38710000E+00, 3.20280000E+00, 3.12960000E+00, 3.02140000E+00,
     3.87960000E+00, 3.33370000E+00, 3.24920000E+00, 3.18830000E+00, 3.18020000E+00, 3.40500000E+00,
     3.07560000E+00, 3.04780000E+00, 3.03220000E+00, 3.03230000E+00, 3.01630000E+00, 3.00190000E+00,
     3.31450000E+00, 3.40500000E+00, 3.46560000E+00, 3.30210000E+00, 3.24330000E+00, 3.14530000E+00,
     3.99910000E+00, 3.50170000E+00, 3.41410000E+00, 3.35200000E+00, 3.55830000E+00, 3.52510000E+00,
     3.42430000E+00, 3.38510000E+00, 3.36620000E+00, 3.35250000E+00, 3.33600000E+00, 3.15640000E+00,
     3.48590000E+00, 3.58230000E+00, 3.65480000E+00, 3.48490000E+00, 3.42040000E+00, 3.32090000E+00,
     4.15850000E+00, 3.79950000E+00, 3.59590000E+00, 3.56280000E+00, 3.53780000E+00, 3.51550000E+00,
     3.49580000E+00, 3.47680000E+00, 3.46090000E+00, 3.24260000E+00, 3.21640000E+00, 3.66660000E+00,
     3.41460000E+00, 3.40090000E+00, 3.39230000E+00, 3.38290000E+00, 3.32430000E+00, 3.53860000E+00,
     3.49730000E+00, 3.48870000E+00, 3.41910000E+00, 3.36940000E+00, 3.42330000E+00, 3.37640000E+00,
     3.31430000E+00, 3.15580000E+00, 3.52470000E+00, 3.62950000E+00, 3.70330000E+00, 3.53300000E+00,
     3.44720000E+00, 3.36460000E+00, 4.16600000E+00, 3.71190000E+00, 3.76740000E+00, 3.54200000E+00,
     3.50930000E+00, 3.48380000E+00, 3.46840000E+00, 3.47830000E+00},
    {2.78460000E+00, 2.53240000E+00, 3.46520000E+00, 2.87590000E+00, 3.00510000E+00, 3.16920000E+00,
     2.92730000E+00, 2.76150000E+00, 2.71640000E+00, 2.62120000E+00, 3.57440000E+00, 3.02750000E+00,
     3.32490000E+00, 3.36270000E+00, 3.36860000E+00, 3.16690000E+00, 3.05840000E+00, 2.99150000E+00,
     3.87730000E+00, 3.30990000E+00, 3.22310000E+00, 3.16000000E+00, 3.15200000E+00, 3.40230000E+00,
     3.04260000E+00, 3.00990000E+00, 2.99200000E+00, 2.98090000E+00, 2.98000000E+00, 2.96460000E+00,
     3.30680000E+00, 3.39300000E+00, 3.44860000E+00, 3.26820000E+00, 3.17290000E+00, 3.11680000E+00,
     3.99520000E+00, 3.47960000E+00, 3.39010000E+00, 3.32550000E+00, 3.55300000E+00, 3.51830000E+00,
     3.40970000E+00, 3.36830000E+00, 3.34920000E+00, 3.33600000E+00, 3.33080000E+00, 3.12240000E+00,
     3.47850000E+00, 3.57270000E+00, 3.64090000E+00, 3.45520000E+00, 3.35530000E+00, 3.29480000E+00,
     4.15360000E+00, 3.78260000E+00, 3.57490000E+00, 3.54080000E+00, 3.51550000E+00, 3.49290000E+00,
     3.47300000E+00, 3.45370000E+00, 3.43770000E+00, 3.21350000E+00, 3.18560000E+00, 3.65170000E+00,
     3.39040000E+00, 3.37660000E+00, 3.36800000E+00, 3.35860000E+00, 3.29750000E+00, 3.51530000E+00,
     3.47150000E+00, 3.46230000E+00, 3.38910000E+00, 3.33390000E+00, 3.40070000E+00, 3.35250000E+00,
     3.30410000E+00, 3.13950000E+00, 3.51530000E+00, 3.61930000E+00, 3.69000000E+00, 3.50380000E+00,
     3.40340000E+00, 3.33900000E+00, 4.16080000E+00, 3.69160000E+00, 3.75210000E+00, 3.51780000E+00,
     3.48530000E+00, 3.45940000E+00, 3.44370000E+00, 3.45390000E+00},
    {2.54240000E+00, 2.66010000E+00, 3.25550000E+00, 3.28070000E+00, 3.13840000E+00, 3.17370000E+00,
     2.93970000E+00, 2.84290000E+00, 2.84920000E+00, 2.72250000E+00, 3.38750000E+00, 3.49100000E+00,
     3.45200000E+00, 3.36080000E+00, 3.30360000E+00, 3.23450000E+00, 3.29990000E+00, 3.14870000E+00,
     3.74090000E+00, 3.83920000E+00, 3.71480000E+00, 3.64390000E+00, 3.61820000E+00, 3.17530000E+00,
     3.52100000E+00, 3.46390000E+00, 3.42650000E+00, 3.40750000E+00, 3.38280000E+00, 3.34740000E+00,
     3.40710000E+00, 3.37540000E+00, 3.36460000E+00, 3.33080000E+00, 3.43930000E+00, 3.29930000E+00,
     3.87680000E+00, 3.98910000E+00, 3.83100000E+00, 3.74830000E+00, 3.34170000E+00, 3.30190000E+00,
     3.22500000E+00, 3.18320000E+00, 3.15780000E+00, 3.15640000E+00, 3.12240000E+00, 3.46200000E+00,
     3.56210000E+00, 3.54190000E+00, 3.53980000E+00, 3.51920000E+00, 3.64870000E+00, 3.50520000E+00,
     4.05790000E+00, 4.22940000E+00, 4.03590000E+00, 4.07800000E+00, 4.05270000E+00, 4.02970000E+00,
     4.01170000E+00, 3.99320000E+00, 3.97880000E+00, 3.74910000E+00, 3.85220000E+00, 3.81740000E+00,
     3.90140000E+00, 3.88270000E+00, 3.86980000E+00, 3.86040000E+00, 3.70710000E+00, 3.78180000E+00,
     3.73400000E+00, 3.69560000E+00, 3.66220000E+00, 3.60020000E+00, 3.23290000E+00, 3.19840000E+00,
     3.10940000E+00, 3.40970000E+00, 3.56540000E+00, 3.56590000E+00, 3.56970000E+00, 3.56390000E+00,
     3.71310000E+00, 3.56370000E+00, 4.07450000E+00, 4.19500000E+00, 4.10620000E+00, 3.88610000E+00,
     3.90750000E+00, 3.88880000E+00, 3.86630000E+00, 3.95530000E+00},
    {2.97430000E+00, 2.80580000E+00, 3.48300000E+00, 3.34740000E+00, 3.68630000E+00, 3.36170000E+00,
     3.16080000E+00, 3.00690000E+00, 2.96400000E+00, 2.84270000E+00, 3.58850000E+00, 3.52190000E+00,
     4.13140000E+00, 3.81200000E+00, 3.60150000E+00, 3.45020000E+00, 3.34980000E+00, 3.27770000E+00,
     3.86350000E+00, 3.82320000E+00, 3.84860000E+00, 3.72150000E+00, 3.64870000E+00, 3.47240000E+00,
     3.56270000E+00, 3.50870000E+00, 3.47570000E+00, 3.45170000E+00, 3.44230000E+00, 3.41390000E+00,
     4.10280000E+00, 3.83880000E+00, 3.67450000E+00, 3.55620000E+00, 3.48060000E+00, 3.42720000E+00,
     4.01820000E+00, 3.99910000E+00, 4.00070000E+00, 3.92820000E+00, 3.72380000E+00, 3.64980000E+00,
     3.56050000E+00, 3.52110000E+00, 3.50090000E+00, 3.48590000E+00, 3.47850000E+00, 3.56210000E+00,
     4.26230000E+00, 4.02450000E+00, 3.87590000E+00, 3.76730000E+00, 3.69730000E+00, 3.64650000E+00,
     4.19800000E+00, 4.33800000E+00, 4.31240000E+00, 4.07270000E+00, 4.04780000E+00, 4.02340000E+00,
     4.00430000E+00, 3.98540000E+00, 3.97120000E+00, 3.90060000E+00, 3.86650000E+00, 3.86080000E+00,
     3.90310000E+00, 3.88680000E+00, 3.87580000E+00, 3.86580000E+00, 3.90470000E+00, 4.13350000E+00,
     4.03160000E+00, 3.95680000E+00, 3.89340000E+00, 3.81640000E+00, 3.67230000E+00, 3.59890000E+00,
     3.51930000E+00, 3.54100000E+00, 4.28620000E+00, 4.06630000E+00, 3.91830000E+00, 3.81920000E+00,
     3.75570000E+00, 3.70530000E+00, 4.22890000E+00, 4.21100000E+00, 4.36330000E+00, 4.14590000E+00,
     4.11150000E+00, 4.08130000E+00, 4.05750000E+00, 3.98180000E+00},
    {3.07750000E+00, 2.82750000E+00, 4.01810000E+00, 3.33850000E+00, 3.53790000E+00, 3.50360000E+00,
     3.25890000E+00, 3.08040000E+00, 3.00940000E+00, 2.90030000E+00, 4.08690000E+00, 3.50880000E+00,
     3.91050000E+00, 3.98330000E+00, 3.71760000E+00, 3.53230000E+00, 3.41020000E+00, 3.32270000E+00,
     4.27020000E+00, 4.08880000E+00, 3.75600000E+00, 3.76870000E+00, 3.66810000E+00, 3.64050000E+00,
     3.55690000E+00, 3.49900000E+00, 3.46590000E+00, 3.44330000E+00, 3.43300000E+00, 3.40920000E+00,
     3.88670000E+00, 4.01900000E+00, 3.79610000E+00, 3.64120000E+00, 3.54050000E+00, 3.46810000E+00,
     4.35380000E+00, 4.21360000E+00, 3.93810000E+00, 3.89120000E+00, 3.96810000E+00, 3.79090000E+00,
     3.67740000E+00, 3.62620000E+00, 3.59990000E+00, 3.58230000E+00, 3.57270000E+00, 3.54190000E+00,
     4.02450000E+00, 4.18740000E+00, 3.98040000E+00, 3.83590000E+00, 3.73980000E+00, 3.66960000E+00,
     4.45640000E+00, 4.43520000E+00, 4.35390000E+00, 4.28360000E+00, 4.26300000E+00, 4.24420000E+00,
     4.22870000E+00, 4.21230000E+00, 4.19970000E+00, 3.83320000E+00, 3.80750000E+00, 4.18050000E+00,
     4.14430000E+00, 4.13160000E+00, 4.12290000E+00, 4.11560000E+00, 3.85140000E+00, 4.01530000E+00,
     3.99580000E+00, 3.89760000E+00, 3.82350000E+00, 3.74780000E+00, 3.68450000E+00, 3.62990000E+00,
     3.57450000E+00, 3.52280000E+00, 4.04370000E+00, 4.23090000E+00, 4.02620000E+00, 3.88830000E+00,
     3.79660000E+00, 3.72660000E+00, 4.48630000E+00, 4.38430000E+00, 4.29910000E+00, 4.11470000E+00,
     4.04730000E+00, 4.02090000E+00, 4.00200000E+00, 4.20400000E+00},
    {3.08930000E+00, 2.79170000E+00, 3.72620000E+00, 3.35180000E+00, 3.42410000E+00, 3.54330000E+00,
     3.27730000E+00, 3.08900000E+00, 2.97750000E+00, 2.90100000E+00, 3.80480000E+00, 3.53620000E+00,
     3.77460000E+00, 3.79110000E+00, 3.75110000E+00, 3.54950000E+00, 3.41490000E+00, 3.31770000E+00,
     4.01290000E+00, 3.83700000E+00, 3.77390000E+00, 3.71250000E+00, 3.71520000E+00, 3.77010000E+00,
     3.58130000E+00, 3.51870000E+00, 3.48350000E+00, 3.45950000E+00, 3.44390000E+00, 3.42420000E+00,
     3.74760000E+00, 3.82390000E+00, 3.83460000E+00, 3.66270000E+00, 3.54790000E+00, 3.46390000E+00,
     4.10260000E+00, 3.97330000E+00, 3.92920000E+00, 3.86670000E+00, 3.95130000E+00, 3.89590000E+00,
     3.76980000E+00, 3.70890000E+00, 3.67650000E+00, 3.65480000E+00, 3.64090000E+00, 3.53980000E+00,
     3.87590000E+00, 3.98040000E+00, 4.01500000E+00, 3.85250000E+00, 3.74050000E+00, 3.65770000E+00,
     4.21920000E+00, 4.19330000E+00, 4.11220000E+00, 4.05530000E+00, 4.03220000E+00, 4.01120000E+00,
     3.99390000E+00, 3.97650000E+00, 3.96240000E+00, 3.80290000E+00, 3.77720000E+00, 3.91020000E+00,
     3.89610000E+00, 3.88070000E+00, 3.87040000E+00, 3.86200000E+00, 3.84220000E+00, 3.91770000E+00,
     3.86870000E+00, 3.82400000E+00, 3.78230000E+00, 3.70280000E+00, 3.68760000E+00, 3.64330000E+00,
     3.60250000E+00, 3.51160000E+00, 3.88710000E+00, 4.01830000E+00, 4.06590000E+00, 3.90920000E+00,
     3.80050000E+00, 3.71770000E+00, 4.25130000E+00, 4.16310000E+00, 4.27670000E+00, 4.07720000E+00,
     4.03180000E+00, 4.00590000E+00, 3.98420000E+00, 3.96040000E+00},
    {2.90910000E+00, 2.76380000E+00, 3.50660000E+00, 3.33770000E+00, 3.34810000E+00, 3.26330000E+00,
     3.18100000E+00, 3.14280000E+00, 2.98720000E+00, 2.88370000E+00, 3.59290000E+00, 3.51830000E+00,
     3.67290000E+00, 3.65960000E+00, 3.60820000E+00, 3.59270000E+00, 3.42240000E+00, 3.29970000E+00,
     3.81900000E+00, 4.18650000E+00, 4.11140000E+00, 4.05400000E+00, 3.63250000E+00, 3.56970000E+00,
     3.55610000E+00, 3.52590000E+00, 3.49010000E+00, 3.45520000E+00, 3.43150000E+00, 3.40910000E+00,
     3.64380000E+00, 3.68790000E+00, 3.68320000E+00, 3.70430000E+00, 3.55570000E+00, 3.44660000E+00,
     3.92030000E+00, 4.29190000E+00, 4.21960000E+00, 4.15420000E+00, 3.75730000E+00, 3.70390000E+00,
     3.65460000E+00, 3.61510000E+00, 3.52930000E+00, 3.48490000E+00, 3.45520000E+00, 3.51920000E+00,
     3.76730000E+00, 3.83590000E+00, 3.85250000E+00, 3.89010000E+00, 3.74590000E+00, 3.63880000E+00,
     4.05280000E+00, 4.45800000E+00, 4.37720000E+00, 4.36470000E+00, 4.34490000E+00, 4.32740000E+00,
     4.31340000E+00, 4.29920000E+00, 4.28770000E+00, 4.14360000E+00, 4.13910000E+00, 3.83940000E+00,
     4.22950000E+00, 4.21640000E+00, 4.20630000E+00, 4.19940000E+00, 3.80220000E+00, 3.86030000E+00,
     3.81150000E+00, 3.76840000E+00, 3.72920000E+00, 3.69520000E+00, 3.61970000E+00, 3.49370000E+00,
     3.43380000E+00, 3.49210000E+00, 3.77410000E+00, 3.86800000E+00, 3.89690000E+00, 3.94780000E+00,
     3.80680000E+00, 3.69960000E+00, 4.08970000E+00, 4.44270000E+00, 4.48570000E+00, 4.31200000E+00,
     4.29990000E+00, 4.28100000E+00, 4.26120000E+00, 4.27180000E+00},
    {2.78060000E+00, 2.72090000E+00, 3.38120000E+00, 3.49580000E+00, 3.29130000E+00, 3.18880000E+00,
     3.09900000E+00, 3.03940000E+00, 2.97890000E+00, 2.85820000E+00, 3.47160000E+00, 3.68830000E+00,
     3.61050000E+00, 3.57040000E+00, 3.50590000E+00, 3.46190000E+00, 3.41380000E+00, 3.27420000E+00,
     3.70800000E+00, 3.97730000E+00, 3.90100000E+00, 3.84090000E+00, 3.79440000E+00, 3.44650000E+00,
     3.72350000E+00, 3.68080000E+00, 3.64530000E+00, 3.61680000E+00, 3.58440000E+00, 3.55760000E+00,
     3.57720000E+00, 3.59590000E+00, 3.57680000E+00, 3.56780000E+00, 3.54860000E+00, 3.42280000E+00,
     3.81070000E+00, 4.08660000E+00, 4.01690000E+00, 3.94760000E+00, 3.63580000E+00, 3.58000000E+00,
     3.52600000E+00, 3.48380000E+00, 3.45010000E+00, 3.42040000E+00, 3.35530000E+00, 3.64870000E+00,
     3.69730000E+00, 3.73980000E+00, 3.74050000E+00, 3.74590000E+00, 3.73800000E+00, 3.61420000E+00,
     3.94890000E+00, 4.25540000E+00, 4.17850000E+00, 4.16220000E+00, 4.14210000E+00, 4.12400000E+00,
     4.10960000E+00, 4.09510000E+00, 4.08310000E+00, 3.94070000E+00, 3.92960000E+00, 3.89680000E+00,
     4.02270000E+00, 4.00920000E+00, 3.99870000E+00, 3.99170000E+00, 3.93230000E+00, 3.93650000E+00,
     3.88620000E+00, 3.84510000E+00, 3.81060000E+00, 3.76690000E+00, 3.47990000E+00, 3.44170000E+00,
     3.34480000E+00, 3.47810000E+00, 3.69850000E+00, 3.76720000E+00, 3.78090000E+00, 3.79950000E+00,
     3.80150000E+00, 3.67750000E+00, 3.98760000E+00, 4.24630000E+00, 4.30390000E+00, 4.12070000E+00,
     4.10500000E+00, 4.08430000E+00, 4.06430000E+00, 4.06890000E+00},
    {2.68480000E+00, 2.67400000E+00, 3.29250000E+00, 3.33860000E+00, 3.24730000E+00, 3.12840000E+00,
     3.03010000E+00, 2.95310000E+00, 2.96020000E+00, 2.82720000E+00, 3.38300000E+00, 3.53580000E+00,
     3.56720000E+00, 3.50490000E+00, 3.42840000E+00, 3.36210000E+00, 3.30010000E+00, 3.24510000E+00,
     3.62090000E+00, 3.82990000E+00, 3.75430000E+00, 3.69200000E+00, 3.64360000E+00, 3.35980000E+00,
     3.57010000E+00, 3.52660000E+00, 3.49040000E+00, 3.45900000E+00, 3.43640000E+00, 3.40770000E+00,
     3.52870000E+00, 3.52800000E+00, 3.49690000E+00, 3.46500000E+00, 3.43040000E+00, 3.39630000E+00,
     3.72290000E+00, 3.94020000E+00, 3.87530000E+00, 3.80350000E+00, 3.54990000E+00, 3.49130000E+00,
     3.43190000E+00, 3.38730000E+00, 3.35200000E+00, 3.32090000E+00, 3.29480000E+00, 3.50520000E+00,
     3.64650000E+00, 3.66960000E+00, 3.65770000E+00, 3.63880000E+00, 3.61420000E+00, 3.58890000E+00,
     3.86420000E+00, 4.10720000E+00, 4.03860000E+00, 4.01780000E+00, 3.99750000E+00, 3.97930000E+00,
     3.96460000E+00, 3.95000000E+00, 3.93780000E+00, 3.79980000E+00, 3.77720000E+00, 3.76730000E+00,
     3.87630000E+00, 3.86270000E+00, 3.85190000E+00, 3.84460000E+00, 3.79320000E+00, 3.79060000E+00,
     3.73790000E+00, 3.69490000E+00, 3.65890000E+00, 3.61370000E+00, 3.37370000E+00, 3.33650000E+00,
     3.29520000E+00, 3.46900000E+00, 3.64130000E+00, 3.69230000E+00, 3.69490000E+00, 3.68960000E+00,
     3.67470000E+00, 3.65580000E+00, 3.90610000E+00, 4.10540000E+00, 4.17620000E+00, 3.99000000E+00,
     3.97100000E+00, 3.94860000E+00, 3.92850000E+00, 3.92530000E+00},
    {3.39680000E+00, 3.01220000E+00, 4.22410000E+00, 3.78870000E+00, 4.00490000E+00, 3.53840000E+00,
     3.26980000E+00, 3.10830000E+00, 2.99170000E+00, 2.90570000E+00, 4.33400000E+00, 3.99000000E+00,
     4.65880000E+00, 4.12780000E+00, 3.81250000E+00, 3.61890000E+00, 3.48510000E+00, 3.38590000E+00,
     4.65310000E+00, 4.31340000E+00, 4.22580000E+00, 4.13090000E+00, 4.06920000E+00, 4.09440000E+00,
     3.98500000E+00, 3.94160000E+00, 3.91120000E+00, 3.88730000E+00, 3.87360000E+00, 3.84730000E+00,
     4.60270000E+00, 4.15380000E+00, 3.89940000E+00, 3.74190000E+00, 3.63560000E+00, 3.55480000E+00,
     4.83530000E+00, 4.54130000E+00, 4.38910000E+00, 4.34160000E+00, 4.32430000E+00, 4.27530000E+00,
     4.20530000E+00, 4.17900000E+00, 4.16850000E+00, 4.15850000E+00, 4.15360000E+00, 4.05790000E+00,
     4.19800000E+00, 4.45640000E+00, 4.21920000E+00, 4.05280000E+00, 3.94890000E+00, 3.86420000E+00,
     5.05670000E+00, 5.04540000E+00, 4.70040000E+00, 4.58020000E+00, 4.54990000E+00, 4.52570000E+00,
     4.50320000E+00, 4.48110000E+00, 4.46550000E+00, 4.16630000E+00, 4.21340000E+00, 4.52740000E+00,
     4.40860000E+00, 4.39360000E+00, 4.38320000E+00, 4.37500000E+00, 4.23430000E+00, 4.79360000E+00,
     4.70910000E+00, 4.63180000E+00, 4.55350000E+00, 4.46980000E+00, 4.43410000E+00, 4.33040000E+00,
     4.21580000E+00, 4.04200000E+00, 4.23450000E+00, 4.52400000E+00, 4.27650000E+00, 4.11650000E+00,
     4.02220000E+00, 3.93310000E+00, 5.06900000E+00, 4.76930000E+00, 4.61970000E+00, 4.47330000E+00,
     4.43200000E+00, 4.41620000E+00, 4.39280000E+00, 4.48690000E+00},
    {3.06300000E+00, 3.32710000E+00, 4.04320000E+00, 4.00460000E+00, 4.15550000E+00, 3.74260000E+00,
     3.51300000E+00, 3.51740000E+00, 3.28840000E+00, 3.13780000E+00, 4.18940000E+00, 4.23210000E+00,
     4.17250000E+00, 4.18330000E+00, 3.89290000E+00, 4.05440000E+00, 3.81180000E+00, 3.64140000E+00,
     4.63730000E+00, 4.62680000E+00, 4.47500000E+00, 4.41340000E+00, 4.34580000E+00, 3.85820000E+00,
     4.25830000E+00, 4.18980000E+00, 4.15620000E+00, 4.11910000E+00, 4.10690000E+00, 4.06390000E+00,
     4.12570000E+00, 4.19740000E+00, 3.95320000E+00, 4.17940000E+00, 3.96600000E+00, 3.81300000E+00,
     4.81600000E+00, 4.82720000E+00, 4.62940000E+00, 4.58400000E+00, 4.07700000E+00, 4.00880000E+00,
     3.91030000E+00, 3.85360000E+00, 3.83240000E+00, 3.79950000E+00, 3.78260000E+00, 4.22940000E+00,
     4.33800000E+00, 4.43520000E+00, 4.19330000E+00, 4.45800000E+00, 4.25540000E+00, 4.10720000E+00,
     5.04540000E+00, 5.18140000E+00, 4.86040000E+00, 4.91250000E+00, 4.88250000E+00, 4.85680000E+00,
     4.83560000E+00, 4.81350000E+00, 4.79740000E+00, 4.53090000E+00, 4.73080000E+00, 4.66820000E+00,
     4.70970000E+00, 4.68710000E+00, 4.67280000E+00, 4.66170000E+00, 4.45830000E+00, 4.74100000E+00,
     4.71560000E+00, 4.65590000E+00, 4.60130000E+00, 4.54880000E+00, 4.05250000E+00, 3.92420000E+00,
     3.78020000E+00, 4.17590000E+00, 4.36630000E+00, 4.48340000E+00, 4.23120000E+00, 4.52320000E+00,
     4.32070000E+00, 4.16550000E+00, 5.04460000E+00, 5.06490000E+00, 4.86540000E+00, 4.61570000E+00,
     4.67060000E+00, 4.65420000E+00, 4.63080000E+00, 4.77920000E+00},
    {3.06320000E+00, 3.26620000E+00, 3.64320000E+00, 3.80880000E+00, 3.79100000E+00, 3.73810000E+00,
     3.50930000E+00, 3.51550000E+00, 3.30470000E+00, 3.16810000E+00, 3.78710000E+00, 3.99240000E+00,
     4.06370000E+00, 4.13820000E+00, 3.85910000E+00, 4.01640000E+00, 3.78780000E+00, 3.63160000E+00,
     4.17410000E+00, 4.31660000E+00, 4.23950000E+00, 4.18310000E+00, 4.11070000E+00, 3.58570000E+00,
     4.02700000E+00, 3.96760000E+00, 3.94630000E+00, 3.91500000E+00, 3.90210000E+00, 3.87080000E+00,
     4.02400000E+00, 4.15510000E+00, 3.91080000E+00, 4.13370000E+00, 3.92890000E+00, 3.78730000E+00,
     4.36660000E+00, 4.50800000E+00, 4.42320000E+00, 4.31550000E+00, 3.84610000E+00, 3.80070000E+00,
     3.69910000E+00, 3.64470000E+00, 3.63080000E+00, 3.59590000E+00, 3.57490000E+00, 4.03590000E+00,
     4.31240000E+00, 4.35390000E+00, 4.11220000E+00, 4.37720000E+00, 4.17850000E+00, 4.03860000E+00,
     4.70040000E+00, 4.86040000E+00, 4.62610000E+00, 4.58610000E+00, 4.56010000E+00, 4.53530000E+00,
     4.51560000E+00, 4.49590000E+00, 4.48130000E+00, 4.34810000E+00, 4.37870000E+00, 4.33440000E+00,
     4.40640000E+00, 4.38820000E+00, 4.37590000E+00, 4.36440000E+00, 4.31150000E+00, 4.54610000E+00,
     4.51990000E+00, 4.45330000E+00, 4.39610000E+00, 4.31680000E+00, 3.90110000E+00, 3.77930000E+00,
     3.64310000E+00, 4.00780000E+00, 4.32570000E+00, 4.15700000E+00, 4.14010000E+00, 4.43570000E+00,
     4.23470000E+00, 4.08790000E+00, 4.61860000E+00, 4.73650000E+00, 4.66330000E+00, 4.45800000E+00,
     4.52730000E+00, 4.50050000E+00, 4.47990000E+00, 4.49180000E+00},
    {2.94550000E+00, 3.24700000E+00, 3.61080000E+00, 3.85220000E+00, 3.66250000E+00, 3.65980000E+00,
     3.44110000E+00, 3.46600000E+00, 3.24150000E+00, 3.09440000E+00, 3.75140000E+00, 4.03970000E+00,
     3.92310000E+00, 4.05610000E+00, 3.78600000E+00, 3.98450000E+00, 3.74540000E+00, 3.58020000E+00,
     4.13660000E+00, 4.35810000E+00, 4.23510000E+00, 4.20110000E+00, 4.14020000E+00, 3.53810000E+00,
     4.06530000E+00, 4.00930000E+00, 3.98830000E+00, 3.95700000E+00, 3.94290000E+00, 3.91120000E+00,
     3.87280000E+00, 4.06820000E+00, 3.83510000E+00, 4.10540000E+00, 3.89280000E+00, 3.74450000E+00,
     4.34150000E+00, 4.54970000E+00, 4.38330000E+00, 4.31220000E+00, 3.80510000E+00, 3.75830000E+00,
     3.66220000E+00, 3.61080000E+00, 3.59710000E+00, 3.56280000E+00, 3.54080000E+00, 4.07800000E+00,
     4.07270000E+00, 4.28360000E+00, 4.05530000E+00, 4.36470000E+00, 4.16220000E+00, 4.01780000E+00,
     4.58020000E+00, 4.91250000E+00, 4.58610000E+00, 4.62010000E+00, 4.59500000E+00, 4.57330000E+00,
     4.55440000E+00, 4.53510000E+00, 4.52090000E+00, 4.29110000E+00, 4.38940000E+00, 4.36390000E+00,
     4.44880000E+00, 4.43160000E+00, 4.41950000E+00, 4.40830000E+00, 4.24570000E+00, 4.56620000E+00,
     4.55420000E+00, 4.49560000E+00, 4.44230000E+00, 4.36460000E+00, 3.89450000E+00, 3.76230000E+00,
     3.61290000E+00, 4.04500000E+00, 4.08690000E+00, 4.32040000E+00, 4.08150000E+00, 4.42260000E+00,
     4.21910000E+00, 4.06810000E+00, 4.60780000E+00, 4.77610000E+00, 4.58780000E+00, 4.41940000E+00,
     4.45810000E+00, 4.44440000E+00, 4.42440000E+00, 4.53420000E+00},
    {2.92440000E+00, 3.22410000E+00, 3.58480000E+00, 3.82930000E+00, 3.63950000E+00, 3.64000000E+00,
     3.42040000E+00, 3.44990000E+00, 3.22530000E+00, 3.07790000E+00, 3.72570000E+00, 4.01700000E+00,
     3.90030000E+00, 4.03720000E+00, 3.76530000E+00, 3.96720000E+00, 3.72830000E+00, 3.56300000E+00,
     4.10920000E+00, 4.33470000E+00, 4.21170000E+00, 4.17930000E+00, 4.11790000E+00, 3.51390000E+00,
     4.04260000E+00, 3.98670000E+00, 3.96610000E+00, 3.93450000E+00, 3.92000000E+00, 3.88830000E+00,
     3.84980000E+00, 4.04960000E+00, 3.81450000E+00, 4.08810000E+00, 3.87560000E+00, 3.72710000E+00,
     4.31280000E+00, 4.52420000E+00, 4.35780000E+00, 4.28700000E+00, 3.77960000E+00, 3.73180000E+00,
     3.63640000E+00, 3.58540000E+00, 3.57260000E+00, 3.53780000E+00, 3.51550000E+00, 4.05270000E+00,
     4.04780000E+00, 4.26300000E+00, 4.03220000E+00, 4.34490000E+00, 4.14210000E+00, 3.99750000E+00,
     4.54990000E+00, 4.88250000E+00, 4.56010000E+00, 4.59500000E+00, 4.57020000E+00, 4.54850000E+00,
     4.52970000E+00, 4.51050000E+00, 4.49640000E+00, 4.26710000E+00, 4.36490000E+00, 4.33840000E+00,
     4.42430000E+00, 4.40730000E+00, 4.39520000E+00, 4.38400000E+00, 4.22130000E+00, 4.53400000E+00,
     4.52300000E+00, 4.46410000E+00, 4.41090000E+00, 4.33380000E+00, 3.86350000E+00, 3.73270000E+00,
     3.58530000E+00, 4.01890000E+00, 4.06120000E+00, 4.29930000E+00, 4.05800000E+00, 4.40310000E+00,
     4.19900000E+00, 4.04790000E+00, 4.57800000E+00, 4.74980000E+00, 4.56400000E+00, 4.39540000E+00,
     4.43320000E+00, 4.41960000E+00, 4.39960000E+00, 4.50900000E+00},
    {2.90460000E+00, 3.20440000E+00, 3.56210000E+00, 3.80780000E+00, 3.61850000E+00, 3.62200000E+00,
     3.40190000E+00, 3.43590000E+00, 3.21100000E+00, 3.06350000E+00, 3.70370000E+00, 3.99580000E+00,
     3.87920000E+00, 4.01940000E+00, 3.74600000E+00, 3.95170000E+00, 3.71280000E+00, 3.54740000E+00,
     4.08720000E+00, 4.31380000E+00, 4.19060000E+00, 4.15930000E+00, 4.09730000E+00, 3.49190000E+00,
     4.02160000E+00, 3.96570000E+00, 3.94540000E+00, 3.91340000E+00, 3.89860000E+00, 3.86690000E+00,
     3.82890000E+00, 4.03230000E+00, 3.79540000E+00, 4.07250000E+00, 3.85980000E+00, 3.71130000E+00,
     4.28960000E+00, 4.50210000E+00, 4.33250000E+00, 4.26450000E+00, 3.75710000E+00, 3.70830000E+00,
     3.61360000E+00, 3.56280000E+00, 3.55070000E+00, 3.51550000E+00, 3.49290000E+00, 4.02970000E+00,
     4.02340000E+00, 4.24420000E+00, 4.01120000E+00, 4.32740000E+00, 4.12400000E+00, 3.97930000E+00,
     4.52570000E+00, 4.85680000E+00, 4.53530000E+00, 4.57330000E+00, 4.54850000E+00, 4.52710000E+00,
     4.50830000E+00, 4.48910000E+00, 4.47500000E+00, 4.24150000E+00, 4.34410000E+00, 4.31620000E+00,
     4.40290000E+00, 4.38580000E+00, 4.37370000E+00, 4.36250000E+00, 4.19450000E+00, 4.50590000E+00,
     4.49590000E+00, 4.43660000E+00, 4.38350000E+00, 4.30670000E+00, 3.83680000E+00, 3.70710000E+00,
     3.56100000E+00, 3.99520000E+00, 4.03640000E+00, 4.28040000E+00, 4.03690000E+00, 4.38600000E+00,
     4.18110000E+00, 4.02990000E+00, 4.55380000E+00, 4.72720000E+00, 4.54220000E+00, 4.37390000E+00,
     4.40640000E+00, 4.39330000E+00, 4.37370000E+00, 4.48680000E+00},
    {2.88780000E+00, 3.18900000E+00, 3.54120000E+00, 3.79080000E+00, 3.59740000E+00, 3.60780000E+00,
     3.38710000E+00, 3.42430000E+00, 3.19920000E+00, 3.05130000E+00, 3.68310000E+00, 3.97840000E+00,
     3.85790000E+00, 4.00490000E+00, 3.73040000E+00, 3.93920000E+00, 3.70020000E+00, 3.53470000E+00,
     4.06570000E+00, 4.29550000E+00, 4.17050000E+00, 4.14240000E+00, 4.08000000E+00, 3.47170000E+00,
     4.00430000E+00, 3.94850000E+00, 3.92860000E+00, 3.89650000E+00, 3.88150000E+00, 3.85000000E+00,
     3.80730000E+00, 4.01800000E+00, 3.77960000E+00, 4.05980000E+00, 3.84700000E+00, 3.69830000E+00,
     4.26780000E+00, 4.48300000E+00, 4.31320000E+00, 4.24440000E+00, 3.73700000E+00, 3.68760000E+00,
     3.59350000E+00, 3.54280000E+00, 3.53140000E+00, 3.49580000E+00, 3.47300000E+00, 4.01170000E+00,
     4.00430000E+00, 4.22870000E+00, 3.99390000E+00, 4.31340000E+00, 4.10960000E+00, 3.96460000E+00,
     4.50320000E+00, 4.83560000E+00, 4.51560000E+00, 4.55440000E+00, 4.52970000E+00, 4.50830000E+00,
     4.48960000E+00, 4.47050000E+00, 4.45650000E+00, 4.22300000E+00, 4.32570000E+00, 4.29720000E+00,
     4.38420000E+00, 4.36720000E+00, 4.35510000E+00, 4.34390000E+00, 4.17560000E+00, 4.48320000E+00,
     4.47500000E+00, 4.41550000E+00, 4.36250000E+00, 4.28600000E+00, 3.81530000E+00, 3.68600000E+00,
     3.54060000E+00, 3.97700000E+00, 4.01700000E+00, 4.26470000E+00, 4.01940000E+00, 4.37210000E+00,
     4.16660000E+00, 4.01520000E+00, 4.53190000E+00, 4.70760000E+00, 4.52310000E+00, 4.35310000E+00,
     4.38730000E+00, 4.37410000E+00, 4.35470000E+00, 4.46800000E+00},
    {2.87090000E+00, 3.17370000E+00, 3.51990000E+00, 3.77340000E+00, 3.58020000E+00, 3.59340000E+00,
     3.37240000E+00, 3.41280000E+00, 3.18770000E+00, 3.03960000E+00, 3.66240000E+00, 3.96080000E+00,
     3.83970000E+00, 3.98930000E+00, 3.71450000E+00, 3.92660000E+00, 3.68770000E+00, 3.52220000E+00,
     4.04480000E+00, 4.27710000E+00, 4.15230000E+00, 4.12470000E+00, 4.06260000E+00, 3.45300000E+00,
     3.98660000E+00, 3.93100000E+00, 3.91150000E+00, 3.87920000E+00, 3.86410000E+00, 3.83260000E+00,
     3.78920000E+00, 4.00250000E+00, 3.76360000E+00, 4.04710000E+00, 3.83430000E+00, 3.68540000E+00,
     4.24640000E+00, 4.46350000E+00, 4.29390000E+00, 4.22520000E+00, 3.71690000E+00, 3.66750000E+00,
     3.57390000E+00, 3.52350000E+00, 3.51260000E+00, 3.47680000E+00, 3.45370000E+00, 3.99320000E+00,
     3.98540000E+00, 4.21230000E+00, 3.97650000E+00, 4.29920000E+00, 4.09510000E+00, 3.95000000E+00,
     4.48110000E+00, 4.81350000E+00, 4.49590000E+00, 4.53510000E+00, 4.51050000E+00, 4.48910000E+00,
     4.47050000E+00, 4.45150000E+00, 4.43750000E+00, 4.20470000E+00, 4.30730000E+00, 4.27790000E+00,
     4.36550000E+00, 4.34850000E+00, 4.33640000E+00, 4.32530000E+00, 4.15690000E+00, 4.46040000E+00,
     4.45290000E+00, 4.39360000E+00, 4.34070000E+00, 4.26450000E+00, 3.79360000E+00, 3.66500000E+00,
     3.52040000E+00, 3.95830000E+00, 3.99780000E+00, 4.24810000E+00, 4.00170000E+00, 4.35800000E+00,
     4.15210000E+00, 4.00060000E+00, 4.51010000E+00, 4.68770000E+00, 4.50420000E+00, 4.33430000E+00,
     4.36840000E+00, 4.35520000E+00, 4.33580000E+00, 4.44860000E+00},
    {2.85680000E+00, 3.16080000E+00, 3.50500000E+00, 3.75980000E+00, 3.56650000E+00, 3.58030000E+00,
     3.36010000E+00, 3.40310000E+00, 3.17790000E+00, 3.02960000E+00, 3.64790000E+00, 3.94710000E+00,
     3.82620000E+00, 3.97730000E+00, 3.70150000E+00, 3.91620000E+00, 3.67710000E+00, 3.51150000E+00,
     4.03060000E+00, 4.26340000E+00, 4.13850000E+00, 4.11160000E+00, 4.04890000E+00, 3.43660000E+00,
     3.97320000E+00, 3.91760000E+00, 3.89830000E+00, 3.86590000E+00, 3.85070000E+00, 3.81910000E+00,
     3.77570000E+00, 3.99070000E+00, 3.75060000E+00, 4.03650000E+00, 3.82350000E+00, 3.67450000E+00,
     4.23140000E+00, 4.44900000E+00, 4.27920000E+00, 4.21050000E+00, 3.70030000E+00, 3.65100000E+00,
     3.55780000E+00, 3.50750000E+00, 3.49710000E+00, 3.46090000E+00, 3.43770000E+00, 3.97880000E+00,
     3.97120000E+00, 4.19970000E+00, 3.96240000E+00, 4.28770000E+00, 4.08310000E+00, 3.93780000E+00,
     4.46550000E+00, 4.79740000E+00, 4.48130000E+00, 4.52090000E+00, 4.49640000E+00, 4.47500000E+00,
     4.45650000E+00, 4.43750000E+00, 4.42340000E+00, 4.19080000E+00, 4.29410000E+00, 4.26360000E+00,
     4.35140000E+00, 4.33440000E+00, 4.32230000E+00, 4.31120000E+00, 4.14240000E+00, 4.44290000E+00,
     4.43610000E+00, 4.37640000E+00, 4.32370000E+00, 4.24780000E+00, 3.77580000E+00, 3.64760000E+00,
     3.50360000E+00, 3.94340000E+00, 3.98340000E+00, 4.23540000E+00, 3.98740000E+00, 4.34660000E+00,
     4.14010000E+00, 3.98830000E+00, 4.49450000E+00, 4.67300000E+00, 4.49010000E+00, 4.31960000E+00,
     4.35370000E+00, 4.34060000E+00, 4.32120000E+00, 4.43410000E+00},
    {2.67980000E+00, 3.01510000E+00, 3.25860000E+00, 3.52920000E+00, 3.53910000E+00, 3.49020000E+00,
     3.28870000E+00, 3.33220000E+00, 3.12280000E+00, 2.98880000E+00, 3.40120000E+00, 3.71450000E+00,
     3.78300000E+00, 3.66650000E+00, 3.58980000E+00, 3.80770000E+00, 3.58100000E+00, 3.42650000E+00,
     3.77260000E+00, 4.03070000E+00, 3.97630000E+00, 3.88900000E+00, 3.84890000E+00, 3.27060000E+00,
     3.75950000E+00, 3.69840000E+00, 3.67720000E+00, 3.64280000E+00, 3.62430000E+00, 3.59510000E+00,
     3.74970000E+00, 3.67750000E+00, 3.63640000E+00, 3.92030000E+00, 3.71570000E+00, 3.57460000E+00,
     3.94940000E+00, 4.20760000E+00, 4.15630000E+00, 4.05080000E+00, 3.53290000E+00, 3.47800000E+00,
     3.37310000E+00, 3.31260000E+00, 3.28460000E+00, 3.24260000E+00, 3.21350000E+00, 3.74910000E+00,
     3.90060000E+00, 3.83320000E+00, 3.80290000E+00, 4.14360000E+00, 3.94070000E+00, 3.79980000E+00,
     4.16630000E+00, 4.53090000E+00, 4.34810000E+00, 4.29110000E+00, 4.26710000E+00, 4.24150000E+00,
     4.22300000E+00, 4.20470000E+00, 4.19080000E+00, 4.12430000E+00, 4.12520000E+00, 4.02530000E+00,
     4.11620000E+00, 4.09840000E+00, 4.08610000E+00, 4.07450000E+00, 4.06200000E+00, 4.23460000E+00,
     4.17740000E+00, 4.13020000E+00, 4.08630000E+00, 4.00670000E+00, 3.49440000E+00, 3.38490000E+00,
     3.26790000E+00, 3.72170000E+00, 3.91370000E+00, 3.86260000E+00, 3.83120000E+00, 4.20360000E+00,
     3.99700000E+00, 3.85000000E+00, 4.19120000E+00, 4.42740000E+00, 4.32820000E+00, 4.21400000E+00,
     4.27280000E+00, 4.24840000E+00, 4.22750000E+00, 4.20230000E+00},
    {2.51890000E+00, 2.97030000E+00, 3.30630000E+00, 3.62350000E+00, 3.45170000E+00, 3.39890000E+00,
     3.21070000E+00, 3.24340000E+00, 3.00940000E+00, 2.85800000E+00, 3.42530000E+00, 3.81570000E+00,
     3.72580000E+00, 3.61320000E+00, 3.52970000E+00, 3.75660000E+00, 3.50950000E+00, 3.33680000E+00,
     3.78900000E+00, 4.12980000E+00, 4.01900000E+00, 3.95730000E+00, 3.92370000E+00, 3.26770000E+00,
     3.84800000E+00, 3.81570000E+00, 3.76560000E+00, 3.73170000E+00, 3.71260000E+00, 3.68140000E+00,
     3.67930000E+00, 3.62180000E+00, 3.57880000E+00, 3.87630000E+00, 3.65720000E+00, 3.50220000E+00,
     3.97370000E+00, 4.32550000E+00, 4.18280000E+00, 4.11580000E+00, 3.50780000E+00, 3.45950000E+00,
     3.36000000E+00, 3.30880000E+00, 3.25750000E+00, 3.21640000E+00, 3.18560000E+00, 3.85220000E+00,
     3.86650000E+00, 3.80750000E+00, 3.77720000E+00, 4.13910000E+00, 3.92960000E+00, 3.77720000E+00,
     4.21340000E+00, 4.73080000E+00, 4.37870000E+00, 4.38940000E+00, 4.36490000E+00, 4.34410000E+00,
     4.32570000E+00, 4.30730000E+00, 4.29410000E+00, 4.12520000E+00, 4.24270000E+00, 4.11680000E+00,
     4.22050000E+00, 4.20360000E+00, 4.19110000E+00, 4.18070000E+00, 4.04940000E+00, 4.42040000E+00,
     4.37740000E+00, 4.33980000E+00, 4.28350000E+00, 4.23490000E+00, 3.48730000E+00, 3.35340000E+00,
     3.21620000E+00, 3.82280000E+00, 3.88250000E+00, 3.84480000E+00, 3.81200000E+00, 4.20370000E+00,
     3.99430000E+00, 3.83590000E+00, 4.23150000E+00, 4.54730000E+00, 4.39780000E+00, 4.23390000E+00,
     4.25490000E+00, 4.24130000E+00, 4.22160000E+00, 4.31220000E+00},
    {3.04810000E+00, 2.95840000E+00, 3.69190000E+00, 3.59900000E+00, 3.88810000E+00, 3.42090000E+00,
     3.16060000E+00, 3.19380000E+00, 2.99750000E+00, 2.86460000E+00, 3.81380000E+00, 3.79350000E+00,
     3.70810000E+00, 3.91550000E+00, 3.59100000E+00, 3.48080000E+00, 3.48860000E+00, 3.33970000E+00,
     4.13360000E+00, 4.11220000E+00, 3.98880000E+00, 3.95430000E+00, 3.89170000E+00, 3.58940000E+00,
     3.81310000E+00, 3.76350000E+00, 3.74190000E+00, 3.70710000E+00, 3.68800000E+00, 3.65740000E+00,
     3.65460000E+00, 3.93750000E+00, 3.65790000E+00, 3.58700000E+00, 3.63610000E+00, 3.50390000E+00,
     4.31490000E+00, 4.29780000E+00, 4.13210000E+00, 4.12980000E+00, 3.81640000E+00, 3.76800000E+00,
     3.71540000E+00, 3.68580000E+00, 3.67090000E+00, 3.66660000E+00, 3.65170000E+00, 3.81740000E+00,
     3.86080000E+00, 4.18050000E+00, 3.91020000E+00, 3.83940000E+00, 3.89680000E+00, 3.76730000E+00,
     4.52740000E+00, 4.66820000E+00, 4.33440000E+00, 4.36390000E+00, 4.33840000E+00, 4.31620000E+00,
     4.29720000E+00, 4.27790000E+00, 4.26360000E+00, 4.02530000E+00, 4.11680000E+00, 4.15410000E+00,
     4.19530000E+00, 4.17910000E+00, 4.16760000E+00, 4.15780000E+00, 3.99530000E+00, 4.31190000E+00,
     4.29630000E+00, 4.22830000E+00, 4.16750000E+00, 4.09580000E+00, 3.90400000E+00, 3.82160000E+00,
     3.70680000E+00, 3.78260000E+00, 3.87580000E+00, 4.22670000E+00, 3.94540000E+00, 3.88670000E+00,
     3.95920000E+00, 3.82490000E+00, 4.55340000E+00, 4.51690000E+00, 4.34830000E+00, 4.17380000E+00,
     4.20320000E+00, 4.19070000E+00, 4.16760000E+00, 4.27100000E+00},
    {2.81360000E+00, 3.09510000E+00, 3.46350000E+00, 3.68750000E+00, 3.49870000E+00, 3.51830000E+00,
     3.29370000E+00, 3.35800000E+00, 3.13250000E+00, 2.98320000E+00, 3.60780000E+00, 3.87570000E+00,
     3.76160000E+00, 3.92220000E+00, 3.63700000E+00, 3.86470000E+00, 3.62560000E+00, 3.45950000E+00,
     3.98740000E+00, 4.19380000E+00, 4.06790000E+00, 4.04300000E+00, 3.97810000E+00, 3.38860000E+00,
     3.90080000E+00, 3.84630000E+00, 3.82880000E+00, 3.79500000E+00, 3.77900000E+00, 3.74720000E+00,
     3.71170000E+00, 3.93710000E+00, 3.68730000E+00, 3.98460000E+00, 3.77090000E+00, 3.62100000E+00,
     4.18120000E+00, 4.37500000E+00, 4.20440000E+00, 4.13400000E+00, 3.64590000E+00, 3.59290000E+00,
     3.50360000E+00, 3.45770000E+00, 3.45280000E+00, 3.41460000E+00, 3.39040000E+00, 3.90140000E+00,
     3.90310000E+00, 4.14430000E+00, 3.89610000E+00, 4.22950000E+00, 4.02270000E+00, 3.87630000E+00,
     4.40860000E+00, 4.70970000E+00, 4.40640000E+00, 4.44880000E+00, 4.42430000E+00, 4.40290000E+00,
     4.38420000E+00, 4.36550000E+00, 4.35140000E+00, 4.11620000E+00, 4.22050000E+00, 4.19530000E+00,
     4.27940000E+00, 4.26220000E+00, 4.25010000E+00, 4.23900000E+00, 4.06940000E+00, 4.34500000E+00,
     4.34060000E+00, 4.27960000E+00, 4.22720000E+00, 4.15430000E+00, 3.71100000E+00, 3.58700000E+00,
     3.44830000E+00, 3.86400000E+00, 3.91430000E+00, 4.17990000E+00, 3.92100000E+00, 4.28950000E+00,
     4.08000000E+00, 3.92690000E+00, 4.43730000E+00, 4.59750000E+00, 4.42020000E+00, 4.24580000E+00,
     4.27940000E+00, 4.26560000E+00, 4.24650000E+00, 4.35870000E+00},
    {2.80320000E+00, 3.08050000E+00, 3.45190000E+00, 3.67000000E+00, 3.48270000E+00, 3.50500000E+00,
     3.27990000E+00, 3.34820000E+00, 3.12330000E+00, 2.97470000E+00, 3.59710000E+00, 3.85860000E+00,
     3.74610000E+00, 3.91000000E+00, 3.62280000E+00, 3.85350000E+00, 3.61470000E+00, 3.44900000E+00,
     3.97640000E+00, 4.17730000E+00, 4.05110000E+00, 4.02700000E+00, 3.96140000E+00, 3.37540000E+00,
     3.88360000E+00, 3.82910000E+00, 3.81210000E+00, 3.77800000E+00, 3.76190000E+00, 3.73000000E+00,
     3.69650000E+00, 3.92530000E+00, 3.67340000E+00, 3.97330000E+00, 3.75970000E+00, 3.60990000E+00,
     4.16830000E+00, 4.35720000E+00, 4.18620000E+00, 4.11530000E+00, 3.63120000E+00, 3.57720000E+00,
     3.48810000E+00, 3.44290000E+00, 3.43950000E+00, 3.40090000E+00, 3.37660000E+00, 3.88270000E+00,
     3.88680000E+00, 4.13160000E+00, 3.88070000E+00, 4.21640000E+00, 4.00920000E+00, 3.86270000E+00,
     4.39360000E+00, 4.68710000E+00, 4.38820000E+00, 4.43160000E+00, 4.40730000E+00, 4.38580000E+00,
     4.36720000E+00, 4.34850000E+00, 4.33440000E+00, 4.09840000E+00, 4.20360000E+00, 4.17910000E+00,
     4.26220000E+00, 4.24500000E+00, 4.23290000E+00, 4.22180000E+00, 4.05160000E+00, 4.31930000E+00,
     4.31590000E+00, 4.25470000E+00, 4.20250000E+00, 4.13020000E+00, 3.69220000E+00, 3.56950000E+00,
     3.43230000E+00, 3.84460000E+00, 3.89760000E+00, 4.16700000E+00, 3.90550000E+00, 4.27660000E+00,
     4.06640000E+00, 3.91330000E+00, 4.42240000E+00, 4.57930000E+00, 4.40390000E+00, 4.22780000E+00,
     4.26130000E+00, 4.24740000E+00, 4.22830000E+00, 4.34070000E+00},
    {2.79670000E+00, 3.06890000E+00, 3.44450000E+00, 3.65810000E+00, 3.47170000E+00, 3.49510000E+00,
     3.26940000E+00, 3.33970000E+00, 3.11470000E+00, 2.96610000E+00, 3.58980000E+00, 3.84680000E+00,
     3.73580000E+00, 3.90140000E+00, 3.61290000E+00, 3.84430000E+00, 3.60540000E+00, 3.43960000E+00,
     3.96830000E+00, 4.16560000E+00, 4.03940000E+00, 4.01580000E+00, 3.94980000E+00, 3.36770000E+00,
     3.87180000E+00, 3.81640000E+00, 3.80050000E+00, 3.76620000E+00, 3.75000000E+00, 3.71810000E+00,
     3.68630000E+00, 3.91700000E+00, 3.66370000E+00, 3.96410000E+00, 3.75030000E+00, 3.60040000E+00,
     4.15900000E+00, 4.34480000E+00, 4.17390000E+00, 4.10290000E+00, 3.62240000E+00, 3.56770000E+00,
     3.47850000E+00, 3.43140000E+00, 3.43130000E+00, 3.39230000E+00, 3.36800000E+00, 3.86980000E+00,
     3.87580000E+00, 4.12290000E+00, 3.87040000E+00, 4.20630000E+00, 3.99870000E+00, 3.85190000E+00,
     4.38320000E+00, 4.67280000E+00, 4.37590000E+00, 4.41950000E+00, 4.39520000E+00, 4.37370000E+00,
     4.35510000E+00, 4.33640000E+00, 4.32230000E+00, 4.08610000E+00, 4.19110000E+00, 4.16760000E+00,
     4.25010000E+00, 4.23290000E+00, 4.22080000E+00, 4.20970000E+00, 4.03960000E+00, 4.30350000E+00,
     4.30060000E+00, 4.23910000E+00, 4.18690000E+00, 4.11410000E+00, 3.67990000E+00, 3.55840000E+00,
     3.42210000E+00, 3.83140000E+00, 3.88640000E+00, 3.87380000E+00, 3.89510000E+00, 4.26620000E+00,
     4.05590000E+00, 3.90250000E+00, 4.41200000E+00, 4.56670000E+00, 4.39260000E+00, 4.21580000E+00,
     4.24910000E+00, 4.23520000E+00, 4.21610000E+00, 4.32810000E+00},
    {2.78970000E+00, 3.06360000E+00, 3.43440000E+00, 3.64800000E+00, 3.46260000E+00, 3.48920000E+00,
     3.26260000E+00, 3.33440000E+00, 3.10880000E+00, 2.95970000E+00, 3.58040000E+00, 3.83590000E+00,
     3.72510000E+00, 3.89400000E+00, 3.60470000E+00, 3.83750000E+00, 3.59900000E+00, 3.43290000E+00,
     3.95970000E+00, 4.15420000E+00, 4.02780000E+00, 4.00480000E+00, 3.93900000E+00, 3.35710000E+00,
     3.86080000E+00, 3.80560000E+00, 3.78990000E+00, 3.75600000E+00, 3.74000000E+00, 3.70810000E+00,
     3.67580000E+00, 3.90950000E+00, 3.65520000E+00, 3.95720000E+00, 3.74360000E+00, 3.59330000E+00,
     4.15080000E+00, 4.33370000E+00, 4.16240000E+00, 4.09160000E+00, 3.61260000E+00, 3.55820000E+00,
     3.46840000E+00, 3.42120000E+00, 3.42070000E+00, 3.38290000E+00, 3.35860000E+00, 3.86040000E+00,
     3.86580000E+00, 4.11560000E+00, 3.86200000E+00, 4.19940000E+00, 3.99170000E+00, 3.84460000E+00,
     4.37500000E+00, 4.66170000E+00, 4.36440000E+00, 4.40830000E+00, 4.38400000E+00, 4.36250000E+00,
     4.34390000E+00, 4.32530000E+00, 4.31120000E+00, 4.07450000E+00, 4.18070000E+00, 4.15780000E+00,
     4.23900000E+00, 4.22180000E+00, 4.20970000E+00, 4.19860000E+00, 4.02800000E+00, 4.29330000E+00,
     4.29100000E+00, 4.22960000E+00, 4.17740000E+00, 4.10480000E+00, 3.67240000E+00, 3.55080000E+00,
     3.41380000E+00, 3.82250000E+00, 3.87680000E+00, 3.86430000E+00, 3.88660000E+00, 4.25980000E+00,
     4.04880000E+00, 3.89480000E+00, 4.40400000E+00, 4.55590000E+00, 4.38070000E+00, 4.20390000E+00,
     4.23750000E+00, 4.22360000E+00, 4.20450000E+00, 4.31740000E+00},
    {2.83950000E+00, 3.00810000E+00, 3.31710000E+00, 3.48780000E+00, 3.53600000E+00, 3.51450000E+00,
     3.28090000E+00, 3.33070000E+00, 3.12600000E+00, 2.99400000E+00, 3.47410000E+00, 3.66750000E+00,
     3.78320000E+00, 3.67870000E+00, 3.61560000E+00, 3.80410000E+00, 3.58130000E+00, 3.43010000E+00,
     3.84800000E+00, 3.98490000E+00, 3.93140000E+00, 3.84050000E+00, 3.80290000E+00, 3.29620000E+00,
     3.71040000E+00, 3.65150000E+00, 3.63780000E+00, 3.60200000E+00, 3.58490000E+00, 3.55500000E+00,
     3.74940000E+00, 3.68930000E+00, 3.66660000E+00, 3.91700000E+00, 3.71500000E+00, 3.57600000E+00,
     4.02680000E+00, 4.15960000E+00, 4.11070000E+00, 3.99950000E+00, 3.55740000E+00, 3.51030000E+00,
     3.41630000E+00, 3.36550000E+00, 3.36770000E+00, 3.32430000E+00, 3.29750000E+00, 3.70710000E+00,
     3.90470000E+00, 3.85140000E+00, 3.84220000E+00, 3.80220000E+00, 3.93230000E+00, 3.79320000E+00,
     4.23430000E+00, 4.45830000E+00, 4.31150000E+00, 4.24570000E+00, 4.22130000E+00, 4.19450000E+00,
     4.17560000E+00, 4.15690000E+00, 4.14240000E+00, 4.06200000E+00, 4.04940000E+00, 3.99530000E+00,
     4.06940000E+00, 4.05160000E+00, 4.03960000E+00, 4.02800000E+00, 4.01300000E+00, 4.15820000E+00,
     4.10080000E+00, 4.06990000E+00, 4.01260000E+00, 3.94100000E+00, 3.56220000E+00, 3.48560000E+00,
     3.36520000E+00, 3.68170000E+00, 3.91900000E+00, 3.87960000E+00, 3.86890000E+00, 3.84080000E+00,
     3.98820000E+00, 3.84220000E+00, 4.26340000E+00, 4.38040000E+00, 4.26490000E+00, 4.16580000E+00,
     4.23220000E+00, 4.20680000E+00, 4.18410000E+00, 4.14990000E+00},
    {2.90070000E+00, 2.96740000E+00, 3.81740000E+00, 3.58560000E+00, 3.64860000E+00, 3.53390000E+00,
     3.28320000E+00, 3.31540000E+00, 3.11440000E+00, 2.98660000E+00, 3.96180000E+00, 3.84300000E+00,
     3.99800000E+00, 3.81340000E+00, 3.66520000E+00, 3.79850000E+00, 3.57560000E+00, 3.42070000E+00,
     4.40610000E+00, 4.28170000E+00, 4.14770000E+00, 4.06160000E+00, 3.99790000E+00, 3.64920000E+00,
     3.88330000E+00, 3.80270000E+00, 3.76600000E+00, 3.71830000E+00, 3.69540000E+00, 3.65250000E+00,
     3.96690000E+00, 3.83710000E+00, 3.73250000E+00, 3.91600000E+00, 3.71560000E+00, 3.57140000E+00,
     4.60360000E+00, 4.46200000E+00, 4.30920000E+00, 4.21220000E+00, 3.84780000E+00, 3.75720000E+00,
     3.65970000E+00, 3.59690000E+00, 3.55750000E+00, 3.53860000E+00, 3.51530000E+00, 3.78180000E+00,
     4.13350000E+00, 4.01530000E+00, 3.91770000E+00, 3.86030000E+00, 3.93650000E+00, 3.79060000E+00,
     4.79360000E+00, 4.74100000E+00, 4.54610000E+00, 4.56620000E+00, 4.53400000E+00, 4.50590000E+00,
     4.48320000E+00, 4.46040000E+00, 4.44290000E+00, 4.23460000E+00, 4.42040000E+00, 4.31190000E+00,
     4.34500000E+00, 4.31930000E+00, 4.30350000E+00, 4.29330000E+00, 4.15820000E+00, 4.24500000E+00,
     4.15680000E+00, 4.10830000E+00, 4.04600000E+00, 3.95950000E+00, 3.60810000E+00, 3.55230000E+00,
     3.45760000E+00, 3.70680000E+00, 4.16130000E+00, 4.05750000E+00, 3.96030000E+00, 3.91690000E+00,
     4.00350000E+00, 3.85090000E+00, 4.77700000E+00, 4.69200000E+00, 4.61350000E+00, 4.35950000E+00,
     4.36650000E+00, 4.34100000E+00, 4.31350000E+00, 4.39400000E+00},
    {2.85590000E+00, 2.90500000E+00, 3.83250000E+00, 3.54420000E+00, 3.50770000E+00, 3.49050000E+00,
     3.23960000E+00, 3.27200000E+00, 3.07260000E+00, 2.94670000E+00, 3.96440000E+00, 3.80500000E+00,
     3.89810000E+00, 3.77620000E+00, 3.62160000E+00, 3.75310000E+00, 3.52970000E+00, 3.37420000E+00,
     4.38140000E+00, 4.28180000E+00, 4.10260000E+00, 4.02940000E+00, 3.96400000E+00, 3.62080000E+00,
     3.84640000E+00, 3.76480000E+00, 3.72810000E+00, 3.67900000E+00, 3.65420000E+00, 3.61170000E+00,
     3.86500000E+00, 3.80100000E+00, 3.68940000E+00, 3.87130000E+00, 3.66990000E+00, 3.52440000E+00,
     4.51510000E+00, 4.45170000E+00, 4.25380000E+00, 4.14830000E+00, 3.86410000E+00, 3.72440000E+00,
     3.62430000E+00, 3.55890000E+00, 3.51720000E+00, 3.49730000E+00, 3.47150000E+00, 3.73400000E+00,
     4.03160000E+00, 3.99580000E+00, 3.86870000E+00, 3.81150000E+00, 3.88620000E+00, 3.73790000E+00,
     4.70910000E+00, 4.71560000E+00, 4.51990000E+00, 4.55420000E+00, 4.52300000E+00, 4.49590000E+00,
     4.47500000E+00, 4.45290000E+00, 4.43610000E+00, 4.17740000E+00, 4.37740000E+00, 4.29630000E+00,
     4.34060000E+00, 4.31590000E+00, 4.30060000E+00, 4.29100000E+00, 4.10080000E+00, 4.15680000E+00,
     4.09800000E+00, 4.03190000E+00, 3.98150000E+00, 3.89410000E+00, 3.54260000E+00, 3.49340000E+00,
     3.40530000E+00, 3.65550000E+00, 4.05660000E+00, 4.03540000E+00, 3.91090000E+00, 3.86810000E+00,
     3.95390000E+00, 3.79900000E+00, 4.68900000E+00, 4.67310000E+00, 4.56050000E+00, 4.28570000E+00,
     4.30610000E+00, 4.28170000E+00, 4.25620000E+00, 4.38950000E+00},
    {2.81100000E+00, 2.85200000E+00, 3.74800000E+00, 3.51050000E+00, 3.43460000E+00, 3.34610000E+00,
     3.19710000E+00, 3.23260000E+00, 3.03290000E+00, 2.90700000E+00, 3.88230000E+00, 3.79280000E+00,
     3.82640000E+00, 3.70060000E+00, 3.57970000E+00, 3.71410000E+00, 3.48940000E+00, 3.33260000E+00,
     4.30480000E+00, 4.22170000E+00, 4.07860000E+00, 3.99000000E+00, 3.93570000E+00, 3.63310000E+00,
     3.83330000E+00, 3.73170000E+00, 3.69570000E+00, 3.64600000E+00, 3.61970000E+00, 3.57790000E+00,
     3.79090000E+00, 3.72570000E+00, 3.64760000E+00, 3.57290000E+00, 3.63040000E+00, 3.48340000E+00,
     4.43680000E+00, 4.39210000E+00, 4.22070000E+00, 4.11330000E+00, 3.80670000E+00, 3.74210000E+00,
     3.61400000E+00, 3.54910000E+00, 3.50770000E+00, 3.48870000E+00, 3.46230000E+00, 3.69560000E+00,
     3.95680000E+00, 3.89760000E+00, 3.82400000E+00, 3.76840000E+00, 3.84510000E+00, 3.69490000E+00,
     4.63180000E+00, 4.65590000E+00, 4.45330000E+00, 4.49560000E+00, 4.46410000E+00, 4.43660000E+00,
     4.41550000E+00, 4.39360000E+00, 4.37640000E+00, 4.13020000E+00, 4.33980000E+00, 4.22830000E+00,
     4.27960000E+00, 4.25470000E+00, 4.23910000E+00, 4.22960000E+00, 4.06990000E+00, 4.10830000E+00,
     4.03190000E+00, 3.98550000E+00, 3.93400000E+00, 3.84650000E+00, 3.49220000E+00, 3.44640000E+00,
     3.36180000E+00, 3.61590000E+00, 3.97840000E+00, 3.93650000E+00, 3.91220000E+00, 3.82500000E+00,
     3.91380000E+00, 3.75700000E+00, 4.61070000E+00, 4.61420000E+00, 4.52320000E+00, 4.24440000E+00,
     4.27140000E+00, 4.24790000E+00, 4.22260000E+00, 4.32600000E+00},
    {2.76760000E+00, 2.80780000E+00, 3.67250000E+00, 3.48040000E+00, 3.37750000E+00, 3.24110000E+00,
     3.15810000E+00, 3.19830000E+00, 2.99730000E+00, 2.87050000E+00, 3.80700000E+00, 3.73920000E+00,
     3.76680000E+00, 3.62630000E+00, 3.54020000E+00, 3.68070000E+00, 3.45450000E+00, 3.29620000E+00,
     4.22830000E+00, 4.16980000E+00, 4.02400000E+00, 3.93410000E+00, 3.87110000E+00, 3.54890000E+00,
     3.77980000E+00, 3.70000000E+00, 3.66540000E+00, 3.61540000E+00, 3.58820000E+00, 3.54720000E+00,
     3.72890000E+00, 3.65100000E+00, 3.60780000E+00, 3.53550000E+00, 3.59630000E+00, 3.44800000E+00,
     4.35870000E+00, 4.33900000E+00, 4.16350000E+00, 4.05360000E+00, 3.71930000E+00, 3.65290000E+00,
     3.55120000E+00, 3.48370000E+00, 3.44000000E+00, 3.41910000E+00, 3.38910000E+00, 3.66220000E+00,
     3.89340000E+00, 3.82350000E+00, 3.78230000E+00, 3.72920000E+00, 3.81060000E+00, 3.65890000E+00,
     4.55350000E+00, 4.60130000E+00, 4.39610000E+00, 4.44230000E+00, 4.41090000E+00, 4.38350000E+00,
     4.36250000E+00, 4.34070000E+00, 4.32370000E+00, 4.08630000E+00, 4.28350000E+00, 4.16750000E+00,
     4.22720000E+00, 4.20250000E+00, 4.18690000E+00, 4.17740000E+00, 4.01260000E+00, 4.04600000E+00,
     3.98150000E+00, 3.93400000E+00, 3.89550000E+00, 3.80890000E+00, 3.44980000E+00, 3.40550000E+00,
     3.32240000E+00, 3.58310000E+00, 3.91160000E+00, 3.86110000E+00, 3.82330000E+00, 3.78550000E+00,
     3.87980000E+00, 3.72190000E+00, 4.53310000E+00, 4.56000000E+00, 4.46760000E+00, 4.18550000E+00,
     4.21550000E+00, 4.19210000E+00, 4.16670000E+00, 4.27250000E+00},
    {2.69120000E+00, 2.76040000E+00, 3.60370000E+00, 3.41940000E+00, 3.30940000E+00, 3.17100000E+00,
     3.08620000E+00, 3.17890000E+00, 2.97380000E+00, 2.84270000E+00, 3.73780000E+00, 3.67420000E+00,
     3.69280000E+00, 3.55120000E+00, 3.46140000E+00, 3.40870000E+00, 3.42010000E+00, 3.26070000E+00,
     4.15270000E+00, 4.09770000E+00, 3.95230000E+00, 3.86280000E+00, 3.80020000E+00, 3.47590000E+00,
     3.71020000E+00, 3.64660000E+00, 3.61060000E+00, 3.55800000E+00, 3.52820000E+00, 3.48780000E+00,
     3.65470000E+00, 3.57630000E+00, 3.52890000E+00, 3.50860000E+00, 3.55930000E+00, 3.40990000E+00,
     4.27880000E+00, 4.26240000E+00, 4.08730000E+00, 3.97700000E+00, 3.64070000E+00, 3.57430000E+00,
     3.51780000E+00, 3.47530000E+00, 3.39310000E+00, 3.36940000E+00, 3.33390000E+00, 3.60020000E+00,
     3.81640000E+00, 3.74780000E+00, 3.70280000E+00, 3.69520000E+00, 3.76690000E+00, 3.61370000E+00,
     4.46980000E+00, 4.54880000E+00, 4.31680000E+00, 4.36460000E+00, 4.33380000E+00, 4.30670000E+00,
     4.28600000E+00, 4.26450000E+00, 4.24780000E+00, 4.00670000E+00, 4.23490000E+00, 4.09580000E+00,
     4.15430000E+00, 4.13020000E+00, 4.11410000E+00, 4.10480000E+00, 3.94100000E+00, 3.95950000E+00,
     3.89410000E+00, 3.84650000E+00, 3.80890000E+00, 3.74900000E+00, 3.39840000E+00, 3.35510000E+00,
     3.27110000E+00, 3.52570000E+00, 3.83260000E+00, 3.78470000E+00, 3.74380000E+00, 3.75010000E+00,
     3.83550000E+00, 3.67620000E+00, 4.44960000E+00, 4.48010000E+00, 4.39480000E+00, 4.11220000E+00,
     4.14160000E+00, 4.11820000E+00, 4.09320000E+00, 4.19610000E+00},
    {2.78950000E+00, 2.58490000E+00, 3.64840000E+00, 3.01620000E+00, 3.12670000E+00, 3.21250000E+00,
     3.00430000E+00, 2.95720000E+00, 2.81970000E+00, 2.72610000E+00, 3.77010000E+00, 3.24460000E+00,
     3.52390000E+00, 3.46960000E+00, 3.42610000E+00, 3.35080000E+00, 3.19680000E+00, 3.08480000E+00,
     4.14960000E+00, 3.65980000E+00, 3.51110000E+00, 3.41990000E+00, 3.38090000E+00, 3.53820000E+00,
     3.25720000E+00, 3.21000000E+00, 3.19170000E+00, 3.15190000E+00, 3.11980000E+00, 3.10050000E+00,
     3.50710000E+00, 3.50860000E+00, 3.50730000E+00, 3.45090000E+00, 3.31200000E+00, 3.20820000E+00,
     4.26110000E+00, 3.81170000E+00, 3.69880000E+00, 3.56460000E+00, 3.69250000E+00, 3.62950000E+00,
     3.53830000E+00, 3.49100000E+00, 3.46250000E+00, 3.42330000E+00, 3.40070000E+00, 3.23290000E+00,
     3.67230000E+00, 3.68450000E+00, 3.68760000E+00, 3.61970000E+00, 3.47990000E+00, 3.37370000E+00,
     4.43410000E+00, 4.05250000E+00, 3.90110000E+00, 3.89450000E+00, 3.86350000E+00, 3.83680000E+00,
     3.81530000E+00, 3.79360000E+00, 3.77580000E+00, 3.49440000E+00, 3.48730000E+00, 3.90400000E+00,
     3.71100000E+00, 3.69220000E+00, 3.67990000E+00, 3.67240000E+00, 3.56220000E+00, 3.60810000E+00,
     3.54260000E+00, 3.49220000E+00, 3.44980000E+00, 3.39840000E+00, 3.44560000E+00, 3.38880000E+00,
     3.33260000E+00, 3.21330000E+00, 3.71220000E+00, 3.73880000E+00, 3.74360000E+00, 3.67530000E+00,
     3.53590000E+00, 3.42600000E+00, 4.40820000E+00, 4.01490000E+00, 4.09890000E+00, 3.78390000E+00,
     3.76600000E+00, 3.73460000E+00, 3.71340000E+00, 3.73610000E+00},
    {2.75220000E+00, 2.55240000E+00, 3.57420000E+00, 2.95080000E+00, 3.07510000E+00, 3.01580000E+00,
     2.96440000E+00, 2.83380000E+00, 2.78910000E+00, 2.69330000E+00, 3.69260000E+00, 3.18140000E+00,
     3.45280000E+00, 3.41860000E+00, 3.38360000E+00, 3.22130000E+00, 3.16260000E+00, 3.05070000E+00,
     4.05480000E+00, 3.53120000E+00, 3.42440000E+00, 3.34090000E+00, 3.28100000E+00, 3.47820000E+00,
     3.19050000E+00, 3.14940000E+00, 3.12210000E+00, 3.11280000E+00, 3.08530000E+00, 3.03840000E+00,
     3.43660000E+00, 3.45620000E+00, 3.46380000E+00, 3.32110000E+00, 3.27620000E+00, 3.17300000E+00,
     4.16320000E+00, 3.68250000E+00, 3.58220000E+00, 3.48700000E+00, 3.63250000E+00, 3.57400000E+00,
     3.47330000E+00, 3.42470000E+00, 3.39690000E+00, 3.37640000E+00, 3.35250000E+00, 3.19840000E+00,
     3.59890000E+00, 3.62990000E+00, 3.64330000E+00, 3.49370000E+00, 3.44170000E+00, 3.33650000E+00,
     4.33040000E+00, 3.92420000E+00, 3.77930000E+00, 3.76230000E+00, 3.73270000E+00, 3.70710000E+00,
     3.68600000E+00, 3.66500000E+00, 3.64760000E+00, 3.38490000E+00, 3.35340000E+00, 3.82160000E+00,
     3.58700000E+00, 3.56950000E+00, 3.55840000E+00, 3.55080000E+00, 3.48560000E+00, 3.55230000E+00,
     3.49340000E+00, 3.44640000E+00, 3.40550000E+00, 3.35510000E+00, 3.38880000E+00, 3.35250000E+00,
     3.29500000E+00, 3.16890000E+00, 3.63780000E+00, 3.68260000E+00, 3.69810000E+00, 3.54990000E+00,
     3.49540000E+00, 3.38660000E+00, 4.30950000E+00, 3.88560000E+00, 3.98640000E+00, 3.68790000E+00,
     3.66280000E+00, 3.63140000E+00, 3.61090000E+00, 3.61930000E+00},
    {2.72020000E+00, 2.51830000E+00, 3.49470000E+00, 2.87310000E+00, 3.01980000E+00, 3.14570000E+00,
     2.92760000E+00, 2.78260000E+00, 2.75740000E+00, 2.66060000E+00, 3.60900000E+00, 3.05810000E+00,
     3.37470000E+00, 3.36770000E+00, 3.34500000E+00, 3.16510000E+00, 3.12590000E+00, 3.01470000E+00,
     3.94980000E+00, 3.38570000E+00, 3.29170000E+00, 3.21540000E+00, 3.16040000E+00, 3.41740000E+00,
     3.07350000E+00, 3.03420000E+00, 3.00960000E+00, 3.01360000E+00, 2.98550000E+00, 2.96800000E+00,
     3.36040000E+00, 3.40370000E+00, 3.42430000E+00, 3.26330000E+00, 3.18100000E+00, 3.13510000E+00,
     4.05570000E+00, 3.53680000E+00, 3.45260000E+00, 3.36990000E+00, 3.57070000E+00, 3.51840000E+00,
     3.40850000E+00, 3.35950000E+00, 3.33330000E+00, 3.31430000E+00, 3.30410000E+00, 3.10940000E+00,
     3.51930000E+00, 3.57450000E+00, 3.60250000E+00, 3.43380000E+00, 3.34480000E+00, 3.29520000E+00,
     4.21580000E+00, 3.78020000E+00, 3.64310000E+00, 3.61290000E+00, 3.58530000E+00, 3.56100000E+00,
     3.54060000E+00, 3.52040000E+00, 3.50360000E+00, 3.26790000E+00, 3.21620000E+00, 3.70680000E+00,
     3.44830000E+00, 3.43230000E+00, 3.42210000E+00, 3.41380000E+00, 3.36520000E+00, 3.45760000E+00,
     3.40530000E+00, 3.36180000E+00, 3.32240000E+00, 3.27110000E+00, 3.33260000E+00, 3.29500000E+00,
     3.25640000E+00, 3.11960000E+00, 3.55760000E+00, 3.62510000E+00, 3.65550000E+00, 3.48720000E+00,
     3.39620000E+00, 3.34250000E+00, 4.20230000E+00, 3.74070000E+00, 3.85960000E+00, 3.58160000E+00,
     3.54760000E+00, 3.51680000E+00, 3.49620000E+00, 3.49160000E+00},
    {2.53150000E+00, 2.61040000E+00, 3.27340000E+00, 3.22990000E+00, 3.10900000E+00, 2.99420000E+00,
     2.91590000E+00, 2.83240000E+00, 2.83500000E+00, 2.72160000E+00, 3.39940000E+00, 3.44750000E+00,
     3.43540000E+00, 3.34380000E+00, 3.28070000E+00, 3.21690000E+00, 3.26770000E+00, 3.12960000E+00,
     3.74930000E+00, 3.80750000E+00, 3.68460000E+00, 3.61040000E+00, 3.55770000E+00, 3.20520000E+00,
     3.48030000E+00, 3.42360000E+00, 3.38450000E+00, 3.36400000E+00, 3.33650000E+00, 3.30100000E+00,
     3.39380000E+00, 3.36240000E+00, 3.34400000E+00, 3.31320000E+00, 3.40350000E+00, 3.27540000E+00,
     3.87010000E+00, 3.95230000E+00, 3.80180000E+00, 3.71490000E+00, 3.36730000E+00, 3.31990000E+00,
     3.24830000E+00, 3.20690000E+00, 3.17930000E+00, 3.15580000E+00, 3.13950000E+00, 3.40970000E+00,
     3.54100000E+00, 3.52280000E+00, 3.51160000E+00, 3.49210000E+00, 3.47810000E+00, 3.46900000E+00,
     4.04200000E+00, 4.17590000E+00, 4.00780000E+00, 4.04500000E+00, 4.01890000E+00, 3.99520000E+00,
     3.97700000E+00, 3.95830000E+00, 3.94340000E+00, 3.72170000E+00, 3.82280000E+00, 3.78260000E+00,
     3.86400000E+00, 3.84460000E+00, 3.83140000E+00, 3.82250000E+00, 3.68170000E+00, 3.70680000E+00,
     3.65550000E+00, 3.61590000E+00, 3.58310000E+00, 3.52570000E+00, 3.21330000E+00, 3.16890000E+00,
     3.11960000E+00, 3.35990000E+00, 3.54570000E+00, 3.54920000E+00, 3.54520000E+00, 3.54010000E+00,
     3.53390000E+00, 3.52940000E+00, 4.05010000E+00, 4.15450000E+00, 4.09420000E+00, 3.86330000E+00,
     3.87900000E+00, 3.85820000E+00, 3.83520000E+00, 3.91150000E+00},
    {2.98520000E+00, 2.78810000E+00, 3.52840000E+00, 3.34930000E+00, 3.69580000E+00, 3.36420000E+00,
     3.15680000E+00, 3.00550000E+00, 2.95580000E+00, 2.83930000E+00, 3.62870000E+00, 3.52830000E+00,
     4.15110000E+00, 3.82590000E+00, 3.60660000E+00, 3.45270000E+00, 3.34800000E+00, 3.27130000E+00,
     3.90370000E+00, 3.83610000E+00, 3.85790000E+00, 3.73110000E+00, 3.65750000E+00, 3.51760000E+00,
     3.56930000E+00, 3.51570000E+00, 3.48140000E+00, 3.45590000E+00, 3.44450000E+00, 3.41600000E+00,
     4.12310000E+00, 3.85430000E+00, 3.68160000E+00, 3.56020000E+00, 3.47980000E+00, 3.42080000E+00,
     4.05420000E+00, 4.01390000E+00, 4.01650000E+00, 3.94120000E+00, 3.76980000E+00, 3.69150000E+00,
     3.60430000E+00, 3.56390000E+00, 3.54160000E+00, 3.52470000E+00, 3.51530000E+00, 3.56540000E+00,
     4.28620000E+00, 4.04370000E+00, 3.88710000E+00, 3.77410000E+00, 3.69850000E+00, 3.64130000E+00,
     4.23450000E+00, 4.36630000E+00, 4.32570000E+00, 4.08690000E+00, 4.06120000E+00, 4.03640000E+00,
     4.01700000E+00, 3.99780000E+00, 3.98340000E+00, 3.91370000E+00, 3.88250000E+00, 3.87580000E+00,
     3.91430000E+00, 3.89760000E+00, 3.88640000E+00, 3.87680000E+00, 3.91900000E+00, 4.16130000E+00,
     4.05660000E+00, 3.97840000E+00, 3.91160000E+00, 3.83260000E+00, 3.71220000E+00, 3.63780000E+00,
     3.55760000E+00, 3.54570000E+00, 4.31270000E+00, 4.08890000E+00, 3.93270000E+00, 3.82880000E+00,
     3.75950000E+00, 3.70220000E+00, 4.25950000E+00, 4.22530000E+00, 4.27200000E+00, 4.15850000E+00,
     4.12330000E+00, 4.09270000E+00, 4.06880000E+00, 3.99140000E+00},
    {3.11600000E+00, 2.84820000E+00, 4.07390000E+00, 3.35990000E+00, 3.56980000E+00, 3.53660000E+00,
     3.28540000E+00, 3.10390000E+00, 2.99530000E+00, 2.91920000E+00, 4.14320000E+00, 3.53200000E+00,
     3.94780000E+00, 4.02310000E+00, 3.75090000E+00, 3.56040000E+00, 3.43400000E+00, 3.34260000E+00,
     4.33280000E+00, 3.82880000E+00, 3.78220000E+00, 3.79090000E+00, 3.69070000E+00, 3.68640000E+00,
     3.57930000E+00, 3.52210000E+00, 3.48830000E+00, 3.46490000E+00, 3.45140000E+00, 3.43010000E+00,
     3.92560000E+00, 4.05960000E+00, 3.83070000E+00, 3.67020000E+00, 3.56510000E+00, 3.48840000E+00,
     4.41820000E+00, 4.25160000E+00, 3.96870000E+00, 3.91860000E+00, 3.94850000E+00, 3.83700000E+00,
     3.72550000E+00, 3.67440000E+00, 3.64760000E+00, 3.62950000E+00, 3.61930000E+00, 3.56590000E+00,
     4.06630000E+00, 4.23090000E+00, 4.01830000E+00, 3.86800000E+00, 3.76720000E+00, 3.69230000E+00,
     4.52400000E+00, 4.48340000E+00, 4.15700000E+00, 4.32040000E+00, 4.29930000E+00, 4.28040000E+00,
     4.26470000E+00, 4.24810000E+00, 4.23540000E+00, 3.86260000E+00, 3.84480000E+00, 4.22670000E+00,
     4.17990000E+00, 4.16700000E+00, 3.87380000E+00, 3.86430000E+00, 3.87960000E+00, 4.05750000E+00,
     4.03540000E+00, 3.93650000E+00, 3.86110000E+00, 3.78470000E+00, 3.73880000E+00, 3.68260000E+00,
     3.62510000E+00, 3.54920000E+00, 4.08890000E+00, 4.27640000E+00, 4.06580000E+00, 3.92170000E+00,
     3.82500000E+00, 3.74970000E+00, 4.54970000E+00, 4.42290000E+00, 4.32700000E+00, 4.14020000E+00,
     4.07380000E+00, 4.04690000E+00, 4.02810000E+00, 3.98090000E+00},
    {3.14160000E+00, 2.83250000E+00, 3.77350000E+00, 3.37870000E+00, 3.46320000E+00, 3.59230000E+00,
     3.32140000E+00, 3.12850000E+00, 3.01470000E+00, 2.93660000E+00, 3.85270000E+00, 3.56020000E+00,
     3.81310000E+00, 3.83490000E+00, 3.79950000E+00, 3.59190000E+00, 3.45390000E+00, 3.35400000E+00,
     4.06540000E+00, 3.86030000E+00, 3.79720000E+00, 3.73580000E+00, 3.73920000E+00, 3.81570000E+00,
     3.60550000E+00, 3.54380000E+00, 3.50890000E+00, 3.48530000E+00, 3.46980000E+00, 3.45080000E+00,
     3.78820000E+00, 3.86820000E+00, 3.88370000E+00, 3.70550000E+00, 3.58700000E+00, 3.50000000E+00,
     4.15730000E+00, 4.00050000E+00, 3.95680000E+00, 3.89360000E+00, 3.99900000E+00, 3.94330000E+00,
     3.81720000E+00, 3.75660000E+00, 3.72460000E+00, 3.70330000E+00, 3.69000000E+00, 3.56970000E+00,
     3.91830000E+00, 4.02620000E+00, 4.06590000E+00, 3.89690000E+00, 3.78090000E+00, 3.69490000E+00,
     4.27650000E+00, 4.23120000E+00, 4.14010000E+00, 4.08150000E+00, 4.05800000E+00, 4.03690000E+00,
     4.01940000E+00, 4.00170000E+00, 3.98740000E+00, 3.83120000E+00, 3.81200000E+00, 3.94540000E+00,
     3.92100000E+00, 3.90550000E+00, 3.89510000E+00, 3.88660000E+00, 3.86890000E+00, 3.96030000E+00,
     3.91090000E+00, 3.91220000E+00, 3.82330000E+00, 3.74380000E+00, 3.74360000E+00, 3.69810000E+00,
     3.65550000E+00, 3.54520000E+00, 3.93270000E+00, 4.06580000E+00, 4.11750000E+00, 3.95380000E+00,
     3.84080000E+00, 3.75420000E+00, 4.30560000E+00, 4.19230000E+00, 4.30220000E+00, 4.10360000E+00,
     4.05750000E+00, 4.03130000E+00, 4.00990000E+00, 3.98660000E+00},
    {2.96640000E+00, 2.82090000E+00, 3.55470000E+00, 3.37960000E+00, 3.39850000E+00, 3.31640000E+00,
     3.23640000E+00, 3.19560000E+00, 3.03700000E+00, 2.93130000E+00, 3.64250000E+00, 3.55650000E+00,
     3.72090000E+00, 3.71080000E+00, 3.66390000E+00, 3.64840000E+00, 3.47450000E+00, 3.34920000E+00,
     3.87550000E+00, 4.24570000E+00, 3.77580000E+00, 3.71610000E+00, 3.66930000E+00, 3.61550000E+00,
     3.59410000E+00, 3.56430000E+00, 3.52920000E+00, 3.49500000E+00, 3.47200000E+00, 3.45030000E+00,
     3.69360000E+00, 3.73920000E+00, 3.73880000E+00, 3.76020000E+00, 3.60780000E+00, 3.49600000E+00,
     3.98000000E+00, 4.35180000E+00, 4.28020000E+00, 3.85800000E+00, 3.80560000E+00, 3.75270000E+00,
     3.70190000E+00, 3.66150000E+00, 3.57680000E+00, 3.53300000E+00, 3.50380000E+00, 3.56390000E+00,
     3.81920000E+00, 3.88830000E+00, 3.90920000E+00, 3.94780000E+00, 3.79950000E+00, 3.68960000E+00,
     4.11650000E+00, 4.52320000E+00, 4.43570000E+00, 4.42260000E+00, 4.40310000E+00, 4.38600000E+00,
     4.37210000E+00, 4.35800000E+00, 4.34660000E+00, 4.20360000E+00, 4.20370000E+00, 3.88670000E+00,
     4.28950000E+00, 4.27660000E+00, 4.26620000E+00, 4.25980000E+00, 3.84080000E+00, 3.91690000E+00,
     3.86810000E+00, 3.82500000E+00, 3.78550000E+00, 3.75010000E+00, 3.67530000E+00, 3.54990000E+00,
     3.48720000E+00, 3.54010000E+00, 3.82880000E+00, 3.92170000E+00, 3.95380000E+00, 4.00540000E+00,
     3.86000000E+00, 3.74940000E+00, 4.15060000E+00, 4.50220000E+00, 4.54100000E+00, 4.36940000E+00,
     4.35750000E+00, 4.33910000E+00, 4.31990000E+00, 4.33290000E+00},
    {2.83880000E+00, 2.78900000E+00, 3.43290000E+00, 3.55930000E+00, 3.34880000E+00, 3.24860000E+00,
     3.16150000E+00, 3.10000000E+00, 3.03940000E+00, 2.91650000E+00, 3.52670000E+00, 3.74790000E+00,
     3.66500000E+00, 3.62630000E+00, 3.56580000E+00, 3.52240000E+00, 3.47620000E+00, 3.33420000E+00,
     3.77380000E+00, 4.03330000E+00, 3.95680000E+00, 3.89750000E+00, 3.85210000E+00, 3.49290000E+00,
     3.78300000E+00, 3.74090000E+00, 3.70620000E+00, 3.67860000E+00, 3.64710000E+00, 3.62080000E+00,
     3.63370000E+00, 3.65190000E+00, 3.63630000E+00, 3.62780000E+00, 3.61100000E+00, 3.48250000E+00,
     3.87950000E+00, 4.14480000E+00, 4.07360000E+00, 4.00450000E+00, 3.68430000E+00, 3.62910000E+00,
     3.57410000E+00, 3.53120000E+00, 3.49740000E+00, 3.44720000E+00, 3.40340000E+00, 3.71310000E+00,
     3.75570000E+00, 3.79660000E+00, 3.80050000E+00, 3.80680000E+00, 3.80150000E+00, 3.67470000E+00,
     4.02220000E+00, 4.32070000E+00, 4.23470000E+00, 4.21910000E+00, 4.19900000E+00, 4.18110000E+00,
     4.16660000E+00, 4.15210000E+00, 4.14010000E+00, 3.99700000E+00, 3.99430000E+00, 3.95920000E+00,
     4.08000000E+00, 4.06640000E+00, 4.05590000E+00, 4.04880000E+00, 3.98820000E+00, 4.00350000E+00,
     3.95390000E+00, 3.91380000E+00, 3.87980000E+00, 3.83550000E+00, 3.53590000E+00, 3.49540000E+00,
     3.39620000E+00, 3.53390000E+00, 3.75950000E+00, 3.82500000E+00, 3.84080000E+00, 3.86000000E+00,
     3.86440000E+00, 3.73700000E+00, 4.05740000E+00, 4.30590000E+00, 4.35760000E+00, 4.17350000E+00,
     4.15860000E+00, 4.13810000E+00, 4.11880000E+00, 4.12760000E+00},
    {2.74120000E+00, 2.74890000E+00, 3.33740000E+00, 3.39500000E+00, 3.30760000E+00, 3.19100000E+00,
     3.09610000E+00, 3.01750000E+00, 3.02800000E+00, 2.89290000E+00, 3.43280000E+00, 3.58830000E+00,
     3.62270000E+00, 3.56160000E+00, 3.48940000E+00, 3.42410000E+00, 3.36410000E+00, 3.31200000E+00,
     3.68150000E+00, 3.87890000E+00, 3.80310000E+00, 3.74130000E+00, 3.69390000E+00, 3.40100000E+00,
     3.62250000E+00, 3.57970000E+00, 3.54430000E+00, 3.51390000E+00, 3.49230000E+00, 3.46420000E+00,
     3.58600000E+00, 3.58490000E+00, 3.55700000E+00, 3.52570000E+00, 3.49360000E+00, 3.46280000E+00,
     3.78740000E+00, 3.99160000E+00, 3.92490000E+00, 3.85300000E+00, 3.59320000E+00, 3.53550000E+00,
     3.47570000E+00, 3.43060000E+00, 3.39530000E+00, 3.36460000E+00, 3.33900000E+00, 3.56370000E+00,
     3.70530000E+00, 3.72660000E+00, 3.71770000E+00, 3.69960000E+00, 3.67750000E+00, 3.65580000E+00,
     3.93310000E+00, 4.16550000E+00, 4.08790000E+00, 4.06810000E+00, 4.04790000E+00, 4.02990000E+00,
     4.01520000E+00, 4.00060000E+00, 3.98830000E+00, 3.85000000E+00, 3.83590000E+00, 3.82490000E+00,
     3.92690000E+00, 3.91330000E+00, 3.90250000E+00, 3.89480000E+00, 3.84220000E+00, 3.85090000E+00,
     3.79900000E+00, 3.75700000E+00, 3.72190000E+00, 3.67620000E+00, 3.42600000E+00, 3.38660000E+00,
     3.34250000E+00, 3.52940000E+00, 3.70220000E+00, 3.74970000E+00, 3.75420000E+00, 3.74940000E+00,
     3.73700000E+00, 3.72160000E+00, 3.97250000E+00, 4.15910000E+00, 4.22350000E+00, 4.03680000E+00,
     4.01830000E+00, 3.99620000E+00, 3.97680000E+00, 3.97820000E+00},
    {3.41550000E+00, 3.05220000E+00, 4.25410000E+00, 3.82180000E+00, 4.04380000E+00, 3.58750000E+00,
     3.32860000E+00, 3.16820000E+00, 3.05660000E+00, 2.97460000E+00, 4.36270000E+00, 4.02490000E+00,
     4.69470000E+00, 4.17180000E+00, 3.86390000E+00, 3.67350000E+00, 3.54350000E+00, 3.44790000E+00,
     4.68060000E+00, 4.34850000E+00, 4.26680000E+00, 4.16900000E+00, 4.10610000E+00, 4.12450000E+00,
     4.02060000E+00, 3.97650000E+00, 3.94580000E+00, 3.92170000E+00, 3.90750000E+00, 3.88130000E+00,
     3.99470000E+00, 4.19890000E+00, 3.95070000E+00, 3.79600000E+00, 3.69250000E+00, 3.61500000E+00,
     4.85350000E+00, 4.56420000E+00, 4.41340000E+00, 4.36880000E+00, 4.33960000E+00, 4.28790000E+00,
     4.21660000E+00, 4.18880000E+00, 4.17680000E+00, 4.16600000E+00, 4.16080000E+00, 4.07450000E+00,
     4.22890000E+00, 4.48630000E+00, 4.25130000E+00, 4.08970000E+00, 3.98760000E+00, 3.90610000E+00,
     5.06900000E+00, 5.04460000E+00, 4.61860000E+00, 4.60780000E+00, 4.57800000E+00, 4.55380000E+00,
     4.53190000E+00, 4.51010000E+00, 4.49450000E+00, 4.19120000E+00, 4.23150000E+00, 4.55340000E+00,
     4.43730000E+00, 4.42240000E+00, 4.41200000E+00, 4.40400000E+00, 4.26340000E+00, 4.77700000E+00,
     4.68900000E+00, 4.61070000E+00, 4.53310000E+00, 4.44960000E+00, 4.40820000E+00, 4.30950000E+00,
     4.20230000E+00, 4.05010000E+00, 4.25950000E+00, 4.54970000E+00, 4.30560000E+00, 4.15060000E+00,
     4.05740000E+00, 3.97250000E+00, 5.07960000E+00, 4.78830000E+00, 4.65450000E+00, 4.50950000E+00,
     4.45930000E+00, 4.44290000E+00, 4.41920000E+00, 4.50970000E+00},
    {3.05480000E+00, 3.32060000E+00, 3.81320000E+00, 3.97200000E+00, 3.76750000E+00, 3.73510000E+00,
     3.51670000E+00, 3.52740000E+00, 3.30850000E+00, 3.16530000E+00, 3.95000000E+00, 4.17300000E+00,
     4.06130000E+00, 4.14930000E+00, 3.88230000E+00, 4.05370000E+00, 3.82000000E+00, 3.65820000E+00,
     4.34220000E+00, 4.51110000E+00, 4.37950000E+00, 4.33620000E+00, 4.27510000E+00, 3.71030000E+00,
     4.19730000E+00, 4.13850000E+00, 4.11290000E+00, 4.08000000E+00, 4.06470000E+00, 4.03080000E+00,
     4.00960000E+00, 4.16190000E+00, 3.93600000E+00, 4.17660000E+00, 3.97050000E+00, 3.82620000E+00,
     4.53480000E+00, 4.70250000E+00, 4.52680000E+00, 4.50760000E+00, 3.95620000E+00, 3.90650000E+00,
     3.81190000E+00, 3.76050000E+00, 3.74470000E+00, 3.71190000E+00, 3.69160000E+00, 4.19500000E+00,
     4.21100000E+00, 4.38430000E+00, 4.16310000E+00, 4.44270000E+00, 4.24630000E+00, 4.10540000E+00,
     4.76930000E+00, 5.06490000E+00, 4.73650000E+00, 4.77610000E+00, 4.74980000E+00, 4.72720000E+00,
     4.70760000E+00, 4.68770000E+00, 4.67300000E+00, 4.42740000E+00, 4.54730000E+00, 4.51690000E+00,
     4.59750000E+00, 4.57930000E+00, 4.56670000E+00, 4.55590000E+00, 4.38040000E+00, 4.69200000E+00,
     4.67310000E+00, 4.61420000E+00, 4.56000000E+00, 4.48010000E+00, 4.01490000E+00, 3.88560000E+00,
     3.74070000E+00, 4.15450000E+00, 4.22530000E+00, 4.42290000E+00, 4.19230000E+00, 4.50220000E+00,
     4.30590000E+00, 4.15910000E+00, 4.78830000E+00, 4.92940000E+00, 4.74470000E+00, 4.55380000E+00,
     4.59270000E+00, 4.57870000E+00, 4.55770000E+00, 4.67690000E+00},
    {3.38500000E+00, 3.42080000E+00, 3.70040000E+00, 3.88000000E+00, 3.98860000E+00, 3.90400000E+00,
     3.67190000E+00, 3.65470000E+00, 3.46250000E+00, 3.33700000E+00, 3.83940000E+00, 4.03350000E+00,
     4.23730000E+00, 4.30230000E+00, 4.03060000E+00, 4.14080000E+00, 3.92970000E+00, 3.78570000E+00,
     4.19070000E+00, 4.32300000E+00, 4.26640000E+00, 4.21730000E+00, 4.14820000E+00, 3.68230000E+00,
     4.07110000E+00, 4.01800000E+00, 4.00170000E+00, 3.97470000E+00, 3.96340000E+00, 3.93830000E+00,
     4.19930000E+00, 4.32050000E+00, 4.08210000E+00, 4.25470000E+00, 4.06590000E+00, 3.93590000E+00,
     4.39520000E+00, 4.51760000E+00, 4.38880000E+00, 4.36070000E+00, 3.95830000E+00, 3.92800000E+00,
     3.83900000E+00, 3.79710000E+00, 3.79550000E+00, 3.76740000E+00, 3.75210000E+00, 4.10620000E+00,
     4.36330000E+00, 4.29910000E+00, 4.27670000E+00, 4.48570000E+00, 4.30390000E+00, 4.17620000E+00,
     4.61970000E+00, 4.86540000E+00, 4.66330000E+00, 4.58780000E+00, 4.56400000E+00, 4.54220000E+00,
     4.52310000E+00, 4.50420000E+00, 4.49010000E+00, 4.32820000E+00, 4.39780000E+00, 4.34830000E+00,
     4.42020000E+00, 4.40390000E+00, 4.39260000E+00, 4.38070000E+00, 4.26490000E+00, 4.61350000E+00,
     4.56050000E+00, 4.52320000E+00, 4.46760000E+00, 4.39480000E+00, 4.09890000E+00, 3.98640000E+00,
     3.85960000E+00, 4.09420000E+00, 4.27200000E+00, 4.32700000E+00, 4.30220000E+00, 4.54100000E+00,
     4.35760000E+00, 4.22350000E+00, 4.65450000E+00, 4.74470000E+00, 4.70430000E+00, 4.52400000E+00,
     4.48650000E+00, 4.47310000E+00, 4.45160000E+00, 4.51580000E+00},
    {3.09420000E+00, 3.20750000E+00, 3.51520000E+00, 3.66590000E+00, 3.82890000E+00, 3.74590000E+00,
     3.51560000E+00, 3.51970000E+00, 3.32900000E+00, 3.20690000E+00, 3.67020000E+00, 3.84480000E+00,
     4.03400000E+00, 3.95090000E+00, 3.85850000E+00, 3.98940000E+00, 3.77870000E+00, 3.63650000E+00,
     4.14250000E+00, 4.16180000E+00, 4.09400000E+00, 4.04660000E+00, 3.99410000E+00, 3.54260000E+00,
     3.89520000E+00, 3.83270000E+00, 3.81260000E+00, 3.77960000E+00, 3.76350000E+00, 3.73560000E+00,
     4.00470000E+00, 3.96550000E+00, 3.91160000E+00, 4.10100000E+00, 3.91020000E+00, 3.78000000E+00,
     4.29640000E+00, 4.33300000E+00, 4.26220000E+00, 4.22540000E+00, 3.81950000E+00, 3.75600000E+00,
     3.65130000E+00, 3.59410000E+00, 3.58100000E+00, 3.54200000E+00, 3.51780000E+00, 3.88610000E+00,
     4.14590000E+00, 4.11470000E+00, 4.07720000E+00, 4.31200000E+00, 4.12070000E+00, 3.99000000E+00,
     4.47330000E+00, 4.61570000E+00, 4.45800000E+00, 4.41940000E+00, 4.39540000E+00, 4.37390000E+00,
     4.35310000E+00, 4.33430000E+00, 4.31960000E+00, 4.21400000E+00, 4.23390000E+00, 4.17380000E+00,
     4.24580000E+00, 4.22780000E+00, 4.21580000E+00, 4.20390000E+00, 4.16580000E+00, 4.35950000E+00,
     4.28570000E+00, 4.24440000E+00, 4.18550000E+00, 4.11220000E+00, 3.78390000E+00, 3.68790000E+00,
     3.58160000E+00, 3.86330000E+00, 4.15850000E+00, 4.14020000E+00, 4.10360000E+00, 4.36940000E+00,
     4.17350000E+00, 4.03680000E+00, 4.50950000E+00, 4.55380000E+00, 4.52400000E+00, 4.42520000E+00,
     4.38130000E+00, 4.35880000E+00, 4.33650000E+00, 4.32910000E+00},
    {3.01870000E+00, 3.19180000E+00, 3.51270000E+00, 3.68750000E+00, 3.74040000E+00, 3.69430000E+00,
     3.47020000E+00, 3.48880000E+00, 3.29140000E+00, 3.16430000E+00, 3.66690000E+00, 3.87240000E+00,
     3.99400000E+00, 4.08160000E+00, 3.80540000E+00, 3.96610000E+00, 3.74920000E+00, 3.60240000E+00,
     4.04280000E+00, 4.19510000E+00, 4.14660000E+00, 4.05150000E+00, 4.00750000E+00, 3.50200000E+00,
     3.91580000E+00, 3.85460000E+00, 3.83420000E+00, 3.80080000E+00, 3.78450000E+00, 3.75490000E+00,
     3.96020000E+00, 3.88720000E+00, 3.85640000E+00, 4.07930000E+00, 3.88350000E+00, 3.74950000E+00,
     4.22130000E+00, 4.37040000E+00, 4.33000000E+00, 4.21210000E+00, 3.76430000E+00, 3.71300000E+00,
     3.61440000E+00, 3.55990000E+00, 3.54740000E+00, 3.50930000E+00, 3.48530000E+00, 3.90750000E+00,
     4.11150000E+00, 4.04730000E+00, 4.03180000E+00, 4.29990000E+00, 4.10500000E+00, 3.97100000E+00,
     4.43200000E+00, 4.67060000E+00, 4.52730000E+00, 4.45810000E+00, 4.43320000E+00, 4.40640000E+00,
     4.38730000E+00, 4.36840000E+00, 4.35370000E+00, 4.27280000E+00, 4.25490000E+00, 4.20320000E+00,
     4.27940000E+00, 4.26130000E+00, 4.24910000E+00, 4.23750000E+00, 4.23220000E+00, 4.36650000E+00,
     4.30610000E+00, 4.27140000E+00, 4.21550000E+00, 4.14160000E+00, 3.76600000E+00, 3.66280000E+00,
     3.54760000E+00, 3.87900000E+00, 4.12330000E+00, 4.07380000E+00, 4.05750000E+00, 4.35750000E+00,
     4.15860000E+00, 4.01830000E+00, 4.45930000E+00, 4.59270000E+00, 4.48650000E+00, 4.38130000E+00,
     4.45940000E+00, 4.42700000E+00, 4.40580000E+00, 4.36090000E+00},
    {2.98750000E+00, 3.16740000E+00, 3.49710000E+00, 3.67150000E+00, 3.71140000E+00, 3.66920000E+00,
     3.44460000E+00, 3.46760000E+00, 3.26850000E+00, 3.14050000E+00, 3.65460000E+00, 3.85790000E+00,
     3.96370000E+00, 4.05810000E+00, 3.77960000E+00, 3.94630000E+00, 3.72750000E+00, 3.57920000E+00,
     4.02950000E+00, 4.18240000E+00, 4.12470000E+00, 4.03570000E+00, 3.99260000E+00, 3.48270000E+00,
     3.90070000E+00, 3.83920000E+00, 3.81910000E+00, 3.78510000E+00, 3.76870000E+00, 3.73870000E+00,
     3.92900000E+00, 3.86060000E+00, 3.83060000E+00, 4.06010000E+00, 3.86250000E+00, 3.72690000E+00,
     4.20620000E+00, 4.35660000E+00, 4.30220000E+00, 4.19290000E+00, 3.74010000E+00, 3.68880000E+00,
     3.59000000E+00, 3.53500000E+00, 3.52260000E+00, 3.48380000E+00, 3.45940000E+00, 3.88880000E+00,
     4.08130000E+00, 4.02090000E+00, 4.00590000E+00, 4.28100000E+00, 4.08430000E+00, 3.94860000E+00,
     4.41620000E+00, 4.65420000E+00, 4.50050000E+00, 4.44440000E+00, 4.41960000E+00, 4.39330000E+00,
     4.37410000E+00, 4.35520000E+00, 4.34060000E+00, 4.24840000E+00, 4.24130000E+00, 4.19070000E+00,
     4.26560000E+00, 4.24740000E+00, 4.23520000E+00, 4.22360000E+00, 4.20680000E+00, 4.34100000E+00,
     4.28170000E+00, 4.24790000E+00, 4.19210000E+00, 4.11820000E+00, 3.73460000E+00, 3.63140000E+00,
     3.51680000E+00, 3.85820000E+00, 4.09270000E+00, 4.04690000E+00, 4.03130000E+00, 4.33910000E+00,
     4.13810000E+00, 3.99620000E+00, 4.44290000E+00, 4.57870000E+00, 4.47310000E+00, 4.35880000E+00,
     4.42700000E+00, 4.39570000E+00, 4.37450000E+00, 4.34620000E+00},
    {2.96590000E+00, 3.14420000E+00, 3.47950000E+00, 3.65030000E+00, 3.68140000E+00, 3.64760000E+00,
     3.42220000E+00, 3.44910000E+00, 3.24940000E+00, 3.12090000E+00, 3.63240000E+00, 3.83750000E+00,
     3.93970000E+00, 3.83110000E+00, 3.75810000E+00, 3.92740000E+00, 3.70850000E+00, 3.55980000E+00,
     4.00800000E+00, 4.16410000E+00, 4.10570000E+00, 4.01580000E+00, 3.97260000E+00, 3.46670000E+00,
     3.88020000E+00, 3.81880000E+00, 3.79890000E+00, 3.76440000E+00, 3.74740000E+00, 3.71730000E+00,
     3.90490000E+00, 3.84240000E+00, 3.80950000E+00, 4.04120000E+00, 3.84360000E+00, 3.70770000E+00,
     4.18370000E+00, 4.33660000E+00, 4.28160000E+00, 4.16860000E+00, 3.72930000E+00, 3.67090000E+00,
     3.57000000E+00, 3.51530000E+00, 3.50390000E+00, 3.46840000E+00, 3.44370000E+00, 3.86630000E+00,
     4.05750000E+00, 4.00200000E+00, 3.98420000E+00, 4.26120000E+00, 4.06430000E+00, 3.92850000E+00,
     4.39280000E+00, 4.63080000E+00, 4.47990000E+00, 4.42440000E+00, 4.39960000E+00, 4.37370000E+00,
     4.35470000E+00, 4.33580000E+00, 4.32120000E+00, 4.22750000E+00, 4.22160000E+00, 4.16760000E+00,
     4.24650000E+00, 4.22830000E+00, 4.21610000E+00, 4.20450000E+00, 4.18410000E+00, 4.31350000E+00,
     4.25620000E+00, 4.22260000E+00, 4.16670000E+00, 4.09320000E+00, 3.71340000E+00, 3.61090000E+00,
     3.49620000E+00, 3.83520000E+00, 4.06880000E+00, 4.02810000E+00, 4.00990000E+00, 4.31990000E+00,
     4.11880000E+00, 3.97680000E+00, 4.41920000E+00, 4.55770000E+00, 4.45160000E+00, 4.33650000E+00,
     4.40580000E+00, 4.37450000E+00, 4.35390000E+00, 4.32650000E+00},
    {2.87630000E+00, 3.12940000E+00, 3.55980000E+00, 3.74650000E+00, 3.56590000E+00, 3.58160000E+00,
     3.35990000E+00, 3.40240000E+00, 3.18770000E+00, 3.04840000E+00, 3.70090000E+00, 3.94510000E+00,
     3.84650000E+00, 3.98730000E+00, 3.70790000E+00, 3.90830000E+00, 3.67560000E+00, 3.51500000E+00,
     4.08290000E+00, 4.27800000E+00, 4.15110000E+00, 4.12600000E+00, 4.05710000E+00, 3.48650000E+00,
     3.97440000E+00, 3.91500000E+00, 3.89300000E+00, 3.85780000E+00, 3.84020000E+00, 3.80730000E+00,
     3.79770000E+00, 4.00360000E+00, 3.76040000E+00, 4.02880000E+00, 3.82100000E+00, 3.67570000E+00,
     4.26460000E+00, 4.45580000E+00, 4.28620000E+00, 4.21220000E+00, 3.70880000E+00, 3.67290000E+00,
     3.58000000E+00, 3.52760000E+00, 3.51650000E+00, 3.47830000E+00, 3.45390000E+00, 3.95530000E+00,
     3.98180000E+00, 4.20400000E+00, 3.96040000E+00, 4.27180000E+00, 4.06890000E+00, 3.92530000E+00,
     4.48690000E+00, 4.77920000E+00, 4.49180000E+00, 4.53420000E+00, 4.50900000E+00, 4.48680000E+00,
     4.46800000E+00, 4.44860000E+00, 4.43410000E+00, 4.20230000E+00, 4.31220000E+00, 4.27100000E+00,
     4.35870000E+00, 4.34070000E+00, 4.32810000E+00, 4.31740000E+00, 4.14990000E+00, 4.39400000E+00,
     4.38950000E+00, 4.32600000E+00, 4.27250000E+00, 4.19610000E+00, 3.73610000E+00, 3.61930000E+00,
     3.49160000E+00, 3.91150000E+00, 3.99140000E+00, 3.98090000E+00, 3.98660000E+00, 4.33290000E+00,
     4.12760000E+00, 3.97820000E+00, 4.50970000E+00, 4.67690000E+00, 4.51580000E+00, 4.32910000E+00,
     4.36090000E+00, 4.34620000E+00, 4.32650000E+00, 4.43410000E+00}};

static constexpr double c6ab_table[N_PARS_ROWS][N_PARS_COLS] = {
    {3.02670000E+00, 1.00000000E+00, 1.00000000E+00, 9.11800000E-01, 9.11800000E-01},
    {2.08350000E+00, 2.00000000E+00, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {1.55830000E+00, 2.00000000E+00, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.89448000E+01, 3.00000000E+00, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.21508000E+01, 3.00000000E+00, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.16344540E+03, 3.00000000E+00, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.44415000E+01, 4.00000000E+00, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {1.48246000E+01, 4.00000000E+00, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.94619000E+02, 4.00000000E+00, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.57486300E+02, 4.00000000E+00, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.73143000E+01, 5.00000000E+00, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {1.10975000E+01, 5.00000000E+00, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.83730800E+02, 5.00000000E+00, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.61597100E+02, 5.00000000E+00, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.07177700E+02, 5.00000000E+00, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.21402000E+01, 6.00000000E+00, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {8.18410000E+00, 6.00000000E+00, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.69903000E+02, 6.00000000E+00, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.02956000E+02, 6.00000000E+00, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.12794000E+01, 6.00000000E+00, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.91130000E+01, 6.00000000E+00, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.71710000E+00, 7.00000000E+00, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {6.13800000E+00, 7.00000000E+00, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.08485400E+02, 7.00000000E+00, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.86458000E+01, 7.00000000E+00, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.91132000E+01, 7.00000000E+00, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.48146000E+01, 7.00000000E+00, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.52685000E+01, 7.00000000E+00, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.71800000E+00, 8.00000000E+00, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {4.89490000E+00, 8.00000000E+00, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.69613000E+01, 8.00000000E+00, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.01252000E+01, 8.00000000E+00, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.67247000E+01, 8.00000000E+00, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.65929000E+01, 8.00000000E+00, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.96546000E+01, 8.00000000E+00, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.55059000E+01, 8.00000000E+00, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.16160000E+00, 9.00000000E+00, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {3.88250000E+00, 9.00000000E+00, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.50933000E+01, 9.00000000E+00, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.67453000E+01, 9.00000000E+00, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.74821000E+01, 9.00000000E+00, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.02827000E+01, 9.00000000E+00, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.52418000E+01, 9.00000000E+00, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.21834000E+01, 9.00000000E+00, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.69160000E+00, 9.00000000E+00, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.01120000E+00, 1.00000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {3.10250000E+00, 1.00000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.04731000E+01, 1.00000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.74867000E+01, 1.00000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.09022000E+01, 1.00000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.56740000E+01, 1.00000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.19479000E+01, 1.00000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.66060000E+00, 1.00000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.76910000E+00, 1.00000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.28960000E+00, 1.00000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.68232000E+01, 1.10000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.68628000E+01, 1.10000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.36732720E+03, 1.10000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.87456300E+02, 1.10000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.38721200E+02, 1.10000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.03763100E+02, 1.10000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.30656300E+02, 1.10000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.30263000E+01, 1.10000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.68423000E+01, 1.10000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.92799000E+01, 1.10000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.60802860E+03, 1.10000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.83531000E+01, 1.20000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.30320000E+01, 1.20000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.30815600E+02, 1.20000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.18216400E+02, 1.20000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.58130300E+02, 1.20000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.62608200E+02, 1.20000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.07615000E+02, 1.20000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.82250000E+01, 1.20000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.71605000E+01, 1.20000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.26771000E+01, 1.20000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.85169700E+02, 1.20000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.83375800E+02, 1.20000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.62909000E+01, 1.30000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.23224000E+01, 1.30000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.05825400E+02, 1.30000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.72630200E+02, 1.30000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.36478000E+02, 1.30000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.52093400E+02, 1.30000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.02200000E+02, 1.30000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.50755000E+01, 1.30000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.53412000E+01, 1.30000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.15966000E+01, 1.30000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.38964800E+02, 1.30000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.03468900E+02, 1.30000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.40540600E+02, 1.30000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.95947000E+01, 1.40000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {1.88500000E+01, 1.40000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.95344900E+02, 1.40000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.79786300E+02, 1.40000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.84511100E+02, 1.40000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.22138700E+02, 1.40000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.38498000E+01, 1.40000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.25349000E+01, 1.40000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.66936000E+01, 1.40000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.54550000E+01, 1.40000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.91045800E+02, 1.40000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.47642300E+02, 1.40000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.08960600E+02, 1.40000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.17857400E+02, 1.40000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.37604000E+01, 1.50000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {1.56689000E+01, 1.50000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.50803000E+02, 1.50000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.08733100E+02, 1.50000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.42348100E+02, 1.50000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.67503000E+01, 1.50000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.77873000E+01, 1.50000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.13098000E+01, 1.50000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.88084000E+01, 1.50000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.97767000E+01, 1.50000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.20006400E+02, 1.50000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.30780100E+02, 1.50000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.07296500E+02, 1.50000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.44354600E+02, 1.50000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.91688700E+02, 1.50000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.00948000E+01, 1.60000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {1.36108000E+01, 1.60000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.73786700E+02, 1.60000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.67951300E+02, 1.60000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.17112100E+02, 1.60000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.10919000E+01, 1.60000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.76734000E+01, 1.60000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.41470000E+01, 1.60000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.37264000E+01, 1.60000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.60940000E+01, 1.60000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.28599000E+02, 1.60000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.64666500E+02, 1.60000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.48500800E+02, 1.60000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.00537400E+02, 1.60000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.59489800E+02, 1.60000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.34006600E+02, 1.60000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.67052000E+01, 1.70000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {1.16302000E+01, 1.70000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.10662600E+02, 1.70000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.32980800E+02, 1.70000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.47612000E+01, 1.70000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.68407000E+01, 1.70000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.82624000E+01, 1.70000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.73688000E+01, 1.70000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.88445000E+01, 1.70000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.25121000E+01, 1.70000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.53513600E+02, 1.70000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.08497800E+02, 1.70000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.97759400E+02, 1.70000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.61868600E+02, 1.70000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30472500E+02, 1.70000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.10700600E+02, 1.70000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.23460000E+01, 1.70000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.38700000E+01, 1.80000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {9.91300000E+00, 1.80000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.63549700E+02, 1.80000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.05722900E+02, 1.80000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.67949000E+01, 1.80000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.50898000E+01, 1.80000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.03435000E+01, 1.80000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.15783000E+01, 1.80000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.46178000E+01, 1.80000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.93774000E+01, 1.80000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.97344000E+02, 1.80000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.65100600E+02, 1.80000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.57959500E+02, 1.80000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30892700E+02, 1.80000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06769800E+02, 1.80000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.14014000E+01, 1.80000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.69383000E+01, 1.80000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.46462000E+01, 1.80000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.62376000E+01, 1.90000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {4.40411000E+01, 1.90000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.38715740E+03, 1.90000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.72319700E+02, 1.90000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.54198000E+02, 1.90000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.32269400E+02, 1.90000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.13200100E+02, 1.90000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.52142000E+02, 1.90000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.09643900E+02, 1.90000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.10861000E+01, 1.90000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.79861240E+03, 1.90000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.64205870E+03, 1.90000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.38793610E+03, 1.90000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.67628300E+02, 1.90000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.84496800E+02, 1.90000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.35238400E+02, 1.90000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.13151200E+02, 1.90000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.22115500E+02, 1.90000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.98350090E+03, 1.90000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.58180000E+01, 2.00000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {3.90701000E+01, 2.00000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.61447190E+03, 2.00000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.57907000E+02, 2.00000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.54843900E+02, 2.00000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.81703500E+02, 2.00000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.84549800E+02, 2.00000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.33384700E+02, 2.00000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.70824000E+01, 2.00000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.23072000E+01, 2.00000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.90770810E+03, 2.00000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.25259320E+03, 2.00000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.08884200E+03, 2.00000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.90634700E+02, 2.00000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.75462700E+02, 2.00000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.56844300E+02, 2.00000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.57473700E+02, 2.00000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.81682300E+02, 2.00000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.24043930E+03, 2.00000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.35268620E+03, 2.00000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.49669000E+01, 2.10000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {3.28995000E+01, 2.10000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.27811830E+03, 2.10000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.17335800E+02, 2.10000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.75041800E+02, 2.10000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.34195000E+02, 2.10000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.54294700E+02, 2.10000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.11935900E+02, 2.10000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.17242000E+01, 2.10000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.10154000E+01, 2.10000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.51253380E+03, 2.10000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.01554540E+03, 2.10000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.88602500E+02, 2.10000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.51204000E+02, 2.10000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.77279700E+02, 2.10000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.80410700E+02, 2.10000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.98768600E+02, 2.10000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.36141700E+02, 2.10000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.54994120E+03, 2.10000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.88879020E+03, 2.10000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.52246760E+03, 2.10000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.36875000E+01, 2.20000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {3.25316000E+01, 2.20000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.19291280E+03, 2.20000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.87489800E+02, 2.20000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.61086500E+02, 2.20000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.27572300E+02, 2.20000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.51019900E+02, 2.20000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.10145200E+02, 2.20000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.07975000E+01, 2.20000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.05594000E+01, 2.20000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.41315700E+03, 2.20000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.62983100E+02, 2.20000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.47461500E+02, 2.20000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.26235300E+02, 2.20000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.62324700E+02, 2.20000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.70263100E+02, 2.20000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.92185300E+02, 2.20000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.31920800E+02, 2.20000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.37466900E+03, 2.20000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.77951620E+03, 2.20000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.43828410E+03, 2.20000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.36191850E+03, 2.20000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.94819000E+01, 2.30000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {3.01851000E+01, 2.30000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.06904260E+03, 2.30000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.33349900E+02, 2.30000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.30102300E+02, 2.30000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.09145000E+02, 2.30000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.39351200E+02, 2.30000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.01932500E+02, 2.30000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.49671000E+01, 2.30000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.63108000E+01, 2.30000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26731390E+03, 2.30000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.72257900E+02, 2.30000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.70343200E+02, 2.30000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.72112900E+02, 2.30000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.24147000E+02, 2.30000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.40602000E+02, 2.30000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.69486300E+02, 2.30000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.14400200E+02, 2.30000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.12419850E+03, 2.30000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.60491780E+03, 2.30000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.29951970E+03, 2.30000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.23232350E+03, 2.30000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.11609840E+03, 2.30000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.91221000E+01, 2.40000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.41463000E+01, 2.40000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.44093600E+02, 2.40000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.18163500E+02, 2.40000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.59302100E+02, 2.40000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.64988200E+02, 2.40000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.10486100E+02, 2.40000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.12000000E+01, 2.40000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.00099000E+01, 2.40000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.52820000E+01, 2.40000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00046520E+03, 2.40000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.84401500E+02, 2.40000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.04407100E+02, 2.40000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.49221600E+02, 2.40000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.33838000E+02, 2.40000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.68802900E+02, 2.40000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.13366800E+02, 2.40000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.70346500E+02, 2.40000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.68367010E+03, 2.40000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26308090E+03, 2.40000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.02168710E+03, 2.40000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.68856500E+02, 2.40000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.77385500E+02, 2.40000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.90742500E+02, 2.40000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.30028000E+01, 2.50000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.64978000E+01, 2.50000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.91975600E+02, 2.50000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.53368600E+02, 2.50000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.83474400E+02, 2.50000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.81003500E+02, 2.50000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.21323300E+02, 2.50000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.91346000E+01, 2.50000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.58112000E+01, 2.50000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.95959000E+01, 2.50000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.05851030E+03, 2.50000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.39041100E+02, 2.50000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.56073800E+02, 2.50000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.90815600E+02, 2.50000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.66120400E+02, 2.50000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.95173700E+02, 2.50000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.34453300E+02, 2.50000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.87172800E+02, 2.50000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.76803220E+03, 2.50000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.35147850E+03, 2.50000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.09715390E+03, 2.50000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04266250E+03, 2.50000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.45607200E+02, 2.50000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.43317900E+02, 2.50000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.02748400E+02, 2.50000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.39110000E+01, 2.60000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.11750000E+01, 2.60000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.98453900E+02, 2.60000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.53535800E+02, 2.60000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.21754700E+02, 2.60000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.42346700E+02, 2.60000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.59782000E+01, 2.60000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.08942000E+01, 2.60000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.26310000E+01, 2.60000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.98659000E+01, 2.60000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.28893600E+02, 2.60000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.76512100E+02, 2.60000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.12082300E+02, 2.60000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.83744500E+02, 2.60000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.87160600E+02, 2.60000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.32262500E+02, 2.60000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.85182300E+02, 2.60000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.48431900E+02, 2.60000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.38895220E+03, 2.60000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.05651930E+03, 2.60000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.57173000E+02, 2.60000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.14840200E+02, 2.60000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.39054200E+02, 2.60000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.81811400E+02, 2.60000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.27540300E+02, 2.60000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.91334900E+02, 2.60000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.63234000E+01, 2.70000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.27233000E+01, 2.70000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.01532400E+02, 2.70000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.69192800E+02, 2.70000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.34869300E+02, 2.70000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.51870300E+02, 2.70000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.02750300E+02, 2.70000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.59912000E+01, 2.70000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.64307000E+01, 2.70000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.27270000E+01, 2.70000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.34182600E+02, 2.70000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.98330200E+02, 2.70000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.35970600E+02, 2.70000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.05990700E+02, 2.70000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.05925600E+02, 2.70000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.48211600E+02, 2.70000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.98357000E+02, 2.70000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.59195700E+02, 2.70000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.38297710E+03, 2.70000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.08176900E+03, 2.70000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.82450300E+02, 2.70000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.41833500E+02, 2.70000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.65317800E+02, 2.70000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.01417800E+02, 2.70000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.51968800E+02, 2.70000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.09776400E+02, 2.70000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.32779400E+02, 2.70000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.71596000E+01, 2.80000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.30792000E+01, 2.80000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.38858000E+02, 2.80000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.83914000E+02, 2.80000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.42417400E+02, 2.80000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.55850700E+02, 2.80000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.04983400E+02, 2.80000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.74010000E+01, 2.80000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.73248000E+01, 2.80000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.33129000E+01, 2.80000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.77935400E+02, 2.80000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.23630500E+02, 2.80000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.56538900E+02, 2.80000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.19359300E+02, 2.80000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.14569800E+02, 2.80000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.54468300E+02, 2.80000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.02770500E+02, 2.80000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.62326400E+02, 2.80000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.45845010E+03, 2.80000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.13210740E+03, 2.80000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.21899000E+02, 2.80000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.78093900E+02, 2.80000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.97519500E+02, 2.80000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.26647200E+02, 2.80000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.78453100E+02, 2.80000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.30322100E+02, 2.80000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.53083400E+02, 2.80000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.74743600E+02, 2.80000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.85940000E+01, 2.90000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {1.80216000E+01, 2.90000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.69452600E+02, 2.90000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.92795500E+02, 2.90000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.85174100E+02, 2.90000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.19635000E+02, 2.90000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.10855000E+01, 2.90000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.01335000E+01, 2.90000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.48098000E+01, 2.90000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.40544000E+01, 2.90000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.76477300E+02, 2.90000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.76242600E+02, 2.90000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.24772300E+02, 2.90000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.20191300E+02, 2.90000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.40797400E+02, 2.90000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.95407500E+02, 2.90000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.56316100E+02, 2.90000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.25677600E+02, 2.90000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12995180E+03, 2.90000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.68367400E+02, 2.90000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.06072000E+02, 2.90000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.72412800E+02, 2.90000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.10570000E+02, 2.90000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.80693800E+02, 2.90000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.19307800E+02, 2.90000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.06726800E+02, 2.90000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.23087700E+02, 2.90000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.39636500E+02, 2.90000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.37180800E+02, 2.90000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.98689000E+01, 3.00000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {1.89695000E+01, 3.00000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.40916000E+02, 3.00000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.93310600E+02, 3.00000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.89629300E+02, 3.00000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.24103200E+02, 3.00000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.47311000E+01, 3.00000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.30774000E+01, 3.00000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.71122000E+01, 3.00000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.58427000E+01, 3.00000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.44340800E+02, 3.00000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.72914300E+02, 3.00000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.27161000E+02, 3.00000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.27281000E+02, 3.00000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.48974300E+02, 3.00000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.03242700E+02, 3.00000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.63384800E+02, 3.00000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.31811900E+02, 3.00000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06259280E+03, 3.00000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.46852700E+02, 3.00000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.93678700E+02, 3.00000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.64100900E+02, 3.00000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.05062300E+02, 3.00000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.75559400E+02, 3.00000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.17091000E+02, 3.00000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.04555800E+02, 3.00000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.24879000E+02, 3.00000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.40090000E+02, 3.00000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.36649300E+02, 3.00000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.40521300E+02, 3.00000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.51697000E+01, 3.10000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.20458000E+01, 3.10000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.56744100E+02, 3.10000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.51114100E+02, 3.10000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.25393300E+02, 3.10000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.46554400E+02, 3.10000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.94407000E+01, 3.10000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.36253000E+01, 3.10000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.46782000E+01, 3.10000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.13681000E+01, 3.10000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.81352400E+02, 3.10000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.67163800E+02, 3.10000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.10585400E+02, 3.10000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.89265600E+02, 3.10000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.94792500E+02, 3.10000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.39818300E+02, 3.10000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.92057800E+02, 3.10000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.54342300E+02, 3.10000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.29281210E+03, 3.10000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.01988350E+03, 3.10000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.33742400E+02, 3.10000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.96861700E+02, 3.10000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.25251400E+02, 3.10000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.69764500E+02, 3.10000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.18838700E+02, 3.10000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.83781500E+02, 3.10000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.07131600E+02, 3.10000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.25816200E+02, 3.10000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.01961300E+02, 3.10000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.05456500E+02, 3.10000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.83751600E+02, 3.10000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.18170000E+01, 3.20000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.04318000E+01, 3.20000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.27926800E+02, 3.20000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.98422000E+02, 3.20000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.97344800E+02, 3.20000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.31095400E+02, 3.20000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.03328000E+01, 3.20000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.75932000E+01, 3.20000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.06412000E+01, 3.20000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.85746000E+01, 3.20000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.30104100E+02, 3.20000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.77374400E+02, 3.20000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.36506000E+02, 3.20000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.39832800E+02, 3.20000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.61823500E+02, 3.20000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.15310100E+02, 3.20000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.74196900E+02, 3.20000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.41206300E+02, 3.20000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.03266510E+03, 3.20000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.43364400E+02, 3.20000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.94702400E+02, 3.20000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.68378600E+02, 3.20000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.10763900E+02, 3.20000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.79963900E+02, 3.20000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.24184800E+02, 3.20000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.10210600E+02, 3.20000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.33850800E+02, 3.20000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.47986800E+02, 3.20000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.42426100E+02, 3.20000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.49988500E+02, 3.20000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.15953000E+02, 3.20000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.63547400E+02, 3.20000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.77884000E+01, 3.30000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {1.83284000E+01, 3.30000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.15153200E+02, 3.30000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.45483100E+02, 3.30000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.66906000E+02, 3.30000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.13261600E+02, 3.30000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.93086000E+01, 3.30000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.00289000E+01, 3.30000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.54180000E+01, 3.30000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.48686000E+01, 3.30000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.96933800E+02, 3.30000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.89492700E+02, 3.30000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.61191300E+02, 3.30000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.86590900E+02, 3.30000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.24477400E+02, 3.30000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.86634800E+02, 3.30000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.52603500E+02, 3.30000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.24856100E+02, 3.30000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.10655400E+02, 3.30000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.79004400E+02, 3.30000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.62694700E+02, 3.30000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.44679700E+02, 3.30000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.99497000E+02, 3.30000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.93192200E+02, 3.30000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.30914600E+02, 3.30000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.38009600E+02, 3.30000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.59715000E+02, 3.30000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.70035900E+02, 3.30000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.83332300E+02, 3.30000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.92503000E+02, 3.30000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.46350800E+02, 3.30000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.07101000E+02, 3.30000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.62949800E+02, 3.30000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.53098000E+01, 3.40000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {1.70243000E+01, 3.40000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.54542900E+02, 3.40000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.15023800E+02, 3.40000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.48800800E+02, 3.40000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.02434600E+02, 3.40000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.25403000E+01, 3.40000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.53627000E+01, 3.40000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.21923000E+01, 3.40000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.25843000E+01, 3.40000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.25186400E+02, 3.40000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.39574200E+02, 3.40000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.17618200E+02, 3.40000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.55009900E+02, 3.40000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.01901900E+02, 3.40000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.69130800E+02, 3.40000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.39316300E+02, 3.40000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.14744900E+02, 3.40000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.92911600E+02, 3.40000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.87964700E+02, 3.40000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.88871500E+02, 3.40000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.75034800E+02, 3.40000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.36571800E+02, 3.40000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.44295600E+02, 3.40000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.77827600E+02, 3.40000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.97039600E+02, 3.40000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.17017400E+02, 3.40000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.25341900E+02, 3.40000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.49638500E+02, 3.40000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.59045900E+02, 3.40000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.05950100E+02, 3.40000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.73653000E+02, 3.40000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.36347300E+02, 3.40000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.13673800E+02, 3.40000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.24834000E+01, 3.50000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {1.54553000E+01, 3.50000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.94789700E+02, 3.50000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.83374200E+02, 3.50000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.29249500E+02, 3.50000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.03424000E+01, 3.50000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.47632000E+01, 3.50000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.98795000E+01, 3.50000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.83225000E+01, 3.50000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.97958000E+01, 3.50000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.54286900E+02, 3.50000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.88273600E+02, 3.50000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.72009800E+02, 3.50000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.21053700E+02, 3.50000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.76993000E+02, 3.50000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.49456300E+02, 3.50000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.24090900E+02, 3.50000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.02947100E+02, 3.50000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.77231800E+02, 3.50000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.95975400E+02, 3.50000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.13746800E+02, 3.50000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.03657600E+02, 3.50000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.71807100E+02, 3.50000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.93918500E+02, 3.50000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.22843300E+02, 3.50000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.54530600E+02, 3.50000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.72305300E+02, 3.50000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.78770500E+02, 3.50000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.14505900E+02, 3.50000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.23634800E+02, 3.50000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.63345800E+02, 3.50000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.37629500E+02, 3.50000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.07077200E+02, 3.50000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.88360500E+02, 3.50000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.67129700E+02, 3.50000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.98182000E+01, 3.60000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {1.39211000E+01, 3.60000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.44555400E+02, 3.60000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.55561600E+02, 3.60000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.11537100E+02, 3.60000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.91080000E+01, 3.60000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.73895000E+01, 3.60000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.46003000E+01, 3.60000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.45452000E+01, 3.60000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.70430000E+01, 3.60000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.94557400E+02, 3.60000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.43596500E+02, 3.60000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.31690400E+02, 3.60000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.90393400E+02, 3.60000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.54057500E+02, 3.60000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.31092800E+02, 3.60000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.09683900E+02, 3.60000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.16430000E+01, 3.60000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.80247200E+02, 3.60000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.16977700E+02, 3.60000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.48847400E+02, 3.60000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.41627300E+02, 3.60000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.15325100E+02, 3.60000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.49941300E+02, 3.60000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.74641900E+02, 3.60000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.17207100E+02, 3.60000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.32759400E+02, 3.60000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.37744700E+02, 3.60000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.83534700E+02, 3.60000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.92048400E+02, 3.60000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.25453700E+02, 3.60000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.05059800E+02, 3.60000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.80181800E+02, 3.60000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.64849400E+02, 3.60000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.47183000E+02, 3.60000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30401700E+02, 3.60000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.59499000E+01, 3.70000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {5.00491000E+01, 3.70000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.64733310E+03, 3.70000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.08270520E+03, 3.70000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.20087800E+02, 3.70000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.73538400E+02, 3.70000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.40694000E+02, 3.70000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.72339800E+02, 3.70000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.24587900E+02, 3.70000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.23827000E+01, 3.70000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.10410160E+03, 3.70000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.82646480E+03, 3.70000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.54692290E+03, 3.70000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.08205630E+03, 3.70000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.68133500E+02, 3.70000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.02222100E+02, 3.70000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.66156500E+02, 3.70000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.64406700E+02, 3.70000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.53028060E+03, 3.70000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.59943490E+03, 3.70000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.83409200E+03, 3.70000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.64124350E+03, 3.70000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.36371750E+03, 3.70000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.87403260E+03, 3.70000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.96876310E+03, 3.70000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.54719820E+03, 3.70000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.54189970E+03, 3.70000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.62510340E+03, 3.70000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.25944270E+03, 3.70000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.18626310E+03, 3.70000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.44238070E+03, 3.70000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.15515260E+03, 3.70000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.09425700E+02, 3.70000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.78920400E+02, 3.70000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.50360600E+02, 3.70000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.42316000E+02, 3.70000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.13877550E+03, 3.70000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.83901000E+01, 3.80000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {4.66658000E+01, 3.80000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.94422600E+03, 3.80000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.03926500E+02, 3.80000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.41572900E+02, 3.80000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.35459700E+02, 3.80000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.19960800E+02, 3.80000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.59146000E+02, 3.80000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.15964700E+02, 3.80000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.64627000E+01, 3.80000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.29619060E+03, 3.80000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.49581410E+03, 3.80000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.29853560E+03, 3.80000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.41412100E+02, 3.80000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.84963200E+02, 3.80000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.43962700E+02, 3.80000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.25899700E+02, 3.80000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.35869000E+02, 3.80000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.91535290E+03, 3.80000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.81949350E+03, 3.80000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.26042170E+03, 3.80000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.12839510E+03, 3.80000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.91869410E+03, 3.80000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.51130430E+03, 3.80000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.61471900E+03, 3.80000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26323270E+03, 3.80000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.29087180E+03, 3.80000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.35137020E+03, 3.80000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.03772840E+03, 3.80000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00968090E+03, 3.80000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.21651380E+03, 3.80000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00448240E+03, 3.80000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.08337000E+02, 3.80000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.00053200E+02, 3.80000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.90732600E+02, 3.80000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.96912000E+02, 3.80000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.34925770E+03, 3.80000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.38136720E+03, 3.80000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.00570000E+01, 3.90000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {4.24195000E+01, 3.90000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.58539500E+03, 3.90000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.71484400E+02, 3.90000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.72436400E+02, 3.90000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.97217100E+02, 3.90000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.97061100E+02, 3.90000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.43660200E+02, 3.90000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.05344700E+02, 3.90000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.89316000E+01, 3.90000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.87686340E+03, 3.90000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26676020E+03, 3.90000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.11229800E+03, 3.90000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.19541800E+02, 3.90000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.03991600E+02, 3.90000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.83379700E+02, 3.90000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.81246200E+02, 3.90000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.02514300E+02, 3.90000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.16606250E+03, 3.90000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.35080180E+03, 3.90000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.89678090E+03, 3.90000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.79431800E+03, 3.90000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.62246890E+03, 3.90000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.27643520E+03, 3.90000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.37144920E+03, 3.90000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.07232650E+03, 3.90000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.10530160E+03, 3.90000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.15357370E+03, 3.90000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.84164800E+02, 3.90000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.70693500E+02, 3.90000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04555050E+03, 3.90000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.74783600E+02, 3.90000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.11750600E+02, 3.90000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.20333400E+02, 3.90000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.26829900E+02, 3.90000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.45701300E+02, 3.90000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.52094660E+03, 3.90000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.81370070E+03, 3.90000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.36589250E+03, 3.90000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.37801000E+01, 4.00000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {3.91088000E+01, 4.00000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.36328420E+03, 4.00000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.81528300E+02, 4.00000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.23230200E+02, 4.00000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.69049000E+02, 4.00000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.79777700E+02, 4.00000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.31785200E+02, 4.00000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.70977000E+01, 4.00000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.30315000E+01, 4.00000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.61628630E+03, 4.00000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.11383390E+03, 4.00000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.85049400E+02, 4.00000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.33180700E+02, 4.00000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.44888200E+02, 4.00000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.38389000E+02, 4.00000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.47533400E+02, 4.00000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.76990800E+02, 4.00000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.71115420E+03, 4.00000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.04852510E+03, 4.00000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.65911560E+03, 4.00000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.57414010E+03, 4.00000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42606870E+03, 4.00000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12154890E+03, 4.00000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.20874510E+03, 4.00000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.45171300E+02, 4.00000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.78976700E+02, 4.00000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.01975890E+03, 4.00000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.81113300E+02, 4.00000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.74587100E+02, 4.00000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.28198200E+02, 4.00000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.82944200E+02, 4.00000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.41556400E+02, 4.00000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.61547400E+02, 4.00000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.79004000E+02, 4.00000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.06872700E+02, 4.00000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.01768560E+03, 4.00000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.44947230E+03, 4.00000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.07261700E+03, 4.00000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.82271810E+03, 4.00000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.86768000E+01, 4.10000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {3.63418000E+01, 4.10000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.20903040E+03, 4.10000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.13985000E+02, 4.10000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.84907200E+02, 4.10000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.46524100E+02, 4.10000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.65694300E+02, 4.10000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.21986800E+02, 4.10000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.02234000E+01, 4.10000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.80769000E+01, 4.10000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.43466410E+03, 4.10000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00056630E+03, 4.10000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.89030200E+02, 4.10000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.66150700E+02, 4.10000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.97970300E+02, 4.10000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.02197400E+02, 4.10000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.20067600E+02, 4.10000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.55976900E+02, 4.10000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.40069180E+03, 4.10000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.83085230E+03, 4.10000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.48606200E+03, 4.10000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.41268180E+03, 4.10000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.28133490E+03, 4.10000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00785710E+03, 4.10000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.08798370E+03, 4.10000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.51063400E+02, 4.10000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.83889000E+02, 4.10000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.19535400E+02, 4.10000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.04378900E+02, 4.10000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.01354900E+02, 4.10000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.39282800E+02, 4.10000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.11682300E+02, 4.10000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.86008400E+02, 4.10000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.14507700E+02, 4.10000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.40299900E+02, 4.10000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.75133900E+02, 4.10000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.67393440E+03, 4.10000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.18828640E+03, 4.10000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.85863770E+03, 4.10000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.63849380E+03, 4.10000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.47525000E+03, 4.10000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.60406000E+01, 4.20000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.91617000E+01, 4.20000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.90512600E+02, 4.20000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.63306500E+02, 4.20000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.95359200E+02, 4.20000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.91952800E+02, 4.20000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.30604500E+02, 4.20000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.70675000E+01, 4.20000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.24186000E+01, 4.20000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.50487000E+01, 4.20000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.05838990E+03, 4.20000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.51662800E+02, 4.20000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.73119300E+02, 4.20000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.10235200E+02, 4.20000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.85589500E+02, 4.20000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.13865700E+02, 4.20000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.51772300E+02, 4.20000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.02857200E+02, 4.20000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.76589460E+03, 4.20000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.36548370E+03, 4.20000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.11200980E+03, 4.20000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06062550E+03, 4.20000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.63943400E+02, 4.20000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.58995500E+02, 4.20000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.20931300E+02, 4.20000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.43124700E+02, 4.20000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.70349000E+02, 4.20000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.95834800E+02, 4.20000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.33653500E+02, 4.20000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.34520700E+02, 4.20000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.37923300E+02, 4.20000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.45784500E+02, 4.20000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.53418600E+02, 4.20000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.00499800E+02, 4.20000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.44982600E+02, 4.20000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.95799400E+02, 4.20000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.96947960E+03, 4.20000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.63154240E+03, 4.20000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.39409760E+03, 4.20000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.23390950E+03, 4.20000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.11409240E+03, 4.20000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.45897200E+02, 4.20000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.10527000E+01, 4.30000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {3.20588000E+01, 4.30000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.01160990E+03, 4.30000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.21424900E+02, 4.30000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.30325700E+02, 4.30000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.13479900E+02, 4.30000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.44569700E+02, 4.30000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.07058800E+02, 4.30000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.96121000E+01, 4.30000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.03521000E+01, 4.30000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.20152340E+03, 4.30000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.47349700E+02, 4.30000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.56619500E+02, 4.30000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.71050100E+02, 4.30000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.29767700E+02, 4.30000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.48774400E+02, 4.30000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.78912300E+02, 4.30000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.24079400E+02, 4.30000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.00761370E+03, 4.30000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.54342580E+03, 4.30000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.25534460E+03, 4.30000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19584340E+03, 4.30000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.08602310E+03, 4.30000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.54747400E+02, 4.30000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.23874300E+02, 4.30000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.23320200E+02, 4.30000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.52967800E+02, 4.30000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.82248300E+02, 4.30000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.99619200E+02, 4.30000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.99314100E+02, 4.30000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.15981100E+02, 4.30000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.10532700E+02, 4.30000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.05509100E+02, 4.30000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.45477800E+02, 4.30000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.82759100E+02, 4.30000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.27382700E+02, 4.30000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.23796650E+03, 4.30000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.84437740E+03, 4.30000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.57237590E+03, 4.30000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.38959790E+03, 4.30000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.25334650E+03, 4.30000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.49649000E+02, 4.30000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06701690E+03, 4.30000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.95745000E+01, 4.40000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.54622000E+01, 4.40000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.30553300E+02, 4.40000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.87268100E+02, 4.40000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.49938200E+02, 4.40000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.64127900E+02, 4.40000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.12637800E+02, 4.40000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.42725000E+01, 4.40000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.32599000E+01, 4.40000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.83429000E+01, 4.40000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.69377400E+02, 4.40000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.26210400E+02, 4.40000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.64071600E+02, 4.40000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.31213200E+02, 4.40000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.28407200E+02, 4.40000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.68782400E+02, 4.40000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.16807000E+02, 4.40000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.75587600E+02, 4.40000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.44730480E+03, 4.40000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.13117690E+03, 4.40000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.23581100E+02, 4.40000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.83149200E+02, 4.40000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.03880700E+02, 4.40000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.33439400E+02, 4.40000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.86177200E+02, 4.40000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.38143500E+02, 4.40000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.62480000E+02, 4.40000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.82901400E+02, 4.40000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.47425900E+02, 4.40000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.50154300E+02, 4.40000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.36109300E+02, 4.40000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.61679500E+02, 4.40000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.85988300E+02, 4.40000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.42381400E+02, 4.40000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.96266800E+02, 4.40000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.55151600E+02, 4.40000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.61583870E+03, 4.40000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.35119880E+03, 4.40000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.15990710E+03, 4.40000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.02976650E+03, 4.40000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.31744500E+02, 4.40000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.10283700E+02, 4.40000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.96195100E+02, 4.40000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.98198800E+02, 4.40000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.31661000E+01, 4.50000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.75745000E+01, 4.50000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.94200600E+02, 4.50000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.24412000E+02, 4.50000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.73719500E+02, 4.50000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.79270800E+02, 4.50000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.22642600E+02, 4.50000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.14959000E+01, 4.50000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.84888000E+01, 4.50000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.22072000E+01, 4.50000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.45422100E+02, 4.50000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.85662800E+02, 4.50000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.17924400E+02, 4.50000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.72373800E+02, 4.50000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.59254700E+02, 4.50000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.93521400E+02, 4.50000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.36275000E+02, 4.50000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.90935200E+02, 4.50000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.56749540E+03, 4.50000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.23460130E+03, 4.50000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00919330E+03, 4.50000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.65230800E+02, 4.50000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.78802700E+02, 4.50000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.91631400E+02, 4.50000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.50325100E+02, 4.50000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.87754400E+02, 4.50000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.15436400E+02, 4.50000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.37763400E+02, 4.50000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.88750100E+02, 4.50000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.92631900E+02, 4.50000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.86773200E+02, 4.50000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.05445200E+02, 4.50000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.22197100E+02, 4.50000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.74063200E+02, 4.50000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.23194100E+02, 4.50000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.77868000E+02, 4.50000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.74976150E+03, 4.50000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.47354870E+03, 4.50000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26684360E+03, 4.50000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12534030E+03, 4.50000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.01829150E+03, 4.50000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.75835300E+02, 4.50000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.69896200E+02, 4.50000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.53177500E+02, 4.50000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.13942700E+02, 4.50000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.02372000E+01, 4.60000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.58716000E+01, 4.60000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.22429400E+02, 4.60000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.90431000E+02, 4.60000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.53342500E+02, 4.60000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.66711100E+02, 4.60000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.14476300E+02, 4.60000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.56427000E+01, 4.60000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.42707000E+01, 4.60000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.90990000E+01, 4.60000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.60620900E+02, 4.60000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.29591200E+02, 4.60000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.69141000E+02, 4.60000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.36946300E+02, 4.60000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.33521900E+02, 4.60000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.73156800E+02, 4.60000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.20409900E+02, 4.60000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.78502100E+02, 4.60000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42402710E+03, 4.60000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12966700E+03, 4.60000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.24855600E+02, 4.60000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.85765700E+02, 4.60000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.07125800E+02, 4.60000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.35325700E+02, 4.60000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.89964700E+02, 4.60000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.40659900E+02, 4.60000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.67104400E+02, 4.60000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.87181900E+02, 4.60000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.50055900E+02, 4.60000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.54803100E+02, 4.60000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.41150100E+02, 4.60000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.67704200E+02, 4.60000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.91852800E+02, 4.60000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.47839300E+02, 4.60000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.01139000E+02, 4.60000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.59396400E+02, 4.60000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.59044150E+03, 4.60000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.34790070E+03, 4.60000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16192890E+03, 4.60000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.03387350E+03, 4.60000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.36552800E+02, 4.60000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.14949500E+02, 4.60000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.01044200E+02, 4.60000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.02795600E+02, 4.60000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.58875300E+02, 4.60000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.08504100E+02, 4.60000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.35413000E+01, 4.70000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.17896000E+01, 4.70000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.14147700E+02, 4.70000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.25577500E+02, 4.70000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.10681100E+02, 4.70000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.38875800E+02, 4.70000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.57137000E+01, 4.70000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.18926000E+01, 4.70000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.41903000E+01, 4.70000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.15780000E+01, 4.70000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.31083200E+02, 4.70000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.26477600E+02, 4.70000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.74571300E+02, 4.70000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.63358500E+02, 4.70000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.77357700E+02, 4.70000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.27502400E+02, 4.70000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.83987600E+02, 4.70000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.49426900E+02, 4.70000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.21846990E+03, 4.70000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.51493500E+02, 4.70000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.76891400E+02, 4.70000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.43204100E+02, 4.70000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.76655300E+02, 4.70000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.33687800E+02, 4.70000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.77812700E+02, 4.70000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.53632700E+02, 4.70000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.73924400E+02, 4.70000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.90978300E+02, 4.70000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.77353100E+02, 4.70000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.79565500E+02, 4.70000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.51588700E+02, 4.70000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.89307500E+02, 4.70000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.26037100E+02, 4.70000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.89646900E+02, 4.70000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.51118900E+02, 4.70000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.16733600E+02, 4.70000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.36075170E+03, 4.70000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.13690990E+03, 4.70000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.76202500E+02, 4.70000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.67005600E+02, 4.70000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.84818800E+02, 4.70000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.99053000E+02, 4.70000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.71171700E+02, 4.70000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.05026900E+02, 4.70000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.51080200E+02, 4.70000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.08774200E+02, 4.70000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.26745000E+02, 4.70000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.56879000E+01, 4.80000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.31563000E+01, 4.80000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.21053600E+02, 4.80000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.40412200E+02, 4.80000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.22627300E+02, 4.80000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.47413800E+02, 4.80000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.01737700E+02, 4.80000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.64092000E+01, 4.80000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.75500000E+01, 4.80000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.41062000E+01, 4.80000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.40578700E+02, 4.80000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.47672100E+02, 4.80000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.97033700E+02, 4.80000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.83679000E+02, 4.80000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.94246500E+02, 4.80000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.41761400E+02, 4.80000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.95704400E+02, 4.80000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.58965000E+02, 4.80000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.22255530E+03, 4.80000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.78425100E+02, 4.80000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.02602000E+02, 4.80000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.70030300E+02, 4.80000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.02421300E+02, 4.80000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.53091900E+02, 4.80000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.01405300E+02, 4.80000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.71532900E+02, 4.80000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.95631300E+02, 4.80000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.12622000E+02, 4.80000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.93051000E+02, 4.80000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.98457200E+02, 4.80000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.73420300E+02, 4.80000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.10905800E+02, 4.80000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.45608500E+02, 4.80000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.07553900E+02, 4.80000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.66970000E+02, 4.80000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.30549300E+02, 4.80000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.36639380E+03, 4.80000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16704710E+03, 4.80000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00944180E+03, 4.80000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.00115400E+02, 4.80000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.16549500E+02, 4.80000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.24950600E+02, 4.80000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.99538200E+02, 4.80000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.27938700E+02, 4.80000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.77010700E+02, 4.80000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.33413400E+02, 4.80000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.45870100E+02, 4.80000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.68190000E+02, 4.80000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.40417000E+01, 4.90000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.79586000E+01, 4.90000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.28967300E+02, 4.90000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.38339600E+02, 4.90000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.81173700E+02, 4.90000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.83336800E+02, 4.90000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.24979800E+02, 4.90000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.29968000E+01, 4.90000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.94558000E+01, 4.90000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.28517000E+01, 4.90000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.86099900E+02, 4.90000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.09356500E+02, 4.90000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.37583200E+02, 4.90000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.85539200E+02, 4.90000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.68027500E+02, 4.90000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.99985300E+02, 4.90000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.40910400E+02, 4.90000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.94264600E+02, 4.90000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.63881530E+03, 4.90000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.28126120E+03, 4.90000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04584010E+03, 4.90000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.99097100E+02, 4.90000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.08975300E+02, 4.90000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.15279900E+02, 4.90000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.75273600E+02, 4.90000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.07096400E+02, 4.90000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.34726300E+02, 4.90000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.58252600E+02, 4.90000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.04380200E+02, 4.90000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.07245200E+02, 4.90000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.04718300E+02, 4.90000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.19350100E+02, 4.90000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.32616700E+02, 4.90000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.82596800E+02, 4.90000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.29925000E+02, 4.90000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.83126900E+02, 4.90000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.82856960E+03, 4.90000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.52968620E+03, 4.90000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.31189220E+03, 4.90000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16359630E+03, 4.90000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.05189170E+03, 4.90000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.99999400E+02, 4.90000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.97645100E+02, 4.90000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.72651600E+02, 4.90000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.35252500E+02, 4.90000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.78099600E+02, 4.90000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.67330300E+02, 4.90000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.93357400E+02, 4.90000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.57739700E+02, 4.90000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.15334000E+01, 5.00000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.67862000E+01, 5.00000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.08459700E+02, 5.00000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.93799100E+02, 5.00000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.58714600E+02, 5.00000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.71436700E+02, 5.00000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.18150900E+02, 5.00000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.85408000E+01, 5.00000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.64996000E+01, 5.00000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.08110000E+01, 5.00000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.45134800E+02, 5.00000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.31943400E+02, 5.00000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.75401800E+02, 5.00000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.45762300E+02, 5.00000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.42416900E+02, 5.00000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.81325500E+02, 5.00000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.27563500E+02, 5.00000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.84584500E+02, 5.00000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.39029120E+03, 5.00000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12306160E+03, 5.00000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.23018200E+02, 5.00000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.86631800E+02, 5.00000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.09413400E+02, 5.00000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.36656100E+02, 5.00000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.93744200E+02, 5.00000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.43362900E+02, 5.00000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.72816900E+02, 5.00000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.92046400E+02, 5.00000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.53208900E+02, 5.00000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.61197100E+02, 5.00000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.48054900E+02, 5.00000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.77103400E+02, 5.00000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.01941500E+02, 5.00000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.57787100E+02, 5.00000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.10509300E+02, 5.00000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.67953200E+02, 5.00000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.55436980E+03, 5.00000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.33856650E+03, 5.00000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16136740E+03, 5.00000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.03728120E+03, 5.00000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.41779200E+02, 5.00000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.21290600E+02, 5.00000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.07235800E+02, 5.00000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.09615200E+02, 5.00000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.66895400E+02, 5.00000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.16752200E+02, 5.00000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.14447700E+02, 5.00000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.41549000E+02, 5.00000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.85613800E+02, 5.00000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.27567700E+02, 5.00000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.77681000E+01, 5.10000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.48583000E+01, 5.10000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.86688800E+02, 5.10000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.40303000E+02, 5.10000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.29042100E+02, 5.10000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.54485100E+02, 5.10000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.07843600E+02, 5.10000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.15358000E+01, 5.10000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.16912000E+01, 5.10000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.74073000E+01, 5.10000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.01690300E+02, 5.10000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.41982600E+02, 5.10000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.99729200E+02, 5.10000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.93689900E+02, 5.10000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.06691200E+02, 5.10000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.54241800E+02, 5.10000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.07404300E+02, 5.10000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.69453300E+02, 5.10000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.14767000E+03, 5.10000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.50654300E+02, 5.10000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.85832300E+02, 5.10000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.58940600E+02, 5.10000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.95068300E+02, 5.10000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.47198100E+02, 5.10000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.98516800E+02, 5.10000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.69456900E+02, 5.10000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.98047200E+02, 5.10000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.13058500E+02, 5.10000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.93024900E+02, 5.10000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.03862100E+02, 5.10000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.78399300E+02, 5.10000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.21861600E+02, 5.10000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.59556400E+02, 5.10000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.22360800E+02, 5.10000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.81805000E+02, 5.10000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.44790200E+02, 5.10000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.28617500E+03, 5.10000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.13217130E+03, 5.10000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.92400100E+02, 5.10000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.92087700E+02, 5.10000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.13434500E+02, 5.10000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.27684900E+02, 5.10000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.00516300E+02, 5.10000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.33380400E+02, 5.10000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.83334600E+02, 5.10000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.40912500E+02, 5.10000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.50642700E+02, 5.10000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.76578300E+02, 5.10000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.98289100E+02, 5.10000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.53359200E+02, 5.10000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.92937900E+02, 5.10000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.54841000E+01, 5.20000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.36911000E+01, 5.20000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.21251100E+02, 5.20000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.09543900E+02, 5.20000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.11489800E+02, 5.20000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.44303200E+02, 5.20000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.01612400E+02, 5.20000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.72970000E+01, 5.20000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.87881000E+01, 5.20000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.53613000E+01, 5.20000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.24407600E+02, 5.20000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.90876000E+02, 5.20000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.56040000E+02, 5.20000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.62951800E+02, 5.20000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.85277900E+02, 5.20000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.37901800E+02, 5.20000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.95186100E+02, 5.20000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.60267400E+02, 5.20000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.01931210E+03, 5.20000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.55188200E+02, 5.20000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.09130300E+02, 5.20000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.87123500E+02, 5.20000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.30490100E+02, 5.20000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.96914300E+02, 5.20000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.44415600E+02, 5.20000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.27643900E+02, 5.20000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.55083600E+02, 5.20000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.67849300E+02, 5.20000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.58816700E+02, 5.20000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.70593600E+02, 5.20000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.38110200E+02, 5.20000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.89299700E+02, 5.20000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.34223400E+02, 5.20000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.01054500E+02, 5.20000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.64451600E+02, 5.20000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.30742600E+02, 5.20000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.14419130E+03, 5.20000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.01834130E+03, 5.20000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.97804300E+02, 5.20000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.10102800E+02, 5.20000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.40621100E+02, 5.20000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.74293200E+02, 5.20000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.39746500E+02, 5.20000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.89695600E+02, 5.20000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.35218900E+02, 5.20000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.97095900E+02, 5.20000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.14151200E+02, 5.20000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.38886100E+02, 5.20000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.48126700E+02, 5.20000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.09969800E+02, 5.20000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.57105700E+02, 5.20000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.25535500E+02, 5.20000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.25171000E+01, 5.30000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.20813000E+01, 5.30000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.49851200E+02, 5.30000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.73739800E+02, 5.30000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.90109400E+02, 5.30000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.31405100E+02, 5.30000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.34562000E+01, 5.30000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.16059000E+01, 5.30000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.47990000E+01, 5.30000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.24954000E+01, 5.30000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.39853600E+02, 5.30000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.32179200E+02, 5.30000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.04763100E+02, 5.30000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.25691200E+02, 5.30000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.58512100E+02, 5.30000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.17033700E+02, 5.30000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.79230600E+02, 5.30000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.48021600E+02, 5.30000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.80174100E+02, 5.30000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.47930600E+02, 5.30000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.22184100E+02, 5.30000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.05050800E+02, 5.30000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.56316600E+02, 5.30000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.39166000E+02, 5.30000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.81807700E+02, 5.30000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.79221200E+02, 5.30000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.04692900E+02, 5.30000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.15130300E+02, 5.30000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.18964200E+02, 5.30000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.31065100E+02, 5.30000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.90454200E+02, 5.30000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.49776000E+02, 5.30000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.02674300E+02, 5.30000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.74063000E+02, 5.30000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.42055600E+02, 5.30000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.12282000E+02, 5.30000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.89892100E+02, 5.30000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.90672300E+02, 5.30000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.90025700E+02, 5.30000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.15731800E+02, 5.30000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.56227600E+02, 5.30000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.11677200E+02, 5.30000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.68790900E+02, 5.30000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.37999200E+02, 5.30000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.78248100E+02, 5.30000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.44966200E+02, 5.30000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.70934400E+02, 5.30000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.93765500E+02, 5.30000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.88970900E+02, 5.30000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.57715600E+02, 5.30000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.12985200E+02, 5.30000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.86099700E+02, 5.30000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.51966700E+02, 5.30000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.96055000E+01, 5.40000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.04594000E+01, 5.40000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.86996900E+02, 5.40000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.40768900E+02, 5.40000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.69857800E+02, 5.40000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.18917800E+02, 5.40000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.54293000E+01, 5.40000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.59403000E+01, 5.40000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.07900000E+01, 5.40000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.95951000E+01, 5.40000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.65281700E+02, 5.40000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.78613400E+02, 5.40000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.57296000E+02, 5.40000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.90498500E+02, 5.40000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.32778600E+02, 5.40000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.96737400E+02, 5.40000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.63536800E+02, 5.40000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.35859200E+02, 5.40000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.58325300E+02, 5.40000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.51539200E+02, 5.40000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.43563100E+02, 5.40000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.30431200E+02, 5.40000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.88652200E+02, 5.40000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.86506700E+02, 5.40000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.24412500E+02, 5.40000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.34825900E+02, 5.40000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.58095400E+02, 5.40000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.66559700E+02, 5.40000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.82288800E+02, 5.40000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.94213800E+02, 5.40000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.46134500E+02, 5.40000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.12426500E+02, 5.40000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.72411000E+02, 5.40000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.47927500E+02, 5.40000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.20158100E+02, 5.40000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.94069900E+02, 5.40000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.54522300E+02, 5.40000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.76085900E+02, 5.40000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.92238600E+02, 5.40000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.29523000E+02, 5.40000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.78787600E+02, 5.40000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.53799800E+02, 5.40000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.03381100E+02, 5.40000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.89953200E+02, 5.40000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.25268400E+02, 5.40000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.96346800E+02, 5.40000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.30764600E+02, 5.40000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.51528500E+02, 5.40000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.34093200E+02, 5.40000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.08595200E+02, 5.40000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.70957200E+02, 5.40000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.48223900E+02, 5.40000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.18890500E+02, 5.40000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.90222300E+02, 5.40000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.05009600E+02, 5.50000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {6.13184000E+01, 5.50000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.24224040E+03, 5.50000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.31621890E+03, 5.50000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.54924300E+02, 5.50000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.55696000E+02, 5.50000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.94148300E+02, 5.50000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.10867200E+02, 5.50000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.52567200E+02, 5.50000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.13176400E+02, 5.50000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.79894390E+03, 5.50000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.22096520E+03, 5.50000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.88171640E+03, 5.50000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.31700560E+03, 5.50000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.36306900E+02, 5.50000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.35006100E+02, 5.50000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.69670100E+02, 5.50000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.45817100E+02, 5.50000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.80839000E+03, 5.50000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.38716690E+03, 5.50000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.45083690E+03, 5.50000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.21585440E+03, 5.50000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.87752790E+03, 5.50000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.28306740E+03, 5.50000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.39634450E+03, 5.50000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.88428240E+03, 5.50000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.87572090E+03, 5.50000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.97686090E+03, 5.50000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.53340630E+03, 5.50000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.44300040E+03, 5.50000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.75551050E+03, 5.50000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.40622060E+03, 5.50000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.10831450E+03, 5.50000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.50197500E+02, 5.50000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.94218400E+02, 5.50000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.62955600E+02, 5.50000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.56043610E+03, 5.50000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.30516270E+03, 5.50000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.28960970E+03, 5.50000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.67485560E+03, 5.50000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.25636330E+03, 5.50000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.39875200E+03, 5.50000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.72604860E+03, 5.50000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.96830660E+03, 5.50000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.12992750E+03, 5.50000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.93583920E+03, 5.50000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.65778290E+03, 5.50000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.66304470E+03, 5.50000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.22648670E+03, 5.50000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.89192070E+03, 5.50000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.56650420E+03, 5.50000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.39457090E+03, 5.50000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.20756720E+03, 5.50000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04339550E+03, 5.50000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.33072940E+03, 5.50000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.94579000E+01, 5.60000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {5.91141000E+01, 5.60000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.55152600E+03, 5.60000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.16044190E+03, 5.60000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.90685200E+02, 5.60000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.26366700E+02, 5.60000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.79074200E+02, 5.60000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.01741100E+02, 5.60000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.46917400E+02, 5.60000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.09500000E+02, 5.60000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.00972330E+03, 5.60000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.92635040E+03, 5.60000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.66576620E+03, 5.60000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.20120210E+03, 5.60000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.71164700E+02, 5.60000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.90865100E+02, 5.60000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.40316000E+02, 5.60000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.25790900E+02, 5.60000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.17129780E+03, 5.60000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.65916550E+03, 5.60000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.92443950E+03, 5.60000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.74897970E+03, 5.60000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.47514270E+03, 5.60000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.95211260E+03, 5.60000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.07949500E+03, 5.60000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.62840400E+03, 5.60000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.65693840E+03, 5.60000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.73639870E+03, 5.60000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.33573190E+03, 5.60000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.29266300E+03, 5.60000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.55970120E+03, 5.60000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.28191520E+03, 5.60000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.02854240E+03, 5.60000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.89614700E+02, 5.60000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.49847500E+02, 5.60000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.30243600E+02, 5.60000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.74356730E+03, 5.60000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.39436070E+03, 5.60000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.63947350E+03, 5.60000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.16017760E+03, 5.60000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.81951470E+03, 5.60000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.09848130E+03, 5.60000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.37391310E+03, 5.60000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.73548270E+03, 5.60000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.89013710E+03, 5.60000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.72727380E+03, 5.60000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.46058580E+03, 5.60000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.49373690E+03, 5.60000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.96409130E+03, 5.60000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.71088320E+03, 5.60000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.44251350E+03, 5.60000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.29563540E+03, 5.60000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.13172210E+03, 5.60000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.85126600E+02, 5.60000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.01649170E+03, 5.60000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.72698870E+03, 5.60000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.91018000E+01, 5.70000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {5.37982000E+01, 5.70000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.08506610E+03, 5.70000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.94031600E+02, 5.70000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.04491500E+02, 5.70000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.78792800E+02, 5.70000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.50557700E+02, 5.70000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.82407500E+02, 5.70000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.33612300E+02, 5.70000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.00027700E+02, 5.70000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.46559650E+03, 5.70000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.63715860E+03, 5.70000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.43174960E+03, 5.70000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04919340E+03, 5.70000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.70492400E+02, 5.70000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.15555900E+02, 5.70000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.84769100E+02, 5.70000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.84230600E+02, 5.70000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.18589680E+03, 5.70000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.05989410E+03, 5.70000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.46176190E+03, 5.70000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.32474540E+03, 5.70000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.09958530E+03, 5.70000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.65344350E+03, 5.70000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.77176530E+03, 5.70000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.38631320E+03, 5.70000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42340230E+03, 5.70000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.48714540E+03, 5.70000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.14140780E+03, 5.70000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.11839730E+03, 5.70000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.34479680E+03, 5.70000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12000210E+03, 5.70000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.08367000E+02, 5.70000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.90485500E+02, 5.70000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.70384300E+02, 5.70000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.66510100E+02, 5.70000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.65374540E+03, 5.70000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.66665560E+03, 5.70000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.06930370E+03, 5.70000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.68208790E+03, 5.70000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.40193260E+03, 5.70000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.79819700E+03, 5.70000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.02961630E+03, 5.70000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.49388950E+03, 5.70000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.62991260E+03, 5.70000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.49348550E+03, 5.70000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.25742970E+03, 5.70000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.29593640E+03, 5.70000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.68940850E+03, 5.70000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.48904400E+03, 5.70000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26826220E+03, 5.70000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.14555950E+03, 5.70000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00651040E+03, 5.70000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.80820200E+02, 5.70000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.67574340E+03, 5.70000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.75448260E+03, 5.70000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.99061720E+03, 5.70000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.45104000E+01, 5.80000000E+01, 1.00000000E+00, 2.79910000E+00, 9.11800000E-01},
    {2.93081000E+01, 5.80000000E+01, 2.00000000E+00, 2.79910000E+00, 0.00000000E+00},
    {7.04334500E+02, 5.80000000E+01, 3.00000000E+00, 2.79910000E+00, 0.00000000E+00},
    {4.03742700E+02, 5.80000000E+01, 4.00000000E+00, 2.79910000E+00, 0.00000000E+00},
    {2.70618700E+02, 5.80000000E+01, 5.00000000E+00, 2.79910000E+00, 0.00000000E+00},
    {1.82216400E+02, 5.80000000E+01, 6.00000000E+00, 2.79910000E+00, 0.00000000E+00},
    {1.27135500E+02, 5.80000000E+01, 7.00000000E+00, 2.79910000E+00, 0.00000000E+00},
    {9.61189000E+01, 5.80000000E+01, 8.00000000E+00, 2.79910000E+00, 0.00000000E+00},
    {7.27359000E+01, 5.80000000E+01, 9.00000000E+00, 2.79910000E+00, 0.00000000E+00},
    {5.59048000E+01, 5.80000000E+01, 1.00000000E+01, 2.79910000E+00, 0.00000000E+00},
    {8.41898600E+02, 5.80000000E+01, 1.10000000E+01, 2.79910000E+00, 0.00000000E+00},
    {6.44222000E+02, 5.80000000E+01, 1.20000000E+01, 2.79910000E+00, 0.00000000E+00},
    {5.92404700E+02, 5.80000000E+01, 1.30000000E+01, 2.79910000E+00, 0.00000000E+00},
    {4.65263100E+02, 5.80000000E+01, 1.40000000E+01, 2.79910000E+00, 0.00000000E+00},
    {3.61801500E+02, 5.80000000E+01, 1.50000000E+01, 2.79910000E+00, 0.00000000E+00},
    {2.99744700E+02, 5.80000000E+01, 1.60000000E+01, 2.79910000E+00, 0.00000000E+00},
    {2.44446800E+02, 5.80000000E+01, 1.70000000E+01, 2.79910000E+00, 0.00000000E+00},
    {1.99713800E+02, 5.80000000E+01, 1.80000000E+01, 2.79910000E+00, 0.00000000E+00},
    {1.38101830E+03, 5.80000000E+01, 1.90000000E+01, 2.79910000E+00, 0.00000000E+00},
    {1.13467480E+03, 5.80000000E+01, 2.00000000E+01, 2.79910000E+00, 0.00000000E+00},
    {9.36406800E+02, 5.80000000E+01, 2.10000000E+01, 2.79910000E+00, 0.00000000E+00},
    {9.03330600E+02, 5.80000000E+01, 2.20000000E+01, 2.79910000E+00, 0.00000000E+00},
    {8.26704700E+02, 5.80000000E+01, 2.30000000E+01, 2.79910000E+00, 0.00000000E+00},
    {6.51144000E+02, 5.80000000E+01, 2.40000000E+01, 2.79910000E+00, 0.00000000E+00},
    {7.11142300E+02, 5.80000000E+01, 2.50000000E+01, 2.79910000E+00, 0.00000000E+00},
    {5.58017700E+02, 5.80000000E+01, 2.60000000E+01, 2.79910000E+00, 0.00000000E+00},
    {5.90738700E+02, 5.80000000E+01, 2.70000000E+01, 2.79910000E+00, 0.00000000E+00},
    {6.08924200E+02, 5.80000000E+01, 2.80000000E+01, 2.79910000E+00, 0.00000000E+00},
    {4.66811000E+02, 5.80000000E+01, 2.90000000E+01, 2.79910000E+00, 0.00000000E+00},
    {4.78335000E+02, 5.80000000E+01, 3.00000000E+01, 2.79910000E+00, 0.00000000E+00},
    {5.66878300E+02, 5.80000000E+01, 3.10000000E+01, 2.79910000E+00, 0.00000000E+00},
    {4.98628400E+02, 5.80000000E+01, 3.20000000E+01, 2.79910000E+00, 0.00000000E+00},
    {4.24290600E+02, 5.80000000E+01, 3.30000000E+01, 2.79910000E+00, 0.00000000E+00},
    {3.80160600E+02, 5.80000000E+01, 3.40000000E+01, 2.79910000E+00, 0.00000000E+00},
    {3.32188400E+02, 5.80000000E+01, 3.50000000E+01, 2.79910000E+00, 0.00000000E+00},
    {2.88501000E+02, 5.80000000E+01, 3.60000000E+01, 2.79910000E+00, 0.00000000E+00},
    {1.54707800E+03, 5.80000000E+01, 3.70000000E+01, 2.79910000E+00, 0.00000000E+00},
    {1.35208030E+03, 5.80000000E+01, 3.80000000E+01, 2.79910000E+00, 0.00000000E+00},
    {1.18201240E+03, 5.80000000E+01, 3.90000000E+01, 2.79910000E+00, 0.00000000E+00},
    {1.06097210E+03, 5.80000000E+01, 4.00000000E+01, 2.79910000E+00, 0.00000000E+00},
    {9.66620100E+02, 5.80000000E+01, 4.10000000E+01, 2.79910000E+00, 0.00000000E+00},
    {7.45117600E+02, 5.80000000E+01, 4.20000000E+01, 2.79910000E+00, 0.00000000E+00},
    {8.31836000E+02, 5.80000000E+01, 4.30000000E+01, 2.79910000E+00, 0.00000000E+00},
    {6.32663900E+02, 5.80000000E+01, 4.40000000E+01, 2.79910000E+00, 0.00000000E+00},
    {6.91564600E+02, 5.80000000E+01, 4.50000000E+01, 2.79910000E+00, 0.00000000E+00},
    {6.40961100E+02, 5.80000000E+01, 4.60000000E+01, 2.79910000E+00, 0.00000000E+00},
    {5.34599300E+02, 5.80000000E+01, 4.70000000E+01, 2.79910000E+00, 0.00000000E+00},
    {5.64391000E+02, 5.80000000E+01, 4.80000000E+01, 2.79910000E+00, 0.00000000E+00},
    {7.09500900E+02, 5.80000000E+01, 4.90000000E+01, 2.79910000E+00, 0.00000000E+00},
    {6.54672300E+02, 5.80000000E+01, 5.00000000E+01, 2.79910000E+00, 0.00000000E+00},
    {5.82161800E+02, 5.80000000E+01, 5.10000000E+01, 2.79910000E+00, 0.00000000E+00},
    {5.39416700E+02, 5.80000000E+01, 5.20000000E+01, 2.79910000E+00, 0.00000000E+00},
    {4.87033300E+02, 5.80000000E+01, 5.30000000E+01, 2.79910000E+00, 0.00000000E+00},
    {4.37283400E+02, 5.80000000E+01, 5.40000000E+01, 2.79910000E+00, 0.00000000E+00},
    {1.88445540E+03, 5.80000000E+01, 5.50000000E+01, 2.79910000E+00, 0.00000000E+00},
    {1.72474150E+03, 5.80000000E+01, 5.60000000E+01, 2.79910000E+00, 0.00000000E+00},
    {1.51225040E+03, 5.80000000E+01, 5.70000000E+01, 2.79910000E+00, 0.00000000E+00},
    {6.88035300E+02, 5.80000000E+01, 5.80000000E+01, 2.79910000E+00, 2.79910000E+00},
    {8.86877000E+01, 5.90000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {5.31523000E+01, 5.90000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.21031680E+03, 5.90000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.01895400E+03, 5.90000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.10761500E+02, 5.90000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.79103800E+02, 5.90000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.49263300E+02, 5.90000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.80838300E+02, 5.90000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.32158400E+02, 5.90000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.88220000E+01, 5.90000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.60904400E+03, 5.90000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.68754190E+03, 5.90000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.46444460E+03, 5.90000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06152130E+03, 5.90000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.73172100E+02, 5.90000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.14865200E+02, 5.90000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.82250900E+02, 5.90000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.81050300E+02, 5.90000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.47377840E+03, 5.90000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.19020860E+03, 5.90000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.55473530E+03, 5.90000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.40505610E+03, 5.90000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.16763250E+03, 5.90000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.70911370E+03, 5.90000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.82378580E+03, 5.90000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42810390E+03, 5.90000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.45703580E+03, 5.90000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.52541410E+03, 5.90000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.17295490E+03, 5.90000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.13943520E+03, 5.90000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.37286490E+03, 5.90000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.13316270E+03, 5.90000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.12526300E+02, 5.90000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.91021400E+02, 5.90000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.68332800E+02, 5.90000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.63015600E+02, 5.90000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.97107780E+03, 5.90000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.82896260E+03, 5.90000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.18144050E+03, 5.90000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.76798940E+03, 5.90000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.47269050E+03, 5.90000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.84407150E+03, 5.90000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.08459750E+03, 5.90000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.52760560E+03, 5.90000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.66446400E+03, 5.90000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.52248060E+03, 5.90000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.28604410E+03, 5.90000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.31827910E+03, 5.90000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.72821620E+03, 5.90000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.51104590E+03, 5.90000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.27834060E+03, 5.90000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.15042000E+03, 5.90000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00701980E+03, 5.90000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.78379800E+02, 5.90000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.07389350E+03, 5.90000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.98331260E+03, 5.90000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.15032030E+03, 5.90000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.52714610E+03, 5.90000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {4.34223860E+03, 5.90000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.54080000E+01, 6.00000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {5.12952000E+01, 6.00000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.09150680E+03, 6.00000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.74616700E+02, 6.00000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.86189600E+02, 6.00000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.64651900E+02, 6.00000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.40113900E+02, 6.00000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.74367800E+02, 6.00000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.27530800E+02, 6.00000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.54202000E+01, 6.00000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.47060200E+03, 6.00000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.61178310E+03, 6.00000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.40131910E+03, 6.00000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.01850980E+03, 6.00000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.43240700E+02, 6.00000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.91675200E+02, 6.00000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.64506200E+02, 6.00000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.67316300E+02, 6.00000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.21639980E+03, 6.00000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.03636620E+03, 6.00000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.43508890E+03, 6.00000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.29428570E+03, 6.00000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.06898090E+03, 6.00000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.63060180E+03, 6.00000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.74219420E+03, 6.00000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.36384560E+03, 6.00000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.39408220E+03, 6.00000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.45873990E+03, 6.00000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12099310E+03, 6.00000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.09157220E+03, 6.00000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.31423060E+03, 6.00000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.08725330E+03, 6.00000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.77006600E+02, 6.00000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.60894900E+02, 6.00000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.43430500E+02, 6.00000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.42438300E+02, 6.00000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.68508460E+03, 6.00000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.64204920E+03, 6.00000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.03291830E+03, 6.00000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.64205260E+03, 6.00000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.36169740E+03, 6.00000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.76311390E+03, 6.00000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.99215340E+03, 6.00000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.46164580E+03, 6.00000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.59336960E+03, 6.00000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.45816710E+03, 6.00000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.23049690E+03, 6.00000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26334820E+03, 6.00000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.65350390E+03, 6.00000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.44887110E+03, 6.00000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.22774550E+03, 6.00000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.10581430E+03, 6.00000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.68794700E+02, 6.00000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.45670300E+02, 6.00000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.71679990E+03, 6.00000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.73341850E+03, 6.00000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.95199560E+03, 6.00000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.46596950E+03, 6.00000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {4.12590230E+03, 6.00000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.92442110E+03, 6.00000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.33310000E+01, 6.10000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {5.00748000E+01, 6.10000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.03127400E+03, 6.10000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.49125500E+02, 6.10000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.71411300E+02, 6.10000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.55671200E+02, 6.10000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.34291300E+02, 6.10000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.70181700E+02, 6.10000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.24494600E+02, 6.10000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.31635000E+01, 6.10000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.39982070E+03, 6.10000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.56898850E+03, 6.10000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.36484940E+03, 6.10000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.92750700E+02, 6.10000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.24819700E+02, 6.10000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.77169600E+02, 6.10000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.53231300E+02, 6.10000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.58472100E+02, 6.10000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.09205750E+03, 6.10000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.95297940E+03, 6.10000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.36912440E+03, 6.10000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.23265030E+03, 6.10000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.01371940E+03, 6.10000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.58685120E+03, 6.10000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.69604410E+03, 6.10000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.32760640E+03, 6.10000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.35773950E+03, 6.10000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42050450E+03, 6.10000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.09142140E+03, 6.10000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06348980E+03, 6.10000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.28017620E+03, 6.10000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.05975360E+03, 6.10000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.55217700E+02, 6.10000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.42164900E+02, 6.10000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.27734200E+02, 6.10000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.29307000E+02, 6.10000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.54707200E+03, 6.10000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.54149300E+03, 6.10000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.95093030E+03, 6.10000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.57149640E+03, 6.10000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.29904810E+03, 6.10000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.71680380E+03, 6.10000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.93962150E+03, 6.10000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42355310E+03, 6.10000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.55205220E+03, 6.10000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42054690E+03, 6.10000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19841940E+03, 6.10000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.23095980E+03, 6.10000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.61041420E+03, 6.10000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.41195880E+03, 6.10000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19701570E+03, 6.10000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.07838540E+03, 6.10000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.44981400E+02, 6.10000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.25047300E+02, 6.10000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.54749000E+03, 6.10000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.60121710E+03, 6.10000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.84403670E+03, 6.10000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42906900E+03, 6.10000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {4.01136350E+03, 6.10000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.81611180E+03, 6.10000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.71093750E+03, 6.10000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.14123000E+01, 6.20000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {4.89406000E+01, 6.20000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.97669550E+03, 6.20000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.25815800E+02, 6.20000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.57838000E+02, 6.20000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.47392100E+02, 6.20000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.28906500E+02, 6.20000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.66300400E+02, 6.20000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.21672700E+02, 6.20000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.10614000E+01, 6.20000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.33564740E+03, 6.20000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.52991200E+03, 6.20000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.33147860E+03, 6.20000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.69102600E+02, 6.20000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.07860400E+02, 6.20000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.63789000E+02, 6.20000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.42809500E+02, 6.20000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.50281200E+02, 6.20000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.97963090E+03, 6.20000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.87706800E+03, 6.20000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.30899620E+03, 6.20000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.17641840E+03, 6.20000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.96327230E+03, 6.20000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.54691950E+03, 6.20000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.65387730E+03, 6.20000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.29449510E+03, 6.20000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.32447700E+03, 6.20000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.38553130E+03, 6.20000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06437980E+03, 6.20000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.03774900E+03, 6.20000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.24899150E+03, 6.20000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.03450240E+03, 6.20000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.35162200E+02, 6.20000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.24898700E+02, 6.20000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.13240500E+02, 6.20000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.17161700E+02, 6.20000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.42226960E+03, 6.20000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.44999060E+03, 6.20000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.87617160E+03, 6.20000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.50708050E+03, 6.20000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.24180850E+03, 6.20000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.67443600E+03, 6.20000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.89158830E+03, 6.20000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.38866680E+03, 6.20000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.51420160E+03, 6.20000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.38606220E+03, 6.20000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16903600E+03, 6.20000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.20124730E+03, 6.20000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.57096430E+03, 6.20000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.37808300E+03, 6.20000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16875240E+03, 6.20000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.05312550E+03, 6.20000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.23019400E+02, 6.20000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.06000200E+02, 6.20000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.39448460E+03, 6.20000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.48104490E+03, 6.20000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.74569210E+03, 6.20000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.39514710E+03, 6.20000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {3.90719230E+03, 6.20000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.71754070E+03, 6.20000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.61520670E+03, 6.20000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.52205080E+03, 6.20000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.97138000E+01, 6.30000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {4.79365000E+01, 6.30000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.92808100E+03, 6.30000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.05110100E+02, 6.30000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.45801500E+02, 6.30000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.40058000E+02, 6.30000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.24138300E+02, 6.30000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.62863800E+02, 6.30000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.19173400E+02, 6.30000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.91988000E+01, 6.30000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.27848980E+03, 6.30000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.49517920E+03, 6.30000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30184380E+03, 6.30000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.48128800E+02, 6.30000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.92833800E+02, 6.30000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.51939000E+02, 6.30000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.33583100E+02, 6.30000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.43030700E+02, 6.30000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.87947290E+03, 6.30000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.80952330E+03, 6.30000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.25551570E+03, 6.30000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.12641900E+03, 6.30000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.91842540E+03, 6.30000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.51141560E+03, 6.30000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.61640230E+03, 6.30000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26506430E+03, 6.30000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.29493130E+03, 6.30000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.35445950E+03, 6.30000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04034900E+03, 6.30000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.01489590E+03, 6.30000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.22130300E+03, 6.30000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.01210530E+03, 6.30000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.17388800E+02, 6.30000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.09603900E+02, 6.30000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.00406400E+02, 6.30000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.06409800E+02, 6.30000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.31109610E+03, 6.30000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.36856460E+03, 6.30000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.80968850E+03, 6.30000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.44981880E+03, 6.30000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.19093870E+03, 6.30000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.63679560E+03, 6.30000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.84890970E+03, 6.30000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.35768100E+03, 6.30000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.48058780E+03, 6.30000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.35544250E+03, 6.30000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.14293520E+03, 6.30000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.17486990E+03, 6.30000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.53592540E+03, 6.30000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.34802160E+03, 6.30000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.14369190E+03, 6.30000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.03073800E+03, 6.30000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.03563000E+02, 6.30000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.89132000E+02, 6.30000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.25820680E+03, 6.30000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.37408260E+03, 6.30000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.65821200E+03, 6.30000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.36506150E+03, 6.30000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {3.81449210E+03, 6.30000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.62982680E+03, 6.30000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.53002280E+03, 6.30000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.43916010E+03, 6.30000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.35831220E+03, 6.30000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.41880000E+01, 6.40000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {3.95786000E+01, 6.40000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.40773350E+03, 6.40000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.89928700E+02, 6.40000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.26489700E+02, 6.40000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.70918300E+02, 6.40000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.81258100E+02, 6.40000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.33143800E+02, 6.40000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.83573000E+01, 6.40000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.41912000E+01, 6.40000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.66738950E+03, 6.40000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.13081980E+03, 6.40000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.96792400E+02, 6.40000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.39020000E+02, 6.40000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.48363300E+02, 6.40000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.41230300E+02, 6.40000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.50036900E+02, 6.40000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.79351500E+02, 6.40000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.81763330E+03, 6.40000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.09453260E+03, 6.40000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.69170520E+03, 6.40000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.60289270E+03, 6.40000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.45071740E+03, 6.40000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.14270150E+03, 6.40000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.22804260E+03, 6.40000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.61560000E+02, 6.40000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.92067300E+02, 6.40000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.03419320E+03, 6.40000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.93876300E+02, 6.40000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.83511400E+02, 6.40000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.39365200E+02, 6.40000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.89636900E+02, 6.40000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.45984900E+02, 6.40000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.65299100E+02, 6.40000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.82317200E+02, 6.40000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.09980300E+02, 6.40000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.13592420E+03, 6.40000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.50775890E+03, 6.40000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.11320760E+03, 6.40000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.85448370E+03, 6.40000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.66543720E+03, 6.40000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.25307300E+03, 6.40000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.41167590E+03, 6.40000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04506170E+03, 6.40000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.14045250E+03, 6.40000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04712790E+03, 6.40000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.80531300E+02, 6.40000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.11079600E+02, 6.40000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.17997200E+03, 6.40000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04809760E+03, 6.40000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.99516200E+02, 6.40000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.16304700E+02, 6.40000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.20984700E+02, 6.40000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.34216300E+02, 6.40000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.82384260E+03, 6.40000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.24365610E+03, 6.40000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.74054780E+03, 6.40000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.07095970E+03, 6.40000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {2.83813570E+03, 6.40000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.70568260E+03, 6.40000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.63266570E+03, 6.40000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.56606070E+03, 6.40000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.50683520E+03, 6.40000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.89167190E+03, 6.40000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.01794000E+01, 6.50000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {4.23679000E+01, 6.50000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.80238520E+03, 6.50000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.08486600E+02, 6.50000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.82797700E+02, 6.50000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.99846900E+02, 6.50000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.97623400E+02, 6.50000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.43775400E+02, 6.50000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.05397000E+02, 6.50000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.90484000E+01, 6.50000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.12408310E+03, 6.50000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.34336510E+03, 6.50000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16215600E+03, 6.50000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.39177600E+02, 6.50000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.10831800E+02, 6.50000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.86231900E+02, 6.50000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.81958400E+02, 6.50000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.02414900E+02, 6.50000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.68763050E+03, 6.50000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.56414430E+03, 6.50000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.04549270E+03, 6.50000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.92283020E+03, 6.50000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.73098900E+03, 6.50000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.36800490E+03, 6.50000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.45416140E+03, 6.50000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.14087130E+03, 6.50000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.15790110E+03, 6.50000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.21320140E+03, 6.50000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.35754400E+02, 6.50000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.03599100E+02, 6.50000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.09009390E+03, 6.50000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.96465100E+02, 6.50000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.21204200E+02, 6.50000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.25448000E+02, 6.50000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.28938100E+02, 6.50000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.46208400E+02, 6.50000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.09856780E+03, 6.50000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.08398990E+03, 6.50000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.54859110E+03, 6.50000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.21149070E+03, 6.50000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.97356480E+03, 6.50000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.47052650E+03, 6.50000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.66313470E+03, 6.50000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.21735110E+03, 6.50000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.32352230E+03, 6.50000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.20968470E+03, 6.50000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.02568350E+03, 6.50000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04657890E+03, 6.50000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.37560670E+03, 6.50000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19755970E+03, 6.50000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.01129750E+03, 6.50000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.09912700E+02, 6.50000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.96699500E+02, 6.50000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.95435000E+02, 6.50000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.02269670E+03, 6.50000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.03009100E+03, 6.50000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.33574350E+03, 6.50000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.20962870E+03, 6.50000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {3.50717160E+03, 6.50000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.32507970E+03, 6.50000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.23132410E+03, 6.50000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.14617400E+03, 6.50000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.07038530E+03, 6.50000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.27625320E+03, 6.50000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.85166770E+03, 6.50000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.79580000E+01, 6.60000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {4.11622000E+01, 6.60000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.72073530E+03, 6.60000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.75771800E+02, 6.60000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.65243900E+02, 6.60000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.89845500E+02, 6.60000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.91455000E+02, 6.60000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.39493700E+02, 6.60000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.02381100E+02, 6.60000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.68557000E+01, 6.60000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.02785680E+03, 6.60000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.28724710E+03, 6.60000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.11603410E+03, 6.60000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.08332200E+02, 6.60000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.89894100E+02, 6.60000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.70297700E+02, 6.60000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.69983300E+02, 6.60000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.93290500E+02, 6.60000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.52697050E+03, 6.60000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.45177340E+03, 6.60000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.95745810E+03, 6.60000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.84158830E+03, 6.60000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.65870190E+03, 6.60000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.31075380E+03, 6.60000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.39448530E+03, 6.60000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.09400160E+03, 6.60000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.11181740E+03, 6.60000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16427550E+03, 6.60000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.97837300E+02, 6.60000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.68757700E+02, 6.60000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04756400E+03, 6.60000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.63586000E+02, 6.60000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.96276000E+02, 6.60000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.04604800E+02, 6.60000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.11971600E+02, 6.60000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.32389300E+02, 6.60000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.92169620E+03, 6.60000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.94856880E+03, 6.60000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.44017620E+03, 6.60000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.11949080E+03, 6.60000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.89278580E+03, 6.60000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.41173630E+03, 6.60000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.59620740E+03, 6.60000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16967850E+03, 6.60000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.27189370E+03, 6.60000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16303740E+03, 6.60000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.85569700E+02, 6.60000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00683820E+03, 6.60000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.32152570E+03, 6.60000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.15279750E+03, 6.60000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.75443500E+02, 6.60000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.78675300E+02, 6.60000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.70290800E+02, 6.60000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.73139400E+02, 6.60000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.81233230E+03, 6.60000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.85229490E+03, 6.60000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.19240390E+03, 6.60000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16616330E+03, 6.60000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {3.35587650E+03, 6.60000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.17996290E+03, 6.60000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.09040720E+03, 6.60000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.00907780E+03, 6.60000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.93670650E+03, 6.60000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.18106750E+03, 6.60000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.72997390E+03, 6.60000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.61733100E+03, 6.60000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.22203000E+01, 6.70000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {4.35711000E+01, 6.70000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.70496790E+03, 6.70000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.11368800E+02, 6.70000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.91887500E+02, 6.70000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.07519800E+02, 6.70000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.03155400E+02, 6.70000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.47835500E+02, 6.70000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.08307600E+02, 6.70000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.11416000E+01, 6.70000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.01633920E+03, 6.70000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.33747830E+03, 6.70000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16793580E+03, 6.70000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.54074000E+02, 6.70000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.25939400E+02, 6.70000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.99457600E+02, 6.70000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.92938500E+02, 6.70000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.11248700E+02, 6.70000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.41951600E+03, 6.70000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.50141640E+03, 6.70000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.01203520E+03, 6.70000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.89920440E+03, 6.70000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.71486350E+03, 6.70000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.35029320E+03, 6.70000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.44660200E+03, 6.70000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.13177100E+03, 6.70000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16149050E+03, 6.70000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.21393480E+03, 6.70000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.31679300E+02, 6.70000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.12008200E+02, 6.70000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.09644500E+03, 6.70000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.11722300E+02, 6.70000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.38212900E+02, 6.70000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.41745200E+02, 6.70000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.43716200E+02, 6.70000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.59120500E+02, 6.70000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.80083040E+03, 6.70000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.99702120E+03, 6.70000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.50735720E+03, 6.70000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.19002720E+03, 6.70000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.96051940E+03, 6.70000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.46680170E+03, 6.70000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.65594660E+03, 6.70000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.21806510E+03, 6.70000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.32914260E+03, 6.70000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.21765820E+03, 6.70000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.02539020E+03, 6.70000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.05637180E+03, 6.70000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.37789010E+03, 6.70000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.21310430E+03, 6.70000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.03180710E+03, 6.70000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.31123900E+02, 6.70000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.17322000E+02, 6.70000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.14647700E+02, 6.70000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.63284280E+03, 6.70000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.88550930E+03, 6.70000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.25991320E+03, 6.70000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.23059890E+03, 6.70000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {3.39154110E+03, 6.70000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.22981680E+03, 6.70000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.14160980E+03, 6.70000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.06125370E+03, 6.70000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.98976600E+03, 6.70000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.23790260E+03, 6.70000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.72436020E+03, 6.70000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.60651360E+03, 6.70000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.66416680E+03, 6.70000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.07154000E+01, 6.80000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {4.26658000E+01, 6.80000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.66477000E+03, 6.80000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.93687700E+02, 6.80000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.81438500E+02, 6.80000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.01069200E+02, 6.80000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.98918500E+02, 6.80000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.44758700E+02, 6.80000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.06055200E+02, 6.80000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.94539000E+01, 6.80000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.96899280E+03, 6.80000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30798140E+03, 6.80000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.14256560E+03, 6.80000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.35896900E+02, 6.80000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.12780900E+02, 6.80000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.89010700E+02, 6.80000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.84749600E+02, 6.80000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.04774800E+02, 6.80000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.33722270E+03, 6.80000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.44467580E+03, 6.80000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.96690500E+03, 6.80000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.85687430E+03, 6.80000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.67681290E+03, 6.80000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.32019120E+03, 6.80000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.41470480E+03, 6.80000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.10672460E+03, 6.80000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.13618970E+03, 6.80000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.18738730E+03, 6.80000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.11170700E+02, 6.80000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.92333200E+02, 6.80000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.07267680E+03, 6.80000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.92301700E+02, 6.80000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.22665400E+02, 6.80000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.28293400E+02, 6.80000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.32364100E+02, 6.80000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.49559200E+02, 6.80000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.70942970E+03, 6.80000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.92871640E+03, 6.80000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.45117750E+03, 6.80000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.14142020E+03, 6.80000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.91722030E+03, 6.80000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.43461300E+03, 6.80000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.61951860E+03, 6.80000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19147160E+03, 6.80000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30026370E+03, 6.80000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19129620E+03, 6.80000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00297970E+03, 6.80000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.03360090E+03, 6.80000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.34784970E+03, 6.80000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.18710400E+03, 6.80000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00995830E+03, 6.80000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.11513700E+02, 6.80000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.00191800E+02, 6.80000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.99723100E+02, 6.80000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.52096450E+03, 6.80000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.79609240E+03, 6.80000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.18623270E+03, 6.80000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.20441850E+03, 6.80000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {3.31388790E+03, 6.80000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.15619920E+03, 6.80000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.07008420E+03, 6.80000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.99162750E+03, 6.80000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.92183100E+03, 6.80000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.18778030E+03, 6.80000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.66114610E+03, 6.80000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.54612490E+03, 6.80000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.60396530E+03, 6.80000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.54517130E+03, 6.80000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.93791000E+01, 6.90000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {4.18727000E+01, 6.90000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.62709510E+03, 6.90000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.77460200E+02, 6.90000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.71986000E+02, 6.90000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.95301400E+02, 6.90000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.95163000E+02, 6.90000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.42047800E+02, 6.90000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.04080000E+02, 6.90000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.79788000E+01, 6.90000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.92465930E+03, 6.90000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.28079110E+03, 6.90000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.11933910E+03, 6.90000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.19428000E+02, 6.90000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.00970100E+02, 6.90000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.79690900E+02, 6.90000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.77487300E+02, 6.90000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.99062200E+02, 6.90000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.25993680E+03, 6.90000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.39199540E+03, 6.90000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.92512630E+03, 6.90000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.81778810E+03, 6.90000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.64173540E+03, 6.90000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.29243840E+03, 6.90000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.38537120E+03, 6.90000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.08369570E+03, 6.90000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.11302590E+03, 6.90000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16303690E+03, 6.90000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.92351600E+02, 6.90000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.74396100E+02, 6.90000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.05097050E+03, 6.90000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.74713200E+02, 6.90000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.08695000E+02, 6.90000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.16265800E+02, 6.90000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.22265500E+02, 6.90000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.41093200E+02, 6.90000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.62365110E+03, 6.90000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.86526130E+03, 6.90000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.39924870E+03, 6.90000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.09664000E+03, 6.90000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.87741740E+03, 6.90000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.40513520E+03, 6.90000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.58611000E+03, 6.90000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16718780E+03, 6.90000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.27390260E+03, 6.90000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16727070E+03, 6.90000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.82521000E+02, 6.90000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.01289040E+03, 6.90000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.32038870E+03, 6.90000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16349880E+03, 6.90000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.90258800E+02, 6.90000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.93908000E+02, 6.90000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.84884100E+02, 6.90000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.86445200E+02, 6.90000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.41596940E+03, 6.90000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.71288120E+03, 6.90000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.11800010E+03, 6.90000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.18077760E+03, 6.90000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {3.24172660E+03, 6.90000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.08784890E+03, 6.90000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.00369220E+03, 6.90000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.92701180E+03, 6.90000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.85879770E+03, 6.90000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.14152640E+03, 6.90000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.60231220E+03, 6.90000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.48996430E+03, 6.90000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.54818020E+03, 6.90000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.49069890E+03, 6.90000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.43745390E+03, 6.90000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.84949000E+01, 7.00000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {4.12866000E+01, 7.00000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.61275810E+03, 7.00000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.69398300E+02, 7.00000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.66609900E+02, 7.00000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.91681700E+02, 7.00000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.92632400E+02, 7.00000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.40130000E+02, 7.00000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.02625100E+02, 7.00000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.68581000E+01, 7.00000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.90753320E+03, 7.00000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26788660E+03, 7.00000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.10752150E+03, 7.00000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.10184500E+02, 7.00000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.93791100E+02, 7.00000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.73739000E+02, 7.00000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.72626700E+02, 7.00000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.95083600E+02, 7.00000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.23171170E+03, 7.00000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.36909580E+03, 7.00000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.90628020E+03, 7.00000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.79963770E+03, 7.00000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.62514300E+03, 7.00000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.27934930E+03, 7.00000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.37111980E+03, 7.00000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.07249590E+03, 7.00000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.10121820E+03, 7.00000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.15085530E+03, 7.00000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.82991900E+02, 7.00000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.64854600E+02, 7.00000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.03966010E+03, 7.00000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.64799600E+02, 7.00000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.00271800E+02, 7.00000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.08717400E+02, 7.00000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.15664900E+02, 7.00000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.35353500E+02, 7.00000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.59204650E+03, 7.00000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.83794230E+03, 7.00000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.37546790E+03, 7.00000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.07534170E+03, 7.00000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.85803530E+03, 7.00000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.39020590E+03, 7.00000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.56944260E+03, 7.00000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.15452090E+03, 7.00000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26007640E+03, 7.00000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.15446390E+03, 7.00000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.71809300E+02, 7.00000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00162140E+03, 7.00000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30620230E+03, 7.00000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.15045710E+03, 7.00000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.78687100E+02, 7.00000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.83191900E+02, 7.00000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.75209100E+02, 7.00000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.77754400E+02, 7.00000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.37747660E+03, 7.00000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.67796060E+03, 7.00000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.08751540E+03, 7.00000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16707990E+03, 7.00000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {3.21084890E+03, 7.00000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.05822770E+03, 7.00000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.97482380E+03, 7.00000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.89883490E+03, 7.00000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.83123440E+03, 7.00000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.11997940E+03, 7.00000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.57780050E+03, 7.00000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.46635830E+03, 7.00000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.52336300E+03, 7.00000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.46641430E+03, 7.00000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.41365140E+03, 7.00000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.39012270E+03, 7.00000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.86519000E+01, 7.10000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {3.59516000E+01, 7.10000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.29475830E+03, 7.10000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.34588100E+02, 7.10000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.91465200E+02, 7.10000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.47955700E+02, 7.10000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.65418400E+02, 7.10000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.21214800E+02, 7.10000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.93405000E+01, 7.10000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.72575000E+01, 7.10000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.53326750E+03, 7.10000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04018250E+03, 7.10000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.16382400E+02, 7.10000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.78554100E+02, 7.10000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.02551700E+02, 7.10000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.03671800E+02, 7.10000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.19628600E+02, 7.10000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.54597900E+02, 7.10000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.59212990E+03, 7.10000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.92554530E+03, 7.10000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.55539390E+03, 7.10000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.47340910E+03, 7.10000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.33340270E+03, 7.10000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04961420E+03, 7.10000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12853090E+03, 7.10000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.83021700E+02, 7.10000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.11415000E+02, 7.10000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.50327800E+02, 7.10000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.28892600E+02, 7.10000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.19524100E+02, 7.10000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.62913900E+02, 7.10000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.24751300E+02, 7.10000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.92058700E+02, 7.10000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.17457600E+02, 7.10000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.40842800E+02, 7.10000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.74140200E+02, 7.10000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.88475850E+03, 7.10000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.30485580E+03, 7.10000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.94213870E+03, 7.10000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.70404400E+03, 7.10000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.52997210E+03, 7.10000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.15016040E+03, 7.10000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.29627010E+03, 7.10000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.58699200E+02, 7.10000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04675970E+03, 7.10000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.60912800E+02, 7.10000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.07440600E+02, 7.10000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.35831000E+02, 7.10000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.08330080E+03, 7.10000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.61840200E+02, 7.10000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.24696400E+02, 7.10000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.47757900E+02, 7.10000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.59704900E+02, 7.10000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.79591100E+02, 7.10000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.51952360E+03, 7.10000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.98064360E+03, 7.10000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.51821920E+03, 7.10000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.81753100E+02, 7.10000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {2.60891930E+03, 7.10000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.48619310E+03, 7.10000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.41911640E+03, 7.10000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.35794390E+03, 7.10000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.30355400E+03, 7.10000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.73765650E+03, 7.10000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.09193800E+03, 7.10000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.00592430E+03, 7.10000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.05653100E+03, 7.10000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.01052030E+03, 7.10000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.96804740E+03, 7.10000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.94831020E+03, 7.10000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.59747960E+03, 7.10000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.88504000E+01, 7.20000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {3.65749000E+01, 7.20000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.17722680E+03, 7.20000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.07585900E+02, 7.20000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.83508400E+02, 7.20000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.46682600E+02, 7.20000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.66247200E+02, 7.20000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.22591300E+02, 7.20000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.07784000E+01, 7.20000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.85505000E+01, 7.20000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.39812540E+03, 7.20000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.87523900E+02, 7.20000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.80790200E+02, 7.20000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.63331400E+02, 7.20000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.97701500E+02, 7.20000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.02794700E+02, 7.20000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.21114500E+02, 7.20000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.57165100E+02, 7.20000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.33005620E+03, 7.20000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.79710240E+03, 7.20000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.46195890E+03, 7.20000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.39198480E+03, 7.20000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26386600E+03, 7.20000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.93628200E+02, 7.20000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.07472870E+03, 7.20000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.40453700E+02, 7.20000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.75433600E+02, 7.20000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.09840700E+02, 7.20000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.96428300E+02, 7.20000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.96213500E+02, 7.20000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.32324900E+02, 7.20000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.08697100E+02, 7.20000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.85425000E+02, 7.20000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.14878400E+02, 7.20000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.41337700E+02, 7.20000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.76528700E+02, 7.20000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.59640330E+03, 7.20000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.14640400E+03, 7.20000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.82975150E+03, 7.20000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.61647370E+03, 7.20000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.45724920E+03, 7.20000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.10256690E+03, 7.20000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.23952010E+03, 7.20000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.23419800E+02, 7.20000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00975270E+03, 7.20000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.29467600E+02, 7.20000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.77799500E+02, 7.20000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.11204000E+02, 7.20000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04229930E+03, 7.20000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.36636100E+02, 7.20000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.11486900E+02, 7.20000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.40066900E+02, 7.20000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.56817100E+02, 7.20000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.80137700E+02, 7.20000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.16064460E+03, 7.20000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.76090940E+03, 7.20000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.36091860E+03, 7.20000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.63477300E+02, 7.20000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {2.42404910E+03, 7.20000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.31705140E+03, 7.20000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.25605840E+03, 7.20000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.20029060E+03, 7.20000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.15074170E+03, 7.20000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.64065670E+03, 7.20000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.93099490E+03, 7.20000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.85295810E+03, 7.20000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.92659250E+03, 7.20000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.88430480E+03, 7.20000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.84550500E+03, 7.20000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.82621490E+03, 7.20000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.50724410E+03, 7.20000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.44123940E+03, 7.20000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.44015000E+01, 7.30000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {3.42834000E+01, 7.30000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.02735770E+03, 7.30000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.44435500E+02, 7.30000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.48628500E+02, 7.30000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.26715900E+02, 7.30000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.54069600E+02, 7.30000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.14296700E+02, 7.30000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.50814000E+01, 7.30000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.45238000E+01, 7.30000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.22203350E+03, 7.30000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.80846900E+02, 7.30000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.91413500E+02, 7.30000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.02134700E+02, 7.30000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.55697000E+02, 7.30000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.70872300E+02, 7.30000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.97283900E+02, 7.30000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.39229100E+02, 7.30000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.02666980E+03, 7.30000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.58948830E+03, 7.30000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.29778260E+03, 7.30000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.23950330E+03, 7.30000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12758120E+03, 7.30000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.86588000E+02, 7.30000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.61519400E+02, 7.30000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.52310400E+02, 7.30000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.87032700E+02, 7.30000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.16345500E+02, 7.30000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.24840800E+02, 7.30000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.28674100E+02, 7.30000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.49977600E+02, 7.30000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.43730100E+02, 7.30000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.35601500E+02, 7.30000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.73168500E+02, 7.30000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.07469100E+02, 7.30000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.49130500E+02, 7.30000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.26084000E+03, 7.30000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.89697990E+03, 7.30000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.62726700E+03, 7.30000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.44319730E+03, 7.30000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30432130E+03, 7.30000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.91165700E+02, 7.30000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.11245110E+03, 7.30000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.32819100E+02, 7.30000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.10842600E+02, 7.30000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.39835100E+02, 7.30000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.01928400E+02, 7.30000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.34565900E+02, 7.30000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.38770200E+02, 7.30000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.49257900E+02, 7.30000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.40510100E+02, 7.30000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.77943000E+02, 7.30000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.04191800E+02, 7.30000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.35777300E+02, 7.30000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.75165090E+03, 7.30000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.43442070E+03, 7.30000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.09485310E+03, 7.30000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.78109800E+02, 7.30000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {2.14176670E+03, 7.30000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.04961250E+03, 7.30000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.99628790E+03, 7.30000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.94746690E+03, 7.30000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.90410850E+03, 7.30000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.46231310E+03, 7.30000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.70267050E+03, 7.30000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.63557520E+03, 7.30000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.70849120E+03, 7.30000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.67131100E+03, 7.30000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.63732250E+03, 7.30000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.61977100E+03, 7.30000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.34286970E+03, 7.30000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.29258360E+03, 7.30000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16382410E+03, 7.30000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.64813000E+01, 7.40000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.98206000E+01, 7.40000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.42412300E+02, 7.40000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.51986100E+02, 7.40000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.92901500E+02, 7.40000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.92598400E+02, 7.40000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.32141000E+02, 7.40000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.87666000E+01, 7.40000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.40309000E+01, 7.40000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.64767000E+01, 7.40000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00295080E+03, 7.40000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.29335200E+02, 7.40000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.58747100E+02, 7.40000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.05189000E+02, 7.40000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.85410900E+02, 7.40000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.15563200E+02, 7.40000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.54528700E+02, 7.40000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.06028600E+02, 7.40000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.66443210E+03, 7.40000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.31176410E+03, 7.40000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.07280500E+03, 7.40000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.02692140E+03, 7.40000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.35385900E+02, 7.40000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.36506300E+02, 7.40000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.99175800E+02, 7.40000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.26338000E+02, 7.40000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.56226000E+02, 7.40000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.79631400E+02, 7.40000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.21082200E+02, 7.40000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.25889400E+02, 7.40000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.26220300E+02, 7.40000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.40698400E+02, 7.40000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.52783700E+02, 7.40000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.01842900E+02, 7.40000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.47796300E+02, 7.40000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.99488000E+02, 7.40000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.85876830E+03, 7.40000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.56594130E+03, 7.40000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.34781630E+03, 7.40000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19832020E+03, 7.40000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.08508780E+03, 7.40000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.27862500E+02, 7.40000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.27740700E+02, 7.40000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.97634600E+02, 7.40000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.62266000E+02, 7.40000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.03761900E+02, 7.40000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.88702900E+02, 7.40000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.16641000E+02, 7.40000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.84709600E+02, 7.40000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.12949400E+02, 7.40000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.24901000E+02, 7.40000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.74158400E+02, 7.40000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.13831500E+02, 7.40000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.57585300E+02, 7.40000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.26390130E+03, 7.40000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.00877510E+03, 7.40000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.73391290E+03, 7.40000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.40686300E+02, 7.40000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {1.76941390E+03, 7.40000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.69389880E+03, 7.40000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.65000040E+03, 7.40000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.60977980E+03, 7.40000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.57406610E+03, 7.40000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.21444480E+03, 7.40000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.40783900E+03, 7.40000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.35334330E+03, 7.40000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.41328210E+03, 7.40000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.38257610E+03, 7.40000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.35457790E+03, 7.40000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.33977110E+03, 7.40000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.11430560E+03, 7.40000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.07638070E+03, 7.40000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.71952400E+02, 7.40000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.14362200E+02, 7.40000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.74814000E+01, 7.50000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {3.05492000E+01, 7.50000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.40302900E+02, 7.50000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.56868700E+02, 7.50000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.97693600E+02, 7.50000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.96407300E+02, 7.50000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.35040600E+02, 7.50000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.01066900E+02, 7.50000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.58337000E+01, 7.50000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.78970000E+01, 7.50000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00118330E+03, 7.50000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.35655300E+02, 7.50000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.66530800E+02, 7.50000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.13214500E+02, 7.50000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.92660400E+02, 7.50000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.22002200E+02, 7.50000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.60080500E+02, 7.50000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.10749700E+02, 7.50000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.65638290E+03, 7.50000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.31729610E+03, 7.50000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.07931680E+03, 7.50000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.03454890E+03, 7.50000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.43142300E+02, 7.50000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.42383900E+02, 7.50000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.06784500E+02, 7.50000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.32208800E+02, 7.50000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.63887100E+02, 7.50000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.87014300E+02, 7.50000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.26485900E+02, 7.50000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.32996200E+02, 7.50000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.34166200E+02, 7.50000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.49328100E+02, 7.50000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.61153100E+02, 7.50000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.09810800E+02, 7.50000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.55137800E+02, 7.50000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.06130600E+02, 7.50000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.85061310E+03, 7.50000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.57168630E+03, 7.50000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.35685330E+03, 7.50000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.20847370E+03, 7.50000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.09542820E+03, 7.50000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.37070800E+02, 7.50000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.37530100E+02, 7.50000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.06235800E+02, 7.50000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.71956700E+02, 7.50000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.13184600E+02, 7.50000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.95998000E+02, 7.50000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.25419900E+02, 7.50000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.94230100E+02, 7.50000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.23672600E+02, 7.50000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.35816100E+02, 7.50000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.84927000E+02, 7.50000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.24127500E+02, 7.50000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.67264200E+02, 7.50000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.25350370E+03, 7.50000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.01347400E+03, 7.50000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.74337910E+03, 7.50000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.53136900E+02, 7.50000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {1.77533570E+03, 7.50000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.70057060E+03, 7.50000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.65677680E+03, 7.50000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.61662660E+03, 7.50000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.58098350E+03, 7.50000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.22342240E+03, 7.50000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.41059040E+03, 7.50000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.35671150E+03, 7.50000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42070810E+03, 7.50000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.38999340E+03, 7.50000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.36203340E+03, 7.50000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.34699510E+03, 7.50000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12257060E+03, 7.50000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.08769310E+03, 7.50000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.83759100E+02, 7.50000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.24961500E+02, 7.50000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.36331000E+02, 7.50000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.35027000E+01, 7.60000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.83546000E+01, 7.60000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.31923400E+02, 7.60000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.07164600E+02, 7.60000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.68730400E+02, 7.60000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.79056300E+02, 7.60000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.24051000E+02, 7.60000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.33597000E+01, 7.60000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.03954000E+01, 7.60000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.39642000E+01, 7.60000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.73358900E+02, 7.60000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.53075000E+02, 7.60000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.95562800E+02, 7.60000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.62684000E+02, 7.60000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.56708800E+02, 7.60000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.94002100E+02, 7.60000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.38641500E+02, 7.60000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.94231400E+02, 7.60000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.43966200E+03, 7.60000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16126180E+03, 7.60000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.54453300E+02, 7.60000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.17469000E+02, 7.60000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.37845500E+02, 7.60000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.59820400E+02, 7.60000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.18505900E+02, 7.60000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.63505900E+02, 7.60000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.93753300E+02, 7.60000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.13351300E+02, 7.60000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.70255300E+02, 7.60000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.78537500E+02, 7.60000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.68225100E+02, 7.60000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.95614600E+02, 7.60000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.18689200E+02, 7.60000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.73559000E+02, 7.60000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.25069400E+02, 7.60000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.81301900E+02, 7.60000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.61037540E+03, 7.60000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.38482540E+03, 7.60000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.20210440E+03, 7.60000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.07434910E+03, 7.60000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.76084200E+02, 7.60000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.48941800E+02, 7.60000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.37527500E+02, 7.60000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.33788600E+02, 7.60000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.92675400E+02, 7.60000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.40896800E+02, 7.60000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.35286900E+02, 7.60000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.63120500E+02, 7.60000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.11697300E+02, 7.60000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.52113500E+02, 7.60000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.76107500E+02, 7.60000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.31782300E+02, 7.60000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.78255700E+02, 7.60000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.27872400E+02, 7.60000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.96101920E+03, 7.60000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.77079840E+03, 7.60000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.54167470E+03, 7.60000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.81786800E+02, 7.60000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {1.56430820E+03, 7.60000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.49987320E+03, 7.60000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.46163150E+03, 7.60000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42652810E+03, 7.60000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.39537630E+03, 7.60000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.08642420E+03, 7.60000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.24132050E+03, 7.60000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19507830E+03, 7.60000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.25567280E+03, 7.60000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.22871110E+03, 7.60000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.20425230E+03, 7.60000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19065280E+03, 7.60000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.96364400E+02, 7.60000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.70801300E+02, 7.60000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.81093800E+02, 7.60000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.40955900E+02, 7.60000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.52157100E+02, 7.60000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.78527800E+02, 7.60000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.01928000E+01, 7.70000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.64936000E+01, 7.70000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.49222500E+02, 7.70000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.67681900E+02, 7.70000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.45238100E+02, 7.70000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.64758600E+02, 7.70000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.14887400E+02, 7.70000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.68793000E+01, 7.70000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.57904000E+01, 7.70000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.06159000E+01, 7.70000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.75633700E+02, 7.70000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.87948700E+02, 7.70000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.38999200E+02, 7.70000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.21783900E+02, 7.70000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.27230400E+02, 7.70000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.70855100E+02, 7.70000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.20776300E+02, 7.70000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.80371100E+02, 7.70000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.27551980E+03, 7.70000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.03989930E+03, 7.70000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.56788600E+02, 7.70000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.25512500E+02, 7.70000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.54919300E+02, 7.70000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.94872800E+02, 7.70000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.48708500E+02, 7.70000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.09218100E+02, 7.70000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.37894600E+02, 7.70000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.54847900E+02, 7.70000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.25678800E+02, 7.70000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.34879200E+02, 7.70000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.15505100E+02, 7.70000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.52125600E+02, 7.70000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.83926500E+02, 7.70000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.43683300E+02, 7.70000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.00117800E+02, 7.70000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.60567200E+02, 7.70000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42824930E+03, 7.70000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.23973510E+03, 7.70000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.08083400E+03, 7.70000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.68648800E+02, 7.70000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.81710100E+02, 7.70000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.78871800E+02, 7.70000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.58183400E+02, 7.70000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.75945200E+02, 7.70000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.29290900E+02, 7.70000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.82962200E+02, 7.70000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.86813100E+02, 7.70000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.13037600E+02, 7.70000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.45862900E+02, 7.70000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.94404900E+02, 7.70000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.27468100E+02, 7.70000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.88231900E+02, 7.70000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.40424700E+02, 7.70000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.95188900E+02, 7.70000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.73949920E+03, 7.70000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.58313330E+03, 7.70000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.38422250E+03, 7.70000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.23811400E+02, 7.70000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {1.40068130E+03, 7.70000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.34395140E+03, 7.70000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30994540E+03, 7.70000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.27869860E+03, 7.70000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.25097670E+03, 7.70000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.78870400E+02, 7.70000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.11069690E+03, 7.70000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.07019120E+03, 7.70000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12693880E+03, 7.70000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.10286140E+03, 7.70000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.08108110E+03, 7.70000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06864600E+03, 7.70000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.97291500E+02, 7.70000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.78057500E+02, 7.70000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.99168600E+02, 7.70000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.73707500E+02, 7.70000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.84585500E+02, 7.70000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.19114800E+02, 7.70000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.66066000E+02, 7.70000000E+01, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.37374000E+01, 7.80000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.26550000E+01, 7.80000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.23292200E+02, 7.80000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.00162400E+02, 7.80000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.02540800E+02, 7.80000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.37576200E+02, 7.80000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.68658000E+01, 7.80000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.38183000E+01, 7.80000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.63053000E+01, 7.80000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.35930000E+01, 7.80000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.25962500E+02, 7.80000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.78853500E+02, 7.80000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.41201400E+02, 7.80000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.47881700E+02, 7.80000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.71987200E+02, 7.80000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.26471000E+02, 7.80000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.85746400E+02, 7.80000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.52656700E+02, 7.80000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.02998990E+03, 7.80000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.44438700E+02, 7.80000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.96927200E+02, 7.80000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.73052700E+02, 7.80000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.16306600E+02, 7.80000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.86521600E+02, 7.80000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.30652700E+02, 7.80000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.17431400E+02, 7.80000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.41402100E+02, 7.80000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.54629500E+02, 7.80000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.49580300E+02, 7.80000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.58012600E+02, 7.80000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.23429200E+02, 7.80000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.73429800E+02, 7.80000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.19040900E+02, 7.80000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.86870300E+02, 7.80000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.51748200E+02, 7.80000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.19655200E+02, 7.80000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.15476380E+03, 7.80000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00706230E+03, 7.80000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.80999300E+02, 7.80000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.91554400E+02, 7.80000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.21934000E+02, 7.80000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.58238900E+02, 7.80000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.22422200E+02, 7.80000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.75071500E+02, 7.80000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.18459500E+02, 7.80000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.80938900E+02, 7.80000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.02191500E+02, 7.80000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.24039000E+02, 7.80000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.31444200E+02, 7.80000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.91010900E+02, 7.80000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.37820900E+02, 7.80000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.06618500E+02, 7.80000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.68259200E+02, 7.80000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.31790400E+02, 7.80000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.40717240E+03, 7.80000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.28543220E+03, 7.80000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12751570E+03, 7.80000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.17685700E+02, 7.80000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {1.13889940E+03, 7.80000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.09324620E+03, 7.80000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06570500E+03, 7.80000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04037420E+03, 7.80000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.01790320E+03, 7.80000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.00353400E+02, 7.80000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.03959400E+02, 7.80000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.71584500E+02, 7.80000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.17604100E+02, 7.80000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.98027200E+02, 7.80000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.80364300E+02, 7.80000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.70036700E+02, 7.80000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.32918500E+02, 7.80000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.19575200E+02, 7.80000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.56821200E+02, 7.80000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.55629700E+02, 7.80000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.65026400E+02, 7.80000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.12426900E+02, 7.80000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.69666400E+02, 7.80000000E+01, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.91144800E+02, 7.80000000E+01, 7.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.16431000E+01, 7.90000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.14057000E+01, 7.90000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.82712000E+02, 7.90000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.78487400E+02, 7.90000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.88785900E+02, 7.90000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.28784300E+02, 7.90000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.10210000E+01, 7.90000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.95772000E+01, 7.90000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.32258000E+01, 7.90000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.13166000E+01, 7.90000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.77755000E+02, 7.90000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.43849100E+02, 7.90000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.09786400E+02, 7.90000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.24092500E+02, 7.90000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.54149600E+02, 7.90000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.12103900E+02, 7.90000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.74380800E+02, 7.90000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.43648600E+02, 7.90000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.50621800E+02, 7.90000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.81604400E+02, 7.90000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.45575000E+02, 7.90000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.24069300E+02, 7.90000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.71774600E+02, 7.90000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.51679900E+02, 7.90000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.92726800E+02, 7.90000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.87919600E+02, 7.90000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.10406800E+02, 7.90000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.22447800E+02, 7.90000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.25119800E+02, 7.90000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.33317700E+02, 7.90000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.93822700E+02, 7.90000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.48092200E+02, 7.90000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.98101300E+02, 7.90000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.68500000E+02, 7.90000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.36076500E+02, 7.90000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.06375600E+02, 7.90000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06632890E+03, 7.90000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.32205500E+02, 7.90000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.16746500E+02, 7.90000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.34615200E+02, 7.90000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.70551700E+02, 7.90000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.19410000E+02, 7.90000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.78746100E+02, 7.90000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.42592100E+02, 7.90000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.82808900E+02, 7.90000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.48121600E+02, 7.90000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.74948600E+02, 7.90000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.95414900E+02, 7.90000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.94653600E+02, 7.90000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.57752900E+02, 7.90000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.08942700E+02, 7.90000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.80291100E+02, 7.90000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.44940800E+02, 7.90000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.11270000E+02, 7.90000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.29959750E+03, 7.90000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.18954050E+03, 7.90000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04489960E+03, 7.90000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.83483100E+02, 7.90000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {1.05463390E+03, 7.90000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.01257350E+03, 7.90000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.87121600E+02, 7.90000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.63702100E+02, 7.90000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.42927400E+02, 7.90000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.42919200E+02, 7.90000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.37292100E+02, 7.90000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.07542100E+02, 7.90000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.50295400E+02, 7.90000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.32171800E+02, 7.90000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.15837600E+02, 7.90000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.06190300E+02, 7.90000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.80073100E+02, 7.90000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.68620000E+02, 7.90000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.11038600E+02, 7.90000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.17608300E+02, 7.90000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.26537000E+02, 7.90000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.78067200E+02, 7.90000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.38608300E+02, 7.90000000E+01, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.65824700E+02, 7.90000000E+01, 7.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.42352600E+02, 7.90000000E+01, 7.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.26475000E+01, 8.00000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.20837000E+01, 8.00000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.87402000E+02, 8.00000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.85061700E+02, 8.00000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.94161500E+02, 8.00000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.32731700E+02, 8.00000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.38836000E+01, 8.00000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.17760000E+01, 8.00000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.49023000E+01, 8.00000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.26085000E+01, 8.00000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.83796000E+02, 8.00000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.53359600E+02, 8.00000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.19838400E+02, 8.00000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.33216700E+02, 8.00000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.61858400E+02, 8.00000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.18718500E+02, 8.00000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.79912400E+02, 8.00000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.48232500E+02, 8.00000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.57361700E+02, 8.00000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.94658500E+02, 8.00000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.57591700E+02, 8.00000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.36521000E+02, 8.00000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.83665700E+02, 8.00000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.60850900E+02, 8.00000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.03554000E+02, 8.00000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.96299900E+02, 8.00000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.20255200E+02, 8.00000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.32274100E+02, 8.00000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.32434000E+02, 8.00000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.41871000E+02, 8.00000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.03705900E+02, 8.00000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.57845500E+02, 8.00000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.07037100E+02, 8.00000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.76769700E+02, 8.00000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.43496000E+02, 8.00000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.12935600E+02, 8.00000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.07441280E+03, 8.00000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.47201100E+02, 8.00000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.32422200E+02, 8.00000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.49982400E+02, 8.00000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.85238500E+02, 8.00000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.31456200E+02, 8.00000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.91921600E+02, 8.00000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.53280200E+02, 8.00000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.94718600E+02, 8.00000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.59428200E+02, 8.00000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.83951800E+02, 8.00000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.05660700E+02, 8.00000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.06645100E+02, 8.00000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.70085000E+02, 8.00000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.20793200E+02, 8.00000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.91674000E+02, 8.00000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.55549700E+02, 8.00000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.21025100E+02, 8.00000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30929210E+03, 8.00000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.20703410E+03, 8.00000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06362760E+03, 8.00000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.97153000E+02, 8.00000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {1.07120930E+03, 8.00000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.02908350E+03, 8.00000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00338460E+03, 8.00000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.79723200E+02, 8.00000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.58739700E+02, 8.00000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.57563100E+02, 8.00000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.49226600E+02, 8.00000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.19526100E+02, 8.00000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.65296300E+02, 8.00000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.46948500E+02, 8.00000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.30441900E+02, 8.00000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.20535600E+02, 8.00000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.93547900E+02, 8.00000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.83919500E+02, 8.00000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.25920100E+02, 8.00000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.30516900E+02, 8.00000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.40050400E+02, 8.00000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.90853100E+02, 8.00000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.50682100E+02, 8.00000000E+01, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.76009600E+02, 8.00000000E+01, 7.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.51935000E+02, 8.00000000E+01, 7.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.62075500E+02, 8.00000000E+01, 8.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.55659000E+01, 8.10000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.93832000E+01, 8.10000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.41562000E+02, 8.10000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.45099000E+02, 8.10000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.87434600E+02, 8.10000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.88913700E+02, 8.10000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.29775900E+02, 8.10000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.71887000E+01, 8.10000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.30353000E+01, 8.10000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.58782000E+01, 8.10000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00134990E+03, 8.10000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.19842800E+02, 8.10000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.48503700E+02, 8.10000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.95889200E+02, 8.10000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.77860600E+02, 8.10000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.09405700E+02, 8.10000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.49715200E+02, 8.10000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.02363900E+02, 8.10000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.66997320E+03, 8.10000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30123250E+03, 8.10000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06212570E+03, 8.10000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.01566760E+03, 8.10000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.24505400E+02, 8.10000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.28752600E+02, 8.10000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.89170400E+02, 8.10000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.19121400E+02, 8.10000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.46884500E+02, 8.10000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.70343600E+02, 8.10000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.14780300E+02, 8.10000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.17771300E+02, 8.10000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.16539200E+02, 8.10000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.31014800E+02, 8.10000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.44131600E+02, 8.10000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.94089300E+02, 8.10000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.41154200E+02, 8.10000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.93946400E+02, 8.10000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.86461640E+03, 8.10000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.55470360E+03, 8.10000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.33414320E+03, 8.10000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.18435580E+03, 8.10000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.07168580E+03, 8.10000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.17138800E+02, 8.10000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.15949300E+02, 8.10000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.88336100E+02, 8.10000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.51411700E+02, 8.10000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.93494200E+02, 8.10000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.81274000E+02, 8.10000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.07456000E+02, 8.10000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.73883300E+02, 8.10000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.01335400E+02, 8.10000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.13796500E+02, 8.10000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.63663300E+02, 8.10000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.04318500E+02, 8.10000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.49157300E+02, 8.10000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.27240780E+03, 8.10000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.99776650E+03, 8.10000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.71879300E+03, 8.10000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.28020700E+02, 8.10000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {1.75847310E+03, 8.10000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.68212810E+03, 8.10000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.63821730E+03, 8.10000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.59800830E+03, 8.10000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.56229390E+03, 8.10000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.20252290E+03, 8.10000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.40266270E+03, 8.10000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.34790370E+03, 8.10000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.40138320E+03, 8.10000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.37074490E+03, 8.10000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.34276540E+03, 8.10000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.32819070E+03, 8.10000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.10309320E+03, 8.10000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06199600E+03, 8.10000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.57833000E+02, 8.10000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.02628500E+02, 8.10000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.12497100E+02, 8.10000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.29240200E+02, 8.10000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.62790900E+02, 8.10000000E+01, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.46917200E+02, 8.10000000E+01, 7.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.09614600E+02, 8.10000000E+01, 7.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.21909100E+02, 8.10000000E+01, 8.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.92237800E+02, 8.10000000E+01, 8.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.51786000E+01, 8.20000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.93963000E+01, 8.20000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.70498500E+02, 8.20000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.25987700E+02, 8.20000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.80171400E+02, 8.20000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.86223500E+02, 8.20000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.28824300E+02, 8.20000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.68801000E+01, 8.20000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.30285000E+01, 8.20000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.59909000E+01, 8.20000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.19125700E+02, 8.20000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.84101100E+02, 8.20000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.22713400E+02, 8.20000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.82588100E+02, 8.20000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.71293900E+02, 8.20000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.05638900E+02, 8.20000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.47814100E+02, 8.20000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.01533000E+02, 8.20000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.51641930E+03, 8.20000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.21870090E+03, 8.20000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00086020E+03, 8.20000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.61371800E+02, 8.20000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.77565500E+02, 8.20000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.91083400E+02, 8.20000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.52115600E+02, 8.20000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.89808500E+02, 8.20000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.20874400E+02, 8.20000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.41676400E+02, 8.20000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.91997600E+02, 8.20000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.99932200E+02, 8.20000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.93738500E+02, 8.20000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.16893200E+02, 8.20000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.35928100E+02, 8.20000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.88537300E+02, 8.20000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.37768300E+02, 8.20000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.92047200E+02, 8.20000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.69569090E+03, 8.20000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.45347450E+03, 8.20000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.25986910E+03, 8.20000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12496170E+03, 8.20000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.02147330E+03, 8.20000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.82990600E+02, 8.20000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.75958200E+02, 8.20000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.62167500E+02, 8.20000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.23704600E+02, 8.20000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.69388200E+02, 8.20000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.59271000E+02, 8.20000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.87936800E+02, 8.20000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.43879700E+02, 8.20000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.80575200E+02, 8.20000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.00371400E+02, 8.20000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.53682600E+02, 8.20000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.97487200E+02, 8.20000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.44700700E+02, 8.20000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.06466870E+03, 8.20000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.85940850E+03, 8.20000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.61646770E+03, 8.20000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.10628900E+02, 8.20000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {1.64198850E+03, 8.20000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.57397630E+03, 8.20000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.53374690E+03, 8.20000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.49682910E+03, 8.20000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.46406240E+03, 8.20000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.13809790E+03, 8.20000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30343160E+03, 8.20000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.25453350E+03, 8.20000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.31701770E+03, 8.20000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.28869070E+03, 8.20000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26296620E+03, 8.20000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.24878910E+03, 8.20000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04390340E+03, 8.20000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.01548870E+03, 8.20000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.20830500E+02, 8.20000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.73852600E+02, 8.20000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.85294500E+02, 8.20000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.07900900E+02, 8.20000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.45547000E+02, 8.20000000E+01, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.34020700E+02, 8.20000000E+01, 7.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.98130100E+02, 8.20000000E+01, 7.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.11314500E+02, 8.20000000E+01, 8.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.61964800E+02, 8.20000000E+01, 8.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.38815600E+02, 8.20000000E+01, 8.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.22419000E+01, 8.30000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.79329000E+01, 8.30000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.61527800E+02, 8.30000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.81030700E+02, 8.30000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.56053700E+02, 8.30000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.72783100E+02, 8.30000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.20799300E+02, 8.30000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.14934000E+01, 8.30000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.93689000E+01, 8.30000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.34204000E+01, 8.30000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.91086800E+02, 8.30000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.07600800E+02, 8.30000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.59433600E+02, 8.30000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.40134300E+02, 8.30000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.42778800E+02, 8.30000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.84294800E+02, 8.30000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.32121900E+02, 8.30000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.89875600E+02, 8.30000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.29662530E+03, 8.30000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06863310E+03, 8.30000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.82515300E+02, 8.30000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.51885500E+02, 8.30000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.79933600E+02, 8.30000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.14450900E+02, 8.30000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.71305500E+02, 8.30000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.26932100E+02, 8.30000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.58174000E+02, 8.30000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.75148700E+02, 8.30000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.41055100E+02, 8.30000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.52367800E+02, 8.30000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.35701300E+02, 8.30000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.71837600E+02, 8.30000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.01992000E+02, 8.30000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.60473500E+02, 8.30000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.15277400E+02, 8.30000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.74075500E+02, 8.30000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.45294180E+03, 8.30000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.27323700E+03, 8.30000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.11439770E+03, 8.30000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00103150E+03, 8.30000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.12482000E+02, 8.30000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.04071300E+02, 8.30000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.85751700E+02, 8.30000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.98279200E+02, 8.30000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.53933700E+02, 8.30000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.06312800E+02, 8.30000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.05757400E+02, 8.30000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.34171400E+02, 8.30000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.70735400E+02, 8.30000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.19582700E+02, 8.30000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.51560500E+02, 8.30000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.11397600E+02, 8.30000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.62083400E+02, 8.30000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.15201100E+02, 8.30000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.76963900E+03, 8.30000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.62347400E+03, 8.30000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42512230E+03, 8.30000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.51712600E+02, 8.30000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {1.43826150E+03, 8.30000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.38094510E+03, 8.30000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.34626740E+03, 8.30000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.31437840E+03, 8.30000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.28609630E+03, 8.30000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.01038600E+03, 8.30000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.13901660E+03, 8.30000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.09831100E+03, 8.30000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.15977740E+03, 8.30000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.13514240E+03, 8.30000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.11291090E+03, 8.30000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.09994100E+03, 8.30000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.26129100E+02, 8.30000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.09815400E+02, 8.30000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.29828500E+02, 8.30000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.00439500E+02, 8.30000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.12417000E+02, 8.30000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.45376500E+02, 8.30000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.90850500E+02, 8.30000000E+01, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.90723400E+02, 8.30000000E+01, 7.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.58472600E+02, 8.30000000E+01, 7.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.71523100E+02, 8.30000000E+01, 8.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.88565500E+02, 8.30000000E+01, 8.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.72690400E+02, 8.30000000E+01, 8.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.17529600E+02, 8.30000000E+01, 8.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.07106000E+01, 8.40000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.72118000E+01, 8.40000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.07315100E+02, 8.40000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.57460300E+02, 8.40000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.43305700E+02, 8.40000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.65732700E+02, 8.40000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.16661500E+02, 8.40000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.87736000E+01, 8.40000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.75672000E+01, 8.40000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.21881000E+01, 8.40000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.27263100E+02, 8.40000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.67807200E+02, 8.40000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.26260000E+02, 8.40000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.17684400E+02, 8.40000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.27713600E+02, 8.40000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.73099500E+02, 8.40000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.23985000E+02, 8.40000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.83922300E+02, 8.40000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.18919020E+03, 8.40000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.92212500E+02, 8.40000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.21782500E+02, 8.40000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.95538400E+02, 8.40000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.29565700E+02, 8.40000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.75173100E+02, 8.40000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.29481000E+02, 8.40000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.94591400E+02, 8.40000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.25497900E+02, 8.40000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.40532700E+02, 8.40000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.14786900E+02, 8.40000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.27461600E+02, 8.40000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.05360000E+02, 8.40000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.48082200E+02, 8.40000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.84085300E+02, 8.40000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.45726600E+02, 8.40000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.03544800E+02, 8.40000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.64797800E+02, 8.40000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.33436920E+03, 8.40000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.18186040E+03, 8.40000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.03986220E+03, 8.40000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.37203600E+02, 8.40000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.56239800E+02, 8.40000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.63360800E+02, 8.40000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.39195700E+02, 8.40000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.65314100E+02, 8.40000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.17699800E+02, 8.40000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.73509900E+02, 8.40000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.78233600E+02, 8.40000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.06168600E+02, 8.40000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.32783300E+02, 8.40000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.87609700E+02, 8.40000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.25872800E+02, 8.40000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.89164000E+02, 8.40000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.43530000E+02, 8.40000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.99833500E+02, 8.40000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.62604190E+03, 8.40000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.50476070E+03, 8.40000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.32773970E+03, 8.40000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.20854800E+02, 8.40000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {1.33543090E+03, 8.40000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.28323360E+03, 8.40000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.25128790E+03, 8.40000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.22187750E+03, 8.40000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19580460E+03, 8.40000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.45157000E+02, 8.40000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.05698640E+03, 8.40000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.02027900E+03, 8.40000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.07968890E+03, 8.40000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.05688310E+03, 8.40000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.03637880E+03, 8.40000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.02403380E+03, 8.40000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.65803000E+02, 8.40000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.55043900E+02, 8.40000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.82480200E+02, 8.40000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.62382000E+02, 8.40000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.74498000E+02, 8.40000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.12777200E+02, 8.40000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.62310000E+02, 8.40000000E+01, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.68252500E+02, 8.40000000E+01, 7.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.37927100E+02, 8.40000000E+01, 7.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.50807400E+02, 8.40000000E+01, 8.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.50742900E+02, 8.40000000E+01, 8.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.38225100E+02, 8.40000000E+01, 8.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.88618800E+02, 8.40000000E+01, 8.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.62601100E+02, 8.40000000E+01, 8.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.80468000E+01, 8.50000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.58071000E+01, 8.50000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.35652100E+02, 8.50000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.23154000E+02, 8.50000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.23397100E+02, 8.50000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.53988300E+02, 8.50000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.09361800E+02, 8.50000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.37437000E+01, 8.50000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.40797000E+01, 8.50000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.97043000E+01, 8.50000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.42559900E+02, 8.50000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.11047200E+02, 8.50000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.77381600E+02, 8.50000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.82891400E+02, 8.50000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.03171700E+02, 8.50000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.54189900E+02, 8.50000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.09695100E+02, 8.50000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.73066800E+02, 8.50000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04867540E+03, 8.50000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.86793800E+02, 8.50000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.36876800E+02, 8.50000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.15824200E+02, 8.50000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.57770000E+02, 8.50000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.19228000E+02, 8.50000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.69185000E+02, 8.50000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.47941200E+02, 8.50000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.77404600E+02, 8.50000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.90029600E+02, 8.50000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.76545000E+02, 8.50000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.90054500E+02, 8.50000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.60132300E+02, 8.50000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.11191400E+02, 8.50000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.55089900E+02, 8.50000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.21157200E+02, 8.50000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.83362600E+02, 8.50000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.48316900E+02, 8.50000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.17878430E+03, 8.50000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.05617810E+03, 8.50000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.34928000E+02, 8.50000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.45958900E+02, 8.50000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.75007700E+02, 8.50000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.03518600E+02, 8.50000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.71209200E+02, 8.50000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.16180700E+02, 8.50000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.63613400E+02, 8.50000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.24171600E+02, 8.50000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.37162800E+02, 8.50000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.63631600E+02, 8.50000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.76486900E+02, 8.50000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.38571300E+02, 8.50000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.85036600E+02, 8.50000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.52972600E+02, 8.50000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.12492900E+02, 8.50000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.73396000E+02, 8.50000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.43752370E+03, 8.50000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.34275570E+03, 8.50000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19180040E+03, 8.50000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.72198800E+02, 8.50000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {1.19415520E+03, 8.50000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.14848790E+03, 8.50000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12016740E+03, 8.50000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.09405800E+03, 8.50000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.07092190E+03, 8.50000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.52630600E+02, 8.50000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.45045000E+02, 8.50000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.13337100E+02, 8.50000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.68258700E+02, 8.50000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.47920200E+02, 8.50000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.29716300E+02, 8.50000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.18344200E+02, 8.50000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.80315100E+02, 8.50000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.75233700E+02, 8.50000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.12288800E+02, 8.50000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.05241300E+02, 8.50000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.17106800E+02, 8.50000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.62588000E+02, 8.50000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.17728500E+02, 8.50000000E+01, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.32648800E+02, 8.50000000E+01, 7.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.05180200E+02, 8.50000000E+01, 7.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.17473700E+02, 8.50000000E+01, 8.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.94349700E+02, 8.50000000E+01, 8.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.85431400E+02, 8.50000000E+01, 8.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.42874400E+02, 8.50000000E+01, 8.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.20587200E+02, 8.50000000E+01, 8.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.83653600E+02, 8.50000000E+01, 8.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.52750000E+01, 8.60000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {2.42886000E+01, 8.60000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.71299400E+02, 8.60000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.90407900E+02, 8.60000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.03662100E+02, 8.60000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.41994600E+02, 8.60000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.01734000E+02, 8.60000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.83993000E+01, 8.60000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.03208000E+01, 8.60000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.69970000E+01, 8.60000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.66304700E+02, 8.60000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.57520700E+02, 8.60000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.30407000E+02, 8.60000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.48532400E+02, 8.60000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.78344100E+02, 8.60000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.34756300E+02, 8.60000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.94777800E+02, 8.60000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.61577800E+02, 8.60000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.23489600E+02, 8.60000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.89465100E+02, 8.60000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.57820900E+02, 8.60000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.41068900E+02, 8.60000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.90136500E+02, 8.60000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.66578600E+02, 8.60000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.12006900E+02, 8.60000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.03716200E+02, 8.60000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.31254300E+02, 8.60000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.41805200E+02, 8.60000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.40102900E+02, 8.60000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.53756900E+02, 8.60000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.16407200E+02, 8.60000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.74737800E+02, 8.60000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.25846800E+02, 8.60000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.96059100E+02, 8.60000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.62467800E+02, 8.60000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.31039200E+02, 8.60000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.03987880E+03, 8.60000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.40376600E+02, 8.60000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.36824400E+02, 8.60000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.59868600E+02, 8.60000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.97907500E+02, 8.60000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.46176200E+02, 8.60000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.06288100E+02, 8.60000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.68752700E+02, 8.60000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.11337000E+02, 8.60000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.76291300E+02, 8.60000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.97512300E+02, 8.60000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.22139300E+02, 8.60000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.22252400E+02, 8.60000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.90457400E+02, 8.60000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.44235100E+02, 8.60000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.16403800E+02, 8.60000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.80746300E+02, 8.60000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.46034000E+02, 8.60000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26915140E+03, 8.60000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19425530E+03, 8.60000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06538860E+03, 8.60000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.23799900E+02, 8.60000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {1.06411130E+03, 8.60000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.02416010E+03, 8.60000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.99100400E+02, 8.60000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.75967100E+02, 8.60000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.55475300E+02, 8.60000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.65775400E+02, 8.60000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.42480800E+02, 8.60000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.15081000E+02, 8.60000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.64860000E+02, 8.60000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.46764100E+02, 8.60000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.30633700E+02, 8.60000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.20222300E+02, 8.60000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.00093900E+02, 8.60000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.99085000E+02, 8.60000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.44671300E+02, 8.60000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.49839200E+02, 8.60000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.61213700E+02, 8.60000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.13276100E+02, 8.60000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.73608700E+02, 8.60000000E+01, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.97190600E+02, 8.60000000E+01, 7.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.72482500E+02, 8.60000000E+01, 7.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.84024100E+02, 8.60000000E+01, 8.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.39905500E+02, 8.60000000E+01, 8.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.33698400E+02, 8.60000000E+01, 8.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.97297200E+02, 8.60000000E+01, 8.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.78311300E+02, 8.60000000E+01, 8.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.46044700E+02, 8.60000000E+01, 8.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.12827500E+02, 8.60000000E+01, 8.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00749400E+02, 8.70000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {5.96594000E+01, 8.70000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.89459660E+03, 8.70000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.21893480E+03, 8.70000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.10658500E+02, 8.70000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.34196700E+02, 8.70000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.82776300E+02, 8.70000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.03972300E+02, 8.70000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.48358400E+02, 8.70000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.10512000E+02, 8.70000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.39864610E+03, 8.70000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.04462960E+03, 8.70000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.74654180E+03, 8.70000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.23782690E+03, 8.70000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.88822500E+02, 8.70000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.01976100E+02, 8.70000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.47271200E+02, 8.70000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.30446800E+02, 8.70000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.02091230E+03, 8.70000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.98855350E+03, 8.70000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.15339330E+03, 8.70000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.94832140E+03, 8.70000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.64413480E+03, 8.70000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.09515630E+03, 8.70000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.20925160E+03, 8.70000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.73597670E+03, 8.70000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.74060130E+03, 8.70000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.83022860E+03, 8.70000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.41687370E+03, 8.70000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.34649750E+03, 8.70000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.63308690E+03, 8.70000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.32196690E+03, 8.70000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.05090640E+03, 8.70000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.05472400E+02, 8.70000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.60708600E+02, 8.70000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.37921000E+02, 8.70000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.68795510E+03, 8.70000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.81406130E+03, 8.70000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.92386300E+03, 8.70000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.37773480E+03, 8.70000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.00120090E+03, 8.70000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.22084040E+03, 8.70000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.51894570E+03, 8.70000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.82860370E+03, 8.70000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.98198220E+03, 8.70000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.80527920E+03, 8.70000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.54030200E+03, 8.70000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.55509640E+03, 8.70000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.06717310E+03, 8.70000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.77326200E+03, 8.70000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.48012550E+03, 8.70000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.32363290E+03, 8.70000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.15163860E+03, 8.70000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.99499300E+02, 8.70000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.23152800E+03, 8.70000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.33972350E+03, 8.70000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.17156280E+03, 8.70000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.77659190E+03, 8.70000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {5.49696010E+03, 8.70000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.18868260E+03, 8.70000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.03768300E+03, 8.70000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.90097390E+03, 8.70000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.77923850E+03, 8.70000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.50046500E+03, 8.70000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.51801450E+03, 8.70000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.32568820E+03, 8.70000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.22162160E+03, 8.70000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.12110130E+03, 8.70000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.02706140E+03, 8.70000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.99087780E+03, 8.70000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.21877040E+03, 8.70000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.92181190E+03, 8.70000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.55632950E+03, 8.70000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.10769910E+03, 8.70000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.10324540E+03, 8.70000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.83838820E+03, 8.70000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.63638840E+03, 8.70000000E+01, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.32711810E+03, 8.70000000E+01, 7.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.22709140E+03, 8.70000000E+01, 7.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.23949860E+03, 8.70000000E+01, 8.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.10947530E+03, 8.70000000E+01, 8.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.93330250E+03, 8.70000000E+01, 8.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.66999650E+03, 8.70000000E+01, 8.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.54083170E+03, 8.70000000E+01, 8.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.36873630E+03, 8.70000000E+01, 8.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.21349590E+03, 8.70000000E+01, 8.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.31473980E+03, 8.70000000E+01, 8.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.94131000E+01, 8.80000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {5.96740000E+01, 8.80000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.43406970E+03, 8.80000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.13203440E+03, 8.80000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.81378900E+02, 8.80000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.24161300E+02, 8.80000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.79360200E+02, 8.80000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.02826000E+02, 8.80000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.48251600E+02, 8.80000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.10812600E+02, 8.80000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.87457170E+03, 8.80000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.87196000E+03, 8.80000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.62794930E+03, 8.80000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.18371030E+03, 8.80000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.64342900E+02, 8.80000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.88375600E+02, 8.80000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.40586800E+02, 8.80000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.27514700E+02, 8.80000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.91285810E+03, 8.80000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.52895900E+03, 8.80000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.82922730E+03, 8.80000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.66559500E+03, 8.80000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.40368780E+03, 8.80000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.89477070E+03, 8.80000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.02389190E+03, 8.80000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.58458380E+03, 8.80000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.61922210E+03, 8.80000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.69424690E+03, 8.80000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30221360E+03, 8.80000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26779660E+03, 8.80000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.52701490E+03, 8.80000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26357300E+03, 8.80000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.01972510E+03, 8.80000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.85043300E+02, 8.80000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.48650200E+02, 8.80000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.31275700E+02, 8.80000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.45940390E+03, 8.80000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.23397360E+03, 8.80000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.52461800E+03, 8.80000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.07002060E+03, 8.80000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.74426750E+03, 8.80000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.04883500E+03, 8.80000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.31493490E+03, 8.80000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.69846720E+03, 8.80000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.85110770E+03, 8.80000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.69393620E+03, 8.80000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42969500E+03, 8.80000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.46746240E+03, 8.80000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.92102150E+03, 8.80000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.68325370E+03, 8.80000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42678310E+03, 8.80000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.28546080E+03, 8.80000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12653190E+03, 8.80000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.83633700E+02, 8.80000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.66460490E+03, 8.80000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.50525820E+03, 8.80000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.59430950E+03, 8.80000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.70371040E+03, 8.80000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {4.79760610E+03, 8.80000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.56238650E+03, 8.80000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.43625920E+03, 8.80000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.32149140E+03, 8.80000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.21936900E+03, 8.80000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.14471840E+03, 8.80000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.86934750E+03, 8.80000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.70069780E+03, 8.80000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.75365340E+03, 8.80000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.66797460E+03, 8.80000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.58843510E+03, 8.80000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.55400670E+03, 8.80000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.88929830E+03, 8.80000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.69212290E+03, 8.80000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.38133180E+03, 8.80000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.96847300E+03, 8.80000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.97606720E+03, 8.80000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.74285220E+03, 8.80000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.56168870E+03, 8.80000000E+01, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.27042580E+03, 8.80000000E+01, 7.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.17662220E+03, 8.80000000E+01, 7.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19575730E+03, 8.80000000E+01, 8.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.95472920E+03, 8.80000000E+01, 8.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.82865150E+03, 8.80000000E+01, 8.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.60463490E+03, 8.80000000E+01, 8.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.49140890E+03, 8.80000000E+01, 8.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.33516320E+03, 8.80000000E+01, 8.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19092860E+03, 8.80000000E+01, 8.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.04571680E+03, 8.80000000E+01, 8.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.30543990E+03, 8.80000000E+01, 8.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.94060000E+01, 8.90000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {5.44613000E+01, 8.90000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.00825910E+03, 8.90000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.77214800E+02, 8.90000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.00053300E+02, 8.90000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.78667200E+02, 8.90000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.51808600E+02, 8.90000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.84031800E+02, 8.90000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.35279200E+02, 8.90000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.01585700E+02, 8.90000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.37746170E+03, 8.90000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.60403000E+03, 8.90000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.40987110E+03, 8.90000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04056810E+03, 8.90000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.68513200E+02, 8.90000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.16131900E+02, 8.90000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.86883500E+02, 8.90000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.87072000E+02, 8.90000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.01747820E+03, 8.90000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.97714780E+03, 8.90000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.40215400E+03, 8.90000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.27327390E+03, 8.90000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.05595890E+03, 8.90000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.61836340E+03, 8.90000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.73844950E+03, 8.90000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.36006630E+03, 8.90000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.40176220E+03, 8.90000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.46253570E+03, 8.90000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12174660E+03, 8.90000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.10495230E+03, 8.90000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.32636010E+03, 8.90000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.11109580E+03, 8.90000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.05534200E+02, 8.90000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.90264500E+02, 8.90000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.72165800E+02, 8.90000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.69547900E+02, 8.90000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.46900530E+03, 8.90000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.56435100E+03, 8.90000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.99777220E+03, 8.90000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.62704460E+03, 8.90000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.35670830E+03, 8.90000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.76925380E+03, 8.90000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.99487290E+03, 8.90000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.47304570E+03, 8.90000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.60817220E+03, 8.90000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.47539240E+03, 8.90000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.24022740E+03, 8.90000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.28228350E+03, 8.90000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.66505020E+03, 8.90000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.47511210E+03, 8.90000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26209170E+03, 8.90000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.14290900E+03, 8.90000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00691410E+03, 8.90000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.83414000E+02, 8.90000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.44789290E+03, 8.90000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.61202470E+03, 8.90000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.88970520E+03, 8.90000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.50315690E+03, 8.90000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {4.03239920E+03, 8.90000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.84335360E+03, 8.90000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.73936460E+03, 8.90000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.64454690E+03, 8.90000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.56022780E+03, 8.90000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.67968020E+03, 8.90000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.23336810E+03, 8.90000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.09680780E+03, 8.90000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.17694060E+03, 8.90000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.10568470E+03, 8.90000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.03984610E+03, 8.90000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.00959990E+03, 8.90000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.46237740E+03, 8.90000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.32022770E+03, 8.90000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.06456700E+03, 8.90000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.71155700E+03, 8.90000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.72322310E+03, 8.90000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.52765350E+03, 8.90000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.37436400E+03, 8.90000000E+01, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12139240E+03, 8.90000000E+01, 7.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.04002380E+03, 8.90000000E+01, 7.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06006070E+03, 8.90000000E+01, 8.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.69466050E+03, 8.90000000E+01, 8.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.60086160E+03, 8.90000000E+01, 8.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.41743460E+03, 8.90000000E+01, 8.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.32363600E+03, 8.90000000E+01, 8.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19132720E+03, 8.90000000E+01, 8.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06750650E+03, 8.90000000E+01, 8.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.98149610E+03, 8.90000000E+01, 8.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.46695840E+03, 8.90000000E+01, 8.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.79965650E+03, 8.90000000E+01, 8.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.20836000E+01, 9.00000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {5.11103000E+01, 9.00000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.67247680E+03, 9.00000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.52524900E+02, 9.00000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.36085800E+02, 9.00000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.44357600E+02, 9.00000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.32072300E+02, 9.00000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.71241800E+02, 9.00000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.26939300E+02, 9.00000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.59845000E+01, 9.00000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.98521030E+03, 9.00000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.38837560E+03, 9.00000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.23526810E+03, 9.00000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.27510700E+02, 9.00000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.94798300E+02, 9.00000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.62063600E+02, 9.00000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.48048600E+02, 9.00000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.58927300E+02, 9.00000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.32081710E+03, 9.00000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.53773080E+03, 9.00000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.06089010E+03, 9.00000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.96028270E+03, 9.00000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.77865460E+03, 9.00000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.39947780E+03, 9.00000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.51108160E+03, 9.00000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.18250930E+03, 9.00000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.22873800E+03, 9.00000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.27778960E+03, 9.00000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.79198400E+02, 9.00000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.75885600E+02, 9.00000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16705830E+03, 9.00000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.91201500E+02, 9.00000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.17555200E+02, 9.00000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.18667900E+02, 9.00000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.15850500E+02, 9.00000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.25424000E+02, 9.00000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.69966370E+03, 9.00000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.03312730E+03, 9.00000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.57870740E+03, 9.00000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.27483250E+03, 9.00000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.04922990E+03, 9.00000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.54917800E+03, 9.00000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.74212020E+03, 9.00000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.29665510E+03, 9.00000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.41684230E+03, 9.00000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30360600E+03, 9.00000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.09259390E+03, 9.00000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.13717450E+03, 9.00000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.46315800E+03, 9.00000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.31158800E+03, 9.00000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.13443850E+03, 9.00000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.03387690E+03, 9.00000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.17092000E+02, 9.00000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.09801800E+02, 9.00000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.50556810E+03, 9.00000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.90712620E+03, 9.00000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.33154730E+03, 9.00000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.34782230E+03, 9.00000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {3.42778940E+03, 9.00000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.27447920E+03, 9.00000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.18774280E+03, 9.00000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.10847690E+03, 9.00000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.03803450E+03, 9.00000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.31220860E+03, 9.00000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.73581130E+03, 9.00000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.62419220E+03, 9.00000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.71913190E+03, 9.00000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.65912680E+03, 9.00000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.60400090E+03, 9.00000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.57697670E+03, 9.00000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.12368190E+03, 9.00000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.02482390E+03, 9.00000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.81375640E+03, 9.00000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.51013610E+03, 9.00000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.52492970E+03, 9.00000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.35985760E+03, 9.00000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.22921640E+03, 9.00000000E+01, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00742250E+03, 9.00000000E+01, 7.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.36095800E+02, 9.00000000E+01, 7.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.56804700E+02, 9.00000000E+01, 8.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.49156980E+03, 9.00000000E+01, 8.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42294310E+03, 9.00000000E+01, 8.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.27266010E+03, 9.00000000E+01, 8.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19514920E+03, 9.00000000E+01, 8.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.08287190E+03, 9.00000000E+01, 8.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.76141700E+02, 9.00000000E+01, 8.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.15559320E+03, 9.00000000E+01, 8.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.80477500E+03, 9.00000000E+01, 8.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.27042580E+03, 9.00000000E+01, 8.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.84727040E+03, 9.00000000E+01, 9.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.07260000E+01, 9.10000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {5.00052000E+01, 9.10000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.73060380E+03, 9.10000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.57637900E+02, 9.10000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.33081800E+02, 9.10000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.39986300E+02, 9.10000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.28133300E+02, 9.10000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.67918700E+02, 9.10000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.24264200E+02, 9.10000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.38663000E+01, 9.10000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.05111510E+03, 9.10000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.40309240E+03, 9.10000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.24030710E+03, 9.10000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.23249600E+02, 9.10000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.87291400E+02, 9.10000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.54127400E+02, 9.10000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.40441800E+02, 9.10000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.52074300E+02, 9.10000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.45704340E+03, 9.10000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.58928880E+03, 9.10000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.09453890E+03, 9.10000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.98692650E+03, 9.10000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.79965770E+03, 9.10000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.41734210E+03, 9.10000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.52510220E+03, 9.10000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19417780E+03, 9.10000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.23449140E+03, 9.10000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.28593000E+03, 9.10000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.86846700E+02, 9.10000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.76686500E+02, 9.10000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16994940E+03, 9.10000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.86640000E+02, 9.10000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.09375400E+02, 9.10000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.09438000E+02, 9.10000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.06302600E+02, 9.10000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.16142500E+02, 9.10000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.84883100E+03, 9.10000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.09878680E+03, 9.10000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.61793610E+03, 9.10000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.30099430E+03, 9.10000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.06841110E+03, 9.10000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.55874810E+03, 9.10000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.75497060E+03, 9.10000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30155980E+03, 9.10000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42072110E+03, 9.10000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30533200E+03, 9.10000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.09681750E+03, 9.10000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.13670420E+03, 9.10000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.46907150E+03, 9.10000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30851460E+03, 9.10000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12584980E+03, 9.10000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.02319420E+03, 9.10000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.05114100E+02, 9.10000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.97331300E+02, 9.10000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.69169640E+03, 9.10000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.00370890E+03, 9.10000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.39156060E+03, 9.10000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.33962480E+03, 9.10000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {3.50593360E+03, 9.10000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.34421620E+03, 9.10000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.25442680E+03, 9.10000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.17247830E+03, 9.10000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.09962070E+03, 9.10000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.34503200E+03, 9.10000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.80841010E+03, 9.10000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.69179850E+03, 9.10000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.76909430E+03, 9.10000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.70731880E+03, 9.10000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.65038670E+03, 9.10000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.62345790E+03, 9.10000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.15385160E+03, 9.10000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.03939590E+03, 9.10000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.82053310E+03, 9.10000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.51338810E+03, 9.10000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.52564090E+03, 9.10000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.35664580E+03, 9.10000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.22366760E+03, 9.10000000E+01, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00147540E+03, 9.10000000E+01, 7.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.29994300E+02, 9.10000000E+01, 7.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.48969800E+02, 9.10000000E+01, 8.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.49760180E+03, 9.10000000E+01, 8.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42069960E+03, 9.10000000E+01, 8.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26424230E+03, 9.10000000E+01, 8.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.18416270E+03, 9.10000000E+01, 8.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06986510E+03, 9.10000000E+01, 8.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.62176800E+02, 9.10000000E+01, 8.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.30396450E+03, 9.10000000E+01, 8.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.88657160E+03, 9.10000000E+01, 8.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.32007640E+03, 9.10000000E+01, 8.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.87251360E+03, 9.10000000E+01, 9.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.90892060E+03, 9.10000000E+01, 9.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.82034000E+01, 9.20000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {4.84980000E+01, 9.20000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.67161560E+03, 9.20000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.29713200E+02, 9.20000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.16019500E+02, 9.20000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.29280200E+02, 9.20000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.21065900E+02, 9.20000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.62792600E+02, 9.20000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.20529000E+02, 9.20000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.10869000E+01, 9.20000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.98146750E+03, 9.20000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.35716280E+03, 9.20000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.20002750E+03, 9.20000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.93644400E+02, 9.20000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.65496500E+02, 9.20000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.36708900E+02, 9.20000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.26735400E+02, 9.20000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.41234000E+02, 9.20000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.33769610E+03, 9.20000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.50335760E+03, 9.20000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.02547400E+03, 9.20000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.92168190E+03, 9.20000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.74072940E+03, 9.20000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.37092640E+03, 9.20000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.47536690E+03, 9.20000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.15527370E+03, 9.20000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19453890E+03, 9.20000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.24420370E+03, 9.20000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.54830900E+02, 9.20000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.45277500E+02, 9.20000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.13211070E+03, 9.20000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.55060700E+02, 9.20000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.83706900E+02, 9.20000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.87083900E+02, 9.20000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.87345000E+02, 9.20000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.00136800E+02, 9.20000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.71606780E+03, 9.20000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.99571860E+03, 9.20000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.53173730E+03, 9.20000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.22568070E+03, 9.20000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.00094390E+03, 9.20000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.50824940E+03, 9.20000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.69794840E+03, 9.20000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.25961420E+03, 9.20000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.37496650E+03, 9.20000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26341360E+03, 9.20000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06154570E+03, 9.20000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.10033990E+03, 9.20000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42160990E+03, 9.20000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26663290E+03, 9.20000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.09010530E+03, 9.20000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.90869600E+02, 9.20000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.76690200E+02, 9.20000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.72451900E+02, 9.20000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.52917270E+03, 9.20000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.86976110E+03, 9.20000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.27933600E+03, 9.20000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.29704610E+03, 9.20000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {3.38903950E+03, 9.20000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.23309750E+03, 9.20000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.14636900E+03, 9.20000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.06720530E+03, 9.20000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.99682370E+03, 9.20000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.26801380E+03, 9.20000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.71402960E+03, 9.20000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.60132260E+03, 9.20000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.67756720E+03, 9.20000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.61787180E+03, 9.20000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.56286450E+03, 9.20000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.53679150E+03, 9.20000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.08303850E+03, 9.20000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.97308210E+03, 9.20000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.76168440E+03, 9.20000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.46465290E+03, 9.20000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.47663370E+03, 9.20000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.31330660E+03, 9.20000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.18475300E+03, 9.20000000E+01, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.69802200E+02, 9.20000000E+01, 7.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.00656200E+02, 9.20000000E+01, 7.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.19082400E+02, 9.20000000E+01, 8.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.44930610E+03, 9.20000000E+01, 8.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.37527520E+03, 9.20000000E+01, 8.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.22413220E+03, 9.20000000E+01, 8.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.14675090E+03, 9.20000000E+01, 8.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.03624960E+03, 9.20000000E+01, 8.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.32112100E+02, 9.20000000E+01, 8.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.15651210E+03, 9.20000000E+01, 8.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.75724300E+03, 9.20000000E+01, 8.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.21075600E+03, 9.20000000E+01, 8.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.77894320E+03, 9.20000000E+01, 9.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.81362690E+03, 9.20000000E+01, 9.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.72152090E+03, 9.20000000E+01, 9.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.94391000E+01, 9.30000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {4.87241000E+01, 9.30000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.80050660E+03, 9.30000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.69564100E+02, 9.30000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.32898800E+02, 9.30000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.36462700E+02, 9.30000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.24209400E+02, 9.30000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.64295900E+02, 9.30000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.21166700E+02, 9.30000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {9.13090000E+01, 9.30000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.13130070E+03, 9.30000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42943080E+03, 9.30000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.25436040E+03, 9.30000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.24197000E+02, 9.30000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.82275000E+02, 9.30000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.47318900E+02, 9.30000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.33010500E+02, 9.30000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.44830500E+02, 9.30000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.60856020E+03, 9.30000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.66111780E+03, 9.30000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.14492320E+03, 9.30000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.02878460E+03, 9.30000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.83418000E+03, 9.30000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.44506820E+03, 9.30000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.55018990E+03, 9.30000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.21387080E+03, 9.30000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.24879090E+03, 9.30000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30333340E+03, 9.30000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.00095490E+03, 9.30000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.83721500E+02, 9.30000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.18044990E+03, 9.30000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.87375100E+02, 9.30000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.04256200E+02, 9.30000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.02031100E+02, 9.30000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.97503500E+02, 9.30000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.06821600E+02, 9.30000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.01389730E+03, 9.30000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.18757340E+03, 9.30000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.67664720E+03, 9.30000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.34375860E+03, 9.30000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.10184170E+03, 9.30000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.57787360E+03, 9.30000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.77903230E+03, 9.30000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.31371580E+03, 9.30000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.43315630E+03, 9.30000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.31465610E+03, 9.30000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.10686170E+03, 9.30000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.14251960E+03, 9.30000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.48401500E+03, 9.30000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.31266530E+03, 9.30000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.12214400E+03, 9.30000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.01600250E+03, 9.30000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.95232100E+02, 9.30000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.85803800E+02, 9.30000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.89304360E+03, 9.30000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.12793300E+03, 9.30000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.47571240E+03, 9.30000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.33728500E+03, 9.30000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {3.60788250E+03, 9.30000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.43785440E+03, 9.30000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.34452790E+03, 9.30000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.25944310E+03, 9.30000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.18376270E+03, 9.30000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.39355840E+03, 9.30000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.89640800E+03, 9.30000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.77303510E+03, 9.30000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.83964690E+03, 9.30000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.77574240E+03, 9.30000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.71665220E+03, 9.30000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.68970330E+03, 9.30000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.19866320E+03, 9.30000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.06800290E+03, 9.30000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.83904640E+03, 9.30000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.52511150E+03, 9.30000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.53479720E+03, 9.30000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.36023780E+03, 9.30000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.22364480E+03, 9.30000000E+01, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.99209600E+02, 9.30000000E+01, 7.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.27028500E+02, 9.30000000E+01, 7.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.44286200E+02, 9.30000000E+01, 8.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.51167140E+03, 9.30000000E+01, 8.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42582660E+03, 9.30000000E+01, 8.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26124150E+03, 9.30000000E+01, 8.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.17740380E+03, 9.30000000E+01, 8.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.05963770E+03, 9.30000000E+01, 8.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.49755400E+02, 9.30000000E+01, 8.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.46977740E+03, 9.30000000E+01, 8.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.99526060E+03, 9.30000000E+01, 8.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.39326010E+03, 9.30000000E+01, 8.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.91709920E+03, 9.30000000E+01, 9.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.96466680E+03, 9.30000000E+01, 9.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.86710250E+03, 9.30000000E+01, 9.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.03297600E+03, 9.30000000E+01, 9.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.71021000E+01, 9.40000000E+01, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {4.73798000E+01, 9.40000000E+01, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.72781400E+03, 9.40000000E+01, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.39447800E+02, 9.40000000E+01, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.15797700E+02, 9.40000000E+01, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {3.26250000E+02, 9.40000000E+01, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.17676500E+02, 9.40000000E+01, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.59643500E+02, 9.40000000E+01, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.17819300E+02, 9.40000000E+01, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {8.88365000E+01, 9.40000000E+01, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.04592840E+03, 9.40000000E+01, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.37858760E+03, 9.40000000E+01, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.21144430E+03, 9.40000000E+01, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.94325500E+02, 9.40000000E+01, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.61209200E+02, 9.40000000E+01, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.30882800E+02, 9.40000000E+01, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.20349800E+02, 9.40000000E+01, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.34975900E+02, 9.40000000E+01, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.45868830E+03, 9.40000000E+01, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.56127500E+03, 9.40000000E+01, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.06617680E+03, 9.40000000E+01, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.95545050E+03, 9.40000000E+01, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.76856490E+03, 9.40000000E+01, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.39313850E+03, 9.40000000E+01, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.49556570E+03, 9.40000000E+01, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.17100130E+03, 9.40000000E+01, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.20601580E+03, 9.40000000E+01, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.25821700E+03, 9.40000000E+01, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.66059900E+02, 9.40000000E+01, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.50860300E+02, 9.40000000E+01, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.14051970E+03, 9.40000000E+01, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.55503900E+02, 9.40000000E+01, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.79297600E+02, 9.40000000E+01, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.80735500E+02, 9.40000000E+01, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.79795600E+02, 9.40000000E+01, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.92112800E+02, 9.40000000E+01, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.84773240E+03, 9.40000000E+01, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.06714630E+03, 9.40000000E+01, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.57900220E+03, 9.40000000E+01, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.26006390E+03, 9.40000000E+01, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.02774400E+03, 9.40000000E+01, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.52337420E+03, 9.40000000E+01, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.71710410E+03, 9.40000000E+01, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26906470E+03, 9.40000000E+01, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.38473150E+03, 9.40000000E+01, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.27065780E+03, 9.40000000E+01, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.06927910E+03, 9.40000000E+01, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.10474670E+03, 9.40000000E+01, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.43343700E+03, 9.40000000E+01, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.26973680E+03, 9.40000000E+01, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.08676710E+03, 9.40000000E+01, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.84629100E+02, 9.40000000E+01, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.68186800E+02, 9.40000000E+01, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.62540700E+02, 9.40000000E+01, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.68949090E+03, 9.40000000E+01, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.96944880E+03, 9.40000000E+01, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.34692780E+03, 9.40000000E+01, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.29469970E+03, 9.40000000E+01, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {3.47081030E+03, 9.40000000E+01, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.30827820E+03, 9.40000000E+01, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.21872530E+03, 9.40000000E+01, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.13705760E+03, 9.40000000E+01, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.06442240E+03, 9.40000000E+01, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.30683660E+03, 9.40000000E+01, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.78425860E+03, 9.40000000E+01, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.66608160E+03, 9.40000000E+01, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.73429740E+03, 9.40000000E+01, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.67290420E+03, 9.40000000E+01, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.61617240E+03, 9.40000000E+01, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.59009570E+03, 9.40000000E+01, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.11898110E+03, 9.40000000E+01, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.99602660E+03, 9.40000000E+01, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.77641870E+03, 9.40000000E+01, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.47375440E+03, 9.40000000E+01, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.48367470E+03, 9.40000000E+01, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.31580800E+03, 9.40000000E+01, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.18429550E+03, 9.40000000E+01, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.67464300E+02, 9.40000000E+01, 7.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.97738600E+02, 9.40000000E+01, 7.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.14791200E+02, 9.40000000E+01, 8.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.46022470E+03, 9.40000000E+01, 8.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.37902630E+03, 9.40000000E+01, 8.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.22126470E+03, 9.40000000E+01, 8.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.14078270E+03, 9.40000000E+01, 8.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.02739480E+03, 9.40000000E+01, 8.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.21404700E+02, 9.40000000E+01, 8.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.28858580E+03, 9.40000000E+01, 8.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.84443250E+03, 9.40000000E+01, 8.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.26952880E+03, 9.40000000E+01, 8.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.81460330E+03, 9.40000000E+01, 9.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.85818750E+03, 9.40000000E+01, 9.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.76425850E+03, 9.40000000E+01, 9.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.92182230E+03, 9.40000000E+01, 9.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.81523660E+03, 9.40000000E+01, 9.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.73790000E+00, 1.01000000E+02, 1.00000000E+00, 0.00000000E+00, 9.11800000E-01},
    {3.12870000E+00, 1.01000000E+02, 2.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {6.89391000E+01, 1.01000000E+02, 3.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {4.13078000E+01, 1.01000000E+02, 4.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {2.82767000E+01, 1.01000000E+02, 5.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.92653000E+01, 1.01000000E+02, 6.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.35164000E+01, 1.01000000E+02, 7.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {1.02371000E+01, 1.01000000E+02, 8.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {7.74410000E+00, 1.01000000E+02, 9.00000000E+00, 0.00000000E+00, 0.00000000E+00},
    {5.94030000E+00, 1.01000000E+02, 1.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.25641000E+01, 1.01000000E+02, 1.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.53703000E+01, 1.01000000E+02, 1.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.08558000E+01, 1.01000000E+02, 1.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.85182000E+01, 1.01000000E+02, 1.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.81397000E+01, 1.01000000E+02, 1.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.17713000E+01, 1.01000000E+02, 1.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.60170000E+01, 1.01000000E+02, 1.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {2.13055000E+01, 1.01000000E+02, 1.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.34444100E+02, 1.01000000E+02, 1.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.13485700E+02, 1.01000000E+02, 2.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.42083000E+01, 1.01000000E+02, 2.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.13334000E+01, 1.01000000E+02, 2.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.38313000E+01, 1.01000000E+02, 2.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.59837000E+01, 1.01000000E+02, 2.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.24111000E+01, 1.01000000E+02, 2.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.67993000E+01, 1.01000000E+02, 2.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.05746000E+01, 1.01000000E+02, 2.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.22532000E+01, 1.01000000E+02, 2.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.76499000E+01, 1.01000000E+02, 2.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.93469000E+01, 1.01000000E+02, 3.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.84207000E+01, 1.01000000E+02, 3.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.19905000E+01, 1.01000000E+02, 3.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.46512000E+01, 1.01000000E+02, 3.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.02010000E+01, 1.01000000E+02, 3.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.52740000E+01, 1.01000000E+02, 3.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {3.07261000E+01, 1.01000000E+02, 3.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.50934000E+02, 1.01000000E+02, 3.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.35066400E+02, 1.01000000E+02, 3.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.19296100E+02, 1.01000000E+02, 3.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.07730600E+02, 1.01000000E+02, 4.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.85183000E+01, 1.01000000E+02, 4.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.63661000E+01, 1.01000000E+02, 4.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {8.50800000E+01, 1.01000000E+02, 4.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.50863000E+01, 1.01000000E+02, 4.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.11985000E+01, 1.01000000E+02, 4.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.61211000E+01, 1.01000000E+02, 4.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.49658000E+01, 1.01000000E+02, 4.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.83568000E+01, 1.01000000E+02, 4.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.29076000E+01, 1.01000000E+02, 4.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.79430000E+01, 1.01000000E+02, 5.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.09470000E+01, 1.01000000E+02, 5.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.67427000E+01, 1.01000000E+02, 5.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.14656000E+01, 1.01000000E+02, 5.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {4.63794000E+01, 1.01000000E+02, 5.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.84022300E+02, 1.01000000E+02, 5.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.71703300E+02, 1.01000000E+02, 5.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.52110600E+02, 1.01000000E+02, 5.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.18830000E+01, 1.01000000E+02, 5.80000000E+01, 0.00000000E+00, 2.79910000E+00},
    {1.52450200E+02, 1.01000000E+02, 5.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.46585100E+02, 1.01000000E+02, 6.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.42962000E+02, 1.01000000E+02, 6.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.39625400E+02, 1.01000000E+02, 6.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.36669500E+02, 1.01000000E+02, 6.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.08366200E+02, 1.01000000E+02, 6.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.20405600E+02, 1.01000000E+02, 6.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.16317800E+02, 1.01000000E+02, 6.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.23522000E+02, 1.01000000E+02, 6.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.20930200E+02, 1.01000000E+02, 6.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.18607000E+02, 1.01000000E+02, 6.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.17181100E+02, 1.01000000E+02, 7.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.92984000E+01, 1.01000000E+02, 7.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.85136000E+01, 1.01000000E+02, 7.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {9.02858000E+01, 1.01000000E+02, 7.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.64150000E+01, 1.01000000E+02, 7.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.78793000E+01, 1.01000000E+02, 7.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.08024000E+01, 1.01000000E+02, 7.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.49890000E+01, 1.01000000E+02, 7.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.40451000E+01, 1.01000000E+02, 7.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.05086000E+01, 1.01000000E+02, 7.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.20547000E+01, 1.01000000E+02, 8.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.48817000E+01, 1.01000000E+02, 8.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.36692000E+01, 1.01000000E+02, 8.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.81031000E+01, 1.01000000E+02, 8.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.51605000E+01, 1.01000000E+02, 8.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {6.03338000E+01, 1.01000000E+02, 8.50000000E+01, 0.00000000E+00, 0.00000000E+00},
    {5.54345000E+01, 1.01000000E+02, 8.60000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.74921200E+02, 1.01000000E+02, 8.70000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.70508200E+02, 1.01000000E+02, 8.80000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.51813900E+02, 1.01000000E+02, 8.90000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.37472400E+02, 1.01000000E+02, 9.00000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.35870100E+02, 1.01000000E+02, 9.10000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.31564400E+02, 1.01000000E+02, 9.20000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.34730400E+02, 1.01000000E+02, 9.30000000E+01, 0.00000000E+00, 0.00000000E+00},
    {1.30594600E+02, 1.01000000E+02, 9.40000000E+01, 0.00000000E+00, 0.00000000E+00},
    {7.59160000E+00, 1.01000000E+02, 1.01000000E+02, 0.00000000E+00, 0.00000000E+00},
    {1.43165000E+01, 1.03000000E+02, 1.00000000E+00, 9.86500000E-01, 9.11800000E-01},
    {8.77730000E+00, 1.03000000E+02, 2.00000000E+00, 9.86500000E-01, 0.00000000E+00},
    {2.82210600E+02, 1.03000000E+02, 3.00000000E+00, 9.86500000E-01, 0.00000000E+00},
    {1.48132300E+02, 1.03000000E+02, 4.00000000E+00, 9.86500000E-01, 0.00000000E+00},
    {9.36729000E+01, 1.03000000E+02, 5.00000000E+00, 9.86500000E-01, 0.00000000E+00},
    {6.00860000E+01, 1.03000000E+02, 6.00000000E+00, 9.86500000E-01, 0.00000000E+00},
    {4.02944000E+01, 1.03000000E+02, 7.00000000E+00, 9.86500000E-01, 0.00000000E+00},
    {2.95575000E+01, 1.03000000E+02, 8.00000000E+00, 9.86500000E-01, 0.00000000E+00},
    {2.17605000E+01, 1.03000000E+02, 9.00000000E+00, 9.86500000E-01, 0.00000000E+00},
    {1.63388000E+01, 1.03000000E+02, 1.00000000E+01, 9.86500000E-01, 0.00000000E+00},
    {3.35345000E+02, 1.03000000E+02, 1.10000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.40162700E+02, 1.03000000E+02, 1.20000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.14726500E+02, 1.03000000E+02, 1.30000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.62050800E+02, 1.03000000E+02, 1.40000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.21506700E+02, 1.03000000E+02, 1.50000000E+01, 9.86500000E-01, 0.00000000E+00},
    {9.81249000E+01, 1.03000000E+02, 1.60000000E+01, 9.86500000E-01, 0.00000000E+00},
    {7.79865000E+01, 1.03000000E+02, 1.70000000E+01, 9.86500000E-01, 0.00000000E+00},
    {6.22205000E+01, 1.03000000E+02, 1.80000000E+01, 9.86500000E-01, 0.00000000E+00},
    {5.55151600E+02, 1.03000000E+02, 1.90000000E+01, 9.86500000E-01, 0.00000000E+00},
    {4.34157600E+02, 1.03000000E+02, 2.00000000E+01, 9.86500000E-01, 0.00000000E+00},
    {3.54031700E+02, 1.03000000E+02, 2.10000000E+01, 9.86500000E-01, 0.00000000E+00},
    {3.37389600E+02, 1.03000000E+02, 2.20000000E+01, 9.86500000E-01, 0.00000000E+00},
    {3.06548200E+02, 1.03000000E+02, 2.30000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.40503300E+02, 1.03000000E+02, 2.40000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.60901900E+02, 1.03000000E+02, 2.50000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.03614900E+02, 1.03000000E+02, 2.60000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.12897800E+02, 1.03000000E+02, 2.70000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.21173900E+02, 1.03000000E+02, 2.80000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.68807500E+02, 1.03000000E+02, 2.90000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.69494900E+02, 1.03000000E+02, 3.00000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.02690000E+02, 1.03000000E+02, 3.10000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.72942500E+02, 1.03000000E+02, 3.20000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.42847100E+02, 1.03000000E+02, 3.30000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.25476800E+02, 1.03000000E+02, 3.40000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.07338000E+02, 1.03000000E+02, 3.50000000E+01, 9.86500000E-01, 0.00000000E+00},
    {9.13333000E+01, 1.03000000E+02, 3.60000000E+01, 9.86500000E-01, 0.00000000E+00},
    {6.18568100E+02, 1.03000000E+02, 3.70000000E+01, 9.86500000E-01, 0.00000000E+00},
    {5.17839300E+02, 1.03000000E+02, 3.80000000E+01, 9.86500000E-01, 0.00000000E+00},
    {4.42944800E+02, 1.03000000E+02, 3.90000000E+01, 9.86500000E-01, 0.00000000E+00},
    {3.91917900E+02, 1.03000000E+02, 4.00000000E+01, 9.86500000E-01, 0.00000000E+00},
    {3.53500800E+02, 1.03000000E+02, 4.10000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.67376100E+02, 1.03000000E+02, 4.20000000E+01, 9.86500000E-01, 0.00000000E+00},
    {3.00653300E+02, 1.03000000E+02, 4.30000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.23887100E+02, 1.03000000E+02, 4.40000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.45256100E+02, 1.03000000E+02, 4.50000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.25804200E+02, 1.03000000E+02, 4.60000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.88337400E+02, 1.03000000E+02, 4.70000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.97094000E+02, 1.03000000E+02, 4.80000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.53137000E+02, 1.03000000E+02, 4.90000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.28081400E+02, 1.03000000E+02, 5.00000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.97769800E+02, 1.03000000E+02, 5.10000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.80305600E+02, 1.03000000E+02, 5.20000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.59867000E+02, 1.03000000E+02, 5.30000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.40981800E+02, 1.03000000E+02, 5.40000000E+01, 9.86500000E-01, 0.00000000E+00},
    {7.52341800E+02, 1.03000000E+02, 5.50000000E+01, 9.86500000E-01, 0.00000000E+00},
    {6.64571900E+02, 1.03000000E+02, 5.60000000E+01, 9.86500000E-01, 0.00000000E+00},
    {5.70426600E+02, 1.03000000E+02, 5.70000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.34527100E+02, 1.03000000E+02, 5.80000000E+01, 9.86500000E-01, 2.79910000E+00},
    {5.83979700E+02, 1.03000000E+02, 5.90000000E+01, 9.86500000E-01, 0.00000000E+00},
    {5.58697200E+02, 1.03000000E+02, 6.00000000E+01, 9.86500000E-01, 0.00000000E+00},
    {5.44123200E+02, 1.03000000E+02, 6.10000000E+01, 9.86500000E-01, 0.00000000E+00},
    {5.30790900E+02, 1.03000000E+02, 6.20000000E+01, 9.86500000E-01, 0.00000000E+00},
    {5.18949800E+02, 1.03000000E+02, 6.30000000E+01, 9.86500000E-01, 0.00000000E+00},
    {3.96722300E+02, 1.03000000E+02, 6.40000000E+01, 9.86500000E-01, 0.00000000E+00},
    {4.63544200E+02, 1.03000000E+02, 6.50000000E+01, 9.86500000E-01, 0.00000000E+00},
    {4.45010100E+02, 1.03000000E+02, 6.60000000E+01, 9.86500000E-01, 0.00000000E+00},
    {4.65410000E+02, 1.03000000E+02, 6.70000000E+01, 9.86500000E-01, 0.00000000E+00},
    {4.55282700E+02, 1.03000000E+02, 6.80000000E+01, 9.86500000E-01, 0.00000000E+00},
    {4.46003000E+02, 1.03000000E+02, 6.90000000E+01, 9.86500000E-01, 0.00000000E+00},
    {4.41322000E+02, 1.03000000E+02, 7.00000000E+01, 9.86500000E-01, 0.00000000E+00},
    {3.64746700E+02, 1.03000000E+02, 7.10000000E+01, 9.86500000E-01, 0.00000000E+00},
    {3.50061700E+02, 1.03000000E+02, 7.20000000E+01, 9.86500000E-01, 0.00000000E+00},
    {3.14248300E+02, 1.03000000E+02, 7.30000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.61383300E+02, 1.03000000E+02, 7.40000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.64369700E+02, 1.03000000E+02, 7.50000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.36025800E+02, 1.03000000E+02, 7.60000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.13464400E+02, 1.03000000E+02, 7.70000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.74613200E+02, 1.03000000E+02, 7.80000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.62134200E+02, 1.03000000E+02, 7.90000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.66048900E+02, 1.03000000E+02, 8.00000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.57382300E+02, 1.03000000E+02, 8.10000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.46835500E+02, 1.03000000E+02, 8.20000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.21432500E+02, 1.03000000E+02, 8.30000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.08126900E+02, 1.03000000E+02, 8.40000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.88607300E+02, 1.03000000E+02, 8.50000000E+01, 9.86500000E-01, 0.00000000E+00},
    {1.69894300E+02, 1.03000000E+02, 8.60000000E+01, 9.86500000E-01, 0.00000000E+00},
    {6.97644900E+02, 1.03000000E+02, 8.70000000E+01, 9.86500000E-01, 0.00000000E+00},
    {6.49032500E+02, 1.03000000E+02, 8.80000000E+01, 9.86500000E-01, 0.00000000E+00},
    {5.61351900E+02, 1.03000000E+02, 8.90000000E+01, 9.86500000E-01, 0.00000000E+00},
    {4.91077700E+02, 1.03000000E+02, 9.00000000E+01, 9.86500000E-01, 0.00000000E+00},
    {4.93465100E+02, 1.03000000E+02, 9.10000000E+01, 9.86500000E-01, 0.00000000E+00},
    {4.77425400E+02, 1.03000000E+02, 9.20000000E+01, 9.86500000E-01, 0.00000000E+00},
    {4.99520200E+02, 1.03000000E+02, 9.30000000E+01, 9.86500000E-01, 0.00000000E+00},
    {4.82346600E+02, 1.03000000E+02, 9.40000000E+01, 9.86500000E-01, 0.00000000E+00},
    {2.40573000E+01, 1.03000000E+02, 1.01000000E+02, 9.86500000E-01, 0.00000000E+00},
    {8.53197000E+01, 1.03000000E+02, 1.03000000E+02, 9.86500000E-01, 9.86500000E-01},
    {1.84656000E+01, 1.04000000E+02, 1.00000000E+00, 9.80800000E-01, 9.11800000E-01},
    {1.14655000E+01, 1.04000000E+02, 2.00000000E+00, 9.80800000E-01, 0.00000000E+00},
    {3.52216000E+02, 1.04000000E+02, 3.00000000E+00, 9.80800000E-01, 0.00000000E+00},
    {1.86345300E+02, 1.04000000E+02, 4.00000000E+00, 9.80800000E-01, 0.00000000E+00},
    {1.19081500E+02, 1.04000000E+02, 5.00000000E+00, 9.80800000E-01, 0.00000000E+00},
    {7.70951000E+01, 1.04000000E+02, 6.00000000E+00, 9.80800000E-01, 0.00000000E+00},
    {5.20852000E+01, 1.04000000E+02, 7.00000000E+00, 9.80800000E-01, 0.00000000E+00},
    {3.84132000E+01, 1.04000000E+02, 8.00000000E+00, 9.80800000E-01, 0.00000000E+00},
    {2.84098000E+01, 1.04000000E+02, 9.00000000E+00, 9.80800000E-01, 0.00000000E+00},
    {2.14067000E+01, 1.04000000E+02, 1.00000000E+01, 9.80800000E-01, 0.00000000E+00},
    {4.18648800E+02, 1.04000000E+02, 1.10000000E+01, 9.80800000E-01, 0.00000000E+00},
    {3.01377700E+02, 1.04000000E+02, 1.20000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.70746100E+02, 1.04000000E+02, 1.30000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.05754400E+02, 1.04000000E+02, 1.40000000E+01, 9.80800000E-01, 0.00000000E+00},
    {1.55364400E+02, 1.04000000E+02, 1.50000000E+01, 9.80800000E-01, 0.00000000E+00},
    {1.26099300E+02, 1.04000000E+02, 1.60000000E+01, 9.80800000E-01, 0.00000000E+00},
    {1.00721200E+02, 1.04000000E+02, 1.70000000E+01, 9.80800000E-01, 0.00000000E+00},
    {8.07174000E+01, 1.04000000E+02, 1.80000000E+01, 9.80800000E-01, 0.00000000E+00},
    {6.94859600E+02, 1.04000000E+02, 1.90000000E+01, 9.80800000E-01, 0.00000000E+00},
    {5.43584200E+02, 1.04000000E+02, 2.00000000E+01, 9.80800000E-01, 0.00000000E+00},
    {4.43719000E+02, 1.04000000E+02, 2.10000000E+01, 9.80800000E-01, 0.00000000E+00},
    {4.23621700E+02, 1.04000000E+02, 2.20000000E+01, 9.80800000E-01, 0.00000000E+00},
    {3.85275400E+02, 1.04000000E+02, 2.30000000E+01, 9.80800000E-01, 0.00000000E+00},
    {3.02606100E+02, 1.04000000E+02, 2.40000000E+01, 9.80800000E-01, 0.00000000E+00},
    {3.28401200E+02, 1.04000000E+02, 2.50000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.56604300E+02, 1.04000000E+02, 2.60000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.68628400E+02, 1.04000000E+02, 2.70000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.78711600E+02, 1.04000000E+02, 2.80000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.12979700E+02, 1.04000000E+02, 2.90000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.14425600E+02, 1.04000000E+02, 3.00000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.56204900E+02, 1.04000000E+02, 3.10000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.19737500E+02, 1.04000000E+02, 3.20000000E+01, 9.80800000E-01, 0.00000000E+00},
    {1.82533800E+02, 1.04000000E+02, 3.30000000E+01, 9.80800000E-01, 0.00000000E+00},
    {1.60968700E+02, 1.04000000E+02, 3.40000000E+01, 9.80800000E-01, 0.00000000E+00},
    {1.38274700E+02, 1.04000000E+02, 3.50000000E+01, 9.80800000E-01, 0.00000000E+00},
    {1.18124000E+02, 1.04000000E+02, 3.60000000E+01, 9.80800000E-01, 0.00000000E+00},
    {7.74965600E+02, 1.04000000E+02, 3.70000000E+01, 9.80800000E-01, 0.00000000E+00},
    {6.48642200E+02, 1.04000000E+02, 3.80000000E+01, 9.80800000E-01, 0.00000000E+00},
    {5.56150300E+02, 1.04000000E+02, 3.90000000E+01, 9.80800000E-01, 0.00000000E+00},
    {4.93010000E+02, 1.04000000E+02, 4.00000000E+01, 9.80800000E-01, 0.00000000E+00},
    {4.45373600E+02, 1.04000000E+02, 4.10000000E+01, 9.80800000E-01, 0.00000000E+00},
    {3.37903500E+02, 1.04000000E+02, 4.20000000E+01, 9.80800000E-01, 0.00000000E+00},
    {3.79525400E+02, 1.04000000E+02, 4.30000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.83569500E+02, 1.04000000E+02, 4.40000000E+01, 9.80800000E-01, 0.00000000E+00},
    {3.10378100E+02, 1.04000000E+02, 4.50000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.86039500E+02, 1.04000000E+02, 4.60000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.38696200E+02, 1.04000000E+02, 4.70000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.49991500E+02, 1.04000000E+02, 4.80000000E+01, 9.80800000E-01, 0.00000000E+00},
    {3.20091200E+02, 1.04000000E+02, 4.90000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.89471300E+02, 1.04000000E+02, 5.00000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.52164900E+02, 1.04000000E+02, 5.10000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.30630900E+02, 1.04000000E+02, 5.20000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.05223700E+02, 1.04000000E+02, 5.30000000E+01, 9.80800000E-01, 0.00000000E+00},
    {1.81626000E+02, 1.04000000E+02, 5.40000000E+01, 9.80800000E-01, 0.00000000E+00},
    {9.43854700E+02, 1.04000000E+02, 5.50000000E+01, 9.80800000E-01, 0.00000000E+00},
    {8.32625100E+02, 1.04000000E+02, 5.60000000E+01, 9.80800000E-01, 0.00000000E+00},
    {7.16075800E+02, 1.04000000E+02, 5.70000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.98878200E+02, 1.04000000E+02, 5.80000000E+01, 9.80800000E-01, 2.79910000E+00},
    {7.32164200E+02, 1.04000000E+02, 5.90000000E+01, 9.80800000E-01, 0.00000000E+00},
    {7.00444200E+02, 1.04000000E+02, 6.00000000E+01, 9.80800000E-01, 0.00000000E+00},
    {6.82190100E+02, 1.04000000E+02, 6.10000000E+01, 9.80800000E-01, 0.00000000E+00},
    {6.65486800E+02, 1.04000000E+02, 6.20000000E+01, 9.80800000E-01, 0.00000000E+00},
    {6.50655700E+02, 1.04000000E+02, 6.30000000E+01, 9.80800000E-01, 0.00000000E+00},
    {4.99183700E+02, 1.04000000E+02, 6.40000000E+01, 9.80800000E-01, 0.00000000E+00},
    {5.82122400E+02, 1.04000000E+02, 6.50000000E+01, 9.80800000E-01, 0.00000000E+00},
    {5.59291700E+02, 1.04000000E+02, 6.60000000E+01, 9.80800000E-01, 0.00000000E+00},
    {5.83706700E+02, 1.04000000E+02, 6.70000000E+01, 9.80800000E-01, 0.00000000E+00},
    {5.71003900E+02, 1.04000000E+02, 6.80000000E+01, 9.80800000E-01, 0.00000000E+00},
    {5.59389800E+02, 1.04000000E+02, 6.90000000E+01, 9.80800000E-01, 0.00000000E+00},
    {5.53426600E+02, 1.04000000E+02, 7.00000000E+01, 9.80800000E-01, 0.00000000E+00},
    {4.58658200E+02, 1.04000000E+02, 7.10000000E+01, 9.80800000E-01, 0.00000000E+00},
    {4.41378900E+02, 1.04000000E+02, 7.20000000E+01, 9.80800000E-01, 0.00000000E+00},
    {3.97131200E+02, 1.04000000E+02, 7.30000000E+01, 9.80800000E-01, 0.00000000E+00},
    {3.31212400E+02, 1.04000000E+02, 7.40000000E+01, 9.80800000E-01, 0.00000000E+00},
    {3.35226100E+02, 1.04000000E+02, 7.50000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.99956500E+02, 1.04000000E+02, 7.60000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.71811900E+02, 1.04000000E+02, 7.70000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.22915700E+02, 1.04000000E+02, 7.80000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.07183100E+02, 1.04000000E+02, 7.90000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.12313200E+02, 1.04000000E+02, 8.00000000E+01, 9.80800000E-01, 0.00000000E+00},
    {3.26106800E+02, 1.04000000E+02, 8.10000000E+01, 9.80800000E-01, 0.00000000E+00},
    {3.13467500E+02, 1.04000000E+02, 8.20000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.82293500E+02, 1.04000000E+02, 8.30000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.66020000E+02, 1.04000000E+02, 8.40000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.41868800E+02, 1.04000000E+02, 8.50000000E+01, 9.80800000E-01, 0.00000000E+00},
    {2.18568700E+02, 1.04000000E+02, 8.60000000E+01, 9.80800000E-01, 0.00000000E+00},
    {8.75856300E+02, 1.04000000E+02, 8.70000000E+01, 9.80800000E-01, 0.00000000E+00},
    {8.14130800E+02, 1.04000000E+02, 8.80000000E+01, 9.80800000E-01, 0.00000000E+00},
    {7.05401100E+02, 1.04000000E+02, 8.90000000E+01, 9.80800000E-01, 0.00000000E+00},
    {6.19060500E+02, 1.04000000E+02, 9.00000000E+01, 9.80800000E-01, 0.00000000E+00},
    {6.21339900E+02, 1.04000000E+02, 9.10000000E+01, 9.80800000E-01, 0.00000000E+00},
    {6.01149600E+02, 1.04000000E+02, 9.20000000E+01, 9.80800000E-01, 0.00000000E+00},
    {6.27638500E+02, 1.04000000E+02, 9.30000000E+01, 9.80800000E-01, 0.00000000E+00},
    {6.06213900E+02, 1.04000000E+02, 9.40000000E+01, 9.80800000E-01, 0.00000000E+00},
    {3.07866000E+01, 1.04000000E+02, 1.01000000E+02, 9.80800000E-01, 0.00000000E+00},
    {1.07508300E+02, 1.04000000E+02, 1.03000000E+02, 9.80800000E-01, 9.86500000E-01},
    {1.35844500E+02, 1.04000000E+02, 1.04000000E+02, 9.80800000E-01, 9.80800000E-01},
    {1.47237000E+01, 1.05000000E+02, 1.00000000E+00, 9.70600000E-01, 9.11800000E-01},
    {9.58360000E+00, 1.05000000E+02, 2.00000000E+00, 9.70600000E-01, 0.00000000E+00},
    {2.31420900E+02, 1.05000000E+02, 3.00000000E+00, 9.70600000E-01, 0.00000000E+00},
    {1.33726500E+02, 1.05000000E+02, 4.00000000E+00, 9.70600000E-01, 0.00000000E+00},
    {8.97429000E+01, 1.05000000E+02, 5.00000000E+00, 9.70600000E-01, 0.00000000E+00},
    {6.02996000E+01, 1.05000000E+02, 6.00000000E+00, 9.70600000E-01, 0.00000000E+00},
    {4.19015000E+01, 1.05000000E+02, 7.00000000E+00, 9.70600000E-01, 0.00000000E+00},
    {3.15354000E+01, 1.05000000E+02, 8.00000000E+00, 9.70600000E-01, 0.00000000E+00},
    {2.37369000E+01, 1.05000000E+02, 9.00000000E+00, 9.70600000E-01, 0.00000000E+00},
    {1.81424000E+01, 1.05000000E+02, 1.00000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.76567100E+02, 1.05000000E+02, 1.10000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.12997900E+02, 1.05000000E+02, 1.20000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.96201400E+02, 1.05000000E+02, 1.30000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.54290100E+02, 1.05000000E+02, 1.40000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.19935300E+02, 1.05000000E+02, 1.50000000E+01, 9.70600000E-01, 0.00000000E+00},
    {9.92159000E+01, 1.05000000E+02, 1.60000000E+01, 9.70600000E-01, 0.00000000E+00},
    {8.07272000E+01, 1.05000000E+02, 1.70000000E+01, 9.70600000E-01, 0.00000000E+00},
    {6.57580000E+01, 1.05000000E+02, 1.80000000E+01, 9.70600000E-01, 0.00000000E+00},
    {4.52519300E+02, 1.05000000E+02, 1.90000000E+01, 9.70600000E-01, 0.00000000E+00},
    {3.73881500E+02, 1.05000000E+02, 2.00000000E+01, 9.70600000E-01, 0.00000000E+00},
    {3.08872100E+02, 1.05000000E+02, 2.10000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.98087100E+02, 1.05000000E+02, 2.20000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.72874700E+02, 1.05000000E+02, 2.30000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.14636000E+02, 1.05000000E+02, 2.40000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.34800000E+02, 1.05000000E+02, 2.50000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.83973700E+02, 1.05000000E+02, 2.60000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.95173100E+02, 1.05000000E+02, 2.70000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.01135100E+02, 1.05000000E+02, 2.80000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.53882400E+02, 1.05000000E+02, 2.90000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.58084800E+02, 1.05000000E+02, 3.00000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.87579700E+02, 1.05000000E+02, 3.10000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.65193000E+02, 1.05000000E+02, 3.20000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.40562500E+02, 1.05000000E+02, 3.30000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.25843700E+02, 1.05000000E+02, 3.40000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.09806000E+02, 1.05000000E+02, 3.50000000E+01, 9.70600000E-01, 0.00000000E+00},
    {9.51763000E+01, 1.05000000E+02, 3.60000000E+01, 9.70600000E-01, 0.00000000E+00},
    {5.06942100E+02, 1.05000000E+02, 3.70000000E+01, 9.70600000E-01, 0.00000000E+00},
    {4.45260800E+02, 1.05000000E+02, 3.80000000E+01, 9.70600000E-01, 0.00000000E+00},
    {3.89882300E+02, 1.05000000E+02, 3.90000000E+01, 9.70600000E-01, 0.00000000E+00},
    {3.50197000E+02, 1.05000000E+02, 4.00000000E+01, 9.70600000E-01, 0.00000000E+00},
    {3.19115100E+02, 1.05000000E+02, 4.10000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.45852700E+02, 1.05000000E+02, 4.20000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.74533400E+02, 1.05000000E+02, 4.30000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.08636200E+02, 1.05000000E+02, 4.40000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.28282200E+02, 1.05000000E+02, 4.50000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.11555900E+02, 1.05000000E+02, 4.60000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.76060300E+02, 1.05000000E+02, 4.70000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.86219900E+02, 1.05000000E+02, 4.80000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.34217100E+02, 1.05000000E+02, 4.90000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.16427600E+02, 1.05000000E+02, 5.00000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.92554200E+02, 1.05000000E+02, 5.10000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.78391900E+02, 1.05000000E+02, 5.20000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.60967900E+02, 1.05000000E+02, 5.30000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.44369700E+02, 1.05000000E+02, 5.40000000E+01, 9.70600000E-01, 0.00000000E+00},
    {6.17605800E+02, 1.05000000E+02, 5.50000000E+01, 9.70600000E-01, 0.00000000E+00},
    {5.67513600E+02, 1.05000000E+02, 5.60000000E+01, 9.70600000E-01, 0.00000000E+00},
    {4.98453400E+02, 1.05000000E+02, 5.70000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.27418900E+02, 1.05000000E+02, 5.80000000E+01, 9.70600000E-01, 2.79910000E+00},
    {5.02496700E+02, 1.05000000E+02, 5.90000000E+01, 9.70600000E-01, 0.00000000E+00},
    {4.82507700E+02, 1.05000000E+02, 6.00000000E+01, 9.70600000E-01, 0.00000000E+00},
    {4.70402900E+02, 1.05000000E+02, 6.10000000E+01, 9.70600000E-01, 0.00000000E+00},
    {4.59275300E+02, 1.05000000E+02, 6.20000000E+01, 9.70600000E-01, 0.00000000E+00},
    {4.49410200E+02, 1.05000000E+02, 6.30000000E+01, 9.70600000E-01, 0.00000000E+00},
    {3.52908700E+02, 1.05000000E+02, 6.40000000E+01, 9.70600000E-01, 0.00000000E+00},
    {3.97449900E+02, 1.05000000E+02, 6.50000000E+01, 9.70600000E-01, 0.00000000E+00},
    {3.83295200E+02, 1.05000000E+02, 6.60000000E+01, 9.70600000E-01, 0.00000000E+00},
    {4.05328500E+02, 1.05000000E+02, 6.70000000E+01, 9.70600000E-01, 0.00000000E+00},
    {3.96737700E+02, 1.05000000E+02, 6.80000000E+01, 9.70600000E-01, 0.00000000E+00},
    {3.88989000E+02, 1.05000000E+02, 6.90000000E+01, 9.70600000E-01, 0.00000000E+00},
    {3.84470100E+02, 1.05000000E+02, 7.00000000E+01, 9.70600000E-01, 0.00000000E+00},
    {3.23645000E+02, 1.05000000E+02, 7.10000000E+01, 9.70600000E-01, 0.00000000E+00},
    {3.18287200E+02, 1.05000000E+02, 7.20000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.90180100E+02, 1.05000000E+02, 7.30000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.44567500E+02, 1.05000000E+02, 7.40000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.48777500E+02, 1.05000000E+02, 7.50000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.25181900E+02, 1.05000000E+02, 7.60000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.05970200E+02, 1.05000000E+02, 7.70000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.70655600E+02, 1.05000000E+02, 7.80000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.59263500E+02, 1.05000000E+02, 7.90000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.63882000E+02, 1.05000000E+02, 8.00000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.40000000E+02, 1.05000000E+02, 8.10000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.34604800E+02, 1.05000000E+02, 8.20000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.15302700E+02, 1.05000000E+02, 8.30000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.05133400E+02, 1.05000000E+02, 8.40000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.88999100E+02, 1.05000000E+02, 8.50000000E+01, 9.70600000E-01, 0.00000000E+00},
    {1.72885400E+02, 1.05000000E+02, 8.60000000E+01, 9.70600000E-01, 0.00000000E+00},
    {5.82981500E+02, 1.05000000E+02, 8.70000000E+01, 9.70600000E-01, 0.00000000E+00},
    {5.61008600E+02, 1.05000000E+02, 8.80000000E+01, 9.70600000E-01, 0.00000000E+00},
    {4.95638900E+02, 1.05000000E+02, 8.90000000E+01, 9.70600000E-01, 0.00000000E+00},
    {4.44810200E+02, 1.05000000E+02, 9.00000000E+01, 9.70600000E-01, 0.00000000E+00},
    {4.41557800E+02, 1.05000000E+02, 9.10000000E+01, 9.70600000E-01, 0.00000000E+00},
    {4.27480700E+02, 1.05000000E+02, 9.20000000E+01, 9.70600000E-01, 0.00000000E+00},
    {4.40271000E+02, 1.05000000E+02, 9.30000000E+01, 9.70600000E-01, 0.00000000E+00},
    {4.26325500E+02, 1.05000000E+02, 9.40000000E+01, 9.70600000E-01, 0.00000000E+00},
    {2.38412000E+01, 1.05000000E+02, 1.01000000E+02, 9.70600000E-01, 0.00000000E+00},
    {7.76633000E+01, 1.05000000E+02, 1.03000000E+02, 9.70600000E-01, 9.86500000E-01},
    {9.89938000E+01, 1.05000000E+02, 1.04000000E+02, 9.70600000E-01, 9.80800000E-01},
    {7.53686000E+01, 1.05000000E+02, 1.05000000E+02, 9.70600000E-01, 9.70600000E-01},
    {1.13932000E+01, 1.06000000E+02, 1.00000000E+00, 9.86800000E-01, 9.11800000E-01},
    {7.70650000E+00, 1.06000000E+02, 2.00000000E+00, 9.86800000E-01, 0.00000000E+00},
    {1.60459000E+02, 1.06000000E+02, 3.00000000E+00, 9.86800000E-01, 0.00000000E+00},
    {9.65431000E+01, 1.06000000E+02, 4.00000000E+00, 9.86800000E-01, 0.00000000E+00},
    {6.67986000E+01, 1.06000000E+02, 5.00000000E+00, 9.86800000E-01, 0.00000000E+00},
    {4.60681000E+01, 1.06000000E+02, 6.00000000E+00, 9.86800000E-01, 0.00000000E+00},
    {3.27009000E+01, 1.06000000E+02, 7.00000000E+00, 9.86800000E-01, 0.00000000E+00},
    {2.50097000E+01, 1.06000000E+02, 8.00000000E+00, 9.86800000E-01, 0.00000000E+00},
    {1.90974000E+01, 1.06000000E+02, 9.00000000E+00, 9.86800000E-01, 0.00000000E+00},
    {1.47716000E+01, 1.06000000E+02, 1.00000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.92372900E+02, 1.06000000E+02, 1.10000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.52613000E+02, 1.06000000E+02, 1.20000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.42609400E+02, 1.06000000E+02, 1.30000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.14440500E+02, 1.06000000E+02, 1.40000000E+01, 9.86800000E-01, 0.00000000E+00},
    {9.06767000E+01, 1.06000000E+02, 1.50000000E+01, 9.86800000E-01, 0.00000000E+00},
    {7.60552000E+01, 1.06000000E+02, 1.60000000E+01, 9.86800000E-01, 0.00000000E+00},
    {6.27465000E+01, 1.06000000E+02, 1.70000000E+01, 9.86800000E-01, 0.00000000E+00},
    {5.17684000E+01, 1.06000000E+02, 1.80000000E+01, 9.86800000E-01, 0.00000000E+00},
    {3.14580000E+02, 1.06000000E+02, 1.90000000E+01, 9.86800000E-01, 0.00000000E+00},
    {2.65120600E+02, 1.06000000E+02, 2.00000000E+01, 9.86800000E-01, 0.00000000E+00},
    {2.20185400E+02, 1.06000000E+02, 2.10000000E+01, 9.86800000E-01, 0.00000000E+00},
    {2.13868200E+02, 1.06000000E+02, 2.20000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.96488500E+02, 1.06000000E+02, 2.30000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.55117100E+02, 1.06000000E+02, 2.40000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.69979600E+02, 1.06000000E+02, 2.50000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.33770000E+02, 1.06000000E+02, 2.60000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.42514000E+02, 1.06000000E+02, 2.70000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.46271100E+02, 1.06000000E+02, 2.80000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.12391400E+02, 1.06000000E+02, 2.90000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.16398700E+02, 1.06000000E+02, 3.00000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.37472300E+02, 1.06000000E+02, 3.10000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.22880500E+02, 1.06000000E+02, 3.20000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.06168600E+02, 1.06000000E+02, 3.30000000E+01, 9.86800000E-01, 0.00000000E+00},
    {9.60599000E+01, 1.06000000E+02, 3.40000000E+01, 9.86800000E-01, 0.00000000E+00},
    {8.47723000E+01, 1.06000000E+02, 3.50000000E+01, 9.86800000E-01, 0.00000000E+00},
    {7.42865000E+01, 1.06000000E+02, 3.60000000E+01, 9.86800000E-01, 0.00000000E+00},
    {3.53630200E+02, 1.06000000E+02, 3.70000000E+01, 9.86800000E-01, 0.00000000E+00},
    {3.15892300E+02, 1.06000000E+02, 3.80000000E+01, 9.86800000E-01, 0.00000000E+00},
    {2.79472700E+02, 1.06000000E+02, 3.90000000E+01, 9.86800000E-01, 0.00000000E+00},
    {2.52821300E+02, 1.06000000E+02, 4.00000000E+01, 9.86800000E-01, 0.00000000E+00},
    {2.31597600E+02, 1.06000000E+02, 4.10000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.80339000E+02, 1.06000000E+02, 4.20000000E+01, 9.86800000E-01, 0.00000000E+00},
    {2.00546400E+02, 1.06000000E+02, 4.30000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.54193800E+02, 1.06000000E+02, 4.40000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.68311500E+02, 1.06000000E+02, 4.50000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.56500600E+02, 1.06000000E+02, 4.60000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.30514400E+02, 1.06000000E+02, 4.70000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.38367200E+02, 1.06000000E+02, 4.80000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.72122900E+02, 1.06000000E+02, 4.90000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.60791000E+02, 1.06000000E+02, 5.00000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.44846200E+02, 1.06000000E+02, 5.10000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.35312200E+02, 1.06000000E+02, 5.20000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.23253800E+02, 1.06000000E+02, 5.30000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.11592300E+02, 1.06000000E+02, 5.40000000E+01, 9.86800000E-01, 0.00000000E+00},
    {4.31517200E+02, 1.06000000E+02, 5.50000000E+01, 9.86800000E-01, 0.00000000E+00},
    {4.01877400E+02, 1.06000000E+02, 5.60000000E+01, 9.86800000E-01, 0.00000000E+00},
    {3.56461000E+02, 1.06000000E+02, 5.70000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.70926200E+02, 1.06000000E+02, 5.80000000E+01, 9.86800000E-01, 2.79910000E+00},
    {3.57139900E+02, 1.06000000E+02, 5.90000000E+01, 9.86800000E-01, 0.00000000E+00},
    {3.43415300E+02, 1.06000000E+02, 6.00000000E+01, 9.86800000E-01, 0.00000000E+00},
    {3.34924500E+02, 1.06000000E+02, 6.10000000E+01, 9.86800000E-01, 0.00000000E+00},
    {3.27099100E+02, 1.06000000E+02, 6.20000000E+01, 9.86800000E-01, 0.00000000E+00},
    {3.20165700E+02, 1.06000000E+02, 6.30000000E+01, 9.86800000E-01, 0.00000000E+00},
    {2.54812200E+02, 1.06000000E+02, 6.40000000E+01, 9.86800000E-01, 0.00000000E+00},
    {2.82854700E+02, 1.06000000E+02, 6.50000000E+01, 9.86800000E-01, 0.00000000E+00},
    {2.73337500E+02, 1.06000000E+02, 6.60000000E+01, 9.86800000E-01, 0.00000000E+00},
    {2.89392000E+02, 1.06000000E+02, 6.70000000E+01, 9.86800000E-01, 0.00000000E+00},
    {2.83300100E+02, 1.06000000E+02, 6.80000000E+01, 9.86800000E-01, 0.00000000E+00},
    {2.77849300E+02, 1.06000000E+02, 6.90000000E+01, 9.86800000E-01, 0.00000000E+00},
    {2.74450000E+02, 1.06000000E+02, 7.00000000E+01, 9.86800000E-01, 0.00000000E+00},
    {2.33139400E+02, 1.06000000E+02, 7.10000000E+01, 9.86800000E-01, 0.00000000E+00},
    {2.31633400E+02, 1.06000000E+02, 7.20000000E+01, 9.86800000E-01, 0.00000000E+00},
    {2.12804100E+02, 1.06000000E+02, 7.30000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.80846200E+02, 1.06000000E+02, 7.40000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.84358600E+02, 1.06000000E+02, 7.50000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.68047700E+02, 1.06000000E+02, 7.60000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.54624800E+02, 1.06000000E+02, 7.70000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.29183500E+02, 1.06000000E+02, 7.80000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.20948500E+02, 1.06000000E+02, 7.90000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.24595100E+02, 1.06000000E+02, 8.00000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.77485300E+02, 1.06000000E+02, 8.10000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.74767500E+02, 1.06000000E+02, 8.20000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.62074000E+02, 1.06000000E+02, 8.30000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.55450600E+02, 1.06000000E+02, 8.40000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.44453200E+02, 1.06000000E+02, 8.50000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.33243200E+02, 1.06000000E+02, 8.60000000E+01, 9.86800000E-01, 0.00000000E+00},
    {4.10586000E+02, 1.06000000E+02, 8.70000000E+01, 9.86800000E-01, 0.00000000E+00},
    {3.99522500E+02, 1.06000000E+02, 8.80000000E+01, 9.86800000E-01, 0.00000000E+00},
    {3.56112700E+02, 1.06000000E+02, 8.90000000E+01, 9.86800000E-01, 0.00000000E+00},
    {3.23495900E+02, 1.06000000E+02, 9.00000000E+01, 9.86800000E-01, 0.00000000E+00},
    {3.19676100E+02, 1.06000000E+02, 9.10000000E+01, 9.86800000E-01, 0.00000000E+00},
    {3.09608200E+02, 1.06000000E+02, 9.20000000E+01, 9.86800000E-01, 0.00000000E+00},
    {3.16596700E+02, 1.06000000E+02, 9.30000000E+01, 9.86800000E-01, 0.00000000E+00},
    {3.06930300E+02, 1.06000000E+02, 9.40000000E+01, 9.86800000E-01, 0.00000000E+00},
    {1.80575000E+01, 1.06000000E+02, 1.01000000E+02, 9.86800000E-01, 0.00000000E+00},
    {5.63468000E+01, 1.06000000E+02, 1.03000000E+02, 9.86800000E-01, 9.86500000E-01},
    {7.23162000E+01, 1.06000000E+02, 1.04000000E+02, 9.86800000E-01, 9.80800000E-01},
    {5.65295000E+01, 1.06000000E+02, 1.05000000E+02, 9.86800000E-01, 9.70600000E-01},
    {4.32452000E+01, 1.06000000E+02, 1.06000000E+02, 9.86800000E-01, 9.86800000E-01},
    {8.14170000E+00, 1.07000000E+02, 1.00000000E+00, 9.94400000E-01, 9.11800000E-01},
    {5.76010000E+00, 1.07000000E+02, 2.00000000E+00, 9.94400000E-01, 0.00000000E+00},
    {1.01270100E+02, 1.07000000E+02, 3.00000000E+00, 9.94400000E-01, 0.00000000E+00},
    {6.38580000E+01, 1.07000000E+02, 4.00000000E+00, 9.94400000E-01, 0.00000000E+00},
    {4.57319000E+01, 1.07000000E+02, 5.00000000E+00, 9.94400000E-01, 0.00000000E+00},
    {3.24848000E+01, 1.07000000E+02, 6.00000000E+00, 9.94400000E-01, 0.00000000E+00},
    {2.36295000E+01, 1.07000000E+02, 7.00000000E+00, 9.94400000E-01, 0.00000000E+00},
    {1.84128000E+01, 1.07000000E+02, 8.00000000E+00, 9.94400000E-01, 0.00000000E+00},
    {1.43015000E+01, 1.07000000E+02, 9.00000000E+00, 9.94400000E-01, 0.00000000E+00},
    {1.12243000E+01, 1.07000000E+02, 1.00000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.21960100E+02, 1.07000000E+02, 1.10000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.00144300E+02, 1.07000000E+02, 1.20000000E+01, 9.94400000E-01, 0.00000000E+00},
    {9.50916000E+01, 1.07000000E+02, 1.30000000E+01, 9.94400000E-01, 0.00000000E+00},
    {7.80466000E+01, 1.07000000E+02, 1.40000000E+01, 9.94400000E-01, 0.00000000E+00},
    {6.31669000E+01, 1.07000000E+02, 1.50000000E+01, 9.94400000E-01, 0.00000000E+00},
    {5.38107000E+01, 1.07000000E+02, 1.60000000E+01, 9.94400000E-01, 0.00000000E+00},
    {4.50952000E+01, 1.07000000E+02, 1.70000000E+01, 9.94400000E-01, 0.00000000E+00},
    {3.77522000E+01, 1.07000000E+02, 1.80000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.99440500E+02, 1.07000000E+02, 1.90000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.72045500E+02, 1.07000000E+02, 2.00000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.43765800E+02, 1.07000000E+02, 2.10000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.40711500E+02, 1.07000000E+02, 2.20000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.29829700E+02, 1.07000000E+02, 2.30000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.03018900E+02, 1.07000000E+02, 2.40000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.13024800E+02, 1.07000000E+02, 2.50000000E+01, 9.94400000E-01, 0.00000000E+00},
    {8.94868000E+01, 1.07000000E+02, 2.60000000E+01, 9.94400000E-01, 0.00000000E+00},
    {9.57051000E+01, 1.07000000E+02, 2.70000000E+01, 9.94400000E-01, 0.00000000E+00},
    {9.77761000E+01, 1.07000000E+02, 2.80000000E+01, 9.94400000E-01, 0.00000000E+00},
    {7.55987000E+01, 1.07000000E+02, 2.90000000E+01, 9.94400000E-01, 0.00000000E+00},
    {7.89183000E+01, 1.07000000E+02, 3.00000000E+01, 9.94400000E-01, 0.00000000E+00},
    {9.26025000E+01, 1.07000000E+02, 3.10000000E+01, 9.94400000E-01, 0.00000000E+00},
    {8.41276000E+01, 1.07000000E+02, 3.20000000E+01, 9.94400000E-01, 0.00000000E+00},
    {7.39142000E+01, 1.07000000E+02, 3.30000000E+01, 9.94400000E-01, 0.00000000E+00},
    {6.76636000E+01, 1.07000000E+02, 3.40000000E+01, 9.94400000E-01, 0.00000000E+00},
    {6.04728000E+01, 1.07000000E+02, 3.50000000E+01, 9.94400000E-01, 0.00000000E+00},
    {5.36496000E+01, 1.07000000E+02, 3.60000000E+01, 9.94400000E-01, 0.00000000E+00},
    {2.25191900E+02, 1.07000000E+02, 3.70000000E+01, 9.94400000E-01, 0.00000000E+00},
    {2.05157400E+02, 1.07000000E+02, 3.80000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.83685100E+02, 1.07000000E+02, 3.90000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.67550100E+02, 1.07000000E+02, 4.00000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.54435600E+02, 1.07000000E+02, 4.10000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.21806600E+02, 1.07000000E+02, 4.20000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.34788700E+02, 1.07000000E+02, 4.30000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.05090500E+02, 1.07000000E+02, 4.40000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.14349600E+02, 1.07000000E+02, 4.50000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.06743800E+02, 1.07000000E+02, 4.60000000E+01, 9.94400000E-01, 0.00000000E+00},
    {8.93419000E+01, 1.07000000E+02, 4.70000000E+01, 9.94400000E-01, 0.00000000E+00},
    {9.48770000E+01, 1.07000000E+02, 4.80000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.16495900E+02, 1.07000000E+02, 4.90000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.10095100E+02, 1.07000000E+02, 5.00000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.00518200E+02, 1.07000000E+02, 5.10000000E+01, 9.94400000E-01, 0.00000000E+00},
    {9.47508000E+01, 1.07000000E+02, 5.20000000E+01, 9.94400000E-01, 0.00000000E+00},
    {8.72028000E+01, 1.07000000E+02, 5.30000000E+01, 9.94400000E-01, 0.00000000E+00},
    {7.97773000E+01, 1.07000000E+02, 5.40000000E+01, 9.94400000E-01, 0.00000000E+00},
    {2.75256700E+02, 1.07000000E+02, 5.50000000E+01, 9.94400000E-01, 0.00000000E+00},
    {2.60448100E+02, 1.07000000E+02, 5.60000000E+01, 9.94400000E-01, 0.00000000E+00},
    {2.33661900E+02, 1.07000000E+02, 5.70000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.18547000E+02, 1.07000000E+02, 5.80000000E+01, 9.94400000E-01, 2.79910000E+00},
    {2.32578300E+02, 1.07000000E+02, 5.90000000E+01, 9.94400000E-01, 0.00000000E+00},
    {2.24004000E+02, 1.07000000E+02, 6.00000000E+01, 9.94400000E-01, 0.00000000E+00},
    {2.18561100E+02, 1.07000000E+02, 6.10000000E+01, 9.94400000E-01, 0.00000000E+00},
    {2.13527700E+02, 1.07000000E+02, 6.20000000E+01, 9.94400000E-01, 0.00000000E+00},
    {2.09070000E+02, 1.07000000E+02, 6.30000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.69055600E+02, 1.07000000E+02, 6.40000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.84590400E+02, 1.07000000E+02, 6.50000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.78807400E+02, 1.07000000E+02, 6.60000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.89453100E+02, 1.07000000E+02, 6.70000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.85493100E+02, 1.07000000E+02, 6.80000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.81982300E+02, 1.07000000E+02, 6.90000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.79620200E+02, 1.07000000E+02, 7.00000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.54229200E+02, 1.07000000E+02, 7.10000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.54910200E+02, 1.07000000E+02, 7.20000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.43576300E+02, 1.07000000E+02, 7.30000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.23230200E+02, 1.07000000E+02, 7.40000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.25912100E+02, 1.07000000E+02, 7.50000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.15693800E+02, 1.07000000E+02, 7.60000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.07177500E+02, 1.07000000E+02, 7.70000000E+01, 9.94400000E-01, 0.00000000E+00},
    {9.04456000E+01, 1.07000000E+02, 7.80000000E+01, 9.94400000E-01, 0.00000000E+00},
    {8.50146000E+01, 1.07000000E+02, 7.90000000E+01, 9.94400000E-01, 0.00000000E+00},
    {8.76490000E+01, 1.07000000E+02, 8.00000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.21080000E+02, 1.07000000E+02, 8.10000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.20126800E+02, 1.07000000E+02, 8.20000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.12648500E+02, 1.07000000E+02, 8.30000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.08813400E+02, 1.07000000E+02, 8.40000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.02052400E+02, 1.07000000E+02, 8.50000000E+01, 9.94400000E-01, 0.00000000E+00},
    {9.49928000E+01, 1.07000000E+02, 8.60000000E+01, 9.94400000E-01, 0.00000000E+00},
    {2.64443000E+02, 1.07000000E+02, 8.70000000E+01, 9.94400000E-01, 0.00000000E+00},
    {2.60652900E+02, 1.07000000E+02, 8.80000000E+01, 9.94400000E-01, 0.00000000E+00},
    {2.34765200E+02, 1.07000000E+02, 8.90000000E+01, 9.94400000E-01, 0.00000000E+00},
    {2.16307400E+02, 1.07000000E+02, 9.00000000E+01, 9.94400000E-01, 0.00000000E+00},
    {2.12745600E+02, 1.07000000E+02, 9.10000000E+01, 9.94400000E-01, 0.00000000E+00},
    {2.06163100E+02, 1.07000000E+02, 9.20000000E+01, 9.94400000E-01, 0.00000000E+00},
    {2.09152400E+02, 1.07000000E+02, 9.30000000E+01, 9.94400000E-01, 0.00000000E+00},
    {2.03041800E+02, 1.07000000E+02, 9.40000000E+01, 9.94400000E-01, 0.00000000E+00},
    {1.25980000E+01, 1.07000000E+02, 1.01000000E+02, 9.94400000E-01, 0.00000000E+00},
    {3.74939000E+01, 1.07000000E+02, 1.03000000E+02, 9.94400000E-01, 9.86500000E-01},
    {4.84885000E+01, 1.07000000E+02, 1.04000000E+02, 9.94400000E-01, 9.80800000E-01},
    {3.90433000E+01, 1.07000000E+02, 1.05000000E+02, 9.94400000E-01, 9.70600000E-01},
    {3.05410000E+01, 1.07000000E+02, 1.06000000E+02, 9.94400000E-01, 9.86800000E-01},
    {2.21241000E+01, 1.07000000E+02, 1.07000000E+02, 9.94400000E-01, 9.94400000E-01},
    {6.05750000E+00, 1.08000000E+02, 1.00000000E+00, 9.92500000E-01, 9.11800000E-01},
    {4.45930000E+00, 1.08000000E+02, 2.00000000E+00, 9.92500000E-01, 0.00000000E+00},
    {6.79312000E+01, 1.08000000E+02, 3.00000000E+00, 9.92500000E-01, 0.00000000E+00},
    {4.44968000E+01, 1.08000000E+02, 4.00000000E+00, 9.92500000E-01, 0.00000000E+00},
    {3.28136000E+01, 1.08000000E+02, 5.00000000E+00, 9.92500000E-01, 0.00000000E+00},
    {2.39120000E+01, 1.08000000E+02, 6.00000000E+00, 9.92500000E-01, 0.00000000E+00},
    {1.77698000E+01, 1.08000000E+02, 7.00000000E+00, 9.92500000E-01, 0.00000000E+00},
    {1.40764000E+01, 1.08000000E+02, 8.00000000E+00, 9.92500000E-01, 0.00000000E+00},
    {1.10994000E+01, 1.08000000E+02, 9.00000000E+00, 9.92500000E-01, 0.00000000E+00},
    {8.82520000E+00, 1.08000000E+02, 1.00000000E+01, 9.92500000E-01, 0.00000000E+00},
    {8.21820000E+01, 1.08000000E+02, 1.10000000E+01, 9.92500000E-01, 0.00000000E+00},
    {6.93675000E+01, 1.08000000E+02, 1.20000000E+01, 9.92500000E-01, 0.00000000E+00},
    {6.67451000E+01, 1.08000000E+02, 1.30000000E+01, 9.92500000E-01, 0.00000000E+00},
    {5.58175000E+01, 1.08000000E+02, 1.40000000E+01, 9.92500000E-01, 0.00000000E+00},
    {4.59937000E+01, 1.08000000E+02, 1.50000000E+01, 9.92500000E-01, 0.00000000E+00},
    {3.97091000E+01, 1.08000000E+02, 1.60000000E+01, 9.92500000E-01, 0.00000000E+00},
    {3.37313000E+01, 1.08000000E+02, 1.70000000E+01, 9.92500000E-01, 0.00000000E+00},
    {2.86000000E+01, 1.08000000E+02, 1.80000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.34650800E+02, 1.08000000E+02, 1.90000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.18241100E+02, 1.08000000E+02, 2.00000000E+01, 9.92500000E-01, 0.00000000E+00},
    {9.92939000E+01, 1.08000000E+02, 2.10000000E+01, 9.92500000E-01, 0.00000000E+00},
    {9.78345000E+01, 1.08000000E+02, 2.20000000E+01, 9.92500000E-01, 0.00000000E+00},
    {9.06007000E+01, 1.08000000E+02, 2.30000000E+01, 9.92500000E-01, 0.00000000E+00},
    {7.22872000E+01, 1.08000000E+02, 2.40000000E+01, 9.92500000E-01, 0.00000000E+00},
    {7.93047000E+01, 1.08000000E+02, 2.50000000E+01, 9.92500000E-01, 0.00000000E+00},
    {6.31865000E+01, 1.08000000E+02, 2.60000000E+01, 9.92500000E-01, 0.00000000E+00},
    {6.77117000E+01, 1.08000000E+02, 2.70000000E+01, 9.92500000E-01, 0.00000000E+00},
    {6.89086000E+01, 1.08000000E+02, 2.80000000E+01, 9.92500000E-01, 0.00000000E+00},
    {5.36420000E+01, 1.08000000E+02, 2.90000000E+01, 9.92500000E-01, 0.00000000E+00},
    {5.62884000E+01, 1.08000000E+02, 3.00000000E+01, 9.92500000E-01, 0.00000000E+00},
    {6.56214000E+01, 1.08000000E+02, 3.10000000E+01, 9.92500000E-01, 0.00000000E+00},
    {6.04041000E+01, 1.08000000E+02, 3.20000000E+01, 9.92500000E-01, 0.00000000E+00},
    {5.38153000E+01, 1.08000000E+02, 3.30000000E+01, 9.92500000E-01, 0.00000000E+00},
    {4.97552000E+01, 1.08000000E+02, 3.40000000E+01, 9.92500000E-01, 0.00000000E+00},
    {4.49510000E+01, 1.08000000E+02, 3.50000000E+01, 9.92500000E-01, 0.00000000E+00},
    {4.03043000E+01, 1.08000000E+02, 3.60000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.52675700E+02, 1.08000000E+02, 3.70000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.41159200E+02, 1.08000000E+02, 3.80000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.27628200E+02, 1.08000000E+02, 3.90000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.17232800E+02, 1.08000000E+02, 4.00000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.08637800E+02, 1.08000000E+02, 4.10000000E+01, 9.92500000E-01, 0.00000000E+00},
    {8.66803000E+01, 1.08000000E+02, 4.20000000E+01, 9.92500000E-01, 0.00000000E+00},
    {9.54966000E+01, 1.08000000E+02, 4.30000000E+01, 9.92500000E-01, 0.00000000E+00},
    {7.53910000E+01, 1.08000000E+02, 4.40000000E+01, 9.92500000E-01, 0.00000000E+00},
    {8.17640000E+01, 1.08000000E+02, 4.50000000E+01, 9.92500000E-01, 0.00000000E+00},
    {7.65871000E+01, 1.08000000E+02, 4.60000000E+01, 9.92500000E-01, 0.00000000E+00},
    {6.43880000E+01, 1.08000000E+02, 4.70000000E+01, 9.92500000E-01, 0.00000000E+00},
    {6.83943000E+01, 1.08000000E+02, 4.80000000E+01, 9.92500000E-01, 0.00000000E+00},
    {8.30278000E+01, 1.08000000E+02, 4.90000000E+01, 9.92500000E-01, 0.00000000E+00},
    {7.91704000E+01, 1.08000000E+02, 5.00000000E+01, 9.92500000E-01, 0.00000000E+00},
    {7.30718000E+01, 1.08000000E+02, 5.10000000E+01, 9.92500000E-01, 0.00000000E+00},
    {6.93901000E+01, 1.08000000E+02, 5.20000000E+01, 9.92500000E-01, 0.00000000E+00},
    {6.44141000E+01, 1.08000000E+02, 5.30000000E+01, 9.92500000E-01, 0.00000000E+00},
    {5.94468000E+01, 1.08000000E+02, 5.40000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.86879100E+02, 1.08000000E+02, 5.50000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.78954900E+02, 1.08000000E+02, 5.60000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.62037100E+02, 1.08000000E+02, 5.70000000E+01, 9.92500000E-01, 0.00000000E+00},
    {8.61665000E+01, 1.08000000E+02, 5.80000000E+01, 9.92500000E-01, 2.79910000E+00},
    {1.60523300E+02, 1.08000000E+02, 5.90000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.54806700E+02, 1.08000000E+02, 6.00000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.51097000E+02, 1.08000000E+02, 6.10000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.47655200E+02, 1.08000000E+02, 6.20000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.44607500E+02, 1.08000000E+02, 6.30000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.18538800E+02, 1.08000000E+02, 6.40000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.27782800E+02, 1.08000000E+02, 6.50000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.24015800E+02, 1.08000000E+02, 6.60000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.31300400E+02, 1.08000000E+02, 6.70000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.28566600E+02, 1.08000000E+02, 6.80000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.26161400E+02, 1.08000000E+02, 6.90000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.24440200E+02, 1.08000000E+02, 7.00000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.07835100E+02, 1.08000000E+02, 7.10000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.09204200E+02, 1.08000000E+02, 7.20000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.01973400E+02, 1.08000000E+02, 7.30000000E+01, 9.92500000E-01, 0.00000000E+00},
    {8.83156000E+01, 1.08000000E+02, 7.40000000E+01, 9.92500000E-01, 0.00000000E+00},
    {9.03918000E+01, 1.08000000E+02, 7.50000000E+01, 9.92500000E-01, 0.00000000E+00},
    {8.36227000E+01, 1.08000000E+02, 7.60000000E+01, 9.92500000E-01, 0.00000000E+00},
    {7.79194000E+01, 1.08000000E+02, 7.70000000E+01, 9.92500000E-01, 0.00000000E+00},
    {6.63591000E+01, 1.08000000E+02, 7.80000000E+01, 9.92500000E-01, 0.00000000E+00},
    {6.26008000E+01, 1.08000000E+02, 7.90000000E+01, 9.92500000E-01, 0.00000000E+00},
    {6.45552000E+01, 1.08000000E+02, 8.00000000E+01, 9.92500000E-01, 0.00000000E+00},
    {8.69552000E+01, 1.08000000E+02, 8.10000000E+01, 9.92500000E-01, 0.00000000E+00},
    {8.67392000E+01, 1.08000000E+02, 8.20000000E+01, 9.92500000E-01, 0.00000000E+00},
    {8.20559000E+01, 1.08000000E+02, 8.30000000E+01, 9.92500000E-01, 0.00000000E+00},
    {7.97133000E+01, 1.08000000E+02, 8.40000000E+01, 9.92500000E-01, 0.00000000E+00},
    {7.53268000E+01, 1.08000000E+02, 8.50000000E+01, 9.92500000E-01, 0.00000000E+00},
    {7.06480000E+01, 1.08000000E+02, 8.60000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.81005500E+02, 1.08000000E+02, 8.70000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.80098800E+02, 1.08000000E+02, 8.80000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.63604200E+02, 1.08000000E+02, 8.90000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.52565000E+02, 1.08000000E+02, 9.00000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.49555100E+02, 1.08000000E+02, 9.10000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.45011300E+02, 1.08000000E+02, 9.20000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.46192800E+02, 1.08000000E+02, 9.30000000E+01, 9.92500000E-01, 0.00000000E+00},
    {1.42077500E+02, 1.08000000E+02, 9.40000000E+01, 9.92500000E-01, 0.00000000E+00},
    {9.18120000E+00, 1.08000000E+02, 1.01000000E+02, 9.92500000E-01, 0.00000000E+00},
    {2.62708000E+01, 1.08000000E+02, 1.03000000E+02, 9.92500000E-01, 9.86500000E-01},
    {3.41983000E+01, 1.08000000E+02, 1.04000000E+02, 9.92500000E-01, 9.80800000E-01},
    {2.82327000E+01, 1.08000000E+02, 1.05000000E+02, 9.92500000E-01, 9.70600000E-01},
    {2.25178000E+01, 1.08000000E+02, 1.06000000E+02, 9.92500000E-01, 9.86800000E-01},
    {1.66775000E+01, 1.08000000E+02, 1.07000000E+02, 9.92500000E-01, 9.94400000E-01},
    {1.28161000E+01, 1.08000000E+02, 1.08000000E+02, 9.92500000E-01, 9.92500000E-01},
    {4.26720000E+00, 1.09000000E+02, 1.00000000E+00, 9.98200000E-01, 9.11800000E-01},
    {3.30770000E+00, 1.09000000E+02, 2.00000000E+00, 9.98200000E-01, 0.00000000E+00},
    {4.20627000E+01, 1.09000000E+02, 3.00000000E+00, 9.98200000E-01, 0.00000000E+00},
    {2.88615000E+01, 1.09000000E+02, 4.00000000E+00, 9.98200000E-01, 0.00000000E+00},
    {2.20931000E+01, 1.09000000E+02, 5.00000000E+00, 9.98200000E-01, 0.00000000E+00},
    {1.66373000E+01, 1.09000000E+02, 6.00000000E+00, 9.98200000E-01, 0.00000000E+00},
    {1.27107000E+01, 1.09000000E+02, 7.00000000E+00, 9.98200000E-01, 0.00000000E+00},
    {1.02867000E+01, 1.09000000E+02, 8.00000000E+00, 9.98200000E-01, 0.00000000E+00},
    {8.27390000E+00, 1.09000000E+02, 9.00000000E+00, 9.98200000E-01, 0.00000000E+00},
    {6.69500000E+00, 1.09000000E+02, 1.00000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.12498000E+01, 1.09000000E+02, 1.10000000E+01, 9.98200000E-01, 0.00000000E+00},
    {4.47141000E+01, 1.09000000E+02, 1.20000000E+01, 9.98200000E-01, 0.00000000E+00},
    {4.37364000E+01, 1.09000000E+02, 1.30000000E+01, 9.98200000E-01, 0.00000000E+00},
    {3.74421000E+01, 1.09000000E+02, 1.40000000E+01, 9.98200000E-01, 0.00000000E+00},
    {3.15577000E+01, 1.09000000E+02, 1.50000000E+01, 9.98200000E-01, 0.00000000E+00},
    {2.77137000E+01, 1.09000000E+02, 1.60000000E+01, 9.98200000E-01, 0.00000000E+00},
    {2.39505000E+01, 1.09000000E+02, 1.70000000E+01, 9.98200000E-01, 0.00000000E+00},
    {2.06394000E+01, 1.09000000E+02, 1.80000000E+01, 9.98200000E-01, 0.00000000E+00},
    {8.43688000E+01, 1.09000000E+02, 1.90000000E+01, 9.98200000E-01, 0.00000000E+00},
    {7.55850000E+01, 1.09000000E+02, 2.00000000E+01, 9.98200000E-01, 0.00000000E+00},
    {6.38564000E+01, 1.09000000E+02, 2.10000000E+01, 9.98200000E-01, 0.00000000E+00},
    {6.34748000E+01, 1.09000000E+02, 2.20000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.90665000E+01, 1.09000000E+02, 2.30000000E+01, 9.98200000E-01, 0.00000000E+00},
    {4.75318000E+01, 1.09000000E+02, 2.40000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.20780000E+01, 1.09000000E+02, 2.50000000E+01, 9.98200000E-01, 0.00000000E+00},
    {4.18942000E+01, 1.09000000E+02, 2.60000000E+01, 9.98200000E-01, 0.00000000E+00},
    {4.49407000E+01, 1.09000000E+02, 2.70000000E+01, 9.98200000E-01, 0.00000000E+00},
    {4.55155000E+01, 1.09000000E+02, 2.80000000E+01, 9.98200000E-01, 0.00000000E+00},
    {3.58100000E+01, 1.09000000E+02, 2.90000000E+01, 9.98200000E-01, 0.00000000E+00},
    {3.77559000E+01, 1.09000000E+02, 3.00000000E+01, 9.98200000E-01, 0.00000000E+00},
    {4.35685000E+01, 1.09000000E+02, 3.10000000E+01, 9.98200000E-01, 0.00000000E+00},
    {4.07500000E+01, 1.09000000E+02, 3.20000000E+01, 9.98200000E-01, 0.00000000E+00},
    {3.69389000E+01, 1.09000000E+02, 3.30000000E+01, 9.98200000E-01, 0.00000000E+00},
    {3.45792000E+01, 1.09000000E+02, 3.40000000E+01, 9.98200000E-01, 0.00000000E+00},
    {3.16699000E+01, 1.09000000E+02, 3.50000000E+01, 9.98200000E-01, 0.00000000E+00},
    {2.87800000E+01, 1.09000000E+02, 3.60000000E+01, 9.98200000E-01, 0.00000000E+00},
    {9.62235000E+01, 1.09000000E+02, 3.70000000E+01, 9.98200000E-01, 0.00000000E+00},
    {9.04081000E+01, 1.09000000E+02, 3.80000000E+01, 9.98200000E-01, 0.00000000E+00},
    {8.27337000E+01, 1.09000000E+02, 3.90000000E+01, 9.98200000E-01, 0.00000000E+00},
    {7.66712000E+01, 1.09000000E+02, 4.00000000E+01, 9.98200000E-01, 0.00000000E+00},
    {7.15493000E+01, 1.09000000E+02, 4.10000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.79784000E+01, 1.09000000E+02, 4.20000000E+01, 9.98200000E-01, 0.00000000E+00},
    {6.35079000E+01, 1.09000000E+02, 4.30000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.09770000E+01, 1.09000000E+02, 4.40000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.50240000E+01, 1.09000000E+02, 4.50000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.17740000E+01, 1.09000000E+02, 4.60000000E+01, 9.98200000E-01, 0.00000000E+00},
    {4.38480000E+01, 1.09000000E+02, 4.70000000E+01, 9.98200000E-01, 0.00000000E+00},
    {4.65340000E+01, 1.09000000E+02, 4.80000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.56589000E+01, 1.09000000E+02, 4.90000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.36243000E+01, 1.09000000E+02, 5.00000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.01473000E+01, 1.09000000E+02, 5.10000000E+01, 9.98200000E-01, 0.00000000E+00},
    {4.80535000E+01, 1.09000000E+02, 5.20000000E+01, 9.98200000E-01, 0.00000000E+00},
    {4.50848000E+01, 1.09000000E+02, 5.30000000E+01, 9.98200000E-01, 0.00000000E+00},
    {4.20634000E+01, 1.09000000E+02, 5.40000000E+01, 9.98200000E-01, 0.00000000E+00},
    {1.17971800E+02, 1.09000000E+02, 5.50000000E+01, 9.98200000E-01, 0.00000000E+00},
    {1.14467600E+02, 1.09000000E+02, 5.60000000E+01, 9.98200000E-01, 0.00000000E+00},
    {1.04809500E+02, 1.09000000E+02, 5.70000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.91379000E+01, 1.09000000E+02, 5.80000000E+01, 9.98200000E-01, 2.79910000E+00},
    {1.03361600E+02, 1.09000000E+02, 5.90000000E+01, 9.98200000E-01, 0.00000000E+00},
    {9.98331000E+01, 1.09000000E+02, 6.00000000E+01, 9.98200000E-01, 0.00000000E+00},
    {9.74789000E+01, 1.09000000E+02, 6.10000000E+01, 9.98200000E-01, 0.00000000E+00},
    {9.52846000E+01, 1.09000000E+02, 6.20000000E+01, 9.98200000E-01, 0.00000000E+00},
    {9.33410000E+01, 1.09000000E+02, 6.30000000E+01, 9.98200000E-01, 0.00000000E+00},
    {7.78742000E+01, 1.09000000E+02, 6.40000000E+01, 9.98200000E-01, 0.00000000E+00},
    {8.27154000E+01, 1.09000000E+02, 6.50000000E+01, 9.98200000E-01, 0.00000000E+00},
    {8.04607000E+01, 1.09000000E+02, 6.60000000E+01, 9.98200000E-01, 0.00000000E+00},
    {8.49478000E+01, 1.09000000E+02, 6.70000000E+01, 9.98200000E-01, 0.00000000E+00},
    {8.31827000E+01, 1.09000000E+02, 6.80000000E+01, 9.98200000E-01, 0.00000000E+00},
    {8.16437000E+01, 1.09000000E+02, 6.90000000E+01, 9.98200000E-01, 0.00000000E+00},
    {8.04575000E+01, 1.09000000E+02, 7.00000000E+01, 9.98200000E-01, 0.00000000E+00},
    {7.05522000E+01, 1.09000000E+02, 7.10000000E+01, 9.98200000E-01, 0.00000000E+00},
    {7.20887000E+01, 1.09000000E+02, 7.20000000E+01, 9.98200000E-01, 0.00000000E+00},
    {6.79590000E+01, 1.09000000E+02, 7.30000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.95758000E+01, 1.09000000E+02, 7.40000000E+01, 9.98200000E-01, 0.00000000E+00},
    {6.10949000E+01, 1.09000000E+02, 7.50000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.70111000E+01, 1.09000000E+02, 7.60000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.35214000E+01, 1.09000000E+02, 7.70000000E+01, 9.98200000E-01, 0.00000000E+00},
    {4.61442000E+01, 1.09000000E+02, 7.80000000E+01, 9.98200000E-01, 0.00000000E+00},
    {4.37468000E+01, 1.09000000E+02, 7.90000000E+01, 9.98200000E-01, 0.00000000E+00},
    {4.51205000E+01, 1.09000000E+02, 8.00000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.89129000E+01, 1.09000000E+02, 8.10000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.91097000E+01, 1.09000000E+02, 8.20000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.65007000E+01, 1.09000000E+02, 8.30000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.52614000E+01, 1.09000000E+02, 8.40000000E+01, 9.98200000E-01, 0.00000000E+00},
    {5.27033000E+01, 1.09000000E+02, 8.50000000E+01, 9.98200000E-01, 0.00000000E+00},
    {4.98927000E+01, 1.09000000E+02, 8.60000000E+01, 9.98200000E-01, 0.00000000E+00},
    {1.15432100E+02, 1.09000000E+02, 8.70000000E+01, 9.98200000E-01, 0.00000000E+00},
    {1.16006200E+02, 1.09000000E+02, 8.80000000E+01, 9.98200000E-01, 0.00000000E+00},
    {1.06530300E+02, 1.09000000E+02, 8.90000000E+01, 9.98200000E-01, 0.00000000E+00},
    {1.00902900E+02, 1.09000000E+02, 9.00000000E+01, 9.98200000E-01, 0.00000000E+00},
    {9.85768000E+01, 1.09000000E+02, 9.10000000E+01, 9.98200000E-01, 0.00000000E+00},
    {9.56625000E+01, 1.09000000E+02, 9.20000000E+01, 9.98200000E-01, 0.00000000E+00},
    {9.57322000E+01, 1.09000000E+02, 9.30000000E+01, 9.98200000E-01, 0.00000000E+00},
    {9.31614000E+01, 1.09000000E+02, 9.40000000E+01, 9.98200000E-01, 0.00000000E+00},
    {6.30000000E+00, 1.09000000E+02, 1.01000000E+02, 9.98200000E-01, 0.00000000E+00},
    {1.71733000E+01, 1.09000000E+02, 1.03000000E+02, 9.98200000E-01, 9.86500000E-01},
    {2.25413000E+01, 1.09000000E+02, 1.04000000E+02, 9.98200000E-01, 9.80800000E-01},
    {1.92029000E+01, 1.09000000E+02, 1.05000000E+02, 9.98200000E-01, 9.70600000E-01},
    {1.56857000E+01, 1.09000000E+02, 1.06000000E+02, 9.98200000E-01, 9.86800000E-01},
    {1.19470000E+01, 1.09000000E+02, 1.07000000E+02, 9.98200000E-01, 9.94400000E-01},
    {9.40480000E+00, 1.09000000E+02, 1.08000000E+02, 9.98200000E-01, 9.92500000E-01},
    {7.13410000E+00, 1.09000000E+02, 1.09000000E+02, 9.98200000E-01, 9.98200000E-01},
    {2.07567000E+01, 1.11000000E+02, 1.00000000E+00, 9.68400000E-01, 9.11800000E-01},
    {1.27287000E+01, 1.11000000E+02, 2.00000000E+00, 9.68400000E-01, 0.00000000E+00},
    {4.25716100E+02, 1.11000000E+02, 3.00000000E+00, 9.68400000E-01, 0.00000000E+00},
    {2.18657200E+02, 1.11000000E+02, 4.00000000E+00, 9.68400000E-01, 0.00000000E+00},
    {1.36953600E+02, 1.11000000E+02, 5.00000000E+00, 9.68400000E-01, 0.00000000E+00},
    {8.73956000E+01, 1.11000000E+02, 6.00000000E+00, 9.68400000E-01, 0.00000000E+00},
    {5.84853000E+01, 1.11000000E+02, 7.00000000E+00, 9.68400000E-01, 0.00000000E+00},
    {4.28911000E+01, 1.11000000E+02, 8.00000000E+00, 9.68400000E-01, 0.00000000E+00},
    {3.16060000E+01, 1.11000000E+02, 9.00000000E+00, 9.68400000E-01, 0.00000000E+00},
    {2.37745000E+01, 1.11000000E+02, 1.00000000E+01, 9.68400000E-01, 0.00000000E+00},
    {5.05436300E+02, 1.11000000E+02, 1.10000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.55918000E+02, 1.11000000E+02, 1.20000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.16458800E+02, 1.11000000E+02, 1.30000000E+01, 9.68400000E-01, 0.00000000E+00},
    {2.37132600E+02, 1.11000000E+02, 1.40000000E+01, 9.68400000E-01, 0.00000000E+00},
    {1.76920500E+02, 1.11000000E+02, 1.50000000E+01, 9.68400000E-01, 0.00000000E+00},
    {1.42543000E+02, 1.11000000E+02, 1.60000000E+01, 9.68400000E-01, 0.00000000E+00},
    {1.13102500E+02, 1.11000000E+02, 1.70000000E+01, 9.68400000E-01, 0.00000000E+00},
    {9.01735000E+01, 1.11000000E+02, 1.80000000E+01, 9.68400000E-01, 0.00000000E+00},
    {8.40829900E+02, 1.11000000E+02, 1.90000000E+01, 9.68400000E-01, 0.00000000E+00},
    {6.48310500E+02, 1.11000000E+02, 2.00000000E+01, 9.68400000E-01, 0.00000000E+00},
    {5.27091700E+02, 1.11000000E+02, 2.10000000E+01, 9.68400000E-01, 0.00000000E+00},
    {5.01239300E+02, 1.11000000E+02, 2.20000000E+01, 9.68400000E-01, 0.00000000E+00},
    {4.54803000E+02, 1.11000000E+02, 2.30000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.57150800E+02, 1.11000000E+02, 2.40000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.86342600E+02, 1.11000000E+02, 2.50000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.01740100E+02, 1.11000000E+02, 2.60000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.14171800E+02, 1.11000000E+02, 2.70000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.26827100E+02, 1.11000000E+02, 2.80000000E+01, 9.68400000E-01, 0.00000000E+00},
    {2.49819700E+02, 1.11000000E+02, 2.90000000E+01, 9.68400000E-01, 0.00000000E+00},
    {2.49392500E+02, 1.11000000E+02, 3.00000000E+01, 9.68400000E-01, 0.00000000E+00},
    {2.98400700E+02, 1.11000000E+02, 3.10000000E+01, 9.68400000E-01, 0.00000000E+00},
    {2.53143300E+02, 1.11000000E+02, 3.20000000E+01, 9.68400000E-01, 0.00000000E+00},
    {2.08180200E+02, 1.11000000E+02, 3.30000000E+01, 9.68400000E-01, 0.00000000E+00},
    {1.82474400E+02, 1.11000000E+02, 3.40000000E+01, 9.68400000E-01, 0.00000000E+00},
    {1.55820200E+02, 1.11000000E+02, 3.50000000E+01, 9.68400000E-01, 0.00000000E+00},
    {1.32434800E+02, 1.11000000E+02, 3.60000000E+01, 9.68400000E-01, 0.00000000E+00},
    {9.36300600E+02, 1.11000000E+02, 3.70000000E+01, 9.68400000E-01, 0.00000000E+00},
    {7.73985400E+02, 1.11000000E+02, 3.80000000E+01, 9.68400000E-01, 0.00000000E+00},
    {6.58796300E+02, 1.11000000E+02, 3.90000000E+01, 9.68400000E-01, 0.00000000E+00},
    {5.81248000E+02, 1.11000000E+02, 4.00000000E+01, 9.68400000E-01, 0.00000000E+00},
    {5.23405800E+02, 1.11000000E+02, 4.10000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.95005800E+02, 1.11000000E+02, 4.20000000E+01, 9.68400000E-01, 0.00000000E+00},
    {4.44526900E+02, 1.11000000E+02, 4.30000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.30227300E+02, 1.11000000E+02, 4.40000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.61428700E+02, 1.11000000E+02, 4.50000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.32439200E+02, 1.11000000E+02, 4.60000000E+01, 9.68400000E-01, 0.00000000E+00},
    {2.77942800E+02, 1.11000000E+02, 4.70000000E+01, 9.68400000E-01, 0.00000000E+00},
    {2.89847400E+02, 1.11000000E+02, 4.80000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.73379400E+02, 1.11000000E+02, 4.90000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.34702300E+02, 1.11000000E+02, 5.00000000E+01, 9.68400000E-01, 0.00000000E+00},
    {2.88988800E+02, 1.11000000E+02, 5.10000000E+01, 9.68400000E-01, 0.00000000E+00},
    {2.62884600E+02, 1.11000000E+02, 5.20000000E+01, 9.68400000E-01, 0.00000000E+00},
    {2.32605300E+02, 1.11000000E+02, 5.30000000E+01, 9.68400000E-01, 0.00000000E+00},
    {2.04798600E+02, 1.11000000E+02, 5.40000000E+01, 9.68400000E-01, 0.00000000E+00},
    {1.13892560E+03, 1.11000000E+02, 5.50000000E+01, 9.68400000E-01, 0.00000000E+00},
    {9.95366000E+02, 1.11000000E+02, 5.60000000E+01, 9.68400000E-01, 0.00000000E+00},
    {8.50076800E+02, 1.11000000E+02, 5.70000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.43138200E+02, 1.11000000E+02, 5.80000000E+01, 9.68400000E-01, 2.79910000E+00},
    {8.73513400E+02, 1.11000000E+02, 5.90000000E+01, 9.68400000E-01, 0.00000000E+00},
    {8.34926300E+02, 1.11000000E+02, 6.00000000E+01, 9.68400000E-01, 0.00000000E+00},
    {8.12935000E+02, 1.11000000E+02, 6.10000000E+01, 9.68400000E-01, 0.00000000E+00},
    {7.92833600E+02, 1.11000000E+02, 6.20000000E+01, 9.68400000E-01, 0.00000000E+00},
    {7.74971900E+02, 1.11000000E+02, 6.30000000E+01, 9.68400000E-01, 0.00000000E+00},
    {5.89682100E+02, 1.11000000E+02, 6.40000000E+01, 9.68400000E-01, 0.00000000E+00},
    {6.95021100E+02, 1.11000000E+02, 6.50000000E+01, 9.68400000E-01, 0.00000000E+00},
    {6.66651500E+02, 1.11000000E+02, 6.60000000E+01, 9.68400000E-01, 0.00000000E+00},
    {6.94074200E+02, 1.11000000E+02, 6.70000000E+01, 9.68400000E-01, 0.00000000E+00},
    {6.78848600E+02, 1.11000000E+02, 6.80000000E+01, 9.68400000E-01, 0.00000000E+00},
    {6.64858000E+02, 1.11000000E+02, 6.90000000E+01, 9.68400000E-01, 0.00000000E+00},
    {6.57992200E+02, 1.11000000E+02, 7.00000000E+01, 9.68400000E-01, 0.00000000E+00},
    {5.42087200E+02, 1.11000000E+02, 7.10000000E+01, 9.68400000E-01, 0.00000000E+00},
    {5.17429500E+02, 1.11000000E+02, 7.20000000E+01, 9.68400000E-01, 0.00000000E+00},
    {4.63272400E+02, 1.11000000E+02, 7.30000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.84909100E+02, 1.11000000E+02, 7.40000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.88797100E+02, 1.11000000E+02, 7.50000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.46399700E+02, 1.11000000E+02, 7.60000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.12813200E+02, 1.11000000E+02, 7.70000000E+01, 9.68400000E-01, 0.00000000E+00},
    {2.55728900E+02, 1.11000000E+02, 7.80000000E+01, 9.68400000E-01, 0.00000000E+00},
    {2.37404400E+02, 1.11000000E+02, 7.90000000E+01, 9.68400000E-01, 0.00000000E+00},
    {2.42791900E+02, 1.11000000E+02, 8.00000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.79720900E+02, 1.11000000E+02, 8.10000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.62564900E+02, 1.11000000E+02, 8.20000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.23935000E+02, 1.11000000E+02, 8.30000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.03830300E+02, 1.11000000E+02, 8.40000000E+01, 9.68400000E-01, 0.00000000E+00},
    {2.74731500E+02, 1.11000000E+02, 8.50000000E+01, 9.68400000E-01, 0.00000000E+00},
    {2.47063500E+02, 1.11000000E+02, 8.60000000E+01, 9.68400000E-01, 0.00000000E+00},
    {1.05213180E+03, 1.11000000E+02, 8.70000000E+01, 9.68400000E-01, 0.00000000E+00},
    {9.69742100E+02, 1.11000000E+02, 8.80000000E+01, 9.68400000E-01, 0.00000000E+00},
    {8.34865000E+02, 1.11000000E+02, 8.90000000E+01, 9.68400000E-01, 0.00000000E+00},
    {7.26938700E+02, 1.11000000E+02, 9.00000000E+01, 9.68400000E-01, 0.00000000E+00},
    {7.32655400E+02, 1.11000000E+02, 9.10000000E+01, 9.68400000E-01, 0.00000000E+00},
    {7.08793600E+02, 1.11000000E+02, 9.20000000E+01, 9.68400000E-01, 0.00000000E+00},
    {7.43919000E+02, 1.11000000E+02, 9.30000000E+01, 9.68400000E-01, 0.00000000E+00},
    {7.17910000E+02, 1.11000000E+02, 9.40000000E+01, 9.68400000E-01, 0.00000000E+00},
    {3.49952000E+01, 1.11000000E+02, 1.01000000E+02, 9.68400000E-01, 0.00000000E+00},
    {1.25824500E+02, 1.11000000E+02, 1.03000000E+02, 9.68400000E-01, 9.86500000E-01},
    {1.58326800E+02, 1.11000000E+02, 1.04000000E+02, 9.68400000E-01, 9.80800000E-01},
    {1.13401600E+02, 1.11000000E+02, 1.05000000E+02, 9.68400000E-01, 9.70600000E-01},
    {8.20144000E+01, 1.11000000E+02, 1.06000000E+02, 9.68400000E-01, 9.86800000E-01},
    {5.44454000E+01, 1.11000000E+02, 1.07000000E+02, 9.68400000E-01, 9.94400000E-01},
    {3.81227000E+01, 1.11000000E+02, 1.08000000E+02, 9.68400000E-01, 9.92500000E-01},
    {2.49560000E+01, 1.11000000E+02, 1.09000000E+02, 9.68400000E-01, 9.98200000E-01},
    {1.86105200E+02, 1.11000000E+02, 1.11000000E+02, 9.68400000E-01, 9.68400000E-01},
    {3.19913000E+01, 1.12000000E+02, 1.00000000E+00, 9.62800000E-01, 9.11800000E-01},
    {1.94648000E+01, 1.12000000E+02, 2.00000000E+00, 9.62800000E-01, 0.00000000E+00},
    {6.70033000E+02, 1.12000000E+02, 3.00000000E+00, 9.62800000E-01, 0.00000000E+00},
    {3.40260200E+02, 1.12000000E+02, 4.00000000E+00, 9.62800000E-01, 0.00000000E+00},
    {2.12218000E+02, 1.12000000E+02, 5.00000000E+00, 9.62800000E-01, 0.00000000E+00},
    {1.34922700E+02, 1.12000000E+02, 6.00000000E+00, 9.62800000E-01, 0.00000000E+00},
    {8.99646000E+01, 1.12000000E+02, 7.00000000E+00, 9.62800000E-01, 0.00000000E+00},
    {6.57609000E+01, 1.12000000E+02, 8.00000000E+00, 9.62800000E-01, 0.00000000E+00},
    {4.82865000E+01, 1.12000000E+02, 9.00000000E+00, 9.62800000E-01, 0.00000000E+00},
    {3.61911000E+01, 1.12000000E+02, 1.00000000E+01, 9.62800000E-01, 0.00000000E+00},
    {7.94759700E+02, 1.12000000E+02, 1.10000000E+01, 9.62800000E-01, 0.00000000E+00},
    {5.54596700E+02, 1.12000000E+02, 1.20000000E+01, 9.62800000E-01, 0.00000000E+00},
    {4.92082900E+02, 1.12000000E+02, 1.30000000E+01, 9.62800000E-01, 0.00000000E+00},
    {3.67581100E+02, 1.12000000E+02, 1.40000000E+01, 9.62800000E-01, 0.00000000E+00},
    {2.73526800E+02, 1.12000000E+02, 1.50000000E+01, 9.62800000E-01, 0.00000000E+00},
    {2.19948900E+02, 1.12000000E+02, 1.60000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.74139900E+02, 1.12000000E+02, 1.70000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.38519000E+02, 1.12000000E+02, 1.80000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.32731760E+03, 1.12000000E+02, 1.90000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.01376950E+03, 1.12000000E+02, 2.00000000E+01, 9.62800000E-01, 0.00000000E+00},
    {8.22895700E+02, 1.12000000E+02, 2.10000000E+01, 9.62800000E-01, 0.00000000E+00},
    {7.81699300E+02, 1.12000000E+02, 2.20000000E+01, 9.62800000E-01, 0.00000000E+00},
    {7.08768400E+02, 1.12000000E+02, 2.30000000E+01, 9.62800000E-01, 0.00000000E+00},
    {5.56612700E+02, 1.12000000E+02, 2.40000000E+01, 9.62800000E-01, 0.00000000E+00},
    {6.01449900E+02, 1.12000000E+02, 2.50000000E+01, 9.62800000E-01, 0.00000000E+00},
    {4.69643700E+02, 1.12000000E+02, 2.60000000E+01, 9.62800000E-01, 0.00000000E+00},
    {4.88171500E+02, 1.12000000E+02, 2.70000000E+01, 9.62800000E-01, 0.00000000E+00},
    {5.08158500E+02, 1.12000000E+02, 2.80000000E+01, 9.62800000E-01, 0.00000000E+00},
    {3.88420900E+02, 1.12000000E+02, 2.90000000E+01, 9.62800000E-01, 0.00000000E+00},
    {3.86888400E+02, 1.12000000E+02, 3.00000000E+01, 9.62800000E-01, 0.00000000E+00},
    {4.63569100E+02, 1.12000000E+02, 3.10000000E+01, 9.62800000E-01, 0.00000000E+00},
    {3.92253300E+02, 1.12000000E+02, 3.20000000E+01, 9.62800000E-01, 0.00000000E+00},
    {3.21873200E+02, 1.12000000E+02, 3.30000000E+01, 9.62800000E-01, 0.00000000E+00},
    {2.81717300E+02, 1.12000000E+02, 3.40000000E+01, 9.62800000E-01, 0.00000000E+00},
    {2.40156500E+02, 1.12000000E+02, 3.50000000E+01, 9.62800000E-01, 0.00000000E+00},
    {2.03747300E+02, 1.12000000E+02, 3.60000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.47765560E+03, 1.12000000E+02, 3.70000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.21095930E+03, 1.12000000E+02, 3.80000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.02813650E+03, 1.12000000E+02, 3.90000000E+01, 9.62800000E-01, 0.00000000E+00},
    {9.05775000E+02, 1.12000000E+02, 4.00000000E+01, 9.62800000E-01, 0.00000000E+00},
    {8.14928900E+02, 1.12000000E+02, 4.10000000E+01, 9.62800000E-01, 0.00000000E+00},
    {6.14026500E+02, 1.12000000E+02, 4.20000000E+01, 9.62800000E-01, 0.00000000E+00},
    {6.91449700E+02, 1.12000000E+02, 4.30000000E+01, 9.62800000E-01, 0.00000000E+00},
    {5.12675900E+02, 1.12000000E+02, 4.40000000E+01, 9.62800000E-01, 0.00000000E+00},
    {5.61001000E+02, 1.12000000E+02, 4.50000000E+01, 9.62800000E-01, 0.00000000E+00},
    {5.15645700E+02, 1.12000000E+02, 4.60000000E+01, 9.62800000E-01, 0.00000000E+00},
    {4.31280700E+02, 1.12000000E+02, 4.70000000E+01, 9.62800000E-01, 0.00000000E+00},
    {4.49156700E+02, 1.12000000E+02, 4.80000000E+01, 9.62800000E-01, 0.00000000E+00},
    {5.79923400E+02, 1.12000000E+02, 4.90000000E+01, 9.62800000E-01, 0.00000000E+00},
    {5.18651800E+02, 1.12000000E+02, 5.00000000E+01, 9.62800000E-01, 0.00000000E+00},
    {4.46923000E+02, 1.12000000E+02, 5.10000000E+01, 9.62800000E-01, 0.00000000E+00},
    {4.06072800E+02, 1.12000000E+02, 5.20000000E+01, 9.62800000E-01, 0.00000000E+00},
    {3.58805200E+02, 1.12000000E+02, 5.30000000E+01, 9.62800000E-01, 0.00000000E+00},
    {3.15458100E+02, 1.12000000E+02, 5.40000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.79901800E+03, 1.12000000E+02, 5.50000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.55950500E+03, 1.12000000E+02, 5.60000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.32825100E+03, 1.12000000E+02, 5.70000000E+01, 9.62800000E-01, 0.00000000E+00},
    {5.30865000E+02, 1.12000000E+02, 5.80000000E+01, 9.62800000E-01, 2.79910000E+00},
    {1.36727140E+03, 1.12000000E+02, 5.90000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.30590000E+03, 1.12000000E+02, 6.00000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.27128830E+03, 1.12000000E+02, 6.10000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.23967490E+03, 1.12000000E+02, 6.20000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.21158250E+03, 1.12000000E+02, 6.30000000E+01, 9.62800000E-01, 0.00000000E+00},
    {9.19632900E+02, 1.12000000E+02, 6.40000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.08982140E+03, 1.12000000E+02, 6.50000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.04519610E+03, 1.12000000E+02, 6.60000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.08423130E+03, 1.12000000E+02, 6.70000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.06033860E+03, 1.12000000E+02, 6.80000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.03836010E+03, 1.12000000E+02, 6.90000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.02773560E+03, 1.12000000E+02, 7.00000000E+01, 9.62800000E-01, 0.00000000E+00},
    {8.45538600E+02, 1.12000000E+02, 7.10000000E+01, 9.62800000E-01, 0.00000000E+00},
    {8.05001400E+02, 1.12000000E+02, 7.20000000E+01, 9.62800000E-01, 0.00000000E+00},
    {7.19715200E+02, 1.12000000E+02, 7.30000000E+01, 9.62800000E-01, 0.00000000E+00},
    {5.97434400E+02, 1.12000000E+02, 7.40000000E+01, 9.62800000E-01, 0.00000000E+00},
    {6.03090700E+02, 1.12000000E+02, 7.50000000E+01, 9.62800000E-01, 0.00000000E+00},
    {5.36622100E+02, 1.12000000E+02, 7.60000000E+01, 9.62800000E-01, 0.00000000E+00},
    {4.84071600E+02, 1.12000000E+02, 7.70000000E+01, 9.62800000E-01, 0.00000000E+00},
    {3.95240100E+02, 1.12000000E+02, 7.80000000E+01, 9.62800000E-01, 0.00000000E+00},
    {3.66701700E+02, 1.12000000E+02, 7.90000000E+01, 9.62800000E-01, 0.00000000E+00},
    {3.74855200E+02, 1.12000000E+02, 8.00000000E+01, 9.62800000E-01, 0.00000000E+00},
    {5.89539500E+02, 1.12000000E+02, 8.10000000E+01, 9.62800000E-01, 0.00000000E+00},
    {5.61729800E+02, 1.12000000E+02, 8.20000000E+01, 9.62800000E-01, 0.00000000E+00},
    {5.00922300E+02, 1.12000000E+02, 8.30000000E+01, 9.62800000E-01, 0.00000000E+00},
    {4.69352300E+02, 1.12000000E+02, 8.40000000E+01, 9.62800000E-01, 0.00000000E+00},
    {4.23850000E+02, 1.12000000E+02, 8.50000000E+01, 9.62800000E-01, 0.00000000E+00},
    {3.80679300E+02, 1.12000000E+02, 8.60000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.65750700E+03, 1.12000000E+02, 8.70000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.51734570E+03, 1.12000000E+02, 8.80000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.30283760E+03, 1.12000000E+02, 8.90000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.13138980E+03, 1.12000000E+02, 9.00000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.14186680E+03, 1.12000000E+02, 9.10000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.10450830E+03, 1.12000000E+02, 9.20000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.16081350E+03, 1.12000000E+02, 9.30000000E+01, 9.62800000E-01, 0.00000000E+00},
    {1.11984660E+03, 1.12000000E+02, 9.40000000E+01, 9.62800000E-01, 0.00000000E+00},
    {5.41021000E+01, 1.12000000E+02, 1.01000000E+02, 9.62800000E-01, 0.00000000E+00},
    {1.95705700E+02, 1.12000000E+02, 1.03000000E+02, 9.62800000E-01, 9.86500000E-01},
    {2.46222700E+02, 1.12000000E+02, 1.04000000E+02, 9.62800000E-01, 9.80800000E-01},
    {1.75564600E+02, 1.12000000E+02, 1.05000000E+02, 9.62800000E-01, 9.70600000E-01},
    {1.26665700E+02, 1.12000000E+02, 1.06000000E+02, 9.62800000E-01, 9.86800000E-01},
    {8.37703000E+01, 1.12000000E+02, 1.07000000E+02, 9.62800000E-01, 9.94400000E-01},
    {5.84303000E+01, 1.12000000E+02, 1.08000000E+02, 9.62800000E-01, 9.92500000E-01},
    {3.79884000E+01, 1.12000000E+02, 1.09000000E+02, 9.62800000E-01, 9.98200000E-01},
    {2.89608200E+02, 1.12000000E+02, 1.11000000E+02, 9.62800000E-01, 9.68400000E-01},
    {4.51270800E+02, 1.12000000E+02, 1.12000000E+02, 9.62800000E-01, 9.62800000E-01},
    {3.31048000E+01, 1.13000000E+02, 1.00000000E+00, 9.64800000E-01, 9.11800000E-01},
    {2.05158000E+01, 1.13000000E+02, 2.00000000E+00, 9.64800000E-01, 0.00000000E+00},
    {6.27530900E+02, 1.13000000E+02, 3.00000000E+00, 9.64800000E-01, 0.00000000E+00},
    {3.34733500E+02, 1.13000000E+02, 4.00000000E+00, 9.64800000E-01, 0.00000000E+00},
    {2.13884800E+02, 1.13000000E+02, 5.00000000E+00, 9.64800000E-01, 0.00000000E+00},
    {1.38325400E+02, 1.13000000E+02, 6.00000000E+00, 9.64800000E-01, 0.00000000E+00},
    {9.33542000E+01, 1.13000000E+02, 7.00000000E+00, 9.64800000E-01, 0.00000000E+00},
    {6.87983000E+01, 1.13000000E+02, 8.00000000E+00, 9.64800000E-01, 0.00000000E+00},
    {5.08577000E+01, 1.13000000E+02, 9.00000000E+00, 9.64800000E-01, 0.00000000E+00},
    {3.83154000E+01, 1.13000000E+02, 1.00000000E+01, 9.64800000E-01, 0.00000000E+00},
    {7.46341900E+02, 1.13000000E+02, 1.10000000E+01, 9.64800000E-01, 0.00000000E+00},
    {5.41028500E+02, 1.13000000E+02, 1.20000000E+01, 9.64800000E-01, 0.00000000E+00},
    {4.86236500E+02, 1.13000000E+02, 1.30000000E+01, 9.64800000E-01, 0.00000000E+00},
    {3.69623200E+02, 1.13000000E+02, 1.40000000E+01, 9.64800000E-01, 0.00000000E+00},
    {2.78927200E+02, 1.13000000E+02, 1.50000000E+01, 9.64800000E-01, 0.00000000E+00},
    {2.26215600E+02, 1.13000000E+02, 1.60000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.80543000E+02, 1.13000000E+02, 1.70000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.44579800E+02, 1.13000000E+02, 1.80000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.23322370E+03, 1.13000000E+02, 1.90000000E+01, 9.64800000E-01, 0.00000000E+00},
    {9.73049300E+02, 1.13000000E+02, 2.00000000E+01, 9.64800000E-01, 0.00000000E+00},
    {7.95220600E+02, 1.13000000E+02, 2.10000000E+01, 9.64800000E-01, 0.00000000E+00},
    {7.59453200E+02, 1.13000000E+02, 2.20000000E+01, 9.64800000E-01, 0.00000000E+00},
    {6.90915600E+02, 1.13000000E+02, 2.30000000E+01, 9.64800000E-01, 0.00000000E+00},
    {5.42245900E+02, 1.13000000E+02, 2.40000000E+01, 9.64800000E-01, 0.00000000E+00},
    {5.89147500E+02, 1.13000000E+02, 2.50000000E+01, 9.64800000E-01, 0.00000000E+00},
    {4.60049600E+02, 1.13000000E+02, 2.60000000E+01, 9.64800000E-01, 0.00000000E+00},
    {4.82299000E+02, 1.13000000E+02, 2.70000000E+01, 9.64800000E-01, 0.00000000E+00},
    {5.00352500E+02, 1.13000000E+02, 2.80000000E+01, 9.64800000E-01, 0.00000000E+00},
    {3.81992300E+02, 1.13000000E+02, 2.90000000E+01, 9.64800000E-01, 0.00000000E+00},
    {3.85144900E+02, 1.13000000E+02, 3.00000000E+01, 9.64800000E-01, 0.00000000E+00},
    {4.59954400E+02, 1.13000000E+02, 3.10000000E+01, 9.64800000E-01, 0.00000000E+00},
    {3.94665000E+02, 1.13000000E+02, 3.20000000E+01, 9.64800000E-01, 0.00000000E+00},
    {3.27722900E+02, 1.13000000E+02, 3.30000000E+01, 9.64800000E-01, 0.00000000E+00},
    {2.88848500E+02, 1.13000000E+02, 3.40000000E+01, 9.64800000E-01, 0.00000000E+00},
    {2.47969200E+02, 1.13000000E+02, 3.50000000E+01, 9.64800000E-01, 0.00000000E+00},
    {2.11694900E+02, 1.13000000E+02, 3.60000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.37525580E+03, 1.13000000E+02, 3.70000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.16025390E+03, 1.13000000E+02, 3.80000000E+01, 9.64800000E-01, 0.00000000E+00},
    {9.96375700E+02, 1.13000000E+02, 3.90000000E+01, 9.64800000E-01, 0.00000000E+00},
    {8.83854100E+02, 1.13000000E+02, 4.00000000E+01, 9.64800000E-01, 0.00000000E+00},
    {7.98609400E+02, 1.13000000E+02, 4.10000000E+01, 9.64800000E-01, 0.00000000E+00},
    {6.05913500E+02, 1.13000000E+02, 4.20000000E+01, 9.64800000E-01, 0.00000000E+00},
    {6.80542700E+02, 1.13000000E+02, 4.30000000E+01, 9.64800000E-01, 0.00000000E+00},
    {5.08528900E+02, 1.13000000E+02, 4.40000000E+01, 9.64800000E-01, 0.00000000E+00},
    {5.57004300E+02, 1.13000000E+02, 4.50000000E+01, 9.64800000E-01, 0.00000000E+00},
    {5.13416300E+02, 1.13000000E+02, 4.60000000E+01, 9.64800000E-01, 0.00000000E+00},
    {4.27996200E+02, 1.13000000E+02, 4.70000000E+01, 9.64800000E-01, 0.00000000E+00},
    {4.48807100E+02, 1.13000000E+02, 4.80000000E+01, 9.64800000E-01, 0.00000000E+00},
    {5.74342000E+02, 1.13000000E+02, 4.90000000E+01, 9.64800000E-01, 0.00000000E+00},
    {5.19842400E+02, 1.13000000E+02, 5.00000000E+01, 9.64800000E-01, 0.00000000E+00},
    {4.52849400E+02, 1.13000000E+02, 5.10000000E+01, 9.64800000E-01, 0.00000000E+00},
    {4.14054400E+02, 1.13000000E+02, 5.20000000E+01, 9.64800000E-01, 0.00000000E+00},
    {3.68279200E+02, 1.13000000E+02, 5.30000000E+01, 9.64800000E-01, 0.00000000E+00},
    {3.25762500E+02, 1.13000000E+02, 5.40000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.67319220E+03, 1.13000000E+02, 5.50000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.48729640E+03, 1.13000000E+02, 5.60000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.28152290E+03, 1.13000000E+02, 5.70000000E+01, 9.64800000E-01, 0.00000000E+00},
    {5.36562600E+02, 1.13000000E+02, 5.80000000E+01, 9.64800000E-01, 2.79910000E+00},
    {1.30860410E+03, 1.13000000E+02, 5.90000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.25269280E+03, 1.13000000E+02, 6.00000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.22022460E+03, 1.13000000E+02, 6.10000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.19050030E+03, 1.13000000E+02, 6.20000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.16410890E+03, 1.13000000E+02, 6.30000000E+01, 9.64800000E-01, 0.00000000E+00},
    {8.93951500E+02, 1.13000000E+02, 6.40000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.03810470E+03, 1.13000000E+02, 6.50000000E+01, 9.64800000E-01, 0.00000000E+00},
    {9.97383700E+02, 1.13000000E+02, 6.60000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.04500380E+03, 1.13000000E+02, 6.70000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.02236740E+03, 1.13000000E+02, 6.80000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.00167840E+03, 1.13000000E+02, 6.90000000E+01, 9.64800000E-01, 0.00000000E+00},
    {9.90980800E+02, 1.13000000E+02, 7.00000000E+01, 9.64800000E-01, 0.00000000E+00},
    {8.21600500E+02, 1.13000000E+02, 7.10000000E+01, 9.64800000E-01, 0.00000000E+00},
    {7.91845600E+02, 1.13000000E+02, 7.20000000E+01, 9.64800000E-01, 0.00000000E+00},
    {7.12742200E+02, 1.13000000E+02, 7.30000000E+01, 9.64800000E-01, 0.00000000E+00},
    {5.94161600E+02, 1.13000000E+02, 7.40000000E+01, 9.64800000E-01, 0.00000000E+00},
    {6.01578000E+02, 1.13000000E+02, 7.50000000E+01, 9.64800000E-01, 0.00000000E+00},
    {5.38385500E+02, 1.13000000E+02, 7.60000000E+01, 9.64800000E-01, 0.00000000E+00},
    {4.87898400E+02, 1.13000000E+02, 7.70000000E+01, 9.64800000E-01, 0.00000000E+00},
    {3.99976800E+02, 1.13000000E+02, 7.80000000E+01, 9.64800000E-01, 0.00000000E+00},
    {3.71721000E+02, 1.13000000E+02, 7.90000000E+01, 9.64800000E-01, 0.00000000E+00},
    {3.81059000E+02, 1.13000000E+02, 8.00000000E+01, 9.64800000E-01, 0.00000000E+00},
    {5.84697200E+02, 1.13000000E+02, 8.10000000E+01, 9.64800000E-01, 0.00000000E+00},
    {5.62687500E+02, 1.13000000E+02, 8.20000000E+01, 9.64800000E-01, 0.00000000E+00},
    {5.06866300E+02, 1.13000000E+02, 8.30000000E+01, 9.64800000E-01, 0.00000000E+00},
    {4.77586200E+02, 1.13000000E+02, 8.40000000E+01, 9.64800000E-01, 0.00000000E+00},
    {4.34096900E+02, 1.13000000E+02, 8.50000000E+01, 9.64800000E-01, 0.00000000E+00},
    {3.92116900E+02, 1.13000000E+02, 8.60000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.55588480E+03, 1.13000000E+02, 8.70000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.45542410E+03, 1.13000000E+02, 8.80000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.26337240E+03, 1.13000000E+02, 8.90000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.11006280E+03, 1.13000000E+02, 9.00000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.11307380E+03, 1.13000000E+02, 9.10000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.07700050E+03, 1.13000000E+02, 9.20000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.12380080E+03, 1.13000000E+02, 9.30000000E+01, 9.64800000E-01, 0.00000000E+00},
    {1.08568460E+03, 1.13000000E+02, 9.40000000E+01, 9.64800000E-01, 0.00000000E+00},
    {5.52636000E+01, 1.13000000E+02, 1.01000000E+02, 9.64800000E-01, 0.00000000E+00},
    {1.93070700E+02, 1.13000000E+02, 1.03000000E+02, 9.64800000E-01, 9.86500000E-01},
    {2.43761200E+02, 1.13000000E+02, 1.04000000E+02, 9.64800000E-01, 9.80800000E-01},
    {1.77725400E+02, 1.13000000E+02, 1.05000000E+02, 9.64800000E-01, 9.70600000E-01},
    {1.29692200E+02, 1.13000000E+02, 1.06000000E+02, 9.64800000E-01, 9.86800000E-01},
    {8.68728000E+01, 1.13000000E+02, 1.07000000E+02, 9.64800000E-01, 9.94400000E-01},
    {6.12175000E+01, 1.13000000E+02, 1.08000000E+02, 9.64800000E-01, 9.92500000E-01},
    {4.03220000E+01, 1.13000000E+02, 1.09000000E+02, 9.64800000E-01, 9.98200000E-01},
    {2.84225100E+02, 1.13000000E+02, 1.11000000E+02, 9.64800000E-01, 9.68400000E-01},
    {4.41732200E+02, 1.13000000E+02, 1.12000000E+02, 9.64800000E-01, 9.62800000E-01},
    {4.37757500E+02, 1.13000000E+02, 1.13000000E+02, 9.64800000E-01, 9.64800000E-01},
    {2.74989000E+01, 1.14000000E+02, 1.00000000E+00, 9.50700000E-01, 9.11800000E-01},
    {1.76356000E+01, 1.14000000E+02, 2.00000000E+00, 9.50700000E-01, 0.00000000E+00},
    {4.50958300E+02, 1.14000000E+02, 3.00000000E+00, 9.50700000E-01, 0.00000000E+00},
    {2.56588100E+02, 1.14000000E+02, 4.00000000E+00, 9.50700000E-01, 0.00000000E+00},
    {1.70189100E+02, 1.14000000E+02, 5.00000000E+00, 9.50700000E-01, 0.00000000E+00},
    {1.13204600E+02, 1.14000000E+02, 6.00000000E+00, 9.50700000E-01, 0.00000000E+00},
    {7.80190000E+01, 1.14000000E+02, 7.00000000E+00, 9.50700000E-01, 0.00000000E+00},
    {5.83557000E+01, 1.14000000E+02, 8.00000000E+00, 9.50700000E-01, 0.00000000E+00},
    {4.36855000E+01, 1.14000000E+02, 9.00000000E+00, 9.50700000E-01, 0.00000000E+00},
    {3.32416000E+01, 1.14000000E+02, 1.00000000E+01, 9.50700000E-01, 0.00000000E+00},
    {5.38341200E+02, 1.14000000E+02, 1.10000000E+01, 9.50700000E-01, 0.00000000E+00},
    {4.09917400E+02, 1.14000000E+02, 1.20000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.75523300E+02, 1.14000000E+02, 1.30000000E+01, 9.50700000E-01, 0.00000000E+00},
    {2.92999400E+02, 1.14000000E+02, 1.40000000E+01, 9.50700000E-01, 0.00000000E+00},
    {2.26065700E+02, 1.14000000E+02, 1.50000000E+01, 9.50700000E-01, 0.00000000E+00},
    {1.86005100E+02, 1.14000000E+02, 1.60000000E+01, 9.50700000E-01, 0.00000000E+00},
    {1.50523100E+02, 1.14000000E+02, 1.70000000E+01, 9.50700000E-01, 0.00000000E+00},
    {1.22001200E+02, 1.14000000E+02, 1.80000000E+01, 9.50700000E-01, 0.00000000E+00},
    {8.81262400E+02, 1.14000000E+02, 1.90000000E+01, 9.50700000E-01, 0.00000000E+00},
    {7.22511100E+02, 1.14000000E+02, 2.00000000E+01, 9.50700000E-01, 0.00000000E+00},
    {5.95672500E+02, 1.14000000E+02, 2.10000000E+01, 9.50700000E-01, 0.00000000E+00},
    {5.73498800E+02, 1.14000000E+02, 2.20000000E+01, 9.50700000E-01, 0.00000000E+00},
    {5.24280800E+02, 1.14000000E+02, 2.30000000E+01, 9.50700000E-01, 0.00000000E+00},
    {4.11887900E+02, 1.14000000E+02, 2.40000000E+01, 9.50700000E-01, 0.00000000E+00},
    {4.50223400E+02, 1.14000000E+02, 2.50000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.52240800E+02, 1.14000000E+02, 2.60000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.73012500E+02, 1.14000000E+02, 2.70000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.85004800E+02, 1.14000000E+02, 2.80000000E+01, 9.50700000E-01, 0.00000000E+00},
    {2.94140600E+02, 1.14000000E+02, 2.90000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.01170900E+02, 1.14000000E+02, 3.00000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.57941100E+02, 1.14000000E+02, 3.10000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.13391100E+02, 1.14000000E+02, 3.20000000E+01, 9.50700000E-01, 0.00000000E+00},
    {2.65071100E+02, 1.14000000E+02, 3.30000000E+01, 9.50700000E-01, 0.00000000E+00},
    {2.36332300E+02, 1.14000000E+02, 3.40000000E+01, 9.50700000E-01, 0.00000000E+00},
    {2.05297600E+02, 1.14000000E+02, 3.50000000E+01, 9.50700000E-01, 0.00000000E+00},
    {1.77181100E+02, 1.14000000E+02, 3.60000000E+01, 9.50700000E-01, 0.00000000E+00},
    {9.86043800E+02, 1.14000000E+02, 3.70000000E+01, 9.50700000E-01, 0.00000000E+00},
    {8.60342100E+02, 1.14000000E+02, 3.80000000E+01, 9.50700000E-01, 0.00000000E+00},
    {7.50381700E+02, 1.14000000E+02, 3.90000000E+01, 9.50700000E-01, 0.00000000E+00},
    {6.72183400E+02, 1.14000000E+02, 4.00000000E+01, 9.50700000E-01, 0.00000000E+00},
    {6.11314700E+02, 1.14000000E+02, 4.10000000E+01, 9.50700000E-01, 0.00000000E+00},
    {4.69109200E+02, 1.14000000E+02, 4.20000000E+01, 9.50700000E-01, 0.00000000E+00},
    {5.24647900E+02, 1.14000000E+02, 4.30000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.96986200E+02, 1.14000000E+02, 4.40000000E+01, 9.50700000E-01, 0.00000000E+00},
    {4.34725100E+02, 1.14000000E+02, 4.50000000E+01, 9.50700000E-01, 0.00000000E+00},
    {4.02366200E+02, 1.14000000E+02, 4.60000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.34667300E+02, 1.14000000E+02, 4.70000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.53597600E+02, 1.14000000E+02, 4.80000000E+01, 9.50700000E-01, 0.00000000E+00},
    {4.46593700E+02, 1.14000000E+02, 4.90000000E+01, 9.50700000E-01, 0.00000000E+00},
    {4.10896000E+02, 1.14000000E+02, 5.00000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.63784600E+02, 1.14000000E+02, 5.10000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.35924000E+02, 1.14000000E+02, 5.20000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.01985200E+02, 1.14000000E+02, 5.30000000E+01, 9.50700000E-01, 0.00000000E+00},
    {2.69837200E+02, 1.14000000E+02, 5.40000000E+01, 9.50700000E-01, 0.00000000E+00},
    {1.20058030E+03, 1.14000000E+02, 5.50000000E+01, 9.50700000E-01, 0.00000000E+00},
    {1.09739800E+03, 1.14000000E+02, 5.60000000E+01, 9.50700000E-01, 0.00000000E+00},
    {9.60233200E+02, 1.14000000E+02, 5.70000000E+01, 9.50700000E-01, 0.00000000E+00},
    {4.29807400E+02, 1.14000000E+02, 5.80000000E+01, 9.50700000E-01, 2.79910000E+00},
    {9.70395700E+02, 1.14000000E+02, 5.90000000E+01, 9.50700000E-01, 0.00000000E+00},
    {9.31291700E+02, 1.14000000E+02, 6.00000000E+01, 9.50700000E-01, 0.00000000E+00},
    {9.07797000E+02, 1.14000000E+02, 6.10000000E+01, 9.50700000E-01, 0.00000000E+00},
    {8.86219300E+02, 1.14000000E+02, 6.20000000E+01, 9.50700000E-01, 0.00000000E+00},
    {8.67084400E+02, 1.14000000E+02, 6.30000000E+01, 9.50700000E-01, 0.00000000E+00},
    {6.77480600E+02, 1.14000000E+02, 6.40000000E+01, 9.50700000E-01, 0.00000000E+00},
    {7.67305500E+02, 1.14000000E+02, 6.50000000E+01, 9.50700000E-01, 0.00000000E+00},
    {7.39397500E+02, 1.14000000E+02, 6.60000000E+01, 9.50700000E-01, 0.00000000E+00},
    {7.81374800E+02, 1.14000000E+02, 6.70000000E+01, 9.50700000E-01, 0.00000000E+00},
    {7.64766800E+02, 1.14000000E+02, 6.80000000E+01, 9.50700000E-01, 0.00000000E+00},
    {7.49741300E+02, 1.14000000E+02, 6.90000000E+01, 9.50700000E-01, 0.00000000E+00},
    {7.41203000E+02, 1.14000000E+02, 7.00000000E+01, 9.50700000E-01, 0.00000000E+00},
    {6.21816900E+02, 1.14000000E+02, 7.10000000E+01, 9.50700000E-01, 0.00000000E+00},
    {6.09060800E+02, 1.14000000E+02, 7.20000000E+01, 9.50700000E-01, 0.00000000E+00},
    {5.53654200E+02, 1.14000000E+02, 7.30000000E+01, 9.50700000E-01, 0.00000000E+00},
    {4.65194200E+02, 1.14000000E+02, 7.40000000E+01, 9.50700000E-01, 0.00000000E+00},
    {4.72792300E+02, 1.14000000E+02, 7.50000000E+01, 9.50700000E-01, 0.00000000E+00},
    {4.26798000E+02, 1.14000000E+02, 7.60000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.89498800E+02, 1.14000000E+02, 7.70000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.21718300E+02, 1.14000000E+02, 7.80000000E+01, 9.50700000E-01, 0.00000000E+00},
    {2.99885800E+02, 1.14000000E+02, 7.90000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.08426300E+02, 1.14000000E+02, 8.00000000E+01, 9.50700000E-01, 0.00000000E+00},
    {4.56587200E+02, 1.14000000E+02, 8.10000000E+01, 9.50700000E-01, 0.00000000E+00},
    {4.45003600E+02, 1.14000000E+02, 8.20000000E+01, 9.50700000E-01, 0.00000000E+00},
    {4.06701700E+02, 1.14000000E+02, 8.30000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.86467500E+02, 1.14000000E+02, 8.40000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.54866600E+02, 1.14000000E+02, 8.50000000E+01, 9.50700000E-01, 0.00000000E+00},
    {3.23541100E+02, 1.14000000E+02, 8.60000000E+01, 9.50700000E-01, 0.00000000E+00},
    {1.12989150E+03, 1.14000000E+02, 8.70000000E+01, 9.50700000E-01, 0.00000000E+00},
    {1.08250180E+03, 1.14000000E+02, 8.80000000E+01, 9.50700000E-01, 0.00000000E+00},
    {9.53142600E+02, 1.14000000E+02, 8.90000000E+01, 9.50700000E-01, 0.00000000E+00},
    {8.51475800E+02, 1.14000000E+02, 9.00000000E+01, 9.50700000E-01, 0.00000000E+00},
    {8.46795900E+02, 1.14000000E+02, 9.10000000E+01, 9.50700000E-01, 0.00000000E+00},
    {8.19685600E+02, 1.14000000E+02, 9.20000000E+01, 9.50700000E-01, 0.00000000E+00},
    {8.46554300E+02, 1.14000000E+02, 9.30000000E+01, 9.50700000E-01, 0.00000000E+00},
    {8.19369800E+02, 1.14000000E+02, 9.40000000E+01, 9.50700000E-01, 0.00000000E+00},
    {4.49030000E+01, 1.14000000E+02, 1.01000000E+02, 9.50700000E-01, 0.00000000E+00},
    {1.48747800E+02, 1.14000000E+02, 1.03000000E+02, 9.50700000E-01, 9.86500000E-01},
    {1.89109000E+02, 1.14000000E+02, 1.04000000E+02, 9.50700000E-01, 9.80800000E-01},
    {1.42512300E+02, 1.14000000E+02, 1.05000000E+02, 9.50700000E-01, 9.70600000E-01},
    {1.06080600E+02, 1.14000000E+02, 1.06000000E+02, 9.50700000E-01, 9.86800000E-01},
    {7.26373000E+01, 1.14000000E+02, 1.07000000E+02, 9.50700000E-01, 9.94400000E-01},
    {5.21311000E+01, 1.14000000E+02, 1.08000000E+02, 9.50700000E-01, 9.92500000E-01},
    {3.51290000E+01, 1.14000000E+02, 1.09000000E+02, 9.50700000E-01, 9.98200000E-01},
    {2.17504900E+02, 1.14000000E+02, 1.11000000E+02, 9.50700000E-01, 9.68400000E-01},
    {3.37025000E+02, 1.14000000E+02, 1.12000000E+02, 9.50700000E-01, 9.62800000E-01},
    {3.39651600E+02, 1.14000000E+02, 1.13000000E+02, 9.50700000E-01, 9.64800000E-01},
    {2.70265000E+02, 1.14000000E+02, 1.14000000E+02, 9.50700000E-01, 9.50700000E-01},
    {2.30908000E+01, 1.15000000E+02, 1.00000000E+00, 9.94700000E-01, 9.11800000E-01},
    {1.52619000E+01, 1.15000000E+02, 2.00000000E+00, 9.94700000E-01, 0.00000000E+00},
    {3.39669300E+02, 1.15000000E+02, 3.00000000E+00, 9.94700000E-01, 0.00000000E+00},
    {2.02228200E+02, 1.15000000E+02, 4.00000000E+00, 9.94700000E-01, 0.00000000E+00},
    {1.38077400E+02, 1.15000000E+02, 5.00000000E+00, 9.94700000E-01, 0.00000000E+00},
    {9.39663000E+01, 1.15000000E+02, 6.00000000E+00, 9.94700000E-01, 0.00000000E+00},
    {6.59127000E+01, 1.15000000E+02, 7.00000000E+00, 9.94700000E-01, 0.00000000E+00},
    {4.99374000E+01, 1.15000000E+02, 8.00000000E+00, 9.94700000E-01, 0.00000000E+00},
    {3.78035000E+01, 1.15000000E+02, 9.00000000E+00, 9.94700000E-01, 0.00000000E+00},
    {2.90280000E+01, 1.15000000E+02, 1.00000000E+01, 9.94700000E-01, 0.00000000E+00},
    {4.06715000E+02, 1.15000000E+02, 1.10000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.20417600E+02, 1.15000000E+02, 1.20000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.97807600E+02, 1.15000000E+02, 1.30000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.36986400E+02, 1.15000000E+02, 1.40000000E+01, 9.94700000E-01, 0.00000000E+00},
    {1.86068400E+02, 1.15000000E+02, 1.50000000E+01, 9.94700000E-01, 0.00000000E+00},
    {1.54922900E+02, 1.15000000E+02, 1.60000000E+01, 9.94700000E-01, 0.00000000E+00},
    {1.26831200E+02, 1.15000000E+02, 1.70000000E+01, 9.94700000E-01, 0.00000000E+00},
    {1.03865300E+02, 1.15000000E+02, 1.80000000E+01, 9.94700000E-01, 0.00000000E+00},
    {6.63118400E+02, 1.15000000E+02, 1.90000000E+01, 9.94700000E-01, 0.00000000E+00},
    {5.57425500E+02, 1.15000000E+02, 2.00000000E+01, 9.94700000E-01, 0.00000000E+00},
    {4.62351100E+02, 1.15000000E+02, 2.10000000E+01, 9.94700000E-01, 0.00000000E+00},
    {4.47959000E+02, 1.15000000E+02, 2.20000000E+01, 9.94700000E-01, 0.00000000E+00},
    {4.11012100E+02, 1.15000000E+02, 2.30000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.23591400E+02, 1.15000000E+02, 2.40000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.54843200E+02, 1.15000000E+02, 2.50000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.78402000E+02, 1.15000000E+02, 2.60000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.96578900E+02, 1.15000000E+02, 2.70000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.04914500E+02, 1.15000000E+02, 2.80000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.33489800E+02, 1.15000000E+02, 2.90000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.41438000E+02, 1.15000000E+02, 3.00000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.85807200E+02, 1.15000000E+02, 3.10000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.53974400E+02, 1.15000000E+02, 3.20000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.17892000E+02, 1.15000000E+02, 3.30000000E+01, 9.94700000E-01, 0.00000000E+00},
    {1.96080600E+02, 1.15000000E+02, 3.40000000E+01, 9.94700000E-01, 0.00000000E+00},
    {1.71991000E+02, 1.15000000E+02, 3.50000000E+01, 9.94700000E-01, 0.00000000E+00},
    {1.49793300E+02, 1.15000000E+02, 3.60000000E+01, 9.94700000E-01, 0.00000000E+00},
    {7.44247500E+02, 1.15000000E+02, 3.70000000E+01, 9.94700000E-01, 0.00000000E+00},
    {6.63565600E+02, 1.15000000E+02, 3.80000000E+01, 9.94700000E-01, 0.00000000E+00},
    {5.85246700E+02, 1.15000000E+02, 3.90000000E+01, 9.94700000E-01, 0.00000000E+00},
    {5.28084100E+02, 1.15000000E+02, 4.00000000E+01, 9.94700000E-01, 0.00000000E+00},
    {4.82703600E+02, 1.15000000E+02, 4.10000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.73942700E+02, 1.15000000E+02, 4.20000000E+01, 9.94700000E-01, 0.00000000E+00},
    {4.16708000E+02, 1.15000000E+02, 4.30000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.18591200E+02, 1.15000000E+02, 4.40000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.48444900E+02, 1.15000000E+02, 4.50000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.23527600E+02, 1.15000000E+02, 4.60000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.69122000E+02, 1.15000000E+02, 4.70000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.85480200E+02, 1.15000000E+02, 4.80000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.56906900E+02, 1.15000000E+02, 4.90000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.32172200E+02, 1.15000000E+02, 5.00000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.97659800E+02, 1.15000000E+02, 5.10000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.76980900E+02, 1.15000000E+02, 5.20000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.51108100E+02, 1.15000000E+02, 5.30000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.26221100E+02, 1.15000000E+02, 5.40000000E+01, 9.94700000E-01, 0.00000000E+00},
    {9.07299300E+02, 1.15000000E+02, 5.50000000E+01, 9.94700000E-01, 0.00000000E+00},
    {8.44012000E+02, 1.15000000E+02, 5.60000000E+01, 9.94700000E-01, 0.00000000E+00},
    {7.46599300E+02, 1.15000000E+02, 5.70000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.51158500E+02, 1.15000000E+02, 5.80000000E+01, 9.94700000E-01, 2.79910000E+00},
    {7.49143800E+02, 1.15000000E+02, 5.90000000E+01, 9.94700000E-01, 0.00000000E+00},
    {7.20145000E+02, 1.15000000E+02, 6.00000000E+01, 9.94700000E-01, 0.00000000E+00},
    {7.02295900E+02, 1.15000000E+02, 6.10000000E+01, 9.94700000E-01, 0.00000000E+00},
    {6.85862000E+02, 1.15000000E+02, 6.20000000E+01, 9.94700000E-01, 0.00000000E+00},
    {6.71300900E+02, 1.15000000E+02, 6.30000000E+01, 9.94700000E-01, 0.00000000E+00},
    {5.31542400E+02, 1.15000000E+02, 6.40000000E+01, 9.94700000E-01, 0.00000000E+00},
    {5.92013500E+02, 1.15000000E+02, 6.50000000E+01, 9.94700000E-01, 0.00000000E+00},
    {5.71748100E+02, 1.15000000E+02, 6.60000000E+01, 9.94700000E-01, 0.00000000E+00},
    {6.06494000E+02, 1.15000000E+02, 6.70000000E+01, 9.94700000E-01, 0.00000000E+00},
    {5.93740300E+02, 1.15000000E+02, 6.80000000E+01, 9.94700000E-01, 0.00000000E+00},
    {5.82295600E+02, 1.15000000E+02, 6.90000000E+01, 9.94700000E-01, 0.00000000E+00},
    {5.75326200E+02, 1.15000000E+02, 7.00000000E+01, 9.94700000E-01, 0.00000000E+00},
    {4.87061900E+02, 1.15000000E+02, 7.10000000E+01, 9.94700000E-01, 0.00000000E+00},
    {4.82461600E+02, 1.15000000E+02, 7.20000000E+01, 9.94700000E-01, 0.00000000E+00},
    {4.41861600E+02, 1.15000000E+02, 7.30000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.73865400E+02, 1.15000000E+02, 7.40000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.80912000E+02, 1.15000000E+02, 7.50000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.46132600E+02, 1.15000000E+02, 7.60000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.17608500E+02, 1.15000000E+02, 7.70000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.64109100E+02, 1.15000000E+02, 7.80000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.46830800E+02, 1.15000000E+02, 7.90000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.54312800E+02, 1.15000000E+02, 8.00000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.66584200E+02, 1.15000000E+02, 8.10000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.60263100E+02, 1.15000000E+02, 8.20000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.32718600E+02, 1.15000000E+02, 8.30000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.18184100E+02, 1.15000000E+02, 8.40000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.94471000E+02, 1.15000000E+02, 8.50000000E+01, 9.94700000E-01, 0.00000000E+00},
    {2.70467700E+02, 1.15000000E+02, 8.60000000E+01, 9.94700000E-01, 0.00000000E+00},
    {8.61382400E+02, 1.15000000E+02, 8.70000000E+01, 9.94700000E-01, 0.00000000E+00},
    {8.37512800E+02, 1.15000000E+02, 8.80000000E+01, 9.94700000E-01, 0.00000000E+00},
    {7.44763200E+02, 1.15000000E+02, 8.90000000E+01, 9.94700000E-01, 0.00000000E+00},
    {6.73568800E+02, 1.15000000E+02, 9.00000000E+01, 9.94700000E-01, 0.00000000E+00},
    {6.66258200E+02, 1.15000000E+02, 9.10000000E+01, 9.94700000E-01, 0.00000000E+00},
    {6.45140500E+02, 1.15000000E+02, 9.20000000E+01, 9.94700000E-01, 0.00000000E+00},
    {6.61280500E+02, 1.15000000E+02, 9.30000000E+01, 9.94700000E-01, 0.00000000E+00},
    {6.40876100E+02, 1.15000000E+02, 9.40000000E+01, 9.94700000E-01, 0.00000000E+00},
    {3.70236000E+01, 1.15000000E+02, 1.01000000E+02, 9.94700000E-01, 0.00000000E+00},
    {1.17747200E+02, 1.15000000E+02, 1.03000000E+02, 9.94700000E-01, 9.86500000E-01},
    {1.50606800E+02, 1.15000000E+02, 1.04000000E+02, 9.94700000E-01, 9.80800000E-01},
    {1.16381800E+02, 1.15000000E+02, 1.05000000E+02, 9.94700000E-01, 9.70600000E-01},
    {8.80769000E+01, 1.15000000E+02, 1.06000000E+02, 9.94700000E-01, 9.86800000E-01},
    {6.14286000E+01, 1.15000000E+02, 1.07000000E+02, 9.94700000E-01, 9.94400000E-01},
    {4.47770000E+01, 1.15000000E+02, 1.08000000E+02, 9.94700000E-01, 9.92500000E-01},
    {3.07688000E+01, 1.15000000E+02, 1.09000000E+02, 9.94700000E-01, 9.98200000E-01},
    {1.71446900E+02, 1.15000000E+02, 1.11000000E+02, 9.94700000E-01, 9.68400000E-01},
    {2.65037800E+02, 1.15000000E+02, 1.12000000E+02, 9.94700000E-01, 9.62800000E-01},
    {2.70358000E+02, 1.15000000E+02, 1.13000000E+02, 9.94700000E-01, 9.64800000E-01},
    {2.19286300E+02, 1.15000000E+02, 1.14000000E+02, 9.94700000E-01, 9.50700000E-01},
    {1.80623800E+02, 1.15000000E+02, 1.15000000E+02, 9.94700000E-01, 9.94700000E-01},
    {1.98667000E+01, 1.16000000E+02, 1.00000000E+00, 9.94800000E-01, 9.11800000E-01},
    {1.34598000E+01, 1.16000000E+02, 2.00000000E+00, 9.94800000E-01, 0.00000000E+00},
    {2.71301100E+02, 1.16000000E+02, 3.00000000E+00, 9.94800000E-01, 0.00000000E+00},
    {1.66180700E+02, 1.16000000E+02, 4.00000000E+00, 9.94800000E-01, 0.00000000E+00},
    {1.15816800E+02, 1.16000000E+02, 5.00000000E+00, 9.94800000E-01, 0.00000000E+00},
    {8.01804000E+01, 1.16000000E+02, 6.00000000E+00, 9.94800000E-01, 0.00000000E+00},
    {5.70255000E+01, 1.16000000E+02, 7.00000000E+00, 9.94800000E-01, 0.00000000E+00},
    {4.36548000E+01, 1.16000000E+02, 8.00000000E+00, 9.94800000E-01, 0.00000000E+00},
    {3.33550000E+01, 1.16000000E+02, 9.00000000E+00, 9.94800000E-01, 0.00000000E+00},
    {2.58107000E+01, 1.16000000E+02, 1.00000000E+01, 9.94800000E-01, 0.00000000E+00},
    {3.25596900E+02, 1.16000000E+02, 1.10000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.61943700E+02, 1.16000000E+02, 1.20000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.45856800E+02, 1.16000000E+02, 1.30000000E+01, 9.94800000E-01, 0.00000000E+00},
    {1.98326600E+02, 1.16000000E+02, 1.40000000E+01, 9.94800000E-01, 0.00000000E+00},
    {1.57697100E+02, 1.16000000E+02, 1.50000000E+01, 9.94800000E-01, 0.00000000E+00},
    {1.32492200E+02, 1.16000000E+02, 1.60000000E+01, 9.94800000E-01, 0.00000000E+00},
    {1.09448600E+02, 1.16000000E+02, 1.70000000E+01, 9.94800000E-01, 0.00000000E+00},
    {9.03716000E+01, 1.16000000E+02, 1.80000000E+01, 9.94800000E-01, 0.00000000E+00},
    {5.30519200E+02, 1.16000000E+02, 1.90000000E+01, 9.94800000E-01, 0.00000000E+00},
    {4.52374500E+02, 1.16000000E+02, 2.00000000E+01, 9.94800000E-01, 0.00000000E+00},
    {3.76614200E+02, 1.16000000E+02, 2.10000000E+01, 9.94800000E-01, 0.00000000E+00},
    {3.66515100E+02, 1.16000000E+02, 2.20000000E+01, 9.94800000E-01, 0.00000000E+00},
    {3.37125000E+02, 1.16000000E+02, 2.30000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.66076200E+02, 1.16000000E+02, 2.40000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.92125700E+02, 1.16000000E+02, 2.50000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.29877600E+02, 1.16000000E+02, 2.60000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.45598800E+02, 1.16000000E+02, 2.70000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.51809700E+02, 1.16000000E+02, 2.80000000E+01, 9.94800000E-01, 0.00000000E+00},
    {1.93386500E+02, 1.16000000E+02, 2.90000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.01070400E+02, 1.16000000E+02, 3.00000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.37256300E+02, 1.16000000E+02, 3.10000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.12943700E+02, 1.16000000E+02, 3.20000000E+01, 9.94800000E-01, 0.00000000E+00},
    {1.84546700E+02, 1.16000000E+02, 3.30000000E+01, 9.94800000E-01, 0.00000000E+00},
    {1.67226400E+02, 1.16000000E+02, 3.40000000E+01, 9.94800000E-01, 0.00000000E+00},
    {1.47767900E+02, 1.16000000E+02, 3.50000000E+01, 9.94800000E-01, 0.00000000E+00},
    {1.29612300E+02, 1.16000000E+02, 3.60000000E+01, 9.94800000E-01, 0.00000000E+00},
    {5.96876300E+02, 1.16000000E+02, 3.70000000E+01, 9.94800000E-01, 0.00000000E+00},
    {5.38672400E+02, 1.16000000E+02, 3.80000000E+01, 9.94800000E-01, 0.00000000E+00},
    {4.78520400E+02, 1.16000000E+02, 3.90000000E+01, 9.94800000E-01, 0.00000000E+00},
    {4.33903600E+02, 1.16000000E+02, 4.00000000E+01, 9.94800000E-01, 0.00000000E+00},
    {3.98041900E+02, 1.16000000E+02, 4.10000000E+01, 9.94800000E-01, 0.00000000E+00},
    {3.10586100E+02, 1.16000000E+02, 4.20000000E+01, 9.94800000E-01, 0.00000000E+00},
    {3.45143400E+02, 1.16000000E+02, 4.30000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.65952800E+02, 1.16000000E+02, 4.40000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.90414600E+02, 1.16000000E+02, 4.50000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.70252400E+02, 1.16000000E+02, 4.60000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.25118100E+02, 1.16000000E+02, 4.70000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.39178400E+02, 1.16000000E+02, 4.80000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.96821400E+02, 1.16000000E+02, 4.90000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.78278700E+02, 1.16000000E+02, 5.00000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.51434400E+02, 1.16000000E+02, 5.10000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.35251900E+02, 1.16000000E+02, 5.20000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.14600200E+02, 1.16000000E+02, 5.30000000E+01, 9.94800000E-01, 0.00000000E+00},
    {1.94523700E+02, 1.16000000E+02, 5.40000000E+01, 9.94800000E-01, 0.00000000E+00},
    {7.28465300E+02, 1.16000000E+02, 5.50000000E+01, 9.94800000E-01, 0.00000000E+00},
    {6.84235300E+02, 1.16000000E+02, 5.60000000E+01, 9.94800000E-01, 0.00000000E+00},
    {6.09442000E+02, 1.16000000E+02, 5.70000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.96458100E+02, 1.16000000E+02, 5.80000000E+01, 9.94800000E-01, 2.79910000E+00},
    {6.08911900E+02, 1.16000000E+02, 5.90000000E+01, 9.94800000E-01, 0.00000000E+00},
    {5.85913400E+02, 1.16000000E+02, 6.00000000E+01, 9.94800000E-01, 0.00000000E+00},
    {5.71539700E+02, 1.16000000E+02, 6.10000000E+01, 9.94800000E-01, 0.00000000E+00},
    {5.58281600E+02, 1.16000000E+02, 6.20000000E+01, 9.94800000E-01, 0.00000000E+00},
    {5.46539500E+02, 1.16000000E+02, 6.30000000E+01, 9.94800000E-01, 0.00000000E+00},
    {4.36778500E+02, 1.16000000E+02, 6.40000000E+01, 9.94800000E-01, 0.00000000E+00},
    {4.81589400E+02, 1.16000000E+02, 6.50000000E+01, 9.94800000E-01, 0.00000000E+00},
    {4.65775800E+02, 1.16000000E+02, 6.60000000E+01, 9.94800000E-01, 0.00000000E+00},
    {4.94528500E+02, 1.16000000E+02, 6.70000000E+01, 9.94800000E-01, 0.00000000E+00},
    {4.84179100E+02, 1.16000000E+02, 6.80000000E+01, 9.94800000E-01, 0.00000000E+00},
    {4.74944300E+02, 1.16000000E+02, 6.90000000E+01, 9.94800000E-01, 0.00000000E+00},
    {4.69056700E+02, 1.16000000E+02, 7.00000000E+01, 9.94800000E-01, 0.00000000E+00},
    {3.99595200E+02, 1.16000000E+02, 7.10000000E+01, 9.94800000E-01, 0.00000000E+00},
    {3.98591500E+02, 1.16000000E+02, 7.20000000E+01, 9.94800000E-01, 0.00000000E+00},
    {3.66947700E+02, 1.16000000E+02, 7.30000000E+01, 9.94800000E-01, 0.00000000E+00},
    {3.12208100E+02, 1.16000000E+02, 7.40000000E+01, 9.94800000E-01, 0.00000000E+00},
    {3.18555600E+02, 1.16000000E+02, 7.50000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.90826300E+02, 1.16000000E+02, 7.60000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.67911400E+02, 1.16000000E+02, 7.70000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.24010000E+02, 1.16000000E+02, 7.80000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.09799800E+02, 1.16000000E+02, 7.90000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.16325300E+02, 1.16000000E+02, 8.00000000E+01, 9.94800000E-01, 0.00000000E+00},
    {3.06150900E+02, 1.16000000E+02, 8.10000000E+01, 9.94800000E-01, 0.00000000E+00},
    {3.02350500E+02, 1.16000000E+02, 8.20000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.81177300E+02, 1.16000000E+02, 8.30000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.70077900E+02, 1.16000000E+02, 8.40000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.51355000E+02, 1.16000000E+02, 8.50000000E+01, 9.94800000E-01, 0.00000000E+00},
    {2.32125800E+02, 1.16000000E+02, 8.60000000E+01, 9.94800000E-01, 0.00000000E+00},
    {6.95535300E+02, 1.16000000E+02, 8.70000000E+01, 9.94800000E-01, 0.00000000E+00},
    {6.81655800E+02, 1.16000000E+02, 8.80000000E+01, 9.94800000E-01, 0.00000000E+00},
    {6.09934100E+02, 1.16000000E+02, 8.90000000E+01, 9.94800000E-01, 0.00000000E+00},
    {5.56249200E+02, 1.16000000E+02, 9.00000000E+01, 9.94800000E-01, 0.00000000E+00},
    {5.48497100E+02, 1.16000000E+02, 9.10000000E+01, 9.94800000E-01, 0.00000000E+00},
    {5.31255400E+02, 1.16000000E+02, 9.20000000E+01, 9.94800000E-01, 0.00000000E+00},
    {5.41870300E+02, 1.16000000E+02, 9.30000000E+01, 9.94800000E-01, 0.00000000E+00},
    {5.25577700E+02, 1.16000000E+02, 9.40000000E+01, 9.94800000E-01, 0.00000000E+00},
    {3.14127000E+01, 1.16000000E+02, 1.01000000E+02, 9.94800000E-01, 0.00000000E+00},
    {9.70849000E+01, 1.16000000E+02, 1.03000000E+02, 9.94800000E-01, 9.86500000E-01},
    {1.24751200E+02, 1.16000000E+02, 1.04000000E+02, 9.94800000E-01, 9.80800000E-01},
    {9.81139000E+01, 1.16000000E+02, 1.05000000E+02, 9.94800000E-01, 9.70600000E-01},
    {7.52039000E+01, 1.16000000E+02, 1.06000000E+02, 9.94800000E-01, 9.86800000E-01},
    {5.32086000E+01, 1.16000000E+02, 1.07000000E+02, 9.94800000E-01, 9.94400000E-01},
    {3.92683000E+01, 1.16000000E+02, 1.08000000E+02, 9.94800000E-01, 9.92500000E-01},
    {2.74117000E+01, 1.16000000E+02, 1.09000000E+02, 9.94800000E-01, 9.98200000E-01},
    {1.41059900E+02, 1.16000000E+02, 1.11000000E+02, 9.94800000E-01, 9.68400000E-01},
    {2.17662900E+02, 1.16000000E+02, 1.12000000E+02, 9.94800000E-01, 9.62800000E-01},
    {2.23794200E+02, 1.16000000E+02, 1.13000000E+02, 9.94800000E-01, 9.64800000E-01},
    {1.83948300E+02, 1.16000000E+02, 1.14000000E+02, 9.94800000E-01, 9.50700000E-01},
    {1.53182300E+02, 1.16000000E+02, 1.15000000E+02, 9.94800000E-01, 9.94700000E-01},
    {1.30996500E+02, 1.16000000E+02, 1.16000000E+02, 9.94800000E-01, 9.94800000E-01},
    {1.65273000E+01, 1.17000000E+02, 1.00000000E+00, 9.97200000E-01, 9.11800000E-01},
    {1.15092000E+01, 1.17000000E+02, 2.00000000E+00, 9.97200000E-01, 0.00000000E+00},
    {2.08774300E+02, 1.17000000E+02, 3.00000000E+00, 9.97200000E-01, 0.00000000E+00},
    {1.31649800E+02, 1.17000000E+02, 4.00000000E+00, 9.97200000E-01, 0.00000000E+00},
    {9.37743000E+01, 1.17000000E+02, 5.00000000E+00, 9.97200000E-01, 0.00000000E+00},
    {6.61354000E+01, 1.17000000E+02, 6.00000000E+00, 9.97200000E-01, 0.00000000E+00},
    {4.77540000E+01, 1.17000000E+02, 7.00000000E+00, 9.97200000E-01, 0.00000000E+00},
    {3.69783000E+01, 1.17000000E+02, 8.00000000E+00, 9.97200000E-01, 0.00000000E+00},
    {2.85468000E+01, 1.17000000E+02, 9.00000000E+00, 9.97200000E-01, 0.00000000E+00},
    {2.22832000E+01, 1.17000000E+02, 1.00000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.51232600E+02, 1.17000000E+02, 1.10000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.06452100E+02, 1.17000000E+02, 1.20000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.95764700E+02, 1.17000000E+02, 1.30000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.60187900E+02, 1.17000000E+02, 1.40000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.29095600E+02, 1.17000000E+02, 1.50000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.09527000E+02, 1.17000000E+02, 1.60000000E+01, 9.97200000E-01, 0.00000000E+00},
    {9.13667000E+01, 1.17000000E+02, 1.70000000E+01, 9.97200000E-01, 0.00000000E+00},
    {7.61256000E+01, 1.17000000E+02, 1.80000000E+01, 9.97200000E-01, 0.00000000E+00},
    {4.09510500E+02, 1.17000000E+02, 1.90000000E+01, 9.97200000E-01, 0.00000000E+00},
    {3.54093500E+02, 1.17000000E+02, 2.00000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.95902800E+02, 1.17000000E+02, 2.10000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.89351200E+02, 1.17000000E+02, 2.20000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.66855700E+02, 1.17000000E+02, 2.30000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.11292800E+02, 1.17000000E+02, 2.40000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.32144700E+02, 1.17000000E+02, 2.50000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.83366100E+02, 1.17000000E+02, 2.60000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.96375500E+02, 1.17000000E+02, 2.70000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.00757100E+02, 1.17000000E+02, 2.80000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.54775600E+02, 1.17000000E+02, 2.90000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.61733700E+02, 1.17000000E+02, 3.00000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.90113900E+02, 1.17000000E+02, 3.10000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.72393300E+02, 1.17000000E+02, 3.20000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.50999900E+02, 1.17000000E+02, 3.30000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.37844300E+02, 1.17000000E+02, 3.40000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.22776300E+02, 1.17000000E+02, 3.50000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.08522600E+02, 1.17000000E+02, 3.60000000E+01, 9.97200000E-01, 0.00000000E+00},
    {4.62025200E+02, 1.17000000E+02, 3.70000000E+01, 9.97200000E-01, 0.00000000E+00},
    {4.21886700E+02, 1.17000000E+02, 3.80000000E+01, 9.97200000E-01, 0.00000000E+00},
    {3.77564200E+02, 1.17000000E+02, 3.90000000E+01, 9.97200000E-01, 0.00000000E+00},
    {3.44131800E+02, 1.17000000E+02, 4.00000000E+01, 9.97200000E-01, 0.00000000E+00},
    {3.16911000E+02, 1.17000000E+02, 4.10000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.49267100E+02, 1.17000000E+02, 4.20000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.76145500E+02, 1.17000000E+02, 4.30000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.14638000E+02, 1.17000000E+02, 4.40000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.33903100E+02, 1.17000000E+02, 4.50000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.18190300E+02, 1.17000000E+02, 4.60000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.82154700E+02, 1.17000000E+02, 4.70000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.93727300E+02, 1.17000000E+02, 4.80000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.38498800E+02, 1.17000000E+02, 4.90000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.25238100E+02, 1.17000000E+02, 5.00000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.05251200E+02, 1.17000000E+02, 5.10000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.93144900E+02, 1.17000000E+02, 5.20000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.77345800E+02, 1.17000000E+02, 5.30000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.61811900E+02, 1.17000000E+02, 5.40000000E+01, 9.97200000E-01, 0.00000000E+00},
    {5.64600600E+02, 1.17000000E+02, 5.50000000E+01, 9.97200000E-01, 0.00000000E+00},
    {5.35267800E+02, 1.17000000E+02, 5.60000000E+01, 9.97200000E-01, 0.00000000E+00},
    {4.80124400E+02, 1.17000000E+02, 5.70000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.41922300E+02, 1.17000000E+02, 5.80000000E+01, 9.97200000E-01, 2.79910000E+00},
    {4.77719700E+02, 1.17000000E+02, 5.90000000E+01, 9.97200000E-01, 0.00000000E+00},
    {4.60125000E+02, 1.17000000E+02, 6.00000000E+01, 9.97200000E-01, 0.00000000E+00},
    {4.48951700E+02, 1.17000000E+02, 6.10000000E+01, 9.97200000E-01, 0.00000000E+00},
    {4.38624300E+02, 1.17000000E+02, 6.20000000E+01, 9.97200000E-01, 0.00000000E+00},
    {4.29481000E+02, 1.17000000E+02, 6.30000000E+01, 9.97200000E-01, 0.00000000E+00},
    {3.46646100E+02, 1.17000000E+02, 6.40000000E+01, 9.97200000E-01, 0.00000000E+00},
    {3.78398400E+02, 1.17000000E+02, 6.50000000E+01, 9.97200000E-01, 0.00000000E+00},
    {3.66516600E+02, 1.17000000E+02, 6.60000000E+01, 9.97200000E-01, 0.00000000E+00},
    {3.89198500E+02, 1.17000000E+02, 6.70000000E+01, 9.97200000E-01, 0.00000000E+00},
    {3.81084700E+02, 1.17000000E+02, 6.80000000E+01, 9.97200000E-01, 0.00000000E+00},
    {3.73887800E+02, 1.17000000E+02, 6.90000000E+01, 9.97200000E-01, 0.00000000E+00},
    {3.69076900E+02, 1.17000000E+02, 7.00000000E+01, 9.97200000E-01, 0.00000000E+00},
    {3.16532700E+02, 1.17000000E+02, 7.10000000E+01, 9.97200000E-01, 0.00000000E+00},
    {3.17923900E+02, 1.17000000E+02, 7.20000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.94298200E+02, 1.17000000E+02, 7.30000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.51961600E+02, 1.17000000E+02, 7.40000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.57444700E+02, 1.17000000E+02, 7.50000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.36206500E+02, 1.17000000E+02, 7.60000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.18513400E+02, 1.17000000E+02, 7.70000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.83843500E+02, 1.17000000E+02, 7.80000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.72595500E+02, 1.17000000E+02, 7.90000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.78060500E+02, 1.17000000E+02, 8.00000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.47219000E+02, 1.17000000E+02, 8.10000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.45295000E+02, 1.17000000E+02, 8.20000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.29726700E+02, 1.17000000E+02, 8.30000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.21656700E+02, 1.17000000E+02, 8.40000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.07500800E+02, 1.17000000E+02, 8.50000000E+01, 9.97200000E-01, 0.00000000E+00},
    {1.92731600E+02, 1.17000000E+02, 8.60000000E+01, 9.97200000E-01, 0.00000000E+00},
    {5.42295300E+02, 1.17000000E+02, 8.70000000E+01, 9.97200000E-01, 0.00000000E+00},
    {5.35470600E+02, 1.17000000E+02, 8.80000000E+01, 9.97200000E-01, 0.00000000E+00},
    {4.82177200E+02, 1.17000000E+02, 8.90000000E+01, 9.97200000E-01, 0.00000000E+00},
    {4.43626200E+02, 1.17000000E+02, 9.00000000E+01, 9.97200000E-01, 0.00000000E+00},
    {4.36152600E+02, 1.17000000E+02, 9.10000000E+01, 9.97200000E-01, 0.00000000E+00},
    {4.22580400E+02, 1.17000000E+02, 9.20000000E+01, 9.97200000E-01, 0.00000000E+00},
    {4.28861900E+02, 1.17000000E+02, 9.30000000E+01, 9.97200000E-01, 0.00000000E+00},
    {4.16311500E+02, 1.17000000E+02, 9.40000000E+01, 9.97200000E-01, 0.00000000E+00},
    {2.57414000E+01, 1.17000000E+02, 1.01000000E+02, 9.97200000E-01, 0.00000000E+00},
    {7.72023000E+01, 1.17000000E+02, 1.03000000E+02, 9.97200000E-01, 9.86500000E-01},
    {9.96995000E+01, 1.17000000E+02, 1.04000000E+02, 9.97200000E-01, 9.80800000E-01},
    {7.98833000E+01, 1.17000000E+02, 1.05000000E+02, 9.97200000E-01, 9.70600000E-01},
    {6.20864000E+01, 1.17000000E+02, 1.06000000E+02, 9.97200000E-01, 9.86800000E-01},
    {4.46217000E+01, 1.17000000E+02, 1.07000000E+02, 9.97200000E-01, 9.94400000E-01},
    {3.33801000E+01, 1.17000000E+02, 1.08000000E+02, 9.97200000E-01, 9.92500000E-01},
    {2.37059000E+01, 1.17000000E+02, 1.09000000E+02, 9.97200000E-01, 9.98200000E-01},
    {1.11982900E+02, 1.17000000E+02, 1.11000000E+02, 9.97200000E-01, 9.68400000E-01},
    {1.72414800E+02, 1.17000000E+02, 1.12000000E+02, 9.97200000E-01, 9.62800000E-01},
    {1.78712700E+02, 1.17000000E+02, 1.13000000E+02, 9.97200000E-01, 9.64800000E-01},
    {1.48955900E+02, 1.17000000E+02, 1.14000000E+02, 9.97200000E-01, 9.50700000E-01},
    {1.25493000E+02, 1.17000000E+02, 1.15000000E+02, 9.97200000E-01, 9.94700000E-01},
    {1.08289300E+02, 1.17000000E+02, 1.16000000E+02, 9.97200000E-01, 9.94800000E-01},
    {9.03985000E+01, 1.17000000E+02, 1.17000000E+02, 9.97200000E-01, 9.97200000E-01},
    {2.93000000E+01, 1.19000000E+02, 1.00000000E+00, 9.76700000E-01, 9.11800000E-01},
    {1.86655000E+01, 1.19000000E+02, 2.00000000E+00, 9.76700000E-01, 0.00000000E+00},
    {5.60295400E+02, 1.19000000E+02, 3.00000000E+00, 9.76700000E-01, 0.00000000E+00},
    {2.92222400E+02, 1.19000000E+02, 4.00000000E+00, 9.76700000E-01, 0.00000000E+00},
    {1.86956600E+02, 1.19000000E+02, 5.00000000E+00, 9.76700000E-01, 0.00000000E+00},
    {1.21929200E+02, 1.19000000E+02, 6.00000000E+00, 9.76700000E-01, 0.00000000E+00},
    {8.32153000E+01, 1.19000000E+02, 7.00000000E+00, 9.76700000E-01, 0.00000000E+00},
    {6.19948000E+01, 1.19000000E+02, 8.00000000E+00, 9.76700000E-01, 0.00000000E+00},
    {4.63520000E+01, 1.19000000E+02, 9.00000000E+00, 9.76700000E-01, 0.00000000E+00},
    {3.52969000E+01, 1.19000000E+02, 1.00000000E+01, 9.76700000E-01, 0.00000000E+00},
    {6.66035700E+02, 1.19000000E+02, 1.10000000E+01, 9.76700000E-01, 0.00000000E+00},
    {4.73766100E+02, 1.19000000E+02, 1.20000000E+01, 9.76700000E-01, 0.00000000E+00},
    {4.24892600E+02, 1.19000000E+02, 1.30000000E+01, 9.76700000E-01, 0.00000000E+00},
    {3.22823300E+02, 1.19000000E+02, 1.40000000E+01, 9.76700000E-01, 0.00000000E+00},
    {2.44572900E+02, 1.19000000E+02, 1.50000000E+01, 9.76700000E-01, 0.00000000E+00},
    {1.99467200E+02, 1.19000000E+02, 1.60000000E+01, 9.76700000E-01, 0.00000000E+00},
    {1.60330400E+02, 1.19000000E+02, 1.70000000E+01, 9.76700000E-01, 0.00000000E+00},
    {1.29428300E+02, 1.19000000E+02, 1.80000000E+01, 9.76700000E-01, 0.00000000E+00},
    {1.11222770E+03, 1.19000000E+02, 1.90000000E+01, 9.76700000E-01, 0.00000000E+00},
    {8.60177100E+02, 1.19000000E+02, 2.00000000E+01, 9.76700000E-01, 0.00000000E+00},
    {7.00728100E+02, 1.19000000E+02, 2.10000000E+01, 9.76700000E-01, 0.00000000E+00},
    {6.68754600E+02, 1.19000000E+02, 2.20000000E+01, 9.76700000E-01, 0.00000000E+00},
    {6.07995100E+02, 1.19000000E+02, 2.30000000E+01, 9.76700000E-01, 0.00000000E+00},
    {4.78974400E+02, 1.19000000E+02, 2.40000000E+01, 9.76700000E-01, 0.00000000E+00},
    {5.18058300E+02, 1.19000000E+02, 2.50000000E+01, 9.76700000E-01, 0.00000000E+00},
    {4.06087600E+02, 1.19000000E+02, 2.60000000E+01, 9.76700000E-01, 0.00000000E+00},
    {4.23380900E+02, 1.19000000E+02, 2.70000000E+01, 9.76700000E-01, 0.00000000E+00},
    {4.39284500E+02, 1.19000000E+02, 2.80000000E+01, 9.76700000E-01, 0.00000000E+00},
    {3.37111700E+02, 1.19000000E+02, 2.90000000E+01, 9.76700000E-01, 0.00000000E+00},
    {3.37898600E+02, 1.19000000E+02, 3.00000000E+01, 9.76700000E-01, 0.00000000E+00},
    {4.03074500E+02, 1.19000000E+02, 3.10000000E+01, 9.76700000E-01, 0.00000000E+00},
    {3.45443400E+02, 1.19000000E+02, 3.20000000E+01, 9.76700000E-01, 0.00000000E+00},
    {2.87553500E+02, 1.19000000E+02, 3.30000000E+01, 9.76700000E-01, 0.00000000E+00},
    {2.54367000E+02, 1.19000000E+02, 3.40000000E+01, 9.76700000E-01, 0.00000000E+00},
    {2.19464700E+02, 1.19000000E+02, 3.50000000E+01, 9.76700000E-01, 0.00000000E+00},
    {1.88481900E+02, 1.19000000E+02, 3.60000000E+01, 9.76700000E-01, 0.00000000E+00},
    {1.24086310E+03, 1.19000000E+02, 3.70000000E+01, 9.76700000E-01, 0.00000000E+00},
    {1.02796710E+03, 1.19000000E+02, 3.80000000E+01, 9.76700000E-01, 0.00000000E+00},
    {8.79034600E+02, 1.19000000E+02, 3.90000000E+01, 9.76700000E-01, 0.00000000E+00},
    {7.78523800E+02, 1.19000000E+02, 4.00000000E+01, 9.76700000E-01, 0.00000000E+00},
    {7.03306000E+02, 1.19000000E+02, 4.10000000E+01, 9.76700000E-01, 0.00000000E+00},
    {5.34630300E+02, 1.19000000E+02, 4.20000000E+01, 9.76700000E-01, 0.00000000E+00},
    {5.99930200E+02, 1.19000000E+02, 4.30000000E+01, 9.76700000E-01, 0.00000000E+00},
    {4.49303000E+02, 1.19000000E+02, 4.40000000E+01, 9.76700000E-01, 0.00000000E+00},
    {4.90580700E+02, 1.19000000E+02, 4.50000000E+01, 9.76700000E-01, 0.00000000E+00},
    {4.52246400E+02, 1.19000000E+02, 4.60000000E+01, 9.76700000E-01, 0.00000000E+00},
    {3.79083600E+02, 1.19000000E+02, 4.70000000E+01, 9.76700000E-01, 0.00000000E+00},
    {3.95513500E+02, 1.19000000E+02, 4.80000000E+01, 9.76700000E-01, 0.00000000E+00},
    {5.05683800E+02, 1.19000000E+02, 4.90000000E+01, 9.76700000E-01, 0.00000000E+00},
    {4.56461300E+02, 1.19000000E+02, 5.00000000E+01, 9.76700000E-01, 0.00000000E+00},
    {3.97841600E+02, 1.19000000E+02, 5.10000000E+01, 9.76700000E-01, 0.00000000E+00},
    {3.64409000E+02, 1.19000000E+02, 5.20000000E+01, 9.76700000E-01, 0.00000000E+00},
    {3.25106800E+02, 1.19000000E+02, 5.30000000E+01, 9.76700000E-01, 0.00000000E+00},
    {2.88727500E+02, 1.19000000E+02, 5.40000000E+01, 9.76700000E-01, 0.00000000E+00},
    {1.51188980E+03, 1.19000000E+02, 5.50000000E+01, 9.76700000E-01, 0.00000000E+00},
    {1.32226190E+03, 1.19000000E+02, 5.60000000E+01, 9.76700000E-01, 0.00000000E+00},
    {1.13377490E+03, 1.19000000E+02, 5.70000000E+01, 9.76700000E-01, 0.00000000E+00},
    {4.72245500E+02, 1.19000000E+02, 5.80000000E+01, 9.76700000E-01, 2.79910000E+00},
    {1.16228350E+03, 1.19000000E+02, 5.90000000E+01, 9.76700000E-01, 0.00000000E+00},
    {1.11126380E+03, 1.19000000E+02, 6.00000000E+01, 9.76700000E-01, 0.00000000E+00},
    {1.08208650E+03, 1.19000000E+02, 6.10000000E+01, 9.76700000E-01, 0.00000000E+00},
    {1.05538910E+03, 1.19000000E+02, 6.20000000E+01, 9.76700000E-01, 0.00000000E+00},
    {1.03167160E+03, 1.19000000E+02, 6.30000000E+01, 9.76700000E-01, 0.00000000E+00},
    {7.90785400E+02, 1.19000000E+02, 6.40000000E+01, 9.76700000E-01, 0.00000000E+00},
    {9.27367200E+02, 1.19000000E+02, 6.50000000E+01, 9.76700000E-01, 0.00000000E+00},
    {8.90505600E+02, 1.19000000E+02, 6.60000000E+01, 9.76700000E-01, 0.00000000E+00},
    {9.24619700E+02, 1.19000000E+02, 6.70000000E+01, 9.76700000E-01, 0.00000000E+00},
    {9.04325700E+02, 1.19000000E+02, 6.80000000E+01, 9.76700000E-01, 0.00000000E+00},
    {8.85753800E+02, 1.19000000E+02, 6.90000000E+01, 9.76700000E-01, 0.00000000E+00},
    {8.76289900E+02, 1.19000000E+02, 7.00000000E+01, 9.76700000E-01, 0.00000000E+00},
    {7.25644100E+02, 1.19000000E+02, 7.10000000E+01, 9.76700000E-01, 0.00000000E+00},
    {6.96183100E+02, 1.19000000E+02, 7.20000000E+01, 9.76700000E-01, 0.00000000E+00},
    {6.26344200E+02, 1.19000000E+02, 7.30000000E+01, 9.76700000E-01, 0.00000000E+00},
    {5.23689200E+02, 1.19000000E+02, 7.40000000E+01, 9.76700000E-01, 0.00000000E+00},
    {5.29624900E+02, 1.19000000E+02, 7.50000000E+01, 9.76700000E-01, 0.00000000E+00},
    {4.74253900E+02, 1.19000000E+02, 7.60000000E+01, 9.76700000E-01, 0.00000000E+00},
    {4.30196400E+02, 1.19000000E+02, 7.70000000E+01, 9.76700000E-01, 0.00000000E+00},
    {3.54141100E+02, 1.19000000E+02, 7.80000000E+01, 9.76700000E-01, 0.00000000E+00},
    {3.29647100E+02, 1.19000000E+02, 7.90000000E+01, 9.76700000E-01, 0.00000000E+00},
    {3.37333500E+02, 1.19000000E+02, 8.00000000E+01, 9.76700000E-01, 0.00000000E+00},
    {5.16953300E+02, 1.19000000E+02, 8.10000000E+01, 9.76700000E-01, 0.00000000E+00},
    {4.95698100E+02, 1.19000000E+02, 8.20000000E+01, 9.76700000E-01, 0.00000000E+00},
    {4.46303300E+02, 1.19000000E+02, 8.30000000E+01, 9.76700000E-01, 0.00000000E+00},
    {4.20868400E+02, 1.19000000E+02, 8.40000000E+01, 9.76700000E-01, 0.00000000E+00},
    {3.83353300E+02, 1.19000000E+02, 8.50000000E+01, 9.76700000E-01, 0.00000000E+00},
    {3.47343700E+02, 1.19000000E+02, 8.60000000E+01, 9.76700000E-01, 0.00000000E+00},
    {1.40016860E+03, 1.19000000E+02, 8.70000000E+01, 9.76700000E-01, 0.00000000E+00},
    {1.29150730E+03, 1.19000000E+02, 8.80000000E+01, 9.76700000E-01, 0.00000000E+00},
    {1.11592880E+03, 1.19000000E+02, 8.90000000E+01, 9.76700000E-01, 0.00000000E+00},
    {9.78192800E+02, 1.19000000E+02, 9.00000000E+01, 9.76700000E-01, 0.00000000E+00},
    {9.83913100E+02, 1.19000000E+02, 9.10000000E+01, 9.76700000E-01, 0.00000000E+00},
    {9.52062700E+02, 1.19000000E+02, 9.20000000E+01, 9.76700000E-01, 0.00000000E+00},
    {9.95352700E+02, 1.19000000E+02, 9.30000000E+01, 9.76700000E-01, 0.00000000E+00},
    {9.61054300E+02, 1.19000000E+02, 9.40000000E+01, 9.76700000E-01, 0.00000000E+00},
    {4.84499000E+01, 1.19000000E+02, 1.01000000E+02, 9.76700000E-01, 0.00000000E+00},
    {1.68741200E+02, 1.19000000E+02, 1.03000000E+02, 9.76700000E-01, 9.86500000E-01},
    {2.13444400E+02, 1.19000000E+02, 1.04000000E+02, 9.76700000E-01, 9.80800000E-01},
    {1.55778100E+02, 1.19000000E+02, 1.05000000E+02, 9.76700000E-01, 9.70600000E-01},
    {1.14572300E+02, 1.19000000E+02, 1.06000000E+02, 9.76700000E-01, 9.86800000E-01},
    {7.76317000E+01, 1.19000000E+02, 1.07000000E+02, 9.76700000E-01, 9.94400000E-01},
    {5.54032000E+01, 1.19000000E+02, 1.08000000E+02, 9.76700000E-01, 9.92500000E-01},
    {3.71982000E+01, 1.19000000E+02, 1.09000000E+02, 9.76700000E-01, 9.98200000E-01},
    {2.49220000E+02, 1.19000000E+02, 1.11000000E+02, 9.76700000E-01, 9.68400000E-01},
    {3.87334900E+02, 1.19000000E+02, 1.12000000E+02, 9.76700000E-01, 9.62800000E-01},
    {3.82642200E+02, 1.19000000E+02, 1.13000000E+02, 9.76700000E-01, 9.64800000E-01},
    {2.96940900E+02, 1.19000000E+02, 1.14000000E+02, 9.76700000E-01, 9.50700000E-01},
    {2.37219100E+02, 1.19000000E+02, 1.15000000E+02, 9.76700000E-01, 9.94700000E-01},
    {1.97380600E+02, 1.19000000E+02, 1.16000000E+02, 9.76700000E-01, 9.94800000E-01},
    {1.58733000E+02, 1.19000000E+02, 1.17000000E+02, 9.76700000E-01, 9.97200000E-01},
    {3.38020700E+02, 1.19000000E+02, 1.19000000E+02, 9.76700000E-01, 9.76700000E-01},
    {5.29871000E+01, 1.20000000E+02, 1.00000000E+00, 9.83100000E-01, 9.11800000E-01},
    {3.18796000E+01, 1.20000000E+02, 2.00000000E+00, 9.83100000E-01, 0.00000000E+00},
    {1.31003140E+03, 1.20000000E+02, 3.00000000E+00, 9.83100000E-01, 0.00000000E+00},
    {6.02566100E+02, 1.20000000E+02, 4.00000000E+00, 9.83100000E-01, 0.00000000E+00},
    {3.62686200E+02, 1.20000000E+02, 5.00000000E+00, 9.83100000E-01, 0.00000000E+00},
    {2.25962900E+02, 1.20000000E+02, 6.00000000E+00, 9.83100000E-01, 0.00000000E+00},
    {1.48979300E+02, 1.20000000E+02, 7.00000000E+00, 9.83100000E-01, 0.00000000E+00},
    {1.08271800E+02, 1.20000000E+02, 8.00000000E+00, 9.83100000E-01, 0.00000000E+00},
    {7.92218000E+01, 1.20000000E+02, 9.00000000E+00, 9.83100000E-01, 0.00000000E+00},
    {5.92779000E+01, 1.20000000E+02, 1.00000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.54570640E+03, 1.20000000E+02, 1.10000000E+01, 9.83100000E-01, 0.00000000E+00},
    {9.97248600E+02, 1.20000000E+02, 1.20000000E+01, 9.83100000E-01, 0.00000000E+00},
    {8.66951700E+02, 1.20000000E+02, 1.30000000E+01, 9.83100000E-01, 0.00000000E+00},
    {6.30045600E+02, 1.20000000E+02, 1.40000000E+01, 9.83100000E-01, 0.00000000E+00},
    {4.60263500E+02, 1.20000000E+02, 1.50000000E+01, 9.83100000E-01, 0.00000000E+00},
    {3.66771100E+02, 1.20000000E+02, 1.60000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.88218200E+02, 1.20000000E+02, 1.70000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.28089200E+02, 1.20000000E+02, 1.80000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.66108300E+03, 1.20000000E+02, 1.90000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.88642210E+03, 1.20000000E+02, 2.00000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.51014270E+03, 1.20000000E+02, 2.10000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.42232440E+03, 1.20000000E+02, 2.20000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.28216100E+03, 1.20000000E+02, 2.30000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.01159550E+03, 1.20000000E+02, 2.40000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.07915320E+03, 1.20000000E+02, 2.50000000E+01, 9.83100000E-01, 0.00000000E+00},
    {8.45468600E+02, 1.20000000E+02, 2.60000000E+01, 9.83100000E-01, 0.00000000E+00},
    {8.62503600E+02, 1.20000000E+02, 2.70000000E+01, 9.83100000E-01, 0.00000000E+00},
    {9.02601400E+02, 1.20000000E+02, 2.80000000E+01, 9.83100000E-01, 0.00000000E+00},
    {6.94505800E+02, 1.20000000E+02, 2.90000000E+01, 9.83100000E-01, 0.00000000E+00},
    {6.75029900E+02, 1.20000000E+02, 3.00000000E+01, 9.83100000E-01, 0.00000000E+00},
    {8.13494300E+02, 1.20000000E+02, 3.10000000E+01, 9.83100000E-01, 0.00000000E+00},
    {6.72691700E+02, 1.20000000E+02, 3.20000000E+01, 9.83100000E-01, 0.00000000E+00},
    {5.43012300E+02, 1.20000000E+02, 3.30000000E+01, 9.83100000E-01, 0.00000000E+00},
    {4.71482200E+02, 1.20000000E+02, 3.40000000E+01, 9.83100000E-01, 0.00000000E+00},
    {3.99022600E+02, 1.20000000E+02, 3.50000000E+01, 9.83100000E-01, 0.00000000E+00},
    {3.36648300E+02, 1.20000000E+02, 3.60000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.95804100E+03, 1.20000000E+02, 3.70000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.26541550E+03, 1.20000000E+02, 3.80000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.88213380E+03, 1.20000000E+02, 3.90000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.63796670E+03, 1.20000000E+02, 4.00000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.46388900E+03, 1.20000000E+02, 4.10000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.09267330E+03, 1.20000000E+02, 4.20000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.23493160E+03, 1.20000000E+02, 4.30000000E+01, 9.83100000E-01, 0.00000000E+00},
    {9.05751500E+02, 1.20000000E+02, 4.40000000E+01, 9.83100000E-01, 0.00000000E+00},
    {9.86372700E+02, 1.20000000E+02, 4.50000000E+01, 9.83100000E-01, 0.00000000E+00},
    {9.02440500E+02, 1.20000000E+02, 4.60000000E+01, 9.83100000E-01, 0.00000000E+00},
    {7.62617400E+02, 1.20000000E+02, 4.70000000E+01, 9.83100000E-01, 0.00000000E+00},
    {7.81660000E+02, 1.20000000E+02, 4.80000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.02416140E+03, 1.20000000E+02, 4.90000000E+01, 9.83100000E-01, 0.00000000E+00},
    {8.96415500E+02, 1.20000000E+02, 5.00000000E+01, 9.83100000E-01, 0.00000000E+00},
    {7.59798800E+02, 1.20000000E+02, 5.10000000E+01, 9.83100000E-01, 0.00000000E+00},
    {6.84707300E+02, 1.20000000E+02, 5.20000000E+01, 9.83100000E-01, 0.00000000E+00},
    {6.00269800E+02, 1.20000000E+02, 5.30000000E+01, 9.83100000E-01, 0.00000000E+00},
    {5.24354700E+02, 1.20000000E+02, 5.40000000E+01, 9.83100000E-01, 0.00000000E+00},
    {3.61926600E+03, 1.20000000E+02, 5.50000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.95133620E+03, 1.20000000E+02, 5.60000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.45676010E+03, 1.20000000E+02, 5.70000000E+01, 9.83100000E-01, 0.00000000E+00},
    {9.07401200E+02, 1.20000000E+02, 5.80000000E+01, 9.83100000E-01, 2.79910000E+00},
    {2.57153190E+03, 1.20000000E+02, 5.90000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.44204950E+03, 1.20000000E+02, 6.00000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.37404450E+03, 1.20000000E+02, 6.10000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.31221470E+03, 1.20000000E+02, 6.20000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.25719930E+03, 1.20000000E+02, 6.30000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.68063380E+03, 1.20000000E+02, 6.40000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.08161730E+03, 1.20000000E+02, 6.50000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.99287280E+03, 1.20000000E+02, 6.60000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.00629580E+03, 1.20000000E+02, 6.70000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.96023910E+03, 1.20000000E+02, 6.80000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.91746400E+03, 1.20000000E+02, 6.90000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.89913060E+03, 1.20000000E+02, 7.00000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.54468520E+03, 1.20000000E+02, 7.10000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.43515030E+03, 1.20000000E+02, 7.20000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.26875180E+03, 1.20000000E+02, 7.30000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.04932960E+03, 1.20000000E+02, 7.40000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.05299220E+03, 1.20000000E+02, 7.50000000E+01, 9.83100000E-01, 0.00000000E+00},
    {9.28452100E+02, 1.20000000E+02, 7.60000000E+01, 9.83100000E-01, 0.00000000E+00},
    {8.31875200E+02, 1.20000000E+02, 7.70000000E+01, 9.83100000E-01, 0.00000000E+00},
    {6.76990200E+02, 1.20000000E+02, 7.80000000E+01, 9.83100000E-01, 0.00000000E+00},
    {6.27085300E+02, 1.20000000E+02, 7.90000000E+01, 9.83100000E-01, 0.00000000E+00},
    {6.37111200E+02, 1.20000000E+02, 8.00000000E+01, 9.83100000E-01, 0.00000000E+00},
    {1.04305560E+03, 1.20000000E+02, 8.10000000E+01, 9.83100000E-01, 0.00000000E+00},
    {9.74284500E+02, 1.20000000E+02, 8.20000000E+01, 9.83100000E-01, 0.00000000E+00},
    {8.54664000E+02, 1.20000000E+02, 8.30000000E+01, 9.83100000E-01, 0.00000000E+00},
    {7.94449100E+02, 1.20000000E+02, 8.40000000E+01, 9.83100000E-01, 0.00000000E+00},
    {7.11414300E+02, 1.20000000E+02, 8.50000000E+01, 9.83100000E-01, 0.00000000E+00},
    {6.34792100E+02, 1.20000000E+02, 8.60000000E+01, 9.83100000E-01, 0.00000000E+00},
    {3.27125630E+03, 1.20000000E+02, 8.70000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.84094470E+03, 1.20000000E+02, 8.80000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.38676370E+03, 1.20000000E+02, 8.90000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.02962450E+03, 1.20000000E+02, 9.00000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.07611720E+03, 1.20000000E+02, 9.10000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.00669150E+03, 1.20000000E+02, 9.20000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.13522690E+03, 1.20000000E+02, 9.30000000E+01, 9.83100000E-01, 0.00000000E+00},
    {2.05401310E+03, 1.20000000E+02, 9.40000000E+01, 9.83100000E-01, 0.00000000E+00},
    {9.07927000E+01, 1.20000000E+02, 1.01000000E+02, 9.83100000E-01, 0.00000000E+00},
    {3.45612000E+02, 1.20000000E+02, 1.03000000E+02, 9.83100000E-01, 9.86500000E-01},
    {4.34028400E+02, 1.20000000E+02, 1.04000000E+02, 9.83100000E-01, 9.80800000E-01},
    {2.98726400E+02, 1.20000000E+02, 1.05000000E+02, 9.83100000E-01, 9.70600000E-01},
    {2.12895700E+02, 1.20000000E+02, 1.06000000E+02, 9.83100000E-01, 9.86800000E-01},
    {1.39021900E+02, 1.20000000E+02, 1.07000000E+02, 9.83100000E-01, 9.94400000E-01},
    {9.61584000E+01, 1.20000000E+02, 1.08000000E+02, 9.83100000E-01, 9.92500000E-01},
    {6.20580000E+01, 1.20000000E+02, 1.09000000E+02, 9.83100000E-01, 9.98200000E-01},
    {5.16691700E+02, 1.20000000E+02, 1.11000000E+02, 9.83100000E-01, 9.68400000E-01},
    {8.09218200E+02, 1.20000000E+02, 1.12000000E+02, 9.83100000E-01, 9.62800000E-01},
    {7.75083200E+02, 1.20000000E+02, 1.13000000E+02, 9.83100000E-01, 9.64800000E-01},
    {5.76296200E+02, 1.20000000E+02, 1.14000000E+02, 9.83100000E-01, 9.50700000E-01},
    {4.46017000E+02, 1.20000000E+02, 1.15000000E+02, 9.83100000E-01, 9.94700000E-01},
    {3.63195100E+02, 1.20000000E+02, 1.16000000E+02, 9.83100000E-01, 9.94800000E-01},
    {2.85490500E+02, 1.20000000E+02, 1.17000000E+02, 9.83100000E-01, 9.97200000E-01},
    {6.88991100E+02, 1.20000000E+02, 1.19000000E+02, 9.83100000E-01, 9.76700000E-01},
    {1.52518910E+03, 1.20000000E+02, 1.20000000E+02, 9.83100000E-01, 9.83100000E-01},
    {2.99953000E+01, 1.21000000E+02, 1.00000000E+00, 1.86270000E+00, 9.11800000E-01},
    {1.93841000E+01, 1.21000000E+02, 2.00000000E+00, 1.86270000E+00, 0.00000000E+00},
    {5.04826800E+02, 1.21000000E+02, 3.00000000E+00, 1.86270000E+00, 0.00000000E+00},
    {2.82442000E+02, 1.21000000E+02, 4.00000000E+00, 1.86270000E+00, 0.00000000E+00},
    {1.86141900E+02, 1.21000000E+02, 5.00000000E+00, 1.86270000E+00, 0.00000000E+00},
    {1.23637900E+02, 1.21000000E+02, 6.00000000E+00, 1.86270000E+00, 0.00000000E+00},
    {8.53404000E+01, 1.21000000E+02, 7.00000000E+00, 1.86270000E+00, 0.00000000E+00},
    {6.40072000E+01, 1.21000000E+02, 8.00000000E+00, 1.86270000E+00, 0.00000000E+00},
    {4.80936000E+01, 1.21000000E+02, 9.00000000E+00, 1.86270000E+00, 0.00000000E+00},
    {3.67468000E+01, 1.21000000E+02, 1.00000000E+01, 1.86270000E+00, 0.00000000E+00},
    {6.02492900E+02, 1.21000000E+02, 1.10000000E+01, 1.86270000E+00, 0.00000000E+00},
    {4.52760000E+02, 1.21000000E+02, 1.20000000E+01, 1.86270000E+00, 0.00000000E+00},
    {4.12905800E+02, 1.21000000E+02, 1.30000000E+01, 1.86270000E+00, 0.00000000E+00},
    {3.20589600E+02, 1.21000000E+02, 1.40000000E+01, 1.86270000E+00, 0.00000000E+00},
    {2.46735800E+02, 1.21000000E+02, 1.50000000E+01, 1.86270000E+00, 0.00000000E+00},
    {2.02967200E+02, 1.21000000E+02, 1.60000000E+01, 1.86270000E+00, 0.00000000E+00},
    {1.64365500E+02, 1.21000000E+02, 1.70000000E+01, 1.86270000E+00, 0.00000000E+00},
    {1.33440400E+02, 1.21000000E+02, 1.80000000E+01, 1.86270000E+00, 0.00000000E+00},
    {9.88588100E+02, 1.21000000E+02, 1.90000000E+01, 1.86270000E+00, 0.00000000E+00},
    {8.03238000E+02, 1.21000000E+02, 2.00000000E+01, 1.86270000E+00, 0.00000000E+00},
    {6.60678000E+02, 1.21000000E+02, 2.10000000E+01, 1.86270000E+00, 0.00000000E+00},
    {6.35045700E+02, 1.21000000E+02, 2.20000000E+01, 1.86270000E+00, 0.00000000E+00},
    {5.79958400E+02, 1.21000000E+02, 2.30000000E+01, 1.86270000E+00, 0.00000000E+00},
    {4.56202300E+02, 1.21000000E+02, 2.40000000E+01, 1.86270000E+00, 0.00000000E+00},
    {4.97347400E+02, 1.21000000E+02, 2.50000000E+01, 1.86270000E+00, 0.00000000E+00},
    {3.89597400E+02, 1.21000000E+02, 2.60000000E+01, 1.86270000E+00, 0.00000000E+00},
    {4.11057200E+02, 1.21000000E+02, 2.70000000E+01, 1.86270000E+00, 0.00000000E+00},
    {4.24669300E+02, 1.21000000E+02, 2.80000000E+01, 1.86270000E+00, 0.00000000E+00},
    {3.25085000E+02, 1.21000000E+02, 2.90000000E+01, 1.86270000E+00, 0.00000000E+00},
    {3.31231000E+02, 1.21000000E+02, 3.00000000E+01, 1.86270000E+00, 0.00000000E+00},
    {3.93530000E+02, 1.21000000E+02, 3.10000000E+01, 1.86270000E+00, 0.00000000E+00},
    {3.43167900E+02, 1.21000000E+02, 3.20000000E+01, 1.86270000E+00, 0.00000000E+00},
    {2.89558200E+02, 1.21000000E+02, 3.30000000E+01, 1.86270000E+00, 0.00000000E+00},
    {2.58008400E+02, 1.21000000E+02, 3.40000000E+01, 1.86270000E+00, 0.00000000E+00},
    {2.24137500E+02, 1.21000000E+02, 3.50000000E+01, 1.86270000E+00, 0.00000000E+00},
    {1.93586800E+02, 1.21000000E+02, 3.60000000E+01, 1.86270000E+00, 0.00000000E+00},
    {1.10546080E+03, 1.21000000E+02, 3.70000000E+01, 1.86270000E+00, 0.00000000E+00},
    {9.57227100E+02, 1.21000000E+02, 3.80000000E+01, 1.86270000E+00, 0.00000000E+00},
    {8.31667900E+02, 1.21000000E+02, 3.90000000E+01, 1.86270000E+00, 0.00000000E+00},
    {7.43431800E+02, 1.21000000E+02, 4.00000000E+01, 1.86270000E+00, 0.00000000E+00},
    {6.75331400E+02, 1.21000000E+02, 4.10000000E+01, 1.86270000E+00, 0.00000000E+00},
    {5.17734000E+02, 1.21000000E+02, 4.20000000E+01, 1.86270000E+00, 0.00000000E+00},
    {5.79162300E+02, 1.21000000E+02, 4.30000000E+01, 1.86270000E+00, 0.00000000E+00},
    {4.37843200E+02, 1.21000000E+02, 4.40000000E+01, 1.86270000E+00, 0.00000000E+00},
    {4.78959800E+02, 1.21000000E+02, 4.50000000E+01, 1.86270000E+00, 0.00000000E+00},
    {4.43081600E+02, 1.21000000E+02, 4.60000000E+01, 1.86270000E+00, 0.00000000E+00},
    {3.69493300E+02, 1.21000000E+02, 4.70000000E+01, 1.86270000E+00, 0.00000000E+00},
    {3.89185900E+02, 1.21000000E+02, 4.80000000E+01, 1.86270000E+00, 0.00000000E+00},
    {4.92190400E+02, 1.21000000E+02, 4.90000000E+01, 1.86270000E+00, 0.00000000E+00},
    {4.51134000E+02, 1.21000000E+02, 5.00000000E+01, 1.86270000E+00, 0.00000000E+00},
    {3.98315200E+02, 1.21000000E+02, 5.10000000E+01, 1.86270000E+00, 0.00000000E+00},
    {3.67390800E+02, 1.21000000E+02, 5.20000000E+01, 1.86270000E+00, 0.00000000E+00},
    {3.30041800E+02, 1.21000000E+02, 5.30000000E+01, 1.86270000E+00, 0.00000000E+00},
    {2.94870600E+02, 1.21000000E+02, 5.40000000E+01, 1.86270000E+00, 0.00000000E+00},
    {1.34452740E+03, 1.21000000E+02, 5.50000000E+01, 1.86270000E+00, 0.00000000E+00},
    {1.22274260E+03, 1.21000000E+02, 5.60000000E+01, 1.86270000E+00, 0.00000000E+00},
    {1.06581400E+03, 1.21000000E+02, 5.70000000E+01, 1.86270000E+00, 0.00000000E+00},
    {4.71185400E+02, 1.21000000E+02, 5.80000000E+01, 1.86270000E+00, 2.79910000E+00},
    {1.07996470E+03, 1.21000000E+02, 5.90000000E+01, 1.86270000E+00, 0.00000000E+00},
    {1.03620550E+03, 1.21000000E+02, 6.00000000E+01, 1.86270000E+00, 0.00000000E+00},
    {1.00988930E+03, 1.21000000E+02, 6.10000000E+01, 1.86270000E+00, 0.00000000E+00},
    {9.85726400E+02, 1.21000000E+02, 6.20000000E+01, 1.86270000E+00, 0.00000000E+00},
    {9.64285500E+02, 1.21000000E+02, 6.30000000E+01, 1.86270000E+00, 0.00000000E+00},
    {7.50784800E+02, 1.21000000E+02, 6.40000000E+01, 1.86270000E+00, 0.00000000E+00},
    {8.55147100E+02, 1.21000000E+02, 6.50000000E+01, 1.86270000E+00, 0.00000000E+00},
    {8.22842000E+02, 1.21000000E+02, 6.60000000E+01, 1.86270000E+00, 0.00000000E+00},
    {8.68110400E+02, 1.21000000E+02, 6.70000000E+01, 1.86270000E+00, 0.00000000E+00},
    {8.49539400E+02, 1.21000000E+02, 6.80000000E+01, 1.86270000E+00, 0.00000000E+00},
    {8.32696100E+02, 1.21000000E+02, 6.90000000E+01, 1.86270000E+00, 0.00000000E+00},
    {8.23308900E+02, 1.21000000E+02, 7.00000000E+01, 1.86270000E+00, 0.00000000E+00},
    {6.88624200E+02, 1.21000000E+02, 7.10000000E+01, 1.86270000E+00, 0.00000000E+00},
    {6.71925800E+02, 1.21000000E+02, 7.20000000E+01, 1.86270000E+00, 0.00000000E+00},
    {6.09751500E+02, 1.21000000E+02, 7.30000000E+01, 1.86270000E+00, 0.00000000E+00},
    {5.12210500E+02, 1.21000000E+02, 7.40000000E+01, 1.86270000E+00, 0.00000000E+00},
    {5.20057800E+02, 1.21000000E+02, 7.50000000E+01, 1.86270000E+00, 0.00000000E+00},
    {4.68956500E+02, 1.21000000E+02, 7.60000000E+01, 1.86270000E+00, 0.00000000E+00},
    {4.27684500E+02, 1.21000000E+02, 7.70000000E+01, 1.86270000E+00, 0.00000000E+00},
    {3.53484500E+02, 1.21000000E+02, 7.80000000E+01, 1.86270000E+00, 0.00000000E+00},
    {3.29598500E+02, 1.21000000E+02, 7.90000000E+01, 1.86270000E+00, 0.00000000E+00},
    {3.38547800E+02, 1.21000000E+02, 8.00000000E+01, 1.86270000E+00, 0.00000000E+00},
    {5.03639300E+02, 1.21000000E+02, 8.10000000E+01, 1.86270000E+00, 0.00000000E+00},
    {4.89254100E+02, 1.21000000E+02, 8.20000000E+01, 1.86270000E+00, 0.00000000E+00},
    {4.45923100E+02, 1.21000000E+02, 8.30000000E+01, 1.86270000E+00, 0.00000000E+00},
    {4.23206300E+02, 1.21000000E+02, 8.40000000E+01, 1.86270000E+00, 0.00000000E+00},
    {3.88212200E+02, 1.21000000E+02, 8.50000000E+01, 1.86270000E+00, 0.00000000E+00},
    {3.53800500E+02, 1.21000000E+02, 8.60000000E+01, 1.86270000E+00, 0.00000000E+00},
    {1.26301910E+03, 1.21000000E+02, 8.70000000E+01, 1.86270000E+00, 0.00000000E+00},
    {1.20398590E+03, 1.21000000E+02, 8.80000000E+01, 1.86270000E+00, 0.00000000E+00},
    {1.05635410E+03, 1.21000000E+02, 8.90000000E+01, 1.86270000E+00, 0.00000000E+00},
    {9.40624100E+02, 1.21000000E+02, 9.00000000E+01, 1.86270000E+00, 0.00000000E+00},
    {9.37657900E+02, 1.21000000E+02, 9.10000000E+01, 1.86270000E+00, 0.00000000E+00},
    {9.07695600E+02, 1.21000000E+02, 9.20000000E+01, 1.86270000E+00, 0.00000000E+00},
    {9.39869100E+02, 1.21000000E+02, 9.30000000E+01, 1.86270000E+00, 0.00000000E+00},
    {9.09289500E+02, 1.21000000E+02, 9.40000000E+01, 1.86270000E+00, 0.00000000E+00},
    {4.89753000E+01, 1.21000000E+02, 1.01000000E+02, 1.86270000E+00, 0.00000000E+00},
    {1.63644600E+02, 1.21000000E+02, 1.03000000E+02, 1.86270000E+00, 9.86500000E-01},
    {2.07810700E+02, 1.21000000E+02, 1.04000000E+02, 1.86270000E+00, 9.80800000E-01},
    {1.55822800E+02, 1.21000000E+02, 1.05000000E+02, 1.86270000E+00, 9.70600000E-01},
    {1.15972600E+02, 1.21000000E+02, 1.06000000E+02, 1.86270000E+00, 9.86800000E-01},
    {7.95383000E+01, 1.21000000E+02, 1.07000000E+02, 1.86270000E+00, 9.94400000E-01},
    {5.72611000E+01, 1.21000000E+02, 1.08000000E+02, 1.86270000E+00, 9.92500000E-01},
    {3.88097000E+01, 1.21000000E+02, 1.09000000E+02, 1.86270000E+00, 9.98200000E-01},
    {2.39971400E+02, 1.21000000E+02, 1.11000000E+02, 1.86270000E+00, 9.68400000E-01},
    {3.71776800E+02, 1.21000000E+02, 1.12000000E+02, 1.86270000E+00, 9.62800000E-01},
    {3.73184500E+02, 1.21000000E+02, 1.13000000E+02, 1.86270000E+00, 9.64800000E-01},
    {2.95610100E+02, 1.21000000E+02, 1.14000000E+02, 1.86270000E+00, 9.50700000E-01},
    {2.39370100E+02, 1.21000000E+02, 1.15000000E+02, 1.86270000E+00, 9.94700000E-01},
    {2.00762200E+02, 1.21000000E+02, 1.16000000E+02, 1.86270000E+00, 9.94800000E-01},
    {1.62680600E+02, 1.21000000E+02, 1.17000000E+02, 1.86270000E+00, 9.97200000E-01},
    {3.27709200E+02, 1.21000000E+02, 1.19000000E+02, 1.86270000E+00, 9.76700000E-01},
    {6.40707600E+02, 1.21000000E+02, 1.20000000E+02, 1.86270000E+00, 9.83100000E-01},
    {3.24506800E+02, 1.21000000E+02, 1.21000000E+02, 1.86270000E+00, 1.86270000E+00},
    {2.90009000E+01, 1.22000000E+02, 1.00000000E+00, 1.82990000E+00, 9.11800000E-01},
    {1.88084000E+01, 1.22000000E+02, 2.00000000E+00, 1.82990000E+00, 0.00000000E+00},
    {4.88293100E+02, 1.22000000E+02, 3.00000000E+00, 1.82990000E+00, 0.00000000E+00},
    {2.72020100E+02, 1.22000000E+02, 4.00000000E+00, 1.82990000E+00, 0.00000000E+00},
    {1.79507700E+02, 1.22000000E+02, 5.00000000E+00, 1.82990000E+00, 0.00000000E+00},
    {1.19437600E+02, 1.22000000E+02, 6.00000000E+00, 1.82990000E+00, 0.00000000E+00},
    {8.25834000E+01, 1.22000000E+02, 7.00000000E+00, 1.82990000E+00, 0.00000000E+00},
    {6.20309000E+01, 1.22000000E+02, 8.00000000E+00, 1.82990000E+00, 0.00000000E+00},
    {4.66765000E+01, 1.22000000E+02, 9.00000000E+00, 1.82990000E+00, 0.00000000E+00},
    {3.57117000E+01, 1.22000000E+02, 1.00000000E+01, 1.82990000E+00, 0.00000000E+00},
    {5.82527000E+02, 1.22000000E+02, 1.10000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.36109100E+02, 1.22000000E+02, 1.20000000E+01, 1.82990000E+00, 0.00000000E+00},
    {3.97861800E+02, 1.22000000E+02, 1.30000000E+01, 1.82990000E+00, 0.00000000E+00},
    {3.09108100E+02, 1.22000000E+02, 1.40000000E+01, 1.82990000E+00, 0.00000000E+00},
    {2.38164400E+02, 1.22000000E+02, 1.50000000E+01, 1.82990000E+00, 0.00000000E+00},
    {1.96112800E+02, 1.22000000E+02, 1.60000000E+01, 1.82990000E+00, 0.00000000E+00},
    {1.58989200E+02, 1.22000000E+02, 1.70000000E+01, 1.82990000E+00, 0.00000000E+00},
    {1.29218000E+02, 1.22000000E+02, 1.80000000E+01, 1.82990000E+00, 0.00000000E+00},
    {9.60337600E+02, 1.22000000E+02, 1.90000000E+01, 1.82990000E+00, 0.00000000E+00},
    {7.74686300E+02, 1.22000000E+02, 2.00000000E+01, 1.82990000E+00, 0.00000000E+00},
    {6.36870600E+02, 1.22000000E+02, 2.10000000E+01, 1.82990000E+00, 0.00000000E+00},
    {6.12222500E+02, 1.22000000E+02, 2.20000000E+01, 1.82990000E+00, 0.00000000E+00},
    {5.59115700E+02, 1.22000000E+02, 2.30000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.40076200E+02, 1.22000000E+02, 2.40000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.79496900E+02, 1.22000000E+02, 2.50000000E+01, 1.82990000E+00, 0.00000000E+00},
    {3.75824200E+02, 1.22000000E+02, 2.60000000E+01, 1.82990000E+00, 0.00000000E+00},
    {3.96274800E+02, 1.22000000E+02, 2.70000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.09355500E+02, 1.22000000E+02, 2.80000000E+01, 1.82990000E+00, 0.00000000E+00},
    {3.13602600E+02, 1.22000000E+02, 2.90000000E+01, 1.82990000E+00, 0.00000000E+00},
    {3.19383400E+02, 1.22000000E+02, 3.00000000E+01, 1.82990000E+00, 0.00000000E+00},
    {3.79410900E+02, 1.22000000E+02, 3.10000000E+01, 1.82990000E+00, 0.00000000E+00},
    {3.30971000E+02, 1.22000000E+02, 3.20000000E+01, 1.82990000E+00, 0.00000000E+00},
    {2.79499000E+02, 1.22000000E+02, 3.30000000E+01, 1.82990000E+00, 0.00000000E+00},
    {2.49225800E+02, 1.22000000E+02, 3.40000000E+01, 1.82990000E+00, 0.00000000E+00},
    {2.16692100E+02, 1.22000000E+02, 3.50000000E+01, 1.82990000E+00, 0.00000000E+00},
    {1.87323100E+02, 1.22000000E+02, 3.60000000E+01, 1.82990000E+00, 0.00000000E+00},
    {1.07419730E+03, 1.22000000E+02, 3.70000000E+01, 1.82990000E+00, 0.00000000E+00},
    {9.23655700E+02, 1.22000000E+02, 3.80000000E+01, 1.82990000E+00, 0.00000000E+00},
    {8.02032400E+02, 1.22000000E+02, 3.90000000E+01, 1.82990000E+00, 0.00000000E+00},
    {7.16860900E+02, 1.22000000E+02, 4.00000000E+01, 1.82990000E+00, 0.00000000E+00},
    {6.51284700E+02, 1.22000000E+02, 4.10000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.99501100E+02, 1.22000000E+02, 4.20000000E+01, 1.82990000E+00, 0.00000000E+00},
    {5.58721100E+02, 1.22000000E+02, 4.30000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.22570800E+02, 1.22000000E+02, 4.40000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.62031400E+02, 1.22000000E+02, 4.50000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.27457700E+02, 1.22000000E+02, 4.60000000E+01, 1.82990000E+00, 0.00000000E+00},
    {3.56724500E+02, 1.22000000E+02, 4.70000000E+01, 1.82990000E+00, 0.00000000E+00},
    {3.75526100E+02, 1.22000000E+02, 4.80000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.74811200E+02, 1.22000000E+02, 4.90000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.35194800E+02, 1.22000000E+02, 5.00000000E+01, 1.82990000E+00, 0.00000000E+00},
    {3.84448700E+02, 1.22000000E+02, 5.10000000E+01, 1.82990000E+00, 0.00000000E+00},
    {3.54779300E+02, 1.22000000E+02, 5.20000000E+01, 1.82990000E+00, 0.00000000E+00},
    {3.18917100E+02, 1.22000000E+02, 5.30000000E+01, 1.82990000E+00, 0.00000000E+00},
    {2.85132500E+02, 1.22000000E+02, 5.40000000E+01, 1.82990000E+00, 0.00000000E+00},
    {1.30864180E+03, 1.22000000E+02, 5.50000000E+01, 1.82990000E+00, 0.00000000E+00},
    {1.18094240E+03, 1.22000000E+02, 5.60000000E+01, 1.82990000E+00, 0.00000000E+00},
    {1.02841690E+03, 1.22000000E+02, 5.70000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.54820000E+02, 1.22000000E+02, 5.80000000E+01, 1.82990000E+00, 2.79910000E+00},
    {1.04331870E+03, 1.22000000E+02, 5.90000000E+01, 1.82990000E+00, 0.00000000E+00},
    {1.00021550E+03, 1.22000000E+02, 6.00000000E+01, 1.82990000E+00, 0.00000000E+00},
    {9.74722700E+02, 1.22000000E+02, 6.10000000E+01, 1.82990000E+00, 0.00000000E+00},
    {9.51325700E+02, 1.22000000E+02, 6.20000000E+01, 1.82990000E+00, 0.00000000E+00},
    {9.30565400E+02, 1.22000000E+02, 6.30000000E+01, 1.82990000E+00, 0.00000000E+00},
    {7.24588600E+02, 1.22000000E+02, 6.40000000E+01, 1.82990000E+00, 0.00000000E+00},
    {8.27797000E+02, 1.22000000E+02, 6.50000000E+01, 1.82990000E+00, 0.00000000E+00},
    {7.97236400E+02, 1.22000000E+02, 6.60000000E+01, 1.82990000E+00, 0.00000000E+00},
    {8.37461800E+02, 1.22000000E+02, 6.70000000E+01, 1.82990000E+00, 0.00000000E+00},
    {8.19497300E+02, 1.22000000E+02, 6.80000000E+01, 1.82990000E+00, 0.00000000E+00},
    {8.03203200E+02, 1.22000000E+02, 6.90000000E+01, 1.82990000E+00, 0.00000000E+00},
    {7.94140400E+02, 1.22000000E+02, 7.00000000E+01, 1.82990000E+00, 0.00000000E+00},
    {6.64772700E+02, 1.22000000E+02, 7.10000000E+01, 1.82990000E+00, 0.00000000E+00},
    {6.47874600E+02, 1.22000000E+02, 7.20000000E+01, 1.82990000E+00, 0.00000000E+00},
    {5.87981300E+02, 1.22000000E+02, 7.30000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.94235700E+02, 1.22000000E+02, 7.40000000E+01, 1.82990000E+00, 0.00000000E+00},
    {5.01773300E+02, 1.22000000E+02, 7.50000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.52572600E+02, 1.22000000E+02, 7.60000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.12855800E+02, 1.22000000E+02, 7.70000000E+01, 1.82990000E+00, 0.00000000E+00},
    {3.41454000E+02, 1.22000000E+02, 7.80000000E+01, 1.82990000E+00, 0.00000000E+00},
    {3.18461900E+02, 1.22000000E+02, 7.90000000E+01, 1.82990000E+00, 0.00000000E+00},
    {3.27077000E+02, 1.22000000E+02, 8.00000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.86211600E+02, 1.22000000E+02, 8.10000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.72155200E+02, 1.22000000E+02, 8.20000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.30480500E+02, 1.22000000E+02, 8.30000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.08695100E+02, 1.22000000E+02, 8.40000000E+01, 1.82990000E+00, 0.00000000E+00},
    {3.75102900E+02, 1.22000000E+02, 8.50000000E+01, 1.82990000E+00, 0.00000000E+00},
    {3.42057500E+02, 1.22000000E+02, 8.60000000E+01, 1.82990000E+00, 0.00000000E+00},
    {1.22639700E+03, 1.22000000E+02, 8.70000000E+01, 1.82990000E+00, 0.00000000E+00},
    {1.16236600E+03, 1.22000000E+02, 8.80000000E+01, 1.82990000E+00, 0.00000000E+00},
    {1.01914220E+03, 1.22000000E+02, 8.90000000E+01, 1.82990000E+00, 0.00000000E+00},
    {9.07236700E+02, 1.22000000E+02, 9.00000000E+01, 1.82990000E+00, 0.00000000E+00},
    {9.04833100E+02, 1.22000000E+02, 9.10000000E+01, 1.82990000E+00, 0.00000000E+00},
    {8.75874800E+02, 1.22000000E+02, 9.20000000E+01, 1.82990000E+00, 0.00000000E+00},
    {9.06961600E+02, 1.22000000E+02, 9.30000000E+01, 1.82990000E+00, 0.00000000E+00},
    {8.77364500E+02, 1.22000000E+02, 9.40000000E+01, 1.82990000E+00, 0.00000000E+00},
    {4.72755000E+01, 1.22000000E+02, 1.01000000E+02, 1.82990000E+00, 0.00000000E+00},
    {1.57666700E+02, 1.22000000E+02, 1.03000000E+02, 1.82990000E+00, 9.86500000E-01},
    {2.00366900E+02, 1.22000000E+02, 1.04000000E+02, 1.82990000E+00, 9.80800000E-01},
    {1.50352000E+02, 1.22000000E+02, 1.05000000E+02, 1.82990000E+00, 9.70600000E-01},
    {1.12052300E+02, 1.22000000E+02, 1.06000000E+02, 1.82990000E+00, 9.86800000E-01},
    {7.69834000E+01, 1.22000000E+02, 1.07000000E+02, 1.82990000E+00, 9.94400000E-01},
    {5.55160000E+01, 1.22000000E+02, 1.08000000E+02, 1.82990000E+00, 9.92500000E-01},
    {3.77222000E+01, 1.22000000E+02, 1.09000000E+02, 1.82990000E+00, 9.98200000E-01},
    {2.31261200E+02, 1.22000000E+02, 1.11000000E+02, 1.82990000E+00, 9.68400000E-01},
    {3.58360200E+02, 1.22000000E+02, 1.12000000E+02, 1.82990000E+00, 9.62800000E-01},
    {3.59649700E+02, 1.22000000E+02, 1.13000000E+02, 1.82990000E+00, 9.64800000E-01},
    {2.85095700E+02, 1.22000000E+02, 1.14000000E+02, 1.82990000E+00, 9.50700000E-01},
    {2.31077300E+02, 1.22000000E+02, 1.15000000E+02, 1.82990000E+00, 9.94700000E-01},
    {1.93984500E+02, 1.22000000E+02, 1.16000000E+02, 1.82990000E+00, 9.94800000E-01},
    {1.57360400E+02, 1.22000000E+02, 1.17000000E+02, 1.82990000E+00, 9.97200000E-01},
    {3.16251800E+02, 1.22000000E+02, 1.19000000E+02, 1.82990000E+00, 9.76700000E-01},
    {6.19430100E+02, 1.22000000E+02, 1.20000000E+02, 1.82990000E+00, 9.83100000E-01},
    {3.12920200E+02, 1.22000000E+02, 1.21000000E+02, 1.82990000E+00, 1.86270000E+00},
    {3.01998800E+02, 1.22000000E+02, 1.22000000E+02, 1.82990000E+00, 1.82990000E+00},
    {2.84067000E+01, 1.23000000E+02, 1.00000000E+00, 1.91380000E+00, 9.11800000E-01},
    {1.84359000E+01, 1.23000000E+02, 2.00000000E+00, 1.91380000E+00, 0.00000000E+00},
    {4.79140200E+02, 1.23000000E+02, 3.00000000E+00, 1.91380000E+00, 0.00000000E+00},
    {2.66705000E+02, 1.23000000E+02, 4.00000000E+00, 1.91380000E+00, 0.00000000E+00},
    {1.75890900E+02, 1.23000000E+02, 5.00000000E+00, 1.91380000E+00, 0.00000000E+00},
    {1.17009800E+02, 1.23000000E+02, 6.00000000E+00, 1.91380000E+00, 0.00000000E+00},
    {8.09144000E+01, 1.23000000E+02, 7.00000000E+00, 1.91380000E+00, 0.00000000E+00},
    {6.07933000E+01, 1.23000000E+02, 8.00000000E+00, 1.91380000E+00, 0.00000000E+00},
    {4.57630000E+01, 1.23000000E+02, 9.00000000E+00, 1.91380000E+00, 0.00000000E+00},
    {3.50290000E+01, 1.23000000E+02, 1.00000000E+01, 1.91380000E+00, 0.00000000E+00},
    {5.71645500E+02, 1.23000000E+02, 1.10000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.27715600E+02, 1.23000000E+02, 1.20000000E+01, 1.91380000E+00, 0.00000000E+00},
    {3.90043200E+02, 1.23000000E+02, 1.30000000E+01, 1.91380000E+00, 0.00000000E+00},
    {3.02898500E+02, 1.23000000E+02, 1.40000000E+01, 1.91380000E+00, 0.00000000E+00},
    {2.33317600E+02, 1.23000000E+02, 1.50000000E+01, 1.91380000E+00, 0.00000000E+00},
    {1.92111300E+02, 1.23000000E+02, 1.60000000E+01, 1.91380000E+00, 0.00000000E+00},
    {1.55750700E+02, 1.23000000E+02, 1.70000000E+01, 1.91380000E+00, 0.00000000E+00},
    {1.26602700E+02, 1.23000000E+02, 1.80000000E+01, 1.91380000E+00, 0.00000000E+00},
    {9.41983700E+02, 1.23000000E+02, 1.90000000E+01, 1.91380000E+00, 0.00000000E+00},
    {7.60071800E+02, 1.23000000E+02, 2.00000000E+01, 1.91380000E+00, 0.00000000E+00},
    {6.24780100E+02, 1.23000000E+02, 2.10000000E+01, 1.91380000E+00, 0.00000000E+00},
    {6.00522100E+02, 1.23000000E+02, 2.20000000E+01, 1.91380000E+00, 0.00000000E+00},
    {5.48390800E+02, 1.23000000E+02, 2.30000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.31668300E+02, 1.23000000E+02, 2.40000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.70252900E+02, 1.23000000E+02, 2.50000000E+01, 1.91380000E+00, 0.00000000E+00},
    {3.68613600E+02, 1.23000000E+02, 2.60000000E+01, 1.91380000E+00, 0.00000000E+00},
    {3.88572900E+02, 1.23000000E+02, 2.70000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.01432900E+02, 1.23000000E+02, 2.80000000E+01, 1.91380000E+00, 0.00000000E+00},
    {3.07576300E+02, 1.23000000E+02, 2.90000000E+01, 1.91380000E+00, 0.00000000E+00},
    {3.13128600E+02, 1.23000000E+02, 3.00000000E+01, 1.91380000E+00, 0.00000000E+00},
    {3.71942700E+02, 1.23000000E+02, 3.10000000E+01, 1.91380000E+00, 0.00000000E+00},
    {3.24343500E+02, 1.23000000E+02, 3.20000000E+01, 1.91380000E+00, 0.00000000E+00},
    {2.73835200E+02, 1.23000000E+02, 3.30000000E+01, 1.91380000E+00, 0.00000000E+00},
    {2.44155400E+02, 1.23000000E+02, 3.40000000E+01, 1.91380000E+00, 0.00000000E+00},
    {2.12279200E+02, 1.23000000E+02, 3.50000000E+01, 1.91380000E+00, 0.00000000E+00},
    {1.83517100E+02, 1.23000000E+02, 3.60000000E+01, 1.91380000E+00, 0.00000000E+00},
    {1.05358340E+03, 1.23000000E+02, 3.70000000E+01, 1.91380000E+00, 0.00000000E+00},
    {9.06236400E+02, 1.23000000E+02, 3.80000000E+01, 1.91380000E+00, 0.00000000E+00},
    {7.86725500E+02, 1.23000000E+02, 3.90000000E+01, 1.91380000E+00, 0.00000000E+00},
    {7.03078400E+02, 1.23000000E+02, 4.00000000E+01, 1.91380000E+00, 0.00000000E+00},
    {6.38701400E+02, 1.23000000E+02, 4.10000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.89816000E+02, 1.23000000E+02, 4.20000000E+01, 1.91380000E+00, 0.00000000E+00},
    {5.47890600E+02, 1.23000000E+02, 4.30000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.14357500E+02, 1.23000000E+02, 4.40000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.53026200E+02, 1.23000000E+02, 4.50000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.19113400E+02, 1.23000000E+02, 4.60000000E+01, 1.91380000E+00, 0.00000000E+00},
    {3.49828800E+02, 1.23000000E+02, 4.70000000E+01, 1.91380000E+00, 0.00000000E+00},
    {3.68187100E+02, 1.23000000E+02, 4.80000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.65564700E+02, 1.23000000E+02, 4.90000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.26593300E+02, 1.23000000E+02, 5.00000000E+01, 1.91380000E+00, 0.00000000E+00},
    {3.76754400E+02, 1.23000000E+02, 5.10000000E+01, 1.91380000E+00, 0.00000000E+00},
    {3.47637400E+02, 1.23000000E+02, 5.20000000E+01, 1.91380000E+00, 0.00000000E+00},
    {3.12471400E+02, 1.23000000E+02, 5.30000000E+01, 1.91380000E+00, 0.00000000E+00},
    {2.79361200E+02, 1.23000000E+02, 5.40000000E+01, 1.91380000E+00, 0.00000000E+00},
    {1.28311930E+03, 1.23000000E+02, 5.50000000E+01, 1.91380000E+00, 0.00000000E+00},
    {1.15866620E+03, 1.23000000E+02, 5.60000000E+01, 1.91380000E+00, 0.00000000E+00},
    {1.00883360E+03, 1.23000000E+02, 5.70000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.45757000E+02, 1.23000000E+02, 5.80000000E+01, 1.91380000E+00, 2.79910000E+00},
    {1.02353950E+03, 1.23000000E+02, 5.90000000E+01, 1.91380000E+00, 0.00000000E+00},
    {9.81345700E+02, 1.23000000E+02, 6.00000000E+01, 1.91380000E+00, 0.00000000E+00},
    {9.56332100E+02, 1.23000000E+02, 6.10000000E+01, 1.91380000E+00, 0.00000000E+00},
    {9.33373800E+02, 1.23000000E+02, 6.20000000E+01, 1.91380000E+00, 0.00000000E+00},
    {9.13001400E+02, 1.23000000E+02, 6.30000000E+01, 1.91380000E+00, 0.00000000E+00},
    {7.10723700E+02, 1.23000000E+02, 6.40000000E+01, 1.91380000E+00, 0.00000000E+00},
    {8.11990500E+02, 1.23000000E+02, 6.50000000E+01, 1.91380000E+00, 0.00000000E+00},
    {7.81827200E+02, 1.23000000E+02, 6.60000000E+01, 1.91380000E+00, 0.00000000E+00},
    {8.21627400E+02, 1.23000000E+02, 6.70000000E+01, 1.91380000E+00, 0.00000000E+00},
    {8.03998900E+02, 1.23000000E+02, 6.80000000E+01, 1.91380000E+00, 0.00000000E+00},
    {7.88006400E+02, 1.23000000E+02, 6.90000000E+01, 1.91380000E+00, 0.00000000E+00},
    {7.79121900E+02, 1.23000000E+02, 7.00000000E+01, 1.91380000E+00, 0.00000000E+00},
    {6.51979400E+02, 1.23000000E+02, 7.10000000E+01, 1.91380000E+00, 0.00000000E+00},
    {6.35299600E+02, 1.23000000E+02, 7.20000000E+01, 1.91380000E+00, 0.00000000E+00},
    {5.76494600E+02, 1.23000000E+02, 7.30000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.84558300E+02, 1.23000000E+02, 7.40000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.91912000E+02, 1.23000000E+02, 7.50000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.43641300E+02, 1.23000000E+02, 7.60000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.04686400E+02, 1.23000000E+02, 7.70000000E+01, 1.91380000E+00, 0.00000000E+00},
    {3.34715300E+02, 1.23000000E+02, 7.80000000E+01, 1.91380000E+00, 0.00000000E+00},
    {3.12187300E+02, 1.23000000E+02, 7.90000000E+01, 1.91380000E+00, 0.00000000E+00},
    {3.20601200E+02, 1.23000000E+02, 8.00000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.76759600E+02, 1.23000000E+02, 8.10000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.62874100E+02, 1.23000000E+02, 8.20000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.21918100E+02, 1.23000000E+02, 8.30000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.00517200E+02, 1.23000000E+02, 8.40000000E+01, 1.91380000E+00, 0.00000000E+00},
    {3.67558800E+02, 1.23000000E+02, 8.50000000E+01, 1.91380000E+00, 0.00000000E+00},
    {3.35161000E+02, 1.23000000E+02, 8.60000000E+01, 1.91380000E+00, 0.00000000E+00},
    {1.20270240E+03, 1.23000000E+02, 8.70000000E+01, 1.91380000E+00, 0.00000000E+00},
    {1.14034300E+03, 1.23000000E+02, 8.80000000E+01, 1.91380000E+00, 0.00000000E+00},
    {9.99655200E+02, 1.23000000E+02, 8.90000000E+01, 1.91380000E+00, 0.00000000E+00},
    {8.89725900E+02, 1.23000000E+02, 9.00000000E+01, 1.91380000E+00, 0.00000000E+00},
    {8.87488000E+02, 1.23000000E+02, 9.10000000E+01, 1.91380000E+00, 0.00000000E+00},
    {8.59099700E+02, 1.23000000E+02, 9.20000000E+01, 1.91380000E+00, 0.00000000E+00},
    {8.89771900E+02, 1.23000000E+02, 9.30000000E+01, 1.91380000E+00, 0.00000000E+00},
    {8.60717800E+02, 1.23000000E+02, 9.40000000E+01, 1.91380000E+00, 0.00000000E+00},
    {4.63089000E+01, 1.23000000E+02, 1.01000000E+02, 1.91380000E+00, 0.00000000E+00},
    {1.54576200E+02, 1.23000000E+02, 1.03000000E+02, 1.91380000E+00, 9.86500000E-01},
    {1.96402700E+02, 1.23000000E+02, 1.04000000E+02, 1.91380000E+00, 9.80800000E-01},
    {1.47315300E+02, 1.23000000E+02, 1.05000000E+02, 1.91380000E+00, 9.70600000E-01},
    {1.09779700E+02, 1.23000000E+02, 1.06000000E+02, 1.91380000E+00, 9.86800000E-01},
    {7.54303000E+01, 1.23000000E+02, 1.07000000E+02, 1.91380000E+00, 9.94400000E-01},
    {5.44109000E+01, 1.23000000E+02, 1.08000000E+02, 1.91380000E+00, 9.92500000E-01},
    {3.69962000E+01, 1.23000000E+02, 1.09000000E+02, 1.91380000E+00, 9.98200000E-01},
    {2.26788600E+02, 1.23000000E+02, 1.11000000E+02, 1.91380000E+00, 9.68400000E-01},
    {3.51392800E+02, 1.23000000E+02, 1.12000000E+02, 1.91380000E+00, 9.62800000E-01},
    {3.52557500E+02, 1.23000000E+02, 1.13000000E+02, 1.91380000E+00, 9.64800000E-01},
    {2.79356300E+02, 1.23000000E+02, 1.14000000E+02, 1.91380000E+00, 9.50700000E-01},
    {2.26376700E+02, 1.23000000E+02, 1.15000000E+02, 1.91380000E+00, 9.94700000E-01},
    {1.90029900E+02, 1.23000000E+02, 1.16000000E+02, 1.91380000E+00, 9.94800000E-01},
    {1.54157600E+02, 1.23000000E+02, 1.17000000E+02, 1.91380000E+00, 9.97200000E-01},
    {3.10110800E+02, 1.23000000E+02, 1.19000000E+02, 1.91380000E+00, 9.76700000E-01},
    {6.07557100E+02, 1.23000000E+02, 1.20000000E+02, 1.91380000E+00, 9.83100000E-01},
    {3.06738900E+02, 1.23000000E+02, 1.21000000E+02, 1.91380000E+00, 1.86270000E+00},
    {2.96002900E+02, 1.23000000E+02, 1.22000000E+02, 1.91380000E+00, 1.82990000E+00},
    {2.90143600E+02, 1.23000000E+02, 1.23000000E+02, 1.91380000E+00, 1.91380000E+00},
    {2.80562000E+01, 1.24000000E+02, 1.00000000E+00, 1.82690000E+00, 9.11800000E-01},
    {1.81754000E+01, 1.24000000E+02, 2.00000000E+00, 1.82690000E+00, 0.00000000E+00},
    {4.80594400E+02, 1.24000000E+02, 3.00000000E+00, 1.82690000E+00, 0.00000000E+00},
    {2.65408800E+02, 1.24000000E+02, 4.00000000E+00, 1.82690000E+00, 0.00000000E+00},
    {1.74371900E+02, 1.24000000E+02, 5.00000000E+00, 1.82690000E+00, 0.00000000E+00},
    {1.15718200E+02, 1.24000000E+02, 6.00000000E+00, 1.82690000E+00, 0.00000000E+00},
    {7.99024000E+01, 1.24000000E+02, 7.00000000E+00, 1.82690000E+00, 0.00000000E+00},
    {5.99829000E+01, 1.24000000E+02, 8.00000000E+00, 1.82690000E+00, 0.00000000E+00},
    {4.51290000E+01, 1.24000000E+02, 9.00000000E+00, 1.82690000E+00, 0.00000000E+00},
    {3.45350000E+01, 1.24000000E+02, 1.00000000E+01, 1.82690000E+00, 0.00000000E+00},
    {5.73143600E+02, 1.24000000E+02, 1.10000000E+01, 1.82690000E+00, 0.00000000E+00},
    {4.26227600E+02, 1.24000000E+02, 1.20000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.87869500E+02, 1.24000000E+02, 1.30000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.00392800E+02, 1.24000000E+02, 1.40000000E+01, 1.82690000E+00, 0.00000000E+00},
    {2.30909000E+02, 1.24000000E+02, 1.50000000E+01, 1.82690000E+00, 0.00000000E+00},
    {1.89905200E+02, 1.24000000E+02, 1.60000000E+01, 1.82690000E+00, 0.00000000E+00},
    {1.53805700E+02, 1.24000000E+02, 1.70000000E+01, 1.82690000E+00, 0.00000000E+00},
    {1.24926500E+02, 1.24000000E+02, 1.80000000E+01, 1.82690000E+00, 0.00000000E+00},
    {9.45870000E+02, 1.24000000E+02, 1.90000000E+01, 1.82690000E+00, 0.00000000E+00},
    {7.59368400E+02, 1.24000000E+02, 2.00000000E+01, 1.82690000E+00, 0.00000000E+00},
    {6.23533200E+02, 1.24000000E+02, 2.10000000E+01, 1.82690000E+00, 0.00000000E+00},
    {5.98798100E+02, 1.24000000E+02, 2.20000000E+01, 1.82690000E+00, 0.00000000E+00},
    {5.46524400E+02, 1.24000000E+02, 2.30000000E+01, 1.82690000E+00, 0.00000000E+00},
    {4.30235300E+02, 1.24000000E+02, 2.40000000E+01, 1.82690000E+00, 0.00000000E+00},
    {4.68296100E+02, 1.24000000E+02, 2.50000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.67078500E+02, 1.24000000E+02, 2.60000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.86448600E+02, 1.24000000E+02, 2.70000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.99453000E+02, 1.24000000E+02, 2.80000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.06118100E+02, 1.24000000E+02, 2.90000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.11060300E+02, 1.24000000E+02, 3.00000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.69630200E+02, 1.24000000E+02, 3.10000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.21640400E+02, 1.24000000E+02, 3.20000000E+01, 1.82690000E+00, 0.00000000E+00},
    {2.71077000E+02, 1.24000000E+02, 3.30000000E+01, 1.82690000E+00, 0.00000000E+00},
    {2.41458500E+02, 1.24000000E+02, 3.40000000E+01, 1.82690000E+00, 0.00000000E+00},
    {2.09739300E+02, 1.24000000E+02, 3.50000000E+01, 1.82690000E+00, 0.00000000E+00},
    {1.81181900E+02, 1.24000000E+02, 3.60000000E+01, 1.82690000E+00, 0.00000000E+00},
    {1.05757180E+03, 1.24000000E+02, 3.70000000E+01, 1.82690000E+00, 0.00000000E+00},
    {9.05618700E+02, 1.24000000E+02, 3.80000000E+01, 1.82690000E+00, 0.00000000E+00},
    {7.84748900E+02, 1.24000000E+02, 3.90000000E+01, 1.82690000E+00, 0.00000000E+00},
    {7.00543200E+02, 1.24000000E+02, 4.00000000E+01, 1.82690000E+00, 0.00000000E+00},
    {6.35963900E+02, 1.24000000E+02, 4.10000000E+01, 1.82690000E+00, 0.00000000E+00},
    {4.87191100E+02, 1.24000000E+02, 4.20000000E+01, 1.82690000E+00, 0.00000000E+00},
    {5.45172400E+02, 1.24000000E+02, 4.30000000E+01, 1.82690000E+00, 0.00000000E+00},
    {4.11818700E+02, 1.24000000E+02, 4.40000000E+01, 1.82690000E+00, 0.00000000E+00},
    {4.50194300E+02, 1.24000000E+02, 4.50000000E+01, 1.82690000E+00, 0.00000000E+00},
    {4.16323000E+02, 1.24000000E+02, 4.60000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.47678600E+02, 1.24000000E+02, 4.70000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.65553100E+02, 1.24000000E+02, 4.80000000E+01, 1.82690000E+00, 0.00000000E+00},
    {4.62830400E+02, 1.24000000E+02, 4.90000000E+01, 1.82690000E+00, 0.00000000E+00},
    {4.23320800E+02, 1.24000000E+02, 5.00000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.73259100E+02, 1.24000000E+02, 5.10000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.44098200E+02, 1.24000000E+02, 5.20000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.09006300E+02, 1.24000000E+02, 5.30000000E+01, 1.82690000E+00, 0.00000000E+00},
    {2.76040600E+02, 1.24000000E+02, 5.40000000E+01, 1.82690000E+00, 0.00000000E+00},
    {1.28796160E+03, 1.24000000E+02, 5.50000000E+01, 1.82690000E+00, 0.00000000E+00},
    {1.15866720E+03, 1.24000000E+02, 5.60000000E+01, 1.82690000E+00, 0.00000000E+00},
    {1.00696360E+03, 1.24000000E+02, 5.70000000E+01, 1.82690000E+00, 0.00000000E+00},
    {4.41780900E+02, 1.24000000E+02, 5.80000000E+01, 1.82690000E+00, 2.79910000E+00},
    {1.02297730E+03, 1.24000000E+02, 5.90000000E+01, 1.82690000E+00, 0.00000000E+00},
    {9.80482900E+02, 1.24000000E+02, 6.00000000E+01, 1.82690000E+00, 0.00000000E+00},
    {9.55404200E+02, 1.24000000E+02, 6.10000000E+01, 1.82690000E+00, 0.00000000E+00},
    {9.32394600E+02, 1.24000000E+02, 6.20000000E+01, 1.82690000E+00, 0.00000000E+00},
    {9.11973500E+02, 1.24000000E+02, 6.30000000E+01, 1.82690000E+00, 0.00000000E+00},
    {7.08576900E+02, 1.24000000E+02, 6.40000000E+01, 1.82690000E+00, 0.00000000E+00},
    {8.12049500E+02, 1.24000000E+02, 6.50000000E+01, 1.82690000E+00, 0.00000000E+00},
    {7.81626000E+02, 1.24000000E+02, 6.60000000E+01, 1.82690000E+00, 0.00000000E+00},
    {8.20307100E+02, 1.24000000E+02, 6.70000000E+01, 1.82690000E+00, 0.00000000E+00},
    {8.02660900E+02, 1.24000000E+02, 6.80000000E+01, 1.82690000E+00, 0.00000000E+00},
    {7.86633600E+02, 1.24000000E+02, 6.90000000E+01, 1.82690000E+00, 0.00000000E+00},
    {7.77823200E+02, 1.24000000E+02, 7.00000000E+01, 1.82690000E+00, 0.00000000E+00},
    {6.50057200E+02, 1.24000000E+02, 7.10000000E+01, 1.82690000E+00, 0.00000000E+00},
    {6.32209900E+02, 1.24000000E+02, 7.20000000E+01, 1.82690000E+00, 0.00000000E+00},
    {5.73094500E+02, 1.24000000E+02, 7.30000000E+01, 1.82690000E+00, 0.00000000E+00},
    {4.81381400E+02, 1.24000000E+02, 7.40000000E+01, 1.82690000E+00, 0.00000000E+00},
    {4.88471900E+02, 1.24000000E+02, 7.50000000E+01, 1.82690000E+00, 0.00000000E+00},
    {4.40165500E+02, 1.24000000E+02, 7.60000000E+01, 1.82690000E+00, 0.00000000E+00},
    {4.01252200E+02, 1.24000000E+02, 7.70000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.31696500E+02, 1.24000000E+02, 7.80000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.09307500E+02, 1.24000000E+02, 7.90000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.17514900E+02, 1.24000000E+02, 8.00000000E+01, 1.82690000E+00, 0.00000000E+00},
    {4.73849600E+02, 1.24000000E+02, 8.10000000E+01, 1.82690000E+00, 0.00000000E+00},
    {4.59374300E+02, 1.24000000E+02, 8.20000000E+01, 1.82690000E+00, 0.00000000E+00},
    {4.18104400E+02, 1.24000000E+02, 8.30000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.96573600E+02, 1.24000000E+02, 8.40000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.63606800E+02, 1.24000000E+02, 8.50000000E+01, 1.82690000E+00, 0.00000000E+00},
    {3.31301400E+02, 1.24000000E+02, 8.60000000E+01, 1.82690000E+00, 0.00000000E+00},
    {1.20544360E+03, 1.24000000E+02, 8.70000000E+01, 1.82690000E+00, 0.00000000E+00},
    {1.13927610E+03, 1.24000000E+02, 8.80000000E+01, 1.82690000E+00, 0.00000000E+00},
    {9.97024900E+02, 1.24000000E+02, 8.90000000E+01, 1.82690000E+00, 0.00000000E+00},
    {8.85766100E+02, 1.24000000E+02, 9.00000000E+01, 1.82690000E+00, 0.00000000E+00},
    {8.84428000E+02, 1.24000000E+02, 9.10000000E+01, 1.82690000E+00, 0.00000000E+00},
    {8.56103900E+02, 1.24000000E+02, 9.20000000E+01, 1.82690000E+00, 0.00000000E+00},
    {8.87706600E+02, 1.24000000E+02, 9.30000000E+01, 1.82690000E+00, 0.00000000E+00},
    {8.58530200E+02, 1.24000000E+02, 9.40000000E+01, 1.82690000E+00, 0.00000000E+00},
    {4.58173000E+01, 1.24000000E+02, 1.01000000E+02, 1.82690000E+00, 0.00000000E+00},
    {1.53753900E+02, 1.24000000E+02, 1.03000000E+02, 1.82690000E+00, 9.86500000E-01},
    {1.95231700E+02, 1.24000000E+02, 1.04000000E+02, 1.82690000E+00, 9.80800000E-01},
    {1.45947700E+02, 1.24000000E+02, 1.05000000E+02, 1.82690000E+00, 9.70600000E-01},
    {1.08582000E+02, 1.24000000E+02, 1.06000000E+02, 1.82690000E+00, 9.86800000E-01},
    {7.44890000E+01, 1.24000000E+02, 1.07000000E+02, 1.82690000E+00, 9.94400000E-01},
    {5.36733000E+01, 1.24000000E+02, 1.08000000E+02, 1.82690000E+00, 9.92500000E-01},
    {3.64594000E+01, 1.24000000E+02, 1.09000000E+02, 1.82690000E+00, 9.98200000E-01},
    {2.25783700E+02, 1.24000000E+02, 1.11000000E+02, 1.82690000E+00, 9.68400000E-01},
    {3.49932800E+02, 1.24000000E+02, 1.12000000E+02, 1.82690000E+00, 9.62800000E-01},
    {3.50429600E+02, 1.24000000E+02, 1.13000000E+02, 1.82690000E+00, 9.64800000E-01},
    {2.76951200E+02, 1.24000000E+02, 1.14000000E+02, 1.82690000E+00, 9.50700000E-01},
    {2.24030300E+02, 1.24000000E+02, 1.15000000E+02, 1.82690000E+00, 9.94700000E-01},
    {1.87857000E+02, 1.24000000E+02, 1.16000000E+02, 1.82690000E+00, 9.94800000E-01},
    {1.52238300E+02, 1.24000000E+02, 1.17000000E+02, 1.82690000E+00, 9.97200000E-01},
    {3.08390100E+02, 1.24000000E+02, 1.19000000E+02, 1.82690000E+00, 9.76700000E-01},
    {6.07070000E+02, 1.24000000E+02, 1.20000000E+02, 1.82690000E+00, 9.83100000E-01},
    {3.04309400E+02, 1.24000000E+02, 1.21000000E+02, 1.82690000E+00, 1.86270000E+00},
    {2.93659800E+02, 1.24000000E+02, 1.22000000E+02, 1.82690000E+00, 1.82990000E+00},
    {2.87865700E+02, 1.24000000E+02, 1.23000000E+02, 1.82690000E+00, 1.91380000E+00},
    {2.85690500E+02, 1.24000000E+02, 1.24000000E+02, 1.82690000E+00, 1.82690000E+00},
    {2.61810000E+01, 1.25000000E+02, 1.00000000E+00, 1.64060000E+00, 9.11800000E-01},
    {1.71526000E+01, 1.25000000E+02, 2.00000000E+00, 1.64060000E+00, 0.00000000E+00},
    {4.16083600E+02, 1.25000000E+02, 3.00000000E+00, 1.64060000E+00, 0.00000000E+00},
    {2.38960600E+02, 1.25000000E+02, 4.00000000E+00, 1.64060000E+00, 0.00000000E+00},
    {1.59849200E+02, 1.25000000E+02, 5.00000000E+00, 1.64060000E+00, 0.00000000E+00},
    {1.07338200E+02, 1.25000000E+02, 6.00000000E+00, 1.64060000E+00, 0.00000000E+00},
    {7.46998000E+01, 1.25000000E+02, 7.00000000E+00, 1.64060000E+00, 0.00000000E+00},
    {5.63622000E+01, 1.25000000E+02, 8.00000000E+00, 1.64060000E+00, 0.00000000E+00},
    {4.25764000E+01, 1.25000000E+02, 9.00000000E+00, 1.64060000E+00, 0.00000000E+00},
    {3.26806000E+01, 1.25000000E+02, 1.00000000E+01, 1.64060000E+00, 0.00000000E+00},
    {4.97358200E+02, 1.25000000E+02, 1.10000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.81296800E+02, 1.25000000E+02, 1.20000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.50452200E+02, 1.25000000E+02, 1.30000000E+01, 1.64060000E+00, 0.00000000E+00},
    {2.74925700E+02, 1.25000000E+02, 1.40000000E+01, 1.64060000E+00, 0.00000000E+00},
    {2.13410100E+02, 1.25000000E+02, 1.50000000E+01, 1.64060000E+00, 0.00000000E+00},
    {1.76516500E+02, 1.25000000E+02, 1.60000000E+01, 1.64060000E+00, 0.00000000E+00},
    {1.43700100E+02, 1.25000000E+02, 1.70000000E+01, 1.64060000E+00, 0.00000000E+00},
    {1.17205700E+02, 1.25000000E+02, 1.80000000E+01, 1.64060000E+00, 0.00000000E+00},
    {8.14246800E+02, 1.25000000E+02, 1.90000000E+01, 1.64060000E+00, 0.00000000E+00},
    {6.70833700E+02, 1.25000000E+02, 2.00000000E+01, 1.64060000E+00, 0.00000000E+00},
    {5.53793100E+02, 1.25000000E+02, 2.10000000E+01, 1.64060000E+00, 0.00000000E+00},
    {5.34126600E+02, 1.25000000E+02, 2.20000000E+01, 1.64060000E+00, 0.00000000E+00},
    {4.88794700E+02, 1.25000000E+02, 2.30000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.84693100E+02, 1.25000000E+02, 2.40000000E+01, 1.64060000E+00, 0.00000000E+00},
    {4.20420700E+02, 1.25000000E+02, 2.50000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.29627300E+02, 1.25000000E+02, 2.60000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.49194100E+02, 1.25000000E+02, 2.70000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.60017200E+02, 1.25000000E+02, 2.80000000E+01, 1.64060000E+00, 0.00000000E+00},
    {2.75720600E+02, 1.25000000E+02, 2.90000000E+01, 1.64060000E+00, 0.00000000E+00},
    {2.82666700E+02, 1.25000000E+02, 3.00000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.35061500E+02, 1.25000000E+02, 3.10000000E+01, 1.64060000E+00, 0.00000000E+00},
    {2.94518400E+02, 1.25000000E+02, 3.20000000E+01, 1.64060000E+00, 0.00000000E+00},
    {2.50283100E+02, 1.25000000E+02, 3.30000000E+01, 1.64060000E+00, 0.00000000E+00},
    {2.23985700E+02, 1.25000000E+02, 3.40000000E+01, 1.64060000E+00, 0.00000000E+00},
    {1.95453600E+02, 1.25000000E+02, 3.50000000E+01, 1.64060000E+00, 0.00000000E+00},
    {1.69509700E+02, 1.25000000E+02, 3.60000000E+01, 1.64060000E+00, 0.00000000E+00},
    {9.11943300E+02, 1.25000000E+02, 3.70000000E+01, 1.64060000E+00, 0.00000000E+00},
    {7.99012500E+02, 1.25000000E+02, 3.80000000E+01, 1.64060000E+00, 0.00000000E+00},
    {6.98676000E+02, 1.25000000E+02, 3.90000000E+01, 1.64060000E+00, 0.00000000E+00},
    {6.27079000E+02, 1.25000000E+02, 4.00000000E+01, 1.64060000E+00, 0.00000000E+00},
    {5.71186200E+02, 1.25000000E+02, 4.10000000E+01, 1.64060000E+00, 0.00000000E+00},
    {4.39932600E+02, 1.25000000E+02, 4.20000000E+01, 1.64060000E+00, 0.00000000E+00},
    {4.91321000E+02, 1.25000000E+02, 4.30000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.73344200E+02, 1.25000000E+02, 4.40000000E+01, 1.64060000E+00, 0.00000000E+00},
    {4.08367700E+02, 1.25000000E+02, 4.50000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.78429500E+02, 1.25000000E+02, 4.60000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.15348200E+02, 1.25000000E+02, 4.70000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.33151800E+02, 1.25000000E+02, 4.80000000E+01, 1.64060000E+00, 0.00000000E+00},
    {4.19070700E+02, 1.25000000E+02, 4.90000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.86618400E+02, 1.25000000E+02, 5.00000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.43520500E+02, 1.25000000E+02, 5.10000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.18048500E+02, 1.25000000E+02, 5.20000000E+01, 1.64060000E+00, 0.00000000E+00},
    {2.86872000E+02, 1.25000000E+02, 5.30000000E+01, 1.64060000E+00, 0.00000000E+00},
    {2.57280300E+02, 1.25000000E+02, 5.40000000E+01, 1.64060000E+00, 0.00000000E+00},
    {1.11035880E+03, 1.25000000E+02, 5.50000000E+01, 1.64060000E+00, 0.00000000E+00},
    {1.01866880E+03, 1.25000000E+02, 5.60000000E+01, 1.64060000E+00, 0.00000000E+00},
    {8.93505600E+02, 1.25000000E+02, 5.70000000E+01, 1.64060000E+00, 0.00000000E+00},
    {4.05873700E+02, 1.25000000E+02, 5.80000000E+01, 1.64060000E+00, 2.79910000E+00},
    {9.02048200E+02, 1.25000000E+02, 5.90000000E+01, 1.64060000E+00, 0.00000000E+00},
    {8.66044200E+02, 1.25000000E+02, 6.00000000E+01, 1.64060000E+00, 0.00000000E+00},
    {8.44285600E+02, 1.25000000E+02, 6.10000000E+01, 1.64060000E+00, 0.00000000E+00},
    {8.24283000E+02, 1.25000000E+02, 6.20000000E+01, 1.64060000E+00, 0.00000000E+00},
    {8.06543200E+02, 1.25000000E+02, 6.30000000E+01, 1.64060000E+00, 0.00000000E+00},
    {6.32574300E+02, 1.25000000E+02, 6.40000000E+01, 1.64060000E+00, 0.00000000E+00},
    {7.13755200E+02, 1.25000000E+02, 6.50000000E+01, 1.64060000E+00, 0.00000000E+00},
    {6.88112500E+02, 1.25000000E+02, 6.60000000E+01, 1.64060000E+00, 0.00000000E+00},
    {7.27237800E+02, 1.25000000E+02, 6.70000000E+01, 1.64060000E+00, 0.00000000E+00},
    {7.11800100E+02, 1.25000000E+02, 6.80000000E+01, 1.64060000E+00, 0.00000000E+00},
    {6.97856500E+02, 1.25000000E+02, 6.90000000E+01, 1.64060000E+00, 0.00000000E+00},
    {6.89784700E+02, 1.25000000E+02, 7.00000000E+01, 1.64060000E+00, 0.00000000E+00},
    {5.80128700E+02, 1.25000000E+02, 7.10000000E+01, 1.64060000E+00, 0.00000000E+00},
    {5.69405600E+02, 1.25000000E+02, 7.20000000E+01, 1.64060000E+00, 0.00000000E+00},
    {5.18790500E+02, 1.25000000E+02, 7.30000000E+01, 1.64060000E+00, 0.00000000E+00},
    {4.37208800E+02, 1.25000000E+02, 7.40000000E+01, 1.64060000E+00, 0.00000000E+00},
    {4.44599700E+02, 1.25000000E+02, 7.50000000E+01, 1.64060000E+00, 0.00000000E+00},
    {4.02309700E+02, 1.25000000E+02, 7.60000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.67945400E+02, 1.25000000E+02, 7.70000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.05050100E+02, 1.25000000E+02, 7.80000000E+01, 1.64060000E+00, 0.00000000E+00},
    {2.84801800E+02, 1.25000000E+02, 7.90000000E+01, 1.64060000E+00, 0.00000000E+00},
    {2.92917700E+02, 1.25000000E+02, 8.00000000E+01, 1.64060000E+00, 0.00000000E+00},
    {4.29595100E+02, 1.25000000E+02, 8.10000000E+01, 1.64060000E+00, 0.00000000E+00},
    {4.19414900E+02, 1.25000000E+02, 8.20000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.84449600E+02, 1.25000000E+02, 8.30000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.66054700E+02, 1.25000000E+02, 8.40000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.37092800E+02, 1.25000000E+02, 8.50000000E+01, 1.64060000E+00, 0.00000000E+00},
    {3.08294700E+02, 1.25000000E+02, 8.60000000E+01, 1.64060000E+00, 0.00000000E+00},
    {1.04720980E+03, 1.25000000E+02, 8.70000000E+01, 1.64060000E+00, 0.00000000E+00},
    {1.00628590E+03, 1.25000000E+02, 8.80000000E+01, 1.64060000E+00, 0.00000000E+00},
    {8.88230000E+02, 1.25000000E+02, 8.90000000E+01, 1.64060000E+00, 0.00000000E+00},
    {7.96310500E+02, 1.25000000E+02, 9.00000000E+01, 1.64060000E+00, 0.00000000E+00},
    {7.91257700E+02, 1.25000000E+02, 9.10000000E+01, 1.64060000E+00, 0.00000000E+00},
    {7.66098400E+02, 1.25000000E+02, 9.20000000E+01, 1.64060000E+00, 0.00000000E+00},
    {7.89923500E+02, 1.25000000E+02, 9.30000000E+01, 1.64060000E+00, 0.00000000E+00},
    {7.64806100E+02, 1.25000000E+02, 9.40000000E+01, 1.64060000E+00, 0.00000000E+00},
    {4.23927000E+01, 1.25000000E+02, 1.01000000E+02, 1.64060000E+00, 0.00000000E+00},
    {1.38749100E+02, 1.25000000E+02, 1.03000000E+02, 1.64060000E+00, 9.86500000E-01},
    {1.76681700E+02, 1.25000000E+02, 1.04000000E+02, 1.64060000E+00, 9.80800000E-01},
    {1.34213600E+02, 1.25000000E+02, 1.05000000E+02, 1.64060000E+00, 9.70600000E-01},
    {1.00646300E+02, 1.25000000E+02, 1.06000000E+02, 1.64060000E+00, 9.86800000E-01},
    {6.96215000E+01, 1.25000000E+02, 1.07000000E+02, 1.64060000E+00, 9.94400000E-01},
    {5.04858000E+01, 1.25000000E+02, 1.08000000E+02, 1.64060000E+00, 9.92500000E-01},
    {3.45481000E+01, 1.25000000E+02, 1.09000000E+02, 1.64060000E+00, 9.98200000E-01},
    {2.02967600E+02, 1.25000000E+02, 1.11000000E+02, 1.64060000E+00, 9.68400000E-01},
    {3.14011900E+02, 1.25000000E+02, 1.12000000E+02, 1.64060000E+00, 9.62800000E-01},
    {3.17328700E+02, 1.25000000E+02, 1.13000000E+02, 1.64060000E+00, 9.64800000E-01},
    {2.53880000E+02, 1.25000000E+02, 1.14000000E+02, 1.64060000E+00, 9.50700000E-01},
    {2.07099500E+02, 1.25000000E+02, 1.15000000E+02, 1.64060000E+00, 9.94700000E-01},
    {1.74577500E+02, 1.25000000E+02, 1.16000000E+02, 1.64060000E+00, 9.94800000E-01},
    {1.42215100E+02, 1.25000000E+02, 1.17000000E+02, 1.64060000E+00, 9.97200000E-01},
    {2.78725900E+02, 1.25000000E+02, 1.19000000E+02, 1.64060000E+00, 9.76700000E-01},
    {5.35814900E+02, 1.25000000E+02, 1.20000000E+02, 1.64060000E+00, 9.83100000E-01},
    {2.78196500E+02, 1.25000000E+02, 1.21000000E+02, 1.64060000E+00, 1.86270000E+00},
    {2.68465700E+02, 1.25000000E+02, 1.22000000E+02, 1.64060000E+00, 1.82990000E+00},
    {2.63112700E+02, 1.25000000E+02, 1.23000000E+02, 1.64060000E+00, 1.91380000E+00},
    {2.60777300E+02, 1.25000000E+02, 1.24000000E+02, 1.64060000E+00, 1.82690000E+00},
    {2.39555400E+02, 1.25000000E+02, 1.25000000E+02, 1.64060000E+00, 1.64060000E+00},
    {2.43452000E+01, 1.26000000E+02, 1.00000000E+00, 1.64830000E+00, 9.11800000E-01},
    {1.60560000E+01, 1.26000000E+02, 2.00000000E+00, 1.64830000E+00, 0.00000000E+00},
    {3.79211700E+02, 1.26000000E+02, 3.00000000E+00, 1.64830000E+00, 0.00000000E+00},
    {2.19477500E+02, 1.26000000E+02, 4.00000000E+00, 1.64830000E+00, 0.00000000E+00},
    {1.47629800E+02, 1.26000000E+02, 5.00000000E+00, 1.64830000E+00, 0.00000000E+00},
    {9.95880000E+01, 1.26000000E+02, 6.00000000E+00, 1.64830000E+00, 0.00000000E+00},
    {6.95638000E+01, 1.26000000E+02, 7.00000000E+00, 1.64830000E+00, 0.00000000E+00},
    {5.26351000E+01, 1.26000000E+02, 8.00000000E+00, 1.64830000E+00, 0.00000000E+00},
    {3.98628000E+01, 1.26000000E+02, 9.00000000E+00, 1.64830000E+00, 0.00000000E+00},
    {3.06648000E+01, 1.26000000E+02, 1.00000000E+01, 1.64830000E+00, 0.00000000E+00},
    {4.53544400E+02, 1.26000000E+02, 1.10000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.49711600E+02, 1.26000000E+02, 1.20000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.22278100E+02, 1.26000000E+02, 1.30000000E+01, 1.64830000E+00, 0.00000000E+00},
    {2.53763800E+02, 1.26000000E+02, 1.40000000E+01, 1.64830000E+00, 0.00000000E+00},
    {1.97656900E+02, 1.26000000E+02, 1.50000000E+01, 1.64830000E+00, 0.00000000E+00},
    {1.63879400E+02, 1.26000000E+02, 1.60000000E+01, 1.64830000E+00, 0.00000000E+00},
    {1.33732400E+02, 1.26000000E+02, 1.70000000E+01, 1.64830000E+00, 0.00000000E+00},
    {1.09315500E+02, 1.26000000E+02, 1.80000000E+01, 1.64830000E+00, 0.00000000E+00},
    {7.42252000E+02, 1.26000000E+02, 1.90000000E+01, 1.64830000E+00, 0.00000000E+00},
    {6.13923300E+02, 1.26000000E+02, 2.00000000E+01, 1.64830000E+00, 0.00000000E+00},
    {5.07340100E+02, 1.26000000E+02, 2.10000000E+01, 1.64830000E+00, 0.00000000E+00},
    {4.89899600E+02, 1.26000000E+02, 2.20000000E+01, 1.64830000E+00, 0.00000000E+00},
    {4.48626200E+02, 1.26000000E+02, 2.30000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.53270900E+02, 1.26000000E+02, 2.40000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.86260200E+02, 1.26000000E+02, 2.50000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.03048000E+02, 1.26000000E+02, 2.60000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.21344800E+02, 1.26000000E+02, 2.70000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.31061400E+02, 1.26000000E+02, 2.80000000E+01, 1.64830000E+00, 0.00000000E+00},
    {2.53703800E+02, 1.26000000E+02, 2.90000000E+01, 1.64830000E+00, 0.00000000E+00},
    {2.60534100E+02, 1.26000000E+02, 3.00000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.08552900E+02, 1.26000000E+02, 3.10000000E+01, 1.64830000E+00, 0.00000000E+00},
    {2.71971100E+02, 1.26000000E+02, 3.20000000E+01, 1.64830000E+00, 0.00000000E+00},
    {2.31761800E+02, 1.26000000E+02, 3.30000000E+01, 1.64830000E+00, 0.00000000E+00},
    {2.07794800E+02, 1.26000000E+02, 3.40000000E+01, 1.64830000E+00, 0.00000000E+00},
    {1.81683600E+02, 1.26000000E+02, 3.50000000E+01, 1.64830000E+00, 0.00000000E+00},
    {1.57866100E+02, 1.26000000E+02, 3.60000000E+01, 1.64830000E+00, 0.00000000E+00},
    {8.31796000E+02, 1.26000000E+02, 3.70000000E+01, 1.64830000E+00, 0.00000000E+00},
    {7.31233000E+02, 1.26000000E+02, 3.80000000E+01, 1.64830000E+00, 0.00000000E+00},
    {6.40664100E+02, 1.26000000E+02, 3.90000000E+01, 1.64830000E+00, 0.00000000E+00},
    {5.75775200E+02, 1.26000000E+02, 4.00000000E+01, 1.64830000E+00, 0.00000000E+00},
    {5.24958800E+02, 1.26000000E+02, 4.10000000E+01, 1.64830000E+00, 0.00000000E+00},
    {4.05081700E+02, 1.26000000E+02, 4.20000000E+01, 1.64830000E+00, 0.00000000E+00},
    {4.52083100E+02, 1.26000000E+02, 4.30000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.44232200E+02, 1.26000000E+02, 4.40000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.76404500E+02, 1.26000000E+02, 4.50000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.49025800E+02, 1.26000000E+02, 4.60000000E+01, 1.64830000E+00, 0.00000000E+00},
    {2.90908300E+02, 1.26000000E+02, 4.70000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.07520900E+02, 1.26000000E+02, 4.80000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.86062800E+02, 1.26000000E+02, 4.90000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.56915900E+02, 1.26000000E+02, 5.00000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.17862400E+02, 1.26000000E+02, 5.10000000E+01, 1.64830000E+00, 0.00000000E+00},
    {2.94734800E+02, 1.26000000E+02, 5.20000000E+01, 1.64830000E+00, 0.00000000E+00},
    {2.66291000E+02, 1.26000000E+02, 5.30000000E+01, 1.64830000E+00, 0.00000000E+00},
    {2.39219000E+02, 1.26000000E+02, 5.40000000E+01, 1.64830000E+00, 0.00000000E+00},
    {1.01304590E+03, 1.26000000E+02, 5.50000000E+01, 1.64830000E+00, 0.00000000E+00},
    {9.31861000E+02, 1.26000000E+02, 5.60000000E+01, 1.64830000E+00, 0.00000000E+00},
    {8.18899400E+02, 1.26000000E+02, 5.70000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.75456800E+02, 1.26000000E+02, 5.80000000E+01, 1.64830000E+00, 2.79910000E+00},
    {8.25773500E+02, 1.26000000E+02, 5.90000000E+01, 1.64830000E+00, 0.00000000E+00},
    {7.93023900E+02, 1.26000000E+02, 6.00000000E+01, 1.64830000E+00, 0.00000000E+00},
    {7.73157500E+02, 1.26000000E+02, 6.10000000E+01, 1.64830000E+00, 0.00000000E+00},
    {7.54886100E+02, 1.26000000E+02, 6.20000000E+01, 1.64830000E+00, 0.00000000E+00},
    {7.38683600E+02, 1.26000000E+02, 6.30000000E+01, 1.64830000E+00, 0.00000000E+00},
    {5.80777300E+02, 1.26000000E+02, 6.40000000E+01, 1.64830000E+00, 0.00000000E+00},
    {6.53475200E+02, 1.26000000E+02, 6.50000000E+01, 1.64830000E+00, 0.00000000E+00},
    {6.30253300E+02, 1.26000000E+02, 6.60000000E+01, 1.64830000E+00, 0.00000000E+00},
    {6.66337800E+02, 1.26000000E+02, 6.70000000E+01, 1.64830000E+00, 0.00000000E+00},
    {6.52215500E+02, 1.26000000E+02, 6.80000000E+01, 1.64830000E+00, 0.00000000E+00},
    {6.39478200E+02, 1.26000000E+02, 6.90000000E+01, 1.64830000E+00, 0.00000000E+00},
    {6.32012300E+02, 1.26000000E+02, 7.00000000E+01, 1.64830000E+00, 0.00000000E+00},
    {5.32441000E+02, 1.26000000E+02, 7.10000000E+01, 1.64830000E+00, 0.00000000E+00},
    {5.23612200E+02, 1.26000000E+02, 7.20000000E+01, 1.64830000E+00, 0.00000000E+00},
    {4.77739700E+02, 1.26000000E+02, 7.30000000E+01, 1.64830000E+00, 0.00000000E+00},
    {4.03186000E+02, 1.26000000E+02, 7.40000000E+01, 1.64830000E+00, 0.00000000E+00},
    {4.10184900E+02, 1.26000000E+02, 7.50000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.71644800E+02, 1.26000000E+02, 7.60000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.40267200E+02, 1.26000000E+02, 7.70000000E+01, 1.64830000E+00, 0.00000000E+00},
    {2.82505600E+02, 1.26000000E+02, 7.80000000E+01, 1.64830000E+00, 0.00000000E+00},
    {2.63903500E+02, 1.26000000E+02, 7.90000000E+01, 1.64830000E+00, 0.00000000E+00},
    {2.71507500E+02, 1.26000000E+02, 8.00000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.96157500E+02, 1.26000000E+02, 8.10000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.87340400E+02, 1.26000000E+02, 8.20000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.55752800E+02, 1.26000000E+02, 8.30000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.39147800E+02, 1.26000000E+02, 8.40000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.12799100E+02, 1.26000000E+02, 8.50000000E+01, 1.64830000E+00, 0.00000000E+00},
    {2.86501700E+02, 1.26000000E+02, 8.60000000E+01, 1.64830000E+00, 0.00000000E+00},
    {9.56820500E+02, 1.26000000E+02, 8.70000000E+01, 1.64830000E+00, 0.00000000E+00},
    {9.21491300E+02, 1.26000000E+02, 8.80000000E+01, 1.64830000E+00, 0.00000000E+00},
    {8.14808000E+02, 1.26000000E+02, 8.90000000E+01, 1.64830000E+00, 0.00000000E+00},
    {7.32154700E+02, 1.26000000E+02, 9.00000000E+01, 1.64830000E+00, 0.00000000E+00},
    {7.26843300E+02, 1.26000000E+02, 9.10000000E+01, 1.64830000E+00, 0.00000000E+00},
    {7.03778700E+02, 1.26000000E+02, 9.20000000E+01, 1.64830000E+00, 0.00000000E+00},
    {7.24688400E+02, 1.26000000E+02, 9.30000000E+01, 1.64830000E+00, 0.00000000E+00},
    {7.01806300E+02, 1.26000000E+02, 9.40000000E+01, 1.64830000E+00, 0.00000000E+00},
    {3.92742000E+01, 1.26000000E+02, 1.01000000E+02, 1.64830000E+00, 0.00000000E+00},
    {1.27549200E+02, 1.26000000E+02, 1.03000000E+02, 1.64830000E+00, 9.86500000E-01},
    {1.62610800E+02, 1.26000000E+02, 1.04000000E+02, 1.64830000E+00, 9.80800000E-01},
    {1.24117000E+02, 1.26000000E+02, 1.05000000E+02, 1.64830000E+00, 9.70600000E-01},
    {9.33848000E+01, 1.26000000E+02, 1.06000000E+02, 1.64830000E+00, 9.86800000E-01},
    {6.48468000E+01, 1.26000000E+02, 1.07000000E+02, 1.64830000E+00, 9.94400000E-01},
    {4.71816000E+01, 1.26000000E+02, 1.08000000E+02, 1.64830000E+00, 9.92500000E-01},
    {3.24321000E+01, 1.26000000E+02, 1.09000000E+02, 1.64830000E+00, 9.98200000E-01},
    {1.86466900E+02, 1.26000000E+02, 1.11000000E+02, 1.64830000E+00, 9.68400000E-01},
    {2.88337100E+02, 1.26000000E+02, 1.12000000E+02, 1.64830000E+00, 9.62800000E-01},
    {2.92024700E+02, 1.26000000E+02, 1.13000000E+02, 1.64830000E+00, 9.64800000E-01},
    {2.34484300E+02, 1.26000000E+02, 1.14000000E+02, 1.64830000E+00, 9.50700000E-01},
    {1.91842900E+02, 1.26000000E+02, 1.15000000E+02, 1.64830000E+00, 9.94700000E-01},
    {1.62075000E+02, 1.26000000E+02, 1.16000000E+02, 1.64830000E+00, 9.94800000E-01},
    {1.32348000E+02, 1.26000000E+02, 1.17000000E+02, 1.64830000E+00, 9.97200000E-01},
    {2.56757900E+02, 1.26000000E+02, 1.19000000E+02, 1.64830000E+00, 9.76700000E-01},
    {4.90748100E+02, 1.26000000E+02, 1.20000000E+02, 1.64830000E+00, 9.83100000E-01},
    {2.56888900E+02, 1.26000000E+02, 1.21000000E+02, 1.64830000E+00, 1.86270000E+00},
    {2.47963600E+02, 1.26000000E+02, 1.22000000E+02, 1.64830000E+00, 1.82990000E+00},
    {2.43014800E+02, 1.26000000E+02, 1.23000000E+02, 1.64830000E+00, 1.91380000E+00},
    {2.40783400E+02, 1.26000000E+02, 1.24000000E+02, 1.64830000E+00, 1.82690000E+00},
    {2.21534000E+02, 1.26000000E+02, 1.25000000E+02, 1.64830000E+00, 1.64060000E+00},
    {2.04992500E+02, 1.26000000E+02, 1.26000000E+02, 1.64830000E+00, 1.64830000E+00},
    {2.32310000E+01, 1.27000000E+02, 1.00000000E+00, 1.71490000E+00, 9.11800000E-01},
    {1.53545000E+01, 1.27000000E+02, 2.00000000E+00, 1.71490000E+00, 0.00000000E+00},
    {3.62374600E+02, 1.27000000E+02, 3.00000000E+00, 1.71490000E+00, 0.00000000E+00},
    {2.09222700E+02, 1.27000000E+02, 4.00000000E+00, 1.71490000E+00, 0.00000000E+00},
    {1.40742800E+02, 1.27000000E+02, 5.00000000E+00, 1.71490000E+00, 0.00000000E+00},
    {9.50051000E+01, 1.27000000E+02, 6.00000000E+00, 1.71490000E+00, 0.00000000E+00},
    {6.64218000E+01, 1.27000000E+02, 7.00000000E+00, 1.71490000E+00, 0.00000000E+00},
    {5.03016000E+01, 1.27000000E+02, 8.00000000E+00, 1.71490000E+00, 0.00000000E+00},
    {3.81313000E+01, 1.27000000E+02, 9.00000000E+00, 1.71490000E+00, 0.00000000E+00},
    {2.93598000E+01, 1.27000000E+02, 1.00000000E+01, 1.71490000E+00, 0.00000000E+00},
    {4.33373600E+02, 1.27000000E+02, 1.10000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.33476700E+02, 1.27000000E+02, 1.20000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.07254000E+02, 1.27000000E+02, 1.30000000E+01, 1.71490000E+00, 0.00000000E+00},
    {2.41915900E+02, 1.27000000E+02, 1.40000000E+01, 1.71490000E+00, 0.00000000E+00},
    {1.88485200E+02, 1.27000000E+02, 1.50000000E+01, 1.71490000E+00, 0.00000000E+00},
    {1.56342300E+02, 1.27000000E+02, 1.60000000E+01, 1.71490000E+00, 0.00000000E+00},
    {1.27651300E+02, 1.27000000E+02, 1.70000000E+01, 1.71490000E+00, 0.00000000E+00},
    {1.04409100E+02, 1.27000000E+02, 1.80000000E+01, 1.71490000E+00, 0.00000000E+00},
    {7.10181800E+02, 1.27000000E+02, 1.90000000E+01, 1.71490000E+00, 0.00000000E+00},
    {5.85957800E+02, 1.27000000E+02, 2.00000000E+01, 1.71490000E+00, 0.00000000E+00},
    {4.84072300E+02, 1.27000000E+02, 2.10000000E+01, 1.71490000E+00, 0.00000000E+00},
    {4.67398400E+02, 1.27000000E+02, 2.20000000E+01, 1.71490000E+00, 0.00000000E+00},
    {4.27993600E+02, 1.27000000E+02, 2.30000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.37143600E+02, 1.27000000E+02, 2.40000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.68471900E+02, 1.27000000E+02, 2.50000000E+01, 1.71490000E+00, 0.00000000E+00},
    {2.89193700E+02, 1.27000000E+02, 2.60000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.06496800E+02, 1.27000000E+02, 2.70000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.15771300E+02, 1.27000000E+02, 2.80000000E+01, 1.71490000E+00, 0.00000000E+00},
    {2.42103000E+02, 1.27000000E+02, 2.90000000E+01, 1.71490000E+00, 0.00000000E+00},
    {2.48485500E+02, 1.27000000E+02, 3.00000000E+01, 1.71490000E+00, 0.00000000E+00},
    {2.94240800E+02, 1.27000000E+02, 3.10000000E+01, 1.71490000E+00, 0.00000000E+00},
    {2.59322400E+02, 1.27000000E+02, 3.20000000E+01, 1.71490000E+00, 0.00000000E+00},
    {2.21023300E+02, 1.27000000E+02, 3.30000000E+01, 1.71490000E+00, 0.00000000E+00},
    {1.98221300E+02, 1.27000000E+02, 3.40000000E+01, 1.71490000E+00, 0.00000000E+00},
    {1.73379300E+02, 1.27000000E+02, 3.50000000E+01, 1.71490000E+00, 0.00000000E+00},
    {1.50719000E+02, 1.27000000E+02, 3.60000000E+01, 1.71490000E+00, 0.00000000E+00},
    {7.95885900E+02, 1.27000000E+02, 3.70000000E+01, 1.71490000E+00, 0.00000000E+00},
    {6.98076200E+02, 1.27000000E+02, 3.80000000E+01, 1.71490000E+00, 0.00000000E+00},
    {6.11340300E+02, 1.27000000E+02, 3.90000000E+01, 1.71490000E+00, 0.00000000E+00},
    {5.49329400E+02, 1.27000000E+02, 4.00000000E+01, 1.71490000E+00, 0.00000000E+00},
    {5.00835500E+02, 1.27000000E+02, 4.10000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.86524500E+02, 1.27000000E+02, 4.20000000E+01, 1.71490000E+00, 0.00000000E+00},
    {4.31347800E+02, 1.27000000E+02, 4.30000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.28501500E+02, 1.27000000E+02, 4.40000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.59105700E+02, 1.27000000E+02, 4.50000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.32989600E+02, 1.27000000E+02, 4.60000000E+01, 1.71490000E+00, 0.00000000E+00},
    {2.77682300E+02, 1.27000000E+02, 4.70000000E+01, 1.71490000E+00, 0.00000000E+00},
    {2.93407400E+02, 1.27000000E+02, 4.80000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.68322400E+02, 1.27000000E+02, 4.90000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.40426500E+02, 1.27000000E+02, 5.00000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.03185600E+02, 1.27000000E+02, 5.10000000E+01, 1.71490000E+00, 0.00000000E+00},
    {2.81163300E+02, 1.27000000E+02, 5.20000000E+01, 1.71490000E+00, 0.00000000E+00},
    {2.54087700E+02, 1.27000000E+02, 5.30000000E+01, 1.71490000E+00, 0.00000000E+00},
    {2.28325600E+02, 1.27000000E+02, 5.40000000E+01, 1.71490000E+00, 0.00000000E+00},
    {9.69546200E+02, 1.27000000E+02, 5.50000000E+01, 1.71490000E+00, 0.00000000E+00},
    {8.89934600E+02, 1.27000000E+02, 5.60000000E+01, 1.71490000E+00, 0.00000000E+00},
    {7.81635000E+02, 1.27000000E+02, 5.70000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.58167700E+02, 1.27000000E+02, 5.80000000E+01, 1.71490000E+00, 2.79910000E+00},
    {7.88580900E+02, 1.27000000E+02, 5.90000000E+01, 1.71490000E+00, 0.00000000E+00},
    {7.57171100E+02, 1.27000000E+02, 6.00000000E+01, 1.71490000E+00, 0.00000000E+00},
    {7.38174800E+02, 1.27000000E+02, 6.10000000E+01, 1.71490000E+00, 0.00000000E+00},
    {7.20705200E+02, 1.27000000E+02, 6.20000000E+01, 1.71490000E+00, 0.00000000E+00},
    {7.05212900E+02, 1.27000000E+02, 6.30000000E+01, 1.71490000E+00, 0.00000000E+00},
    {5.54346200E+02, 1.27000000E+02, 6.40000000E+01, 1.71490000E+00, 0.00000000E+00},
    {6.24448100E+02, 1.27000000E+02, 6.50000000E+01, 1.71490000E+00, 0.00000000E+00},
    {6.02267200E+02, 1.27000000E+02, 6.60000000E+01, 1.71490000E+00, 0.00000000E+00},
    {6.36035700E+02, 1.27000000E+02, 6.70000000E+01, 1.71490000E+00, 0.00000000E+00},
    {6.22537200E+02, 1.27000000E+02, 6.80000000E+01, 1.71490000E+00, 0.00000000E+00},
    {6.10360000E+02, 1.27000000E+02, 6.90000000E+01, 1.71490000E+00, 0.00000000E+00},
    {6.03235200E+02, 1.27000000E+02, 7.00000000E+01, 1.71490000E+00, 0.00000000E+00},
    {5.08163100E+02, 1.27000000E+02, 7.10000000E+01, 1.71490000E+00, 0.00000000E+00},
    {4.99470100E+02, 1.27000000E+02, 7.20000000E+01, 1.71490000E+00, 0.00000000E+00},
    {4.55686600E+02, 1.27000000E+02, 7.30000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.84672300E+02, 1.27000000E+02, 7.40000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.91311000E+02, 1.27000000E+02, 7.50000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.54558200E+02, 1.27000000E+02, 7.60000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.24649100E+02, 1.27000000E+02, 7.70000000E+01, 1.71490000E+00, 0.00000000E+00},
    {2.69633800E+02, 1.27000000E+02, 7.80000000E+01, 1.71490000E+00, 0.00000000E+00},
    {2.51915100E+02, 1.27000000E+02, 7.90000000E+01, 1.71490000E+00, 0.00000000E+00},
    {2.59138200E+02, 1.27000000E+02, 8.00000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.78091300E+02, 1.27000000E+02, 8.10000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.69550200E+02, 1.27000000E+02, 8.20000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.39393900E+02, 1.27000000E+02, 8.30000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.23571000E+02, 1.27000000E+02, 8.40000000E+01, 1.71490000E+00, 0.00000000E+00},
    {2.98479900E+02, 1.27000000E+02, 8.50000000E+01, 1.71490000E+00, 0.00000000E+00},
    {2.73450100E+02, 1.27000000E+02, 8.60000000E+01, 1.71490000E+00, 0.00000000E+00},
    {9.15161000E+02, 1.27000000E+02, 8.70000000E+01, 1.71490000E+00, 0.00000000E+00},
    {8.79833900E+02, 1.27000000E+02, 8.80000000E+01, 1.71490000E+00, 0.00000000E+00},
    {7.77616500E+02, 1.27000000E+02, 8.90000000E+01, 1.71490000E+00, 0.00000000E+00},
    {6.98561900E+02, 1.27000000E+02, 9.00000000E+01, 1.71490000E+00, 0.00000000E+00},
    {6.93714300E+02, 1.27000000E+02, 9.10000000E+01, 1.71490000E+00, 0.00000000E+00},
    {6.71700700E+02, 1.27000000E+02, 9.20000000E+01, 1.71490000E+00, 0.00000000E+00},
    {6.91798500E+02, 1.27000000E+02, 9.30000000E+01, 1.71490000E+00, 0.00000000E+00},
    {6.69915800E+02, 1.27000000E+02, 9.40000000E+01, 1.71490000E+00, 0.00000000E+00},
    {3.74496000E+01, 1.27000000E+02, 1.01000000E+02, 1.71490000E+00, 0.00000000E+00},
    {1.21603700E+02, 1.27000000E+02, 1.03000000E+02, 1.71490000E+00, 9.86500000E-01},
    {1.55055400E+02, 1.27000000E+02, 1.04000000E+02, 1.71490000E+00, 9.80800000E-01},
    {1.18353000E+02, 1.27000000E+02, 1.05000000E+02, 1.71490000E+00, 9.70600000E-01},
    {8.90992000E+01, 1.27000000E+02, 1.06000000E+02, 1.71490000E+00, 9.86800000E-01},
    {6.19256000E+01, 1.27000000E+02, 1.07000000E+02, 1.71490000E+00, 9.94400000E-01},
    {4.51002000E+01, 1.27000000E+02, 1.08000000E+02, 1.71490000E+00, 9.92500000E-01},
    {3.10516000E+01, 1.27000000E+02, 1.09000000E+02, 1.71490000E+00, 9.98200000E-01},
    {1.77835500E+02, 1.27000000E+02, 1.11000000E+02, 1.71490000E+00, 9.68400000E-01},
    {2.74984000E+02, 1.27000000E+02, 1.12000000E+02, 1.71490000E+00, 9.62800000E-01},
    {2.78417300E+02, 1.27000000E+02, 1.13000000E+02, 1.71490000E+00, 9.64800000E-01},
    {2.23554800E+02, 1.27000000E+02, 1.14000000E+02, 1.71490000E+00, 9.50700000E-01},
    {1.82951100E+02, 1.27000000E+02, 1.15000000E+02, 1.71490000E+00, 9.94700000E-01},
    {1.54624300E+02, 1.27000000E+02, 1.16000000E+02, 1.71490000E+00, 9.94800000E-01},
    {1.26331900E+02, 1.27000000E+02, 1.17000000E+02, 1.71490000E+00, 9.97200000E-01},
    {2.45018800E+02, 1.27000000E+02, 1.19000000E+02, 1.71490000E+00, 9.76700000E-01},
    {4.68689900E+02, 1.27000000E+02, 1.20000000E+02, 1.71490000E+00, 9.83100000E-01},
    {2.45003500E+02, 1.27000000E+02, 1.21000000E+02, 1.71490000E+00, 1.86270000E+00},
    {2.36532500E+02, 1.27000000E+02, 1.22000000E+02, 1.71490000E+00, 1.82990000E+00},
    {2.31818000E+02, 1.27000000E+02, 1.23000000E+02, 1.71490000E+00, 1.91380000E+00},
    {2.29702200E+02, 1.27000000E+02, 1.24000000E+02, 1.71490000E+00, 1.82690000E+00},
    {2.11302500E+02, 1.27000000E+02, 1.25000000E+02, 1.71490000E+00, 1.64060000E+00},
    {1.95543000E+02, 1.27000000E+02, 1.26000000E+02, 1.71490000E+00, 1.64830000E+00},
    {1.86546200E+02, 1.27000000E+02, 1.27000000E+02, 1.71490000E+00, 1.71490000E+00},
    {2.26822000E+01, 1.28000000E+02, 1.00000000E+00, 1.79370000E+00, 9.11800000E-01},
    {1.49833000E+01, 1.28000000E+02, 2.00000000E+00, 1.79370000E+00, 0.00000000E+00},
    {3.55960600E+02, 1.28000000E+02, 3.00000000E+00, 1.79370000E+00, 0.00000000E+00},
    {2.04908600E+02, 1.28000000E+02, 4.00000000E+00, 1.79370000E+00, 0.00000000E+00},
    {1.37625400E+02, 1.28000000E+02, 5.00000000E+00, 1.79370000E+00, 0.00000000E+00},
    {9.28107000E+01, 1.28000000E+02, 6.00000000E+00, 1.79370000E+00, 0.00000000E+00},
    {6.48519000E+01, 1.28000000E+02, 7.00000000E+00, 1.79370000E+00, 0.00000000E+00},
    {4.90996000E+01, 1.28000000E+02, 8.00000000E+00, 1.79370000E+00, 0.00000000E+00},
    {3.72158000E+01, 1.28000000E+02, 9.00000000E+00, 1.79370000E+00, 0.00000000E+00},
    {2.86554000E+01, 1.28000000E+02, 1.00000000E+01, 1.79370000E+00, 0.00000000E+00},
    {4.25645500E+02, 1.28000000E+02, 1.10000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.26786400E+02, 1.28000000E+02, 1.20000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.00827100E+02, 1.28000000E+02, 1.30000000E+01, 1.79370000E+00, 0.00000000E+00},
    {2.36595200E+02, 1.28000000E+02, 1.40000000E+01, 1.79370000E+00, 0.00000000E+00},
    {1.84184700E+02, 1.28000000E+02, 1.50000000E+01, 1.79370000E+00, 0.00000000E+00},
    {1.52703600E+02, 1.28000000E+02, 1.60000000E+01, 1.79370000E+00, 0.00000000E+00},
    {1.24631700E+02, 1.28000000E+02, 1.70000000E+01, 1.79370000E+00, 0.00000000E+00},
    {1.01911600E+02, 1.28000000E+02, 1.80000000E+01, 1.79370000E+00, 0.00000000E+00},
    {6.97765000E+02, 1.28000000E+02, 1.90000000E+01, 1.79370000E+00, 0.00000000E+00},
    {5.74752200E+02, 1.28000000E+02, 2.00000000E+01, 1.79370000E+00, 0.00000000E+00},
    {4.74625400E+02, 1.28000000E+02, 2.10000000E+01, 1.79370000E+00, 0.00000000E+00},
    {4.58113600E+02, 1.28000000E+02, 2.20000000E+01, 1.79370000E+00, 0.00000000E+00},
    {4.19403700E+02, 1.28000000E+02, 2.30000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.30380800E+02, 1.28000000E+02, 2.40000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.60969200E+02, 1.28000000E+02, 2.50000000E+01, 1.79370000E+00, 0.00000000E+00},
    {2.83301100E+02, 1.28000000E+02, 2.60000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.00105500E+02, 1.28000000E+02, 2.70000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.09254300E+02, 1.28000000E+02, 2.80000000E+01, 1.79370000E+00, 0.00000000E+00},
    {2.37120000E+02, 1.28000000E+02, 2.90000000E+01, 1.79370000E+00, 0.00000000E+00},
    {2.43196000E+02, 1.28000000E+02, 3.00000000E+01, 1.79370000E+00, 0.00000000E+00},
    {2.88007300E+02, 1.28000000E+02, 3.10000000E+01, 1.79370000E+00, 0.00000000E+00},
    {2.53613400E+02, 1.28000000E+02, 3.20000000E+01, 1.79370000E+00, 0.00000000E+00},
    {2.16004300E+02, 1.28000000E+02, 3.30000000E+01, 1.79370000E+00, 0.00000000E+00},
    {1.93643200E+02, 1.28000000E+02, 3.40000000E+01, 1.79370000E+00, 0.00000000E+00},
    {1.69313300E+02, 1.28000000E+02, 3.50000000E+01, 1.79370000E+00, 0.00000000E+00},
    {1.47141700E+02, 1.28000000E+02, 3.60000000E+01, 1.79370000E+00, 0.00000000E+00},
    {7.81844100E+02, 1.28000000E+02, 3.70000000E+01, 1.79370000E+00, 0.00000000E+00},
    {6.84771500E+02, 1.28000000E+02, 3.80000000E+01, 1.79370000E+00, 0.00000000E+00},
    {5.99265200E+02, 1.28000000E+02, 3.90000000E+01, 1.79370000E+00, 0.00000000E+00},
    {5.38248000E+02, 1.28000000E+02, 4.00000000E+01, 1.79370000E+00, 0.00000000E+00},
    {4.90597300E+02, 1.28000000E+02, 4.10000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.78459400E+02, 1.28000000E+02, 4.20000000E+01, 1.79370000E+00, 0.00000000E+00},
    {4.22414400E+02, 1.28000000E+02, 4.30000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.21550600E+02, 1.28000000E+02, 4.40000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.51498600E+02, 1.28000000E+02, 4.50000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.25885800E+02, 1.28000000E+02, 4.60000000E+01, 1.79370000E+00, 0.00000000E+00},
    {2.71808800E+02, 1.28000000E+02, 4.70000000E+01, 1.79370000E+00, 0.00000000E+00},
    {2.87095800E+02, 1.28000000E+02, 4.80000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.60573100E+02, 1.28000000E+02, 4.90000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.33030100E+02, 1.28000000E+02, 5.00000000E+01, 1.79370000E+00, 0.00000000E+00},
    {2.96406400E+02, 1.28000000E+02, 5.10000000E+01, 1.79370000E+00, 0.00000000E+00},
    {2.74775100E+02, 1.28000000E+02, 5.20000000E+01, 1.79370000E+00, 0.00000000E+00},
    {2.48223700E+02, 1.28000000E+02, 5.30000000E+01, 1.79370000E+00, 0.00000000E+00},
    {2.22985600E+02, 1.28000000E+02, 5.40000000E+01, 1.79370000E+00, 0.00000000E+00},
    {9.52340100E+02, 1.28000000E+02, 5.50000000E+01, 1.79370000E+00, 0.00000000E+00},
    {8.73162200E+02, 1.28000000E+02, 5.60000000E+01, 1.79370000E+00, 0.00000000E+00},
    {7.66367700E+02, 1.28000000E+02, 5.70000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.50205400E+02, 1.28000000E+02, 5.80000000E+01, 1.79370000E+00, 2.79910000E+00},
    {7.73554900E+02, 1.28000000E+02, 5.90000000E+01, 1.79370000E+00, 0.00000000E+00},
    {7.42675700E+02, 1.28000000E+02, 6.00000000E+01, 1.79370000E+00, 0.00000000E+00},
    {7.24020700E+02, 1.28000000E+02, 6.10000000E+01, 1.79370000E+00, 0.00000000E+00},
    {7.06867100E+02, 1.28000000E+02, 6.20000000E+01, 1.79370000E+00, 0.00000000E+00},
    {6.91653800E+02, 1.28000000E+02, 6.30000000E+01, 1.79370000E+00, 0.00000000E+00},
    {5.43274200E+02, 1.28000000E+02, 6.40000000E+01, 1.79370000E+00, 0.00000000E+00},
    {6.12632900E+02, 1.28000000E+02, 6.50000000E+01, 1.79370000E+00, 0.00000000E+00},
    {5.90769400E+02, 1.28000000E+02, 6.60000000E+01, 1.79370000E+00, 0.00000000E+00},
    {6.23699600E+02, 1.28000000E+02, 6.70000000E+01, 1.79370000E+00, 0.00000000E+00},
    {6.10451100E+02, 1.28000000E+02, 6.80000000E+01, 1.79370000E+00, 0.00000000E+00},
    {5.98493500E+02, 1.28000000E+02, 6.90000000E+01, 1.79370000E+00, 0.00000000E+00},
    {5.91525500E+02, 1.28000000E+02, 7.00000000E+01, 1.79370000E+00, 0.00000000E+00},
    {4.98025300E+02, 1.28000000E+02, 7.10000000E+01, 1.79370000E+00, 0.00000000E+00},
    {4.89152500E+02, 1.28000000E+02, 7.20000000E+01, 1.79370000E+00, 0.00000000E+00},
    {4.46091600E+02, 1.28000000E+02, 7.30000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.76466500E+02, 1.28000000E+02, 7.40000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.82901200E+02, 1.28000000E+02, 7.50000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.46825100E+02, 1.28000000E+02, 7.60000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.17488400E+02, 1.28000000E+02, 7.70000000E+01, 1.79370000E+00, 0.00000000E+00},
    {2.63632200E+02, 1.28000000E+02, 7.80000000E+01, 1.79370000E+00, 0.00000000E+00},
    {2.46290200E+02, 1.28000000E+02, 7.90000000E+01, 1.79370000E+00, 0.00000000E+00},
    {2.53314900E+02, 1.28000000E+02, 8.00000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.70093300E+02, 1.28000000E+02, 8.10000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.61535700E+02, 1.28000000E+02, 8.20000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.31839300E+02, 1.28000000E+02, 8.30000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.16265300E+02, 1.28000000E+02, 8.40000000E+01, 1.79370000E+00, 0.00000000E+00},
    {2.91634500E+02, 1.28000000E+02, 8.50000000E+01, 1.79370000E+00, 0.00000000E+00},
    {2.67097400E+02, 1.28000000E+02, 8.60000000E+01, 1.79370000E+00, 0.00000000E+00},
    {8.98470600E+02, 1.28000000E+02, 8.70000000E+01, 1.79370000E+00, 0.00000000E+00},
    {8.62940200E+02, 1.28000000E+02, 8.80000000E+01, 1.79370000E+00, 0.00000000E+00},
    {7.62207300E+02, 1.28000000E+02, 8.90000000E+01, 1.79370000E+00, 0.00000000E+00},
    {6.84242100E+02, 1.28000000E+02, 9.00000000E+01, 1.79370000E+00, 0.00000000E+00},
    {6.79749200E+02, 1.28000000E+02, 9.10000000E+01, 1.79370000E+00, 0.00000000E+00},
    {6.58172700E+02, 1.28000000E+02, 9.20000000E+01, 1.79370000E+00, 0.00000000E+00},
    {6.78185600E+02, 1.28000000E+02, 9.30000000E+01, 1.79370000E+00, 0.00000000E+00},
    {6.56680000E+02, 1.28000000E+02, 9.40000000E+01, 1.79370000E+00, 0.00000000E+00},
    {3.65901000E+01, 1.28000000E+02, 1.01000000E+02, 1.79370000E+00, 0.00000000E+00},
    {1.19072500E+02, 1.28000000E+02, 1.03000000E+02, 1.79370000E+00, 9.86500000E-01},
    {1.51779800E+02, 1.28000000E+02, 1.04000000E+02, 1.79370000E+00, 9.80800000E-01},
    {1.15700000E+02, 1.28000000E+02, 1.05000000E+02, 1.79370000E+00, 9.70600000E-01},
    {8.70432000E+01, 1.28000000E+02, 1.06000000E+02, 1.79370000E+00, 9.86800000E-01},
    {6.04608000E+01, 1.28000000E+02, 1.07000000E+02, 1.79370000E+00, 9.94400000E-01},
    {4.40173000E+01, 1.28000000E+02, 1.08000000E+02, 1.79370000E+00, 9.92500000E-01},
    {3.03004000E+01, 1.28000000E+02, 1.09000000E+02, 1.79370000E+00, 9.98200000E-01},
    {1.74200100E+02, 1.28000000E+02, 1.11000000E+02, 1.79370000E+00, 9.68400000E-01},
    {2.69377900E+02, 1.28000000E+02, 1.12000000E+02, 1.79370000E+00, 9.62800000E-01},
    {2.72540900E+02, 1.28000000E+02, 1.13000000E+02, 1.79370000E+00, 9.64800000E-01},
    {2.18606600E+02, 1.28000000E+02, 1.14000000E+02, 1.79370000E+00, 9.50700000E-01},
    {1.78773500E+02, 1.28000000E+02, 1.15000000E+02, 1.79370000E+00, 9.94700000E-01},
    {1.51028600E+02, 1.28000000E+02, 1.16000000E+02, 1.79370000E+00, 9.94800000E-01},
    {1.23345700E+02, 1.28000000E+02, 1.17000000E+02, 1.79370000E+00, 9.97200000E-01},
    {2.39884900E+02, 1.28000000E+02, 1.19000000E+02, 1.79370000E+00, 9.76700000E-01},
    {4.59681200E+02, 1.28000000E+02, 1.20000000E+02, 1.79370000E+00, 9.83100000E-01},
    {2.39656300E+02, 1.28000000E+02, 1.21000000E+02, 1.79370000E+00, 1.86270000E+00},
    {2.31363100E+02, 1.28000000E+02, 1.22000000E+02, 1.79370000E+00, 1.82990000E+00},
    {2.26760100E+02, 1.28000000E+02, 1.23000000E+02, 1.79370000E+00, 1.91380000E+00},
    {2.24717600E+02, 1.28000000E+02, 1.24000000E+02, 1.79370000E+00, 1.82690000E+00},
    {2.06612600E+02, 1.28000000E+02, 1.25000000E+02, 1.79370000E+00, 1.64060000E+00},
    {1.91179800E+02, 1.28000000E+02, 1.26000000E+02, 1.79370000E+00, 1.64830000E+00},
    {1.82387600E+02, 1.28000000E+02, 1.27000000E+02, 1.79370000E+00, 1.71490000E+00},
    {1.78331100E+02, 1.28000000E+02, 1.28000000E+02, 1.79370000E+00, 1.79370000E+00},
    {2.21550000E+01, 1.29000000E+02, 1.00000000E+00, 9.57600000E-01, 9.11800000E-01},
    {1.45147000E+01, 1.29000000E+02, 2.00000000E+00, 9.57600000E-01, 0.00000000E+00},
    {3.67430800E+02, 1.29000000E+02, 3.00000000E+00, 9.57600000E-01, 0.00000000E+00},
    {2.05931300E+02, 1.29000000E+02, 4.00000000E+00, 9.57600000E-01, 0.00000000E+00},
    {1.36370500E+02, 1.29000000E+02, 5.00000000E+00, 9.57600000E-01, 0.00000000E+00},
    {9.11025000E+01, 1.29000000E+02, 6.00000000E+00, 9.57600000E-01, 0.00000000E+00},
    {6.32709000E+01, 1.29000000E+02, 7.00000000E+00, 9.57600000E-01, 0.00000000E+00},
    {4.77239000E+01, 1.29000000E+02, 8.00000000E+00, 9.57600000E-01, 0.00000000E+00},
    {3.60749000E+01, 1.29000000E+02, 9.00000000E+00, 9.57600000E-01, 0.00000000E+00},
    {2.77282000E+01, 1.29000000E+02, 1.00000000E+01, 9.57600000E-01, 0.00000000E+00},
    {4.38719800E+02, 1.29000000E+02, 1.10000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.30001800E+02, 1.29000000E+02, 1.20000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.01465000E+02, 1.29000000E+02, 1.30000000E+01, 9.57600000E-01, 0.00000000E+00},
    {2.34754800E+02, 1.29000000E+02, 1.40000000E+01, 9.57600000E-01, 0.00000000E+00},
    {1.81326700E+02, 1.29000000E+02, 1.50000000E+01, 9.57600000E-01, 0.00000000E+00},
    {1.49642100E+02, 1.29000000E+02, 1.60000000E+01, 9.57600000E-01, 0.00000000E+00},
    {1.21632900E+02, 1.29000000E+02, 1.70000000E+01, 9.57600000E-01, 0.00000000E+00},
    {9.91398000E+01, 1.29000000E+02, 1.80000000E+01, 9.57600000E-01, 0.00000000E+00},
    {7.22000200E+02, 1.29000000E+02, 1.90000000E+01, 9.57600000E-01, 0.00000000E+00},
    {5.85370400E+02, 1.29000000E+02, 2.00000000E+01, 9.57600000E-01, 0.00000000E+00},
    {4.81647200E+02, 1.29000000E+02, 2.10000000E+01, 9.57600000E-01, 0.00000000E+00},
    {4.63392400E+02, 1.29000000E+02, 2.20000000E+01, 9.57600000E-01, 0.00000000E+00},
    {4.23419300E+02, 1.29000000E+02, 2.30000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.33501400E+02, 1.29000000E+02, 2.40000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.63417100E+02, 1.29000000E+02, 2.50000000E+01, 9.57600000E-01, 0.00000000E+00},
    {2.85101800E+02, 1.29000000E+02, 2.60000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.00735600E+02, 1.29000000E+02, 2.70000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.10513300E+02, 1.29000000E+02, 2.80000000E+01, 9.57600000E-01, 0.00000000E+00},
    {2.38120700E+02, 1.29000000E+02, 2.90000000E+01, 9.57600000E-01, 0.00000000E+00},
    {2.42689300E+02, 1.29000000E+02, 3.00000000E+01, 9.57600000E-01, 0.00000000E+00},
    {2.87852000E+02, 1.29000000E+02, 3.10000000E+01, 9.57600000E-01, 0.00000000E+00},
    {2.51538600E+02, 1.29000000E+02, 3.20000000E+01, 9.57600000E-01, 0.00000000E+00},
    {2.12834600E+02, 1.29000000E+02, 3.30000000E+01, 9.57600000E-01, 0.00000000E+00},
    {1.90077100E+02, 1.29000000E+02, 3.40000000E+01, 9.57600000E-01, 0.00000000E+00},
    {1.65586000E+02, 1.29000000E+02, 3.50000000E+01, 9.57600000E-01, 0.00000000E+00},
    {1.43452500E+02, 1.29000000E+02, 3.60000000E+01, 9.57600000E-01, 0.00000000E+00},
    {8.07878600E+02, 1.29000000E+02, 3.70000000E+01, 9.57600000E-01, 0.00000000E+00},
    {6.97862300E+02, 1.29000000E+02, 3.80000000E+01, 9.57600000E-01, 0.00000000E+00},
    {6.06858500E+02, 1.29000000E+02, 3.90000000E+01, 9.57600000E-01, 0.00000000E+00},
    {5.42949900E+02, 1.29000000E+02, 4.00000000E+01, 9.57600000E-01, 0.00000000E+00},
    {4.93636300E+02, 1.29000000E+02, 4.10000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.79229900E+02, 1.29000000E+02, 4.20000000E+01, 9.57600000E-01, 0.00000000E+00},
    {4.23915400E+02, 1.29000000E+02, 4.30000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.21250200E+02, 1.29000000E+02, 4.40000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.51122200E+02, 1.29000000E+02, 4.50000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.25046600E+02, 1.29000000E+02, 4.60000000E+01, 9.57600000E-01, 0.00000000E+00},
    {2.71463400E+02, 1.29000000E+02, 4.70000000E+01, 9.57600000E-01, 0.00000000E+00},
    {2.85818000E+02, 1.29000000E+02, 4.80000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.60662600E+02, 1.29000000E+02, 4.90000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.31006300E+02, 1.29000000E+02, 5.00000000E+01, 9.57600000E-01, 0.00000000E+00},
    {2.92858900E+02, 1.29000000E+02, 5.10000000E+01, 9.57600000E-01, 0.00000000E+00},
    {2.70552600E+02, 1.29000000E+02, 5.20000000E+01, 9.57600000E-01, 0.00000000E+00},
    {2.43547500E+02, 1.29000000E+02, 5.30000000E+01, 9.57600000E-01, 0.00000000E+00},
    {2.18092900E+02, 1.29000000E+02, 5.40000000E+01, 9.57600000E-01, 0.00000000E+00},
    {9.83591600E+02, 1.29000000E+02, 5.50000000E+01, 9.57600000E-01, 0.00000000E+00},
    {8.91688500E+02, 1.29000000E+02, 5.60000000E+01, 9.57600000E-01, 0.00000000E+00},
    {7.77707000E+02, 1.29000000E+02, 5.70000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.46437400E+02, 1.29000000E+02, 5.80000000E+01, 9.57600000E-01, 2.79910000E+00},
    {7.88376100E+02, 1.29000000E+02, 5.90000000E+01, 9.57600000E-01, 0.00000000E+00},
    {7.56157700E+02, 1.29000000E+02, 6.00000000E+01, 9.57600000E-01, 0.00000000E+00},
    {7.36950900E+02, 1.29000000E+02, 6.10000000E+01, 9.57600000E-01, 0.00000000E+00},
    {7.19311800E+02, 1.29000000E+02, 6.20000000E+01, 9.57600000E-01, 0.00000000E+00},
    {7.03659200E+02, 1.29000000E+02, 6.30000000E+01, 9.57600000E-01, 0.00000000E+00},
    {5.48911900E+02, 1.29000000E+02, 6.40000000E+01, 9.57600000E-01, 0.00000000E+00},
    {6.25239600E+02, 1.29000000E+02, 6.50000000E+01, 9.57600000E-01, 0.00000000E+00},
    {6.02141600E+02, 1.29000000E+02, 6.60000000E+01, 9.57600000E-01, 0.00000000E+00},
    {6.33526000E+02, 1.29000000E+02, 6.70000000E+01, 9.57600000E-01, 0.00000000E+00},
    {6.19959700E+02, 1.29000000E+02, 6.80000000E+01, 9.57600000E-01, 0.00000000E+00},
    {6.07663300E+02, 1.29000000E+02, 6.90000000E+01, 9.57600000E-01, 0.00000000E+00},
    {6.00757200E+02, 1.29000000E+02, 7.00000000E+01, 9.57600000E-01, 0.00000000E+00},
    {5.03384100E+02, 1.29000000E+02, 7.10000000E+01, 9.57600000E-01, 0.00000000E+00},
    {4.91221200E+02, 1.29000000E+02, 7.20000000E+01, 9.57600000E-01, 0.00000000E+00},
    {4.46295500E+02, 1.29000000E+02, 7.30000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.75608400E+02, 1.29000000E+02, 7.40000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.81458400E+02, 1.29000000E+02, 7.50000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.44442900E+02, 1.29000000E+02, 7.60000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.14532500E+02, 1.29000000E+02, 7.70000000E+01, 9.57600000E-01, 0.00000000E+00},
    {2.60578500E+02, 1.29000000E+02, 7.80000000E+01, 9.57600000E-01, 0.00000000E+00},
    {2.43220400E+02, 1.29000000E+02, 7.90000000E+01, 9.57600000E-01, 0.00000000E+00},
    {2.49823100E+02, 1.29000000E+02, 8.00000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.69713400E+02, 1.29000000E+02, 8.10000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.59382200E+02, 1.29000000E+02, 8.20000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.28096100E+02, 1.29000000E+02, 8.30000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.11754300E+02, 1.29000000E+02, 8.40000000E+01, 9.57600000E-01, 0.00000000E+00},
    {2.86480300E+02, 1.29000000E+02, 8.50000000E+01, 9.57600000E-01, 0.00000000E+00},
    {2.61592300E+02, 1.29000000E+02, 8.60000000E+01, 9.57600000E-01, 0.00000000E+00},
    {9.23444800E+02, 1.29000000E+02, 8.70000000E+01, 9.57600000E-01, 0.00000000E+00},
    {8.78388800E+02, 1.29000000E+02, 8.80000000E+01, 9.57600000E-01, 0.00000000E+00},
    {7.71371000E+02, 1.29000000E+02, 8.90000000E+01, 9.57600000E-01, 0.00000000E+00},
    {6.87985100E+02, 1.29000000E+02, 9.00000000E+01, 9.57600000E-01, 0.00000000E+00},
    {6.85761400E+02, 1.29000000E+02, 9.10000000E+01, 9.57600000E-01, 0.00000000E+00},
    {6.63908500E+02, 1.29000000E+02, 9.20000000E+01, 9.57600000E-01, 0.00000000E+00},
    {6.86940900E+02, 1.29000000E+02, 9.30000000E+01, 9.57600000E-01, 0.00000000E+00},
    {6.64658100E+02, 1.29000000E+02, 9.40000000E+01, 9.57600000E-01, 0.00000000E+00},
    {3.59871000E+01, 1.29000000E+02, 1.01000000E+02, 9.57600000E-01, 0.00000000E+00},
    {1.19439700E+02, 1.29000000E+02, 1.03000000E+02, 9.57600000E-01, 9.86500000E-01},
    {1.51842600E+02, 1.29000000E+02, 1.04000000E+02, 9.57600000E-01, 9.80800000E-01},
    {1.14346600E+02, 1.29000000E+02, 1.05000000E+02, 9.57600000E-01, 9.70600000E-01},
    {8.54731000E+01, 1.29000000E+02, 1.06000000E+02, 9.57600000E-01, 9.86800000E-01},
    {5.89903000E+01, 1.29000000E+02, 1.07000000E+02, 9.57600000E-01, 9.94400000E-01},
    {4.27458000E+01, 1.29000000E+02, 1.08000000E+02, 9.57600000E-01, 9.92500000E-01},
    {2.92808000E+01, 1.29000000E+02, 1.09000000E+02, 9.57600000E-01, 9.98200000E-01},
    {1.75253100E+02, 1.29000000E+02, 1.11000000E+02, 9.57600000E-01, 9.68400000E-01},
    {2.71275500E+02, 1.29000000E+02, 1.12000000E+02, 9.57600000E-01, 9.62800000E-01},
    {2.72638400E+02, 1.29000000E+02, 1.13000000E+02, 9.57600000E-01, 9.64800000E-01},
    {2.16615400E+02, 1.29000000E+02, 1.14000000E+02, 9.57600000E-01, 9.50700000E-01},
    {1.75962800E+02, 1.29000000E+02, 1.15000000E+02, 9.57600000E-01, 9.94700000E-01},
    {1.48024400E+02, 1.29000000E+02, 1.16000000E+02, 9.57600000E-01, 9.94800000E-01},
    {1.20392700E+02, 1.29000000E+02, 1.17000000E+02, 9.57600000E-01, 9.97200000E-01},
    {2.40179100E+02, 1.29000000E+02, 1.19000000E+02, 9.57600000E-01, 9.76700000E-01},
    {4.67915600E+02, 1.29000000E+02, 1.20000000E+02, 9.57600000E-01, 9.83100000E-01},
    {2.38012800E+02, 1.29000000E+02, 1.21000000E+02, 9.57600000E-01, 1.86270000E+00},
    {2.29722600E+02, 1.29000000E+02, 1.22000000E+02, 9.57600000E-01, 1.82990000E+00},
    {2.25202000E+02, 1.29000000E+02, 1.23000000E+02, 9.57600000E-01, 1.91380000E+00},
    {2.23400900E+02, 1.29000000E+02, 1.24000000E+02, 9.57600000E-01, 1.82690000E+00},
    {2.04448000E+02, 1.29000000E+02, 1.25000000E+02, 9.57600000E-01, 1.64060000E+00},
    {1.88944500E+02, 1.29000000E+02, 1.26000000E+02, 9.57600000E-01, 1.64830000E+00},
    {1.80271900E+02, 1.29000000E+02, 1.27000000E+02, 9.57600000E-01, 1.71490000E+00},
    {1.76335800E+02, 1.29000000E+02, 1.28000000E+02, 9.57600000E-01, 1.79370000E+00},
    {1.74996900E+02, 1.29000000E+02, 1.29000000E+02, 9.57600000E-01, 9.57600000E-01},
    {2.12139000E+01, 1.30000000E+02, 1.00000000E+00, 1.94190000E+00, 9.11800000E-01},
    {1.41185000E+01, 1.30000000E+02, 2.00000000E+00, 1.94190000E+00, 0.00000000E+00},
    {3.17882600E+02, 1.30000000E+02, 3.00000000E+00, 1.94190000E+00, 0.00000000E+00},
    {1.87301200E+02, 1.30000000E+02, 4.00000000E+00, 1.94190000E+00, 0.00000000E+00},
    {1.27248800E+02, 1.30000000E+02, 5.00000000E+00, 1.94190000E+00, 0.00000000E+00},
    {8.64709000E+01, 1.30000000E+02, 6.00000000E+00, 1.94190000E+00, 0.00000000E+00},
    {6.07331000E+01, 1.30000000E+02, 7.00000000E+00, 1.94190000E+00, 0.00000000E+00},
    {4.61355000E+01, 1.30000000E+02, 8.00000000E+00, 1.94190000E+00, 0.00000000E+00},
    {3.50629000E+01, 1.30000000E+02, 9.00000000E+00, 1.94190000E+00, 0.00000000E+00},
    {2.70524000E+01, 1.30000000E+02, 1.00000000E+01, 1.94190000E+00, 0.00000000E+00},
    {3.80656800E+02, 1.30000000E+02, 1.10000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.97544000E+02, 1.30000000E+02, 1.20000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.75629900E+02, 1.30000000E+02, 1.30000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.18528200E+02, 1.30000000E+02, 1.40000000E+01, 1.94190000E+00, 0.00000000E+00},
    {1.71188900E+02, 1.30000000E+02, 1.50000000E+01, 1.94190000E+00, 0.00000000E+00},
    {1.42459100E+02, 1.30000000E+02, 1.60000000E+01, 1.94190000E+00, 0.00000000E+00},
    {1.16663700E+02, 1.30000000E+02, 1.70000000E+01, 1.94190000E+00, 0.00000000E+00},
    {9.56586000E+01, 1.30000000E+02, 1.80000000E+01, 1.94190000E+00, 0.00000000E+00},
    {6.21484100E+02, 1.30000000E+02, 1.90000000E+01, 1.94190000E+00, 0.00000000E+00},
    {5.19518200E+02, 1.30000000E+02, 2.00000000E+01, 1.94190000E+00, 0.00000000E+00},
    {4.30367500E+02, 1.30000000E+02, 2.10000000E+01, 1.94190000E+00, 0.00000000E+00},
    {4.16532800E+02, 1.30000000E+02, 2.20000000E+01, 1.94190000E+00, 0.00000000E+00},
    {3.81962900E+02, 1.30000000E+02, 2.30000000E+01, 1.94190000E+00, 0.00000000E+00},
    {3.00932300E+02, 1.30000000E+02, 2.40000000E+01, 1.94190000E+00, 0.00000000E+00},
    {3.29519600E+02, 1.30000000E+02, 2.50000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.58729000E+02, 1.30000000E+02, 2.60000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.75038100E+02, 1.30000000E+02, 2.70000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.82965600E+02, 1.30000000E+02, 2.80000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.16957000E+02, 1.30000000E+02, 2.90000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.23662200E+02, 1.30000000E+02, 3.00000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.64472400E+02, 1.30000000E+02, 3.10000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.34340700E+02, 1.30000000E+02, 3.20000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.00640800E+02, 1.30000000E+02, 3.30000000E+01, 1.94190000E+00, 0.00000000E+00},
    {1.80418900E+02, 1.30000000E+02, 3.40000000E+01, 1.94190000E+00, 0.00000000E+00},
    {1.58221300E+02, 1.30000000E+02, 3.50000000E+01, 1.94190000E+00, 0.00000000E+00},
    {1.37859000E+02, 1.30000000E+02, 3.60000000E+01, 1.94190000E+00, 0.00000000E+00},
    {6.97201200E+02, 1.30000000E+02, 3.70000000E+01, 1.94190000E+00, 0.00000000E+00},
    {6.18592800E+02, 1.30000000E+02, 3.80000000E+01, 1.94190000E+00, 0.00000000E+00},
    {5.44314000E+02, 1.30000000E+02, 3.90000000E+01, 1.94190000E+00, 0.00000000E+00},
    {4.90509700E+02, 1.30000000E+02, 4.00000000E+01, 1.94190000E+00, 0.00000000E+00},
    {4.48032800E+02, 1.30000000E+02, 4.10000000E+01, 1.94190000E+00, 0.00000000E+00},
    {3.46838700E+02, 1.30000000E+02, 4.20000000E+01, 1.94190000E+00, 0.00000000E+00},
    {3.86625600E+02, 1.30000000E+02, 4.30000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.95430700E+02, 1.30000000E+02, 4.40000000E+01, 1.94190000E+00, 0.00000000E+00},
    {3.22983100E+02, 1.30000000E+02, 4.50000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.99831700E+02, 1.30000000E+02, 4.60000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.49834800E+02, 1.30000000E+02, 4.70000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.64571600E+02, 1.30000000E+02, 4.80000000E+01, 1.94190000E+00, 0.00000000E+00},
    {3.30958500E+02, 1.30000000E+02, 4.90000000E+01, 1.94190000E+00, 0.00000000E+00},
    {3.07263500E+02, 1.30000000E+02, 5.00000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.74781600E+02, 1.30000000E+02, 5.10000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.55428900E+02, 1.30000000E+02, 5.20000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.31400600E+02, 1.30000000E+02, 5.30000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.08405500E+02, 1.30000000E+02, 5.40000000E+01, 1.94190000E+00, 0.00000000E+00},
    {8.49344600E+02, 1.30000000E+02, 5.50000000E+01, 1.94190000E+00, 0.00000000E+00},
    {7.87277100E+02, 1.30000000E+02, 5.60000000E+01, 1.94190000E+00, 0.00000000E+00},
    {6.94795900E+02, 1.30000000E+02, 5.70000000E+01, 1.94190000E+00, 0.00000000E+00},
    {3.24336500E+02, 1.30000000E+02, 5.80000000E+01, 1.94190000E+00, 2.79910000E+00},
    {6.98724300E+02, 1.30000000E+02, 5.90000000E+01, 1.94190000E+00, 0.00000000E+00},
    {6.71473200E+02, 1.30000000E+02, 6.00000000E+01, 1.94190000E+00, 0.00000000E+00},
    {6.54777600E+02, 1.30000000E+02, 6.10000000E+01, 1.94190000E+00, 0.00000000E+00},
    {6.39407500E+02, 1.30000000E+02, 6.20000000E+01, 1.94190000E+00, 0.00000000E+00},
    {6.25782000E+02, 1.30000000E+02, 6.30000000E+01, 1.94190000E+00, 0.00000000E+00},
    {4.94424300E+02, 1.30000000E+02, 6.40000000E+01, 1.94190000E+00, 0.00000000E+00},
    {5.52580300E+02, 1.30000000E+02, 6.50000000E+01, 1.94190000E+00, 0.00000000E+00},
    {5.33393800E+02, 1.30000000E+02, 6.60000000E+01, 1.94190000E+00, 0.00000000E+00},
    {5.65085800E+02, 1.30000000E+02, 6.70000000E+01, 1.94190000E+00, 0.00000000E+00},
    {5.53168700E+02, 1.30000000E+02, 6.80000000E+01, 1.94190000E+00, 0.00000000E+00},
    {5.42451600E+02, 1.30000000E+02, 6.90000000E+01, 1.94190000E+00, 0.00000000E+00},
    {5.36007700E+02, 1.30000000E+02, 7.00000000E+01, 1.94190000E+00, 0.00000000E+00},
    {4.53076200E+02, 1.30000000E+02, 7.10000000E+01, 1.94190000E+00, 0.00000000E+00},
    {4.47443200E+02, 1.30000000E+02, 7.20000000E+01, 1.94190000E+00, 0.00000000E+00},
    {4.09336900E+02, 1.30000000E+02, 7.30000000E+01, 1.94190000E+00, 0.00000000E+00},
    {3.46236000E+02, 1.30000000E+02, 7.40000000E+01, 1.94190000E+00, 0.00000000E+00},
    {3.52588800E+02, 1.30000000E+02, 7.50000000E+01, 1.94190000E+00, 0.00000000E+00},
    {3.20195900E+02, 1.30000000E+02, 7.60000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.93711300E+02, 1.30000000E+02, 7.70000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.44372100E+02, 1.30000000E+02, 7.80000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.28478500E+02, 1.30000000E+02, 7.90000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.35246900E+02, 1.30000000E+02, 8.00000000E+01, 1.94190000E+00, 0.00000000E+00},
    {3.40066100E+02, 1.30000000E+02, 8.10000000E+01, 1.94190000E+00, 0.00000000E+00},
    {3.33557700E+02, 1.30000000E+02, 8.20000000E+01, 1.94190000E+00, 0.00000000E+00},
    {3.07487100E+02, 1.30000000E+02, 8.30000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.93762200E+02, 1.30000000E+02, 8.40000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.71635200E+02, 1.30000000E+02, 8.50000000E+01, 1.94190000E+00, 0.00000000E+00},
    {2.49380900E+02, 1.30000000E+02, 8.60000000E+01, 1.94190000E+00, 0.00000000E+00},
    {8.04990200E+02, 1.30000000E+02, 8.70000000E+01, 1.94190000E+00, 0.00000000E+00},
    {7.80267500E+02, 1.30000000E+02, 8.80000000E+01, 1.94190000E+00, 0.00000000E+00},
    {6.92682800E+02, 1.30000000E+02, 8.90000000E+01, 1.94190000E+00, 0.00000000E+00},
    {6.25293300E+02, 1.30000000E+02, 9.00000000E+01, 1.94190000E+00, 0.00000000E+00},
    {6.19438400E+02, 1.30000000E+02, 9.10000000E+01, 1.94190000E+00, 0.00000000E+00},
    {5.99855900E+02, 1.30000000E+02, 9.20000000E+01, 1.94190000E+00, 0.00000000E+00},
    {6.15960700E+02, 1.30000000E+02, 9.30000000E+01, 1.94190000E+00, 0.00000000E+00},
    {5.96816900E+02, 1.30000000E+02, 9.40000000E+01, 1.94190000E+00, 0.00000000E+00},
    {3.40312000E+01, 1.30000000E+02, 1.01000000E+02, 1.94190000E+00, 0.00000000E+00},
    {1.09012700E+02, 1.30000000E+02, 1.03000000E+02, 1.94190000E+00, 9.86500000E-01},
    {1.39242000E+02, 1.30000000E+02, 1.04000000E+02, 1.94190000E+00, 9.80800000E-01},
    {1.07202600E+02, 1.30000000E+02, 1.05000000E+02, 1.94190000E+00, 9.70600000E-01},
    {8.10686000E+01, 1.30000000E+02, 1.06000000E+02, 1.94190000E+00, 9.86800000E-01},
    {5.66140000E+01, 1.30000000E+02, 1.07000000E+02, 1.94190000E+00, 9.94400000E-01},
    {4.13862000E+01, 1.30000000E+02, 1.08000000E+02, 1.94190000E+00, 9.92500000E-01},
    {2.86291000E+01, 1.30000000E+02, 1.09000000E+02, 1.94190000E+00, 9.98200000E-01},
    {1.59122800E+02, 1.30000000E+02, 1.11000000E+02, 1.94190000E+00, 9.68400000E-01},
    {2.45805400E+02, 1.30000000E+02, 1.12000000E+02, 1.94190000E+00, 9.62800000E-01},
    {2.50073000E+02, 1.30000000E+02, 1.13000000E+02, 1.94190000E+00, 9.64800000E-01},
    {2.02130600E+02, 1.30000000E+02, 1.14000000E+02, 1.94190000E+00, 9.50700000E-01},
    {1.66188700E+02, 1.30000000E+02, 1.15000000E+02, 1.94190000E+00, 9.94700000E-01},
    {1.40880400E+02, 1.30000000E+02, 1.16000000E+02, 1.94190000E+00, 9.94800000E-01},
    {1.15450100E+02, 1.30000000E+02, 1.17000000E+02, 1.94190000E+00, 9.97200000E-01},
    {2.19985200E+02, 1.30000000E+02, 1.19000000E+02, 1.94190000E+00, 9.76700000E-01},
    {4.15563800E+02, 1.30000000E+02, 1.20000000E+02, 1.94190000E+00, 9.83100000E-01},
    {2.21246700E+02, 1.30000000E+02, 1.21000000E+02, 1.94190000E+00, 1.86270000E+00},
    {2.13619100E+02, 1.30000000E+02, 1.22000000E+02, 1.94190000E+00, 1.82990000E+00},
    {2.09342800E+02, 1.30000000E+02, 1.23000000E+02, 1.94190000E+00, 1.91380000E+00},
    {2.07292600E+02, 1.30000000E+02, 1.24000000E+02, 1.94190000E+00, 1.82690000E+00},
    {1.91312500E+02, 1.30000000E+02, 1.25000000E+02, 1.94190000E+00, 1.64060000E+00},
    {1.77203400E+02, 1.30000000E+02, 1.26000000E+02, 1.94190000E+00, 1.64830000E+00},
    {1.69046100E+02, 1.30000000E+02, 1.27000000E+02, 1.94190000E+00, 1.71490000E+00},
    {1.65236100E+02, 1.30000000E+02, 1.28000000E+02, 1.94190000E+00, 1.79370000E+00},
    {1.62923600E+02, 1.30000000E+02, 1.29000000E+02, 1.94190000E+00, 9.57600000E-01},
    {1.53452800E+02, 1.30000000E+02, 1.30000000E+02, 1.94190000E+00, 1.94190000E+00},
    {3.32515000E+01, 1.31000000E+02, 1.00000000E+00, 9.60100000E-01, 9.11800000E-01},
    {2.09874000E+01, 1.31000000E+02, 2.00000000E+00, 9.60100000E-01, 0.00000000E+00},
    {6.02810400E+02, 1.31000000E+02, 3.00000000E+00, 9.60100000E-01, 0.00000000E+00},
    {3.26667200E+02, 1.31000000E+02, 4.00000000E+00, 9.60100000E-01, 0.00000000E+00},
    {2.11275600E+02, 1.31000000E+02, 5.00000000E+00, 9.60100000E-01, 0.00000000E+00},
    {1.38149700E+02, 1.31000000E+02, 6.00000000E+00, 9.60100000E-01, 0.00000000E+00},
    {9.41329000E+01, 1.31000000E+02, 7.00000000E+00, 9.60100000E-01, 0.00000000E+00},
    {6.99039000E+01, 1.31000000E+02, 8.00000000E+00, 9.60100000E-01, 0.00000000E+00},
    {5.20476000E+01, 1.31000000E+02, 9.00000000E+00, 9.60100000E-01, 0.00000000E+00},
    {3.94587000E+01, 1.31000000E+02, 1.00000000E+01, 9.60100000E-01, 0.00000000E+00},
    {7.17753100E+02, 1.31000000E+02, 1.10000000E+01, 9.60100000E-01, 0.00000000E+00},
    {5.26423600E+02, 1.31000000E+02, 1.20000000E+01, 9.60100000E-01, 0.00000000E+00},
    {4.75731800E+02, 1.31000000E+02, 1.30000000E+01, 9.60100000E-01, 0.00000000E+00},
    {3.64611000E+02, 1.31000000E+02, 1.40000000E+01, 9.60100000E-01, 0.00000000E+00},
    {2.77352300E+02, 1.31000000E+02, 1.50000000E+01, 9.60100000E-01, 0.00000000E+00},
    {2.26279600E+02, 1.31000000E+02, 1.60000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.81717800E+02, 1.31000000E+02, 1.70000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.46385500E+02, 1.31000000E+02, 1.80000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.18483270E+03, 1.31000000E+02, 1.90000000E+01, 9.60100000E-01, 0.00000000E+00},
    {9.42582800E+02, 1.31000000E+02, 2.00000000E+01, 9.60100000E-01, 0.00000000E+00},
    {7.71976300E+02, 1.31000000E+02, 2.10000000E+01, 9.60100000E-01, 0.00000000E+00},
    {7.39028100E+02, 1.31000000E+02, 2.20000000E+01, 9.60100000E-01, 0.00000000E+00},
    {6.73282300E+02, 1.31000000E+02, 2.30000000E+01, 9.60100000E-01, 0.00000000E+00},
    {5.29000600E+02, 1.31000000E+02, 2.40000000E+01, 9.60100000E-01, 0.00000000E+00},
    {5.75322900E+02, 1.31000000E+02, 2.50000000E+01, 9.60100000E-01, 0.00000000E+00},
    {4.49904000E+02, 1.31000000E+02, 2.60000000E+01, 9.60100000E-01, 0.00000000E+00},
    {4.72640100E+02, 1.31000000E+02, 2.70000000E+01, 9.60100000E-01, 0.00000000E+00},
    {4.89549900E+02, 1.31000000E+02, 2.80000000E+01, 9.60100000E-01, 0.00000000E+00},
    {3.74253700E+02, 1.31000000E+02, 2.90000000E+01, 9.60100000E-01, 0.00000000E+00},
    {3.78732900E+02, 1.31000000E+02, 3.00000000E+01, 9.60100000E-01, 0.00000000E+00},
    {4.51400300E+02, 1.31000000E+02, 3.10000000E+01, 9.60100000E-01, 0.00000000E+00},
    {3.89737000E+02, 1.31000000E+02, 3.20000000E+01, 9.60100000E-01, 0.00000000E+00},
    {3.25726800E+02, 1.31000000E+02, 3.30000000E+01, 9.60100000E-01, 0.00000000E+00},
    {2.88393400E+02, 1.31000000E+02, 3.40000000E+01, 9.60100000E-01, 0.00000000E+00},
    {2.48821100E+02, 1.31000000E+02, 3.50000000E+01, 9.60100000E-01, 0.00000000E+00},
    {2.13482100E+02, 1.31000000E+02, 3.60000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.32273220E+03, 1.31000000E+02, 3.70000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.12391670E+03, 1.31000000E+02, 3.80000000E+01, 9.60100000E-01, 0.00000000E+00},
    {9.69077700E+02, 1.31000000E+02, 3.90000000E+01, 9.60100000E-01, 0.00000000E+00},
    {8.62036300E+02, 1.31000000E+02, 4.00000000E+01, 9.60100000E-01, 0.00000000E+00},
    {7.80480500E+02, 1.31000000E+02, 4.10000000E+01, 9.60100000E-01, 0.00000000E+00},
    {5.94576500E+02, 1.31000000E+02, 4.20000000E+01, 9.60100000E-01, 0.00000000E+00},
    {6.66760100E+02, 1.31000000E+02, 4.30000000E+01, 9.60100000E-01, 0.00000000E+00},
    {5.00516200E+02, 1.31000000E+02, 4.40000000E+01, 9.60100000E-01, 0.00000000E+00},
    {5.47834600E+02, 1.31000000E+02, 4.50000000E+01, 9.60100000E-01, 0.00000000E+00},
    {5.05669800E+02, 1.31000000E+02, 4.60000000E+01, 9.60100000E-01, 0.00000000E+00},
    {4.21742000E+02, 1.31000000E+02, 4.70000000E+01, 9.60100000E-01, 0.00000000E+00},
    {4.42863900E+02, 1.31000000E+02, 4.80000000E+01, 9.60100000E-01, 0.00000000E+00},
    {5.64156400E+02, 1.31000000E+02, 4.90000000E+01, 9.60100000E-01, 0.00000000E+00},
    {5.13031500E+02, 1.31000000E+02, 5.00000000E+01, 9.60100000E-01, 0.00000000E+00},
    {4.49300600E+02, 1.31000000E+02, 5.10000000E+01, 9.60100000E-01, 0.00000000E+00},
    {4.12279300E+02, 1.31000000E+02, 5.20000000E+01, 9.60100000E-01, 0.00000000E+00},
    {3.68218300E+02, 1.31000000E+02, 5.30000000E+01, 9.60100000E-01, 0.00000000E+00},
    {3.27083800E+02, 1.31000000E+02, 5.40000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.60989510E+03, 1.31000000E+02, 5.50000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.43936100E+03, 1.31000000E+02, 5.60000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.24503820E+03, 1.31000000E+02, 5.70000000E+01, 9.60100000E-01, 0.00000000E+00},
    {5.32068300E+02, 1.31000000E+02, 5.80000000E+01, 9.60100000E-01, 2.79910000E+00},
    {1.26825810E+03, 1.31000000E+02, 5.90000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.21478670E+03, 1.31000000E+02, 6.00000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.18349310E+03, 1.31000000E+02, 6.10000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.15481720E+03, 1.31000000E+02, 6.20000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.12936230E+03, 1.31000000E+02, 6.30000000E+01, 9.60100000E-01, 0.00000000E+00},
    {8.71696600E+02, 1.31000000E+02, 6.40000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.00612730E+03, 1.31000000E+02, 6.50000000E+01, 9.60100000E-01, 0.00000000E+00},
    {9.67419300E+02, 1.31000000E+02, 6.60000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.01474430E+03, 1.31000000E+02, 6.70000000E+01, 9.60100000E-01, 0.00000000E+00},
    {9.92839800E+02, 1.31000000E+02, 6.80000000E+01, 9.60100000E-01, 0.00000000E+00},
    {9.72875300E+02, 1.31000000E+02, 6.90000000E+01, 9.60100000E-01, 0.00000000E+00},
    {9.62268100E+02, 1.31000000E+02, 7.00000000E+01, 9.60100000E-01, 0.00000000E+00},
    {8.00560700E+02, 1.31000000E+02, 7.10000000E+01, 9.60100000E-01, 0.00000000E+00},
    {7.74794900E+02, 1.31000000E+02, 7.20000000E+01, 9.60100000E-01, 0.00000000E+00},
    {6.99548200E+02, 1.31000000E+02, 7.30000000E+01, 9.60100000E-01, 0.00000000E+00},
    {5.85006900E+02, 1.31000000E+02, 7.40000000E+01, 9.60100000E-01, 0.00000000E+00},
    {5.92906600E+02, 1.31000000E+02, 7.50000000E+01, 9.60100000E-01, 0.00000000E+00},
    {5.32195000E+02, 1.31000000E+02, 7.60000000E+01, 9.60100000E-01, 0.00000000E+00},
    {4.83509000E+02, 1.31000000E+02, 7.70000000E+01, 9.60100000E-01, 0.00000000E+00},
    {3.97739100E+02, 1.31000000E+02, 7.80000000E+01, 9.60100000E-01, 0.00000000E+00},
    {3.70151700E+02, 1.31000000E+02, 7.90000000E+01, 9.60100000E-01, 0.00000000E+00},
    {3.79709200E+02, 1.31000000E+02, 8.00000000E+01, 9.60100000E-01, 0.00000000E+00},
    {5.75628100E+02, 1.31000000E+02, 8.10000000E+01, 9.60100000E-01, 0.00000000E+00},
    {5.55830700E+02, 1.31000000E+02, 8.20000000E+01, 9.60100000E-01, 0.00000000E+00},
    {5.02988500E+02, 1.31000000E+02, 8.30000000E+01, 9.60100000E-01, 0.00000000E+00},
    {4.75313400E+02, 1.31000000E+02, 8.40000000E+01, 9.60100000E-01, 0.00000000E+00},
    {4.33662500E+02, 1.31000000E+02, 8.50000000E+01, 9.60100000E-01, 0.00000000E+00},
    {3.93185000E+02, 1.31000000E+02, 8.60000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.50145410E+03, 1.31000000E+02, 8.70000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.41152170E+03, 1.31000000E+02, 8.80000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.22973070E+03, 1.31000000E+02, 8.90000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.08573940E+03, 1.31000000E+02, 9.00000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.08653800E+03, 1.31000000E+02, 9.10000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.05149420E+03, 1.31000000E+02, 9.20000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.09410310E+03, 1.31000000E+02, 9.30000000E+01, 9.60100000E-01, 0.00000000E+00},
    {1.05750950E+03, 1.31000000E+02, 9.40000000E+01, 9.60100000E-01, 0.00000000E+00},
    {5.49919000E+01, 1.31000000E+02, 1.01000000E+02, 9.60100000E-01, 0.00000000E+00},
    {1.88762200E+02, 1.31000000E+02, 1.03000000E+02, 9.60100000E-01, 9.86500000E-01},
    {2.38906800E+02, 1.31000000E+02, 1.04000000E+02, 9.60100000E-01, 9.80800000E-01},
    {1.76097600E+02, 1.31000000E+02, 1.05000000E+02, 9.60100000E-01, 9.70600000E-01},
    {1.29567000E+02, 1.31000000E+02, 1.06000000E+02, 9.60100000E-01, 9.86800000E-01},
    {8.76645000E+01, 1.31000000E+02, 1.07000000E+02, 9.60100000E-01, 9.94400000E-01},
    {6.23522000E+01, 1.31000000E+02, 1.08000000E+02, 9.60100000E-01, 9.92500000E-01},
    {4.15909000E+01, 1.31000000E+02, 1.09000000E+02, 9.60100000E-01, 9.98200000E-01},
    {2.77514300E+02, 1.31000000E+02, 1.11000000E+02, 9.60100000E-01, 9.68400000E-01},
    {4.30829700E+02, 1.31000000E+02, 1.12000000E+02, 9.60100000E-01, 9.62800000E-01},
    {4.28952100E+02, 1.31000000E+02, 1.13000000E+02, 9.60100000E-01, 9.64800000E-01},
    {3.35520600E+02, 1.31000000E+02, 1.14000000E+02, 9.60100000E-01, 9.50700000E-01},
    {2.68937000E+02, 1.31000000E+02, 1.15000000E+02, 9.60100000E-01, 9.94700000E-01},
    {2.23845300E+02, 1.31000000E+02, 1.16000000E+02, 9.60100000E-01, 9.94800000E-01},
    {1.79868300E+02, 1.31000000E+02, 1.17000000E+02, 9.60100000E-01, 9.97200000E-01},
    {3.75849500E+02, 1.31000000E+02, 1.19000000E+02, 9.60100000E-01, 9.76700000E-01},
    {7.51814400E+02, 1.31000000E+02, 1.20000000E+02, 9.60100000E-01, 9.83100000E-01},
    {3.68541700E+02, 1.31000000E+02, 1.21000000E+02, 9.60100000E-01, 1.86270000E+00},
    {3.55349800E+02, 1.31000000E+02, 1.22000000E+02, 9.60100000E-01, 1.82990000E+00},
    {3.48329300E+02, 1.31000000E+02, 1.23000000E+02, 9.60100000E-01, 1.91380000E+00},
    {3.45979600E+02, 1.31000000E+02, 1.24000000E+02, 9.60100000E-01, 1.82690000E+00},
    {3.14424400E+02, 1.31000000E+02, 1.25000000E+02, 9.60100000E-01, 1.64060000E+00},
    {2.89758800E+02, 1.31000000E+02, 1.26000000E+02, 9.60100000E-01, 1.64830000E+00},
    {2.76313000E+02, 1.31000000E+02, 1.27000000E+02, 9.60100000E-01, 1.71490000E+00},
    {2.70402300E+02, 1.31000000E+02, 1.28000000E+02, 9.60100000E-01, 1.79370000E+00},
    {2.69734400E+02, 1.31000000E+02, 1.29000000E+02, 9.60100000E-01, 9.57600000E-01},
    {2.48700300E+02, 1.31000000E+02, 1.30000000E+02, 9.60100000E-01, 1.94190000E+00},
    {4.21649700E+02, 1.31000000E+02, 1.31000000E+02, 9.60100000E-01, 9.60100000E-01},
    {2.99527000E+01, 1.32000000E+02, 1.00000000E+00, 9.43400000E-01, 9.11800000E-01},
    {1.93761000E+01, 1.32000000E+02, 2.00000000E+00, 9.43400000E-01, 0.00000000E+00},
    {4.84016500E+02, 1.32000000E+02, 3.00000000E+00, 9.43400000E-01, 0.00000000E+00},
    {2.76571700E+02, 1.32000000E+02, 4.00000000E+00, 9.43400000E-01, 0.00000000E+00},
    {1.84205500E+02, 1.32000000E+02, 5.00000000E+00, 9.43400000E-01, 0.00000000E+00},
    {1.23057400E+02, 1.32000000E+02, 6.00000000E+00, 9.43400000E-01, 0.00000000E+00},
    {8.51621000E+01, 1.32000000E+02, 7.00000000E+00, 9.43400000E-01, 0.00000000E+00},
    {6.39245000E+01, 1.32000000E+02, 8.00000000E+00, 9.43400000E-01, 0.00000000E+00},
    {4.80233000E+01, 1.32000000E+02, 9.00000000E+00, 9.43400000E-01, 0.00000000E+00},
    {3.66615000E+01, 1.32000000E+02, 1.00000000E+01, 9.43400000E-01, 0.00000000E+00},
    {5.78088200E+02, 1.32000000E+02, 1.10000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.41523900E+02, 1.32000000E+02, 1.20000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.05161100E+02, 1.32000000E+02, 1.30000000E+01, 9.43400000E-01, 0.00000000E+00},
    {3.16970000E+02, 1.32000000E+02, 1.40000000E+01, 9.43400000E-01, 0.00000000E+00},
    {2.45268300E+02, 1.32000000E+02, 1.50000000E+01, 9.43400000E-01, 0.00000000E+00},
    {2.02286300E+02, 1.32000000E+02, 1.60000000E+01, 9.43400000E-01, 0.00000000E+00},
    {1.64126600E+02, 1.32000000E+02, 1.70000000E+01, 9.43400000E-01, 0.00000000E+00},
    {1.33379400E+02, 1.32000000E+02, 1.80000000E+01, 9.43400000E-01, 0.00000000E+00},
    {9.46575600E+02, 1.32000000E+02, 1.90000000E+01, 9.43400000E-01, 0.00000000E+00},
    {7.77536100E+02, 1.32000000E+02, 2.00000000E+01, 9.43400000E-01, 0.00000000E+00},
    {6.41406400E+02, 1.32000000E+02, 2.10000000E+01, 9.43400000E-01, 0.00000000E+00},
    {6.18040100E+02, 1.32000000E+02, 2.20000000E+01, 9.43400000E-01, 0.00000000E+00},
    {5.65263900E+02, 1.32000000E+02, 2.30000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.44419600E+02, 1.32000000E+02, 2.40000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.85764800E+02, 1.32000000E+02, 2.50000000E+01, 9.43400000E-01, 0.00000000E+00},
    {3.80385300E+02, 1.32000000E+02, 2.60000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.02914600E+02, 1.32000000E+02, 2.70000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.15643900E+02, 1.32000000E+02, 2.80000000E+01, 9.43400000E-01, 0.00000000E+00},
    {3.17862400E+02, 1.32000000E+02, 2.90000000E+01, 9.43400000E-01, 0.00000000E+00},
    {3.25692800E+02, 1.32000000E+02, 3.00000000E+01, 9.43400000E-01, 0.00000000E+00},
    {3.86712900E+02, 1.32000000E+02, 3.10000000E+01, 9.43400000E-01, 0.00000000E+00},
    {3.39240400E+02, 1.32000000E+02, 3.20000000E+01, 9.43400000E-01, 0.00000000E+00},
    {2.87584700E+02, 1.32000000E+02, 3.30000000E+01, 9.43400000E-01, 0.00000000E+00},
    {2.56851700E+02, 1.32000000E+02, 3.40000000E+01, 9.43400000E-01, 0.00000000E+00},
    {2.23574400E+02, 1.32000000E+02, 3.50000000E+01, 9.43400000E-01, 0.00000000E+00},
    {1.93363400E+02, 1.32000000E+02, 3.60000000E+01, 9.43400000E-01, 0.00000000E+00},
    {1.05960440E+03, 1.32000000E+02, 3.70000000E+01, 9.43400000E-01, 0.00000000E+00},
    {9.26002500E+02, 1.32000000E+02, 3.80000000E+01, 9.43400000E-01, 0.00000000E+00},
    {8.08599100E+02, 1.32000000E+02, 3.90000000E+01, 9.43400000E-01, 0.00000000E+00},
    {7.24980500E+02, 1.32000000E+02, 4.00000000E+01, 9.43400000E-01, 0.00000000E+00},
    {6.59803900E+02, 1.32000000E+02, 4.10000000E+01, 9.43400000E-01, 0.00000000E+00},
    {5.07150300E+02, 1.32000000E+02, 4.20000000E+01, 9.43400000E-01, 0.00000000E+00},
    {5.66829400E+02, 1.32000000E+02, 4.30000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.29699600E+02, 1.32000000E+02, 4.40000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.70307800E+02, 1.32000000E+02, 4.50000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.35526800E+02, 1.32000000E+02, 4.60000000E+01, 9.43400000E-01, 0.00000000E+00},
    {3.62507600E+02, 1.32000000E+02, 4.70000000E+01, 9.43400000E-01, 0.00000000E+00},
    {3.83020400E+02, 1.32000000E+02, 4.80000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.82911500E+02, 1.32000000E+02, 4.90000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.44904800E+02, 1.32000000E+02, 5.00000000E+01, 9.43400000E-01, 0.00000000E+00},
    {3.94582200E+02, 1.32000000E+02, 5.10000000E+01, 9.43400000E-01, 0.00000000E+00},
    {3.64824300E+02, 1.32000000E+02, 5.20000000E+01, 9.43400000E-01, 0.00000000E+00},
    {3.28475000E+02, 1.32000000E+02, 5.30000000E+01, 9.43400000E-01, 0.00000000E+00},
    {2.93996600E+02, 1.32000000E+02, 5.40000000E+01, 9.43400000E-01, 0.00000000E+00},
    {1.29032490E+03, 1.32000000E+02, 5.50000000E+01, 9.43400000E-01, 0.00000000E+00},
    {1.18097290E+03, 1.32000000E+02, 5.60000000E+01, 9.43400000E-01, 0.00000000E+00},
    {1.03449000E+03, 1.32000000E+02, 5.70000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.66188000E+02, 1.32000000E+02, 5.80000000E+01, 9.43400000E-01, 2.79910000E+00},
    {1.04486240E+03, 1.32000000E+02, 5.90000000E+01, 9.43400000E-01, 0.00000000E+00},
    {1.00291380E+03, 1.32000000E+02, 6.00000000E+01, 9.43400000E-01, 0.00000000E+00},
    {9.77652300E+02, 1.32000000E+02, 6.10000000E+01, 9.43400000E-01, 0.00000000E+00},
    {9.54443100E+02, 1.32000000E+02, 6.20000000E+01, 9.43400000E-01, 0.00000000E+00},
    {9.33861500E+02, 1.32000000E+02, 6.30000000E+01, 9.43400000E-01, 0.00000000E+00},
    {7.30922700E+02, 1.32000000E+02, 6.40000000E+01, 9.43400000E-01, 0.00000000E+00},
    {8.26510400E+02, 1.32000000E+02, 6.50000000E+01, 9.43400000E-01, 0.00000000E+00},
    {7.96629900E+02, 1.32000000E+02, 6.60000000E+01, 9.43400000E-01, 0.00000000E+00},
    {8.41752600E+02, 1.32000000E+02, 6.70000000E+01, 9.43400000E-01, 0.00000000E+00},
    {8.23868700E+02, 1.32000000E+02, 6.80000000E+01, 9.43400000E-01, 0.00000000E+00},
    {8.07703100E+02, 1.32000000E+02, 6.90000000E+01, 9.43400000E-01, 0.00000000E+00},
    {7.98438000E+02, 1.32000000E+02, 7.00000000E+01, 9.43400000E-01, 0.00000000E+00},
    {6.70609700E+02, 1.32000000E+02, 7.10000000E+01, 9.43400000E-01, 0.00000000E+00},
    {6.57556500E+02, 1.32000000E+02, 7.20000000E+01, 9.43400000E-01, 0.00000000E+00},
    {5.98367800E+02, 1.32000000E+02, 7.30000000E+01, 9.43400000E-01, 0.00000000E+00},
    {5.03443100E+02, 1.32000000E+02, 7.40000000E+01, 9.43400000E-01, 0.00000000E+00},
    {5.11797500E+02, 1.32000000E+02, 7.50000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.62504200E+02, 1.32000000E+02, 7.60000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.22487600E+02, 1.32000000E+02, 7.70000000E+01, 9.43400000E-01, 0.00000000E+00},
    {3.49516000E+02, 1.32000000E+02, 7.80000000E+01, 9.43400000E-01, 0.00000000E+00},
    {3.26007200E+02, 1.32000000E+02, 7.90000000E+01, 9.43400000E-01, 0.00000000E+00},
    {3.35304500E+02, 1.32000000E+02, 8.00000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.94293400E+02, 1.32000000E+02, 8.10000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.82154800E+02, 1.32000000E+02, 8.20000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.41286800E+02, 1.32000000E+02, 8.30000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.19740700E+02, 1.32000000E+02, 8.40000000E+01, 9.43400000E-01, 0.00000000E+00},
    {3.85941900E+02, 1.32000000E+02, 8.50000000E+01, 9.43400000E-01, 0.00000000E+00},
    {3.52374900E+02, 1.32000000E+02, 8.60000000E+01, 9.43400000E-01, 0.00000000E+00},
    {1.21544540E+03, 1.32000000E+02, 8.70000000E+01, 9.43400000E-01, 0.00000000E+00},
    {1.16571070E+03, 1.32000000E+02, 8.80000000E+01, 9.43400000E-01, 0.00000000E+00},
    {1.02748590E+03, 1.32000000E+02, 8.90000000E+01, 9.43400000E-01, 0.00000000E+00},
    {9.19351700E+02, 1.32000000E+02, 9.00000000E+01, 9.43400000E-01, 0.00000000E+00},
    {9.13902300E+02, 1.32000000E+02, 9.10000000E+01, 9.43400000E-01, 0.00000000E+00},
    {8.84715300E+02, 1.32000000E+02, 9.20000000E+01, 9.43400000E-01, 0.00000000E+00},
    {9.12970500E+02, 1.32000000E+02, 9.30000000E+01, 9.43400000E-01, 0.00000000E+00},
    {8.83778000E+02, 1.32000000E+02, 9.40000000E+01, 9.43400000E-01, 0.00000000E+00},
    {4.87247000E+01, 1.32000000E+02, 1.01000000E+02, 9.43400000E-01, 0.00000000E+00},
    {1.60450200E+02, 1.32000000E+02, 1.03000000E+02, 9.43400000E-01, 9.86500000E-01},
    {2.04169000E+02, 1.32000000E+02, 1.04000000E+02, 9.43400000E-01, 9.80800000E-01},
    {1.54440900E+02, 1.32000000E+02, 1.05000000E+02, 9.43400000E-01, 9.70600000E-01},
    {1.15348800E+02, 1.32000000E+02, 1.06000000E+02, 9.43400000E-01, 9.86800000E-01},
    {7.93278000E+01, 1.32000000E+02, 1.07000000E+02, 9.43400000E-01, 9.94400000E-01},
    {5.71752000E+01, 1.32000000E+02, 1.08000000E+02, 9.43400000E-01, 9.92500000E-01},
    {3.87583000E+01, 1.32000000E+02, 1.09000000E+02, 9.43400000E-01, 9.98200000E-01},
    {2.34604800E+02, 1.32000000E+02, 1.11000000E+02, 9.43400000E-01, 9.68400000E-01},
    {3.63332000E+02, 1.32000000E+02, 1.12000000E+02, 9.43400000E-01, 9.62800000E-01},
    {3.66656100E+02, 1.32000000E+02, 1.13000000E+02, 9.43400000E-01, 9.64800000E-01},
    {2.92533500E+02, 1.32000000E+02, 1.14000000E+02, 9.43400000E-01, 9.50700000E-01},
    {2.37958000E+02, 1.32000000E+02, 1.15000000E+02, 9.43400000E-01, 9.94700000E-01},
    {2.00052600E+02, 1.32000000E+02, 1.16000000E+02, 9.43400000E-01, 9.94800000E-01},
    {1.62420600E+02, 1.32000000E+02, 1.17000000E+02, 9.43400000E-01, 9.97200000E-01},
    {3.21157400E+02, 1.32000000E+02, 1.19000000E+02, 9.43400000E-01, 9.76700000E-01},
    {6.20673500E+02, 1.32000000E+02, 1.20000000E+02, 9.43400000E-01, 9.83100000E-01},
    {3.20134200E+02, 1.32000000E+02, 1.21000000E+02, 9.43400000E-01, 1.86270000E+00},
    {3.08830400E+02, 1.32000000E+02, 1.22000000E+02, 9.43400000E-01, 1.82990000E+00},
    {3.02627200E+02, 1.32000000E+02, 1.23000000E+02, 9.43400000E-01, 1.91380000E+00},
    {2.99970600E+02, 1.32000000E+02, 1.24000000E+02, 9.43400000E-01, 1.82690000E+00},
    {2.75260700E+02, 1.32000000E+02, 1.25000000E+02, 9.43400000E-01, 1.64060000E+00},
    {2.54373500E+02, 1.32000000E+02, 1.26000000E+02, 9.43400000E-01, 1.64830000E+00},
    {2.42557200E+02, 1.32000000E+02, 1.27000000E+02, 9.43400000E-01, 1.71490000E+00},
    {2.37173700E+02, 1.32000000E+02, 1.28000000E+02, 9.43400000E-01, 1.79370000E+00},
    {2.34832700E+02, 1.32000000E+02, 1.29000000E+02, 9.43400000E-01, 9.57600000E-01},
    {2.19450800E+02, 1.32000000E+02, 1.30000000E+02, 9.43400000E-01, 1.94190000E+00},
    {3.62683200E+02, 1.32000000E+02, 1.31000000E+02, 9.43400000E-01, 9.60100000E-01},
    {3.16852100E+02, 1.32000000E+02, 1.32000000E+02, 9.43400000E-01, 9.43400000E-01},
    {2.73574000E+01, 1.33000000E+02, 1.00000000E+00, 9.88900000E-01, 9.11800000E-01},
    {1.80816000E+01, 1.33000000E+02, 2.00000000E+00, 9.88900000E-01, 0.00000000E+00},
    {4.06342700E+02, 1.33000000E+02, 3.00000000E+00, 9.88900000E-01, 0.00000000E+00},
    {2.40755900E+02, 1.33000000E+02, 4.00000000E+00, 9.88900000E-01, 0.00000000E+00},
    {1.63965500E+02, 1.33000000E+02, 5.00000000E+00, 9.88900000E-01, 0.00000000E+00},
    {1.11425700E+02, 1.33000000E+02, 6.00000000E+00, 9.88900000E-01, 0.00000000E+00},
    {7.81138000E+01, 1.33000000E+02, 7.00000000E+00, 9.88900000E-01, 0.00000000E+00},
    {5.91764000E+01, 1.33000000E+02, 8.00000000E+00, 9.88900000E-01, 0.00000000E+00},
    {4.48079000E+01, 1.33000000E+02, 9.00000000E+00, 9.88900000E-01, 0.00000000E+00},
    {3.44226000E+01, 1.33000000E+02, 1.00000000E+01, 9.88900000E-01, 0.00000000E+00},
    {4.86463800E+02, 1.33000000E+02, 1.10000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.81839600E+02, 1.33000000E+02, 1.20000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.54370600E+02, 1.33000000E+02, 1.30000000E+01, 9.88900000E-01, 0.00000000E+00},
    {2.81489300E+02, 1.33000000E+02, 1.40000000E+01, 9.88900000E-01, 0.00000000E+00},
    {2.20714600E+02, 1.33000000E+02, 1.50000000E+01, 9.88900000E-01, 0.00000000E+00},
    {1.83646200E+02, 1.33000000E+02, 1.60000000E+01, 9.88900000E-01, 0.00000000E+00},
    {1.50274200E+02, 1.33000000E+02, 1.70000000E+01, 9.88900000E-01, 0.00000000E+00},
    {1.23035800E+02, 1.33000000E+02, 1.80000000E+01, 9.88900000E-01, 0.00000000E+00},
    {7.93630600E+02, 1.33000000E+02, 1.90000000E+01, 9.88900000E-01, 0.00000000E+00},
    {6.65322100E+02, 1.33000000E+02, 2.00000000E+01, 9.88900000E-01, 0.00000000E+00},
    {5.51499900E+02, 1.33000000E+02, 2.10000000E+01, 9.88900000E-01, 0.00000000E+00},
    {5.34026600E+02, 1.33000000E+02, 2.20000000E+01, 9.88900000E-01, 0.00000000E+00},
    {4.89821000E+02, 1.33000000E+02, 2.30000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.85657700E+02, 1.33000000E+02, 2.40000000E+01, 9.88900000E-01, 0.00000000E+00},
    {4.22687400E+02, 1.33000000E+02, 2.50000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.31638900E+02, 1.33000000E+02, 2.60000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.53007400E+02, 1.33000000E+02, 2.70000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.63056600E+02, 1.33000000E+02, 2.80000000E+01, 9.88900000E-01, 0.00000000E+00},
    {2.78057800E+02, 1.33000000E+02, 2.90000000E+01, 9.88900000E-01, 0.00000000E+00},
    {2.87177800E+02, 1.33000000E+02, 3.00000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.39962500E+02, 1.33000000E+02, 3.10000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.01680000E+02, 1.33000000E+02, 3.20000000E+01, 9.88900000E-01, 0.00000000E+00},
    {2.58526300E+02, 1.33000000E+02, 3.30000000E+01, 9.88900000E-01, 0.00000000E+00},
    {2.32507200E+02, 1.33000000E+02, 3.40000000E+01, 9.88900000E-01, 0.00000000E+00},
    {2.03840000E+02, 1.33000000E+02, 3.50000000E+01, 9.88900000E-01, 0.00000000E+00},
    {1.77471600E+02, 1.33000000E+02, 3.60000000E+01, 9.88900000E-01, 0.00000000E+00},
    {8.90493100E+02, 1.33000000E+02, 3.70000000E+01, 9.88900000E-01, 0.00000000E+00},
    {7.92081600E+02, 1.33000000E+02, 3.80000000E+01, 9.88900000E-01, 0.00000000E+00},
    {6.97803700E+02, 1.33000000E+02, 3.90000000E+01, 9.88900000E-01, 0.00000000E+00},
    {6.29219000E+02, 1.33000000E+02, 4.00000000E+01, 9.88900000E-01, 0.00000000E+00},
    {5.74900900E+02, 1.33000000E+02, 4.10000000E+01, 9.88900000E-01, 0.00000000E+00},
    {4.45079900E+02, 1.33000000E+02, 4.20000000E+01, 9.88900000E-01, 0.00000000E+00},
    {4.96102300E+02, 1.33000000E+02, 4.30000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.79042700E+02, 1.33000000E+02, 4.40000000E+01, 9.88900000E-01, 0.00000000E+00},
    {4.14541100E+02, 1.33000000E+02, 4.50000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.84815200E+02, 1.33000000E+02, 4.60000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.20223100E+02, 1.33000000E+02, 4.70000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.39481400E+02, 1.33000000E+02, 4.80000000E+01, 9.88900000E-01, 0.00000000E+00},
    {4.24700800E+02, 1.33000000E+02, 4.90000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.94816500E+02, 1.33000000E+02, 5.00000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.53424500E+02, 1.33000000E+02, 5.10000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.28675600E+02, 1.33000000E+02, 5.20000000E+01, 9.88900000E-01, 0.00000000E+00},
    {2.97806400E+02, 1.33000000E+02, 5.30000000E+01, 9.88900000E-01, 0.00000000E+00},
    {2.68170000E+02, 1.33000000E+02, 5.40000000E+01, 9.88900000E-01, 0.00000000E+00},
    {1.08536770E+03, 1.33000000E+02, 5.50000000E+01, 9.88900000E-01, 0.00000000E+00},
    {1.00779130E+03, 1.33000000E+02, 5.60000000E+01, 9.88900000E-01, 0.00000000E+00},
    {8.90476400E+02, 1.33000000E+02, 5.70000000E+01, 9.88900000E-01, 0.00000000E+00},
    {4.17035000E+02, 1.33000000E+02, 5.80000000E+01, 9.88900000E-01, 2.79910000E+00},
    {8.94281500E+02, 1.33000000E+02, 5.90000000E+01, 9.88900000E-01, 0.00000000E+00},
    {8.59524800E+02, 1.33000000E+02, 6.00000000E+01, 9.88900000E-01, 0.00000000E+00},
    {8.38183700E+02, 1.33000000E+02, 6.10000000E+01, 9.88900000E-01, 0.00000000E+00},
    {8.18538100E+02, 1.33000000E+02, 6.20000000E+01, 9.88900000E-01, 0.00000000E+00},
    {8.01128600E+02, 1.33000000E+02, 6.30000000E+01, 9.88900000E-01, 0.00000000E+00},
    {6.33580900E+02, 1.33000000E+02, 6.40000000E+01, 9.88900000E-01, 0.00000000E+00},
    {7.06870700E+02, 1.33000000E+02, 6.50000000E+01, 9.88900000E-01, 0.00000000E+00},
    {6.82512600E+02, 1.33000000E+02, 6.60000000E+01, 9.88900000E-01, 0.00000000E+00},
    {7.23601300E+02, 1.33000000E+02, 6.70000000E+01, 9.88900000E-01, 0.00000000E+00},
    {7.08365300E+02, 1.33000000E+02, 6.80000000E+01, 9.88900000E-01, 0.00000000E+00},
    {6.94680700E+02, 1.33000000E+02, 6.90000000E+01, 9.88900000E-01, 0.00000000E+00},
    {6.86401100E+02, 1.33000000E+02, 7.00000000E+01, 9.88900000E-01, 0.00000000E+00},
    {5.80611700E+02, 1.33000000E+02, 7.10000000E+01, 9.88900000E-01, 0.00000000E+00},
    {5.74409000E+02, 1.33000000E+02, 7.20000000E+01, 9.88900000E-01, 0.00000000E+00},
    {5.25737500E+02, 1.33000000E+02, 7.30000000E+01, 9.88900000E-01, 0.00000000E+00},
    {4.44648600E+02, 1.33000000E+02, 7.40000000E+01, 9.88900000E-01, 0.00000000E+00},
    {4.52917700E+02, 1.33000000E+02, 7.50000000E+01, 9.88900000E-01, 0.00000000E+00},
    {4.11366900E+02, 1.33000000E+02, 7.60000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.77332800E+02, 1.33000000E+02, 7.70000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.13704000E+02, 1.33000000E+02, 7.80000000E+01, 9.88900000E-01, 0.00000000E+00},
    {2.93165600E+02, 1.33000000E+02, 7.90000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.01972300E+02, 1.33000000E+02, 8.00000000E+01, 9.88900000E-01, 0.00000000E+00},
    {4.36158400E+02, 1.33000000E+02, 8.10000000E+01, 9.88900000E-01, 0.00000000E+00},
    {4.28263900E+02, 1.33000000E+02, 8.20000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.95149500E+02, 1.33000000E+02, 8.30000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.77687300E+02, 1.33000000E+02, 8.40000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.49340700E+02, 1.33000000E+02, 8.50000000E+01, 9.88900000E-01, 0.00000000E+00},
    {3.20719700E+02, 1.33000000E+02, 8.60000000E+01, 9.88900000E-01, 0.00000000E+00},
    {1.02953990E+03, 1.33000000E+02, 8.70000000E+01, 9.88900000E-01, 0.00000000E+00},
    {9.99439500E+02, 1.33000000E+02, 8.80000000E+01, 9.88900000E-01, 0.00000000E+00},
    {8.87911200E+02, 1.33000000E+02, 8.90000000E+01, 9.88900000E-01, 0.00000000E+00},
    {8.02155300E+02, 1.33000000E+02, 9.00000000E+01, 9.88900000E-01, 0.00000000E+00},
    {7.93947300E+02, 1.33000000E+02, 9.10000000E+01, 9.88900000E-01, 0.00000000E+00},
    {7.68782100E+02, 1.33000000E+02, 9.20000000E+01, 9.88900000E-01, 0.00000000E+00},
    {7.88640200E+02, 1.33000000E+02, 9.30000000E+01, 9.88900000E-01, 0.00000000E+00},
    {7.64211000E+02, 1.33000000E+02, 9.40000000E+01, 9.88900000E-01, 0.00000000E+00},
    {4.39067000E+01, 1.33000000E+02, 1.01000000E+02, 9.88900000E-01, 0.00000000E+00},
    {1.40135300E+02, 1.33000000E+02, 1.03000000E+02, 9.88900000E-01, 9.86500000E-01},
    {1.79139000E+02, 1.33000000E+02, 1.04000000E+02, 9.88900000E-01, 9.80800000E-01},
    {1.38144600E+02, 1.33000000E+02, 1.05000000E+02, 9.88900000E-01, 9.70600000E-01},
    {1.04453200E+02, 1.33000000E+02, 1.06000000E+02, 9.88900000E-01, 9.86800000E-01},
    {7.28073000E+01, 1.33000000E+02, 1.07000000E+02, 9.88900000E-01, 9.94400000E-01},
    {5.30645000E+01, 1.33000000E+02, 1.08000000E+02, 9.88900000E-01, 9.92500000E-01},
    {3.64728000E+01, 1.33000000E+02, 1.09000000E+02, 9.88900000E-01, 9.98200000E-01},
    {2.04193700E+02, 1.33000000E+02, 1.11000000E+02, 9.88900000E-01, 9.68400000E-01},
    {3.15668800E+02, 1.33000000E+02, 1.12000000E+02, 9.88900000E-01, 9.62800000E-01},
    {3.21604300E+02, 1.33000000E+02, 1.13000000E+02, 9.88900000E-01, 9.64800000E-01},
    {2.60404400E+02, 1.33000000E+02, 1.14000000E+02, 9.88900000E-01, 9.50700000E-01},
    {2.14251400E+02, 1.33000000E+02, 1.15000000E+02, 9.88900000E-01, 9.94700000E-01},
    {1.81590500E+02, 1.33000000E+02, 1.16000000E+02, 9.88900000E-01, 9.94800000E-01},
    {1.48694300E+02, 1.33000000E+02, 1.17000000E+02, 9.88900000E-01, 9.97200000E-01},
    {2.82307000E+02, 1.33000000E+02, 1.19000000E+02, 9.88900000E-01, 9.76700000E-01},
    {5.32246100E+02, 1.33000000E+02, 1.20000000E+02, 9.88900000E-01, 9.83100000E-01},
    {2.84451600E+02, 1.33000000E+02, 1.21000000E+02, 9.88900000E-01, 1.86270000E+00},
    {2.74592900E+02, 1.33000000E+02, 1.22000000E+02, 9.88900000E-01, 1.82990000E+00},
    {2.69027000E+02, 1.33000000E+02, 1.23000000E+02, 9.88900000E-01, 1.91380000E+00},
    {2.66293900E+02, 1.33000000E+02, 1.24000000E+02, 9.88900000E-01, 1.82690000E+00},
    {2.45971300E+02, 1.33000000E+02, 1.25000000E+02, 9.88900000E-01, 1.64060000E+00},
    {2.27811300E+02, 1.33000000E+02, 1.26000000E+02, 9.88900000E-01, 1.64830000E+00},
    {2.17263500E+02, 1.33000000E+02, 1.27000000E+02, 9.88900000E-01, 1.71490000E+00},
    {2.12322100E+02, 1.33000000E+02, 1.28000000E+02, 9.88900000E-01, 1.79370000E+00},
    {2.09132200E+02, 1.33000000E+02, 1.29000000E+02, 9.88900000E-01, 9.57600000E-01},
    {1.97277100E+02, 1.33000000E+02, 1.30000000E+02, 9.88900000E-01, 1.94190000E+00},
    {3.19787100E+02, 1.33000000E+02, 1.31000000E+02, 9.88900000E-01, 9.60100000E-01},
    {2.82571700E+02, 1.33000000E+02, 1.32000000E+02, 9.88900000E-01, 9.43400000E-01},
    {2.54192400E+02, 1.33000000E+02, 1.33000000E+02, 9.88900000E-01, 9.88900000E-01},
    {2.53149000E+01, 1.34000000E+02, 1.00000000E+00, 9.90100000E-01, 9.11800000E-01},
    {1.70280000E+01, 1.34000000E+02, 2.00000000E+00, 9.90100000E-01, 0.00000000E+00},
    {3.55045000E+02, 1.34000000E+02, 3.00000000E+00, 9.90100000E-01, 0.00000000E+00},
    {2.15167100E+02, 1.34000000E+02, 4.00000000E+00, 9.90100000E-01, 0.00000000E+00},
    {1.48859000E+02, 1.34000000E+02, 5.00000000E+00, 9.90100000E-01, 0.00000000E+00},
    {1.02462000E+02, 1.34000000E+02, 6.00000000E+00, 9.90100000E-01, 0.00000000E+00},
    {7.25567000E+01, 1.34000000E+02, 7.00000000E+00, 9.90100000E-01, 0.00000000E+00},
    {5.53752000E+01, 1.34000000E+02, 8.00000000E+00, 9.90100000E-01, 0.00000000E+00},
    {4.22027000E+01, 1.34000000E+02, 9.00000000E+00, 9.90100000E-01, 0.00000000E+00},
    {3.25935000E+01, 1.34000000E+02, 1.00000000E+01, 9.90100000E-01, 0.00000000E+00},
    {4.25773300E+02, 1.34000000E+02, 1.10000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.39842700E+02, 1.34000000E+02, 1.20000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.17813300E+02, 1.34000000E+02, 1.30000000E+01, 9.90100000E-01, 0.00000000E+00},
    {2.55115300E+02, 1.34000000E+02, 1.40000000E+01, 9.90100000E-01, 0.00000000E+00},
    {2.01960300E+02, 1.34000000E+02, 1.50000000E+01, 9.90100000E-01, 0.00000000E+00},
    {1.69171200E+02, 1.34000000E+02, 1.60000000E+01, 9.90100000E-01, 0.00000000E+00},
    {1.39345300E+02, 1.34000000E+02, 1.70000000E+01, 9.90100000E-01, 0.00000000E+00},
    {1.14767700E+02, 1.34000000E+02, 1.80000000E+01, 9.90100000E-01, 0.00000000E+00},
    {6.93981800E+02, 1.34000000E+02, 1.90000000E+01, 9.90100000E-01, 0.00000000E+00},
    {5.88576300E+02, 1.34000000E+02, 2.00000000E+01, 9.90100000E-01, 0.00000000E+00},
    {4.89330500E+02, 1.34000000E+02, 2.10000000E+01, 9.90100000E-01, 0.00000000E+00},
    {4.75445300E+02, 1.34000000E+02, 2.20000000E+01, 9.90100000E-01, 0.00000000E+00},
    {4.36929300E+02, 1.34000000E+02, 2.30000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.44585800E+02, 1.34000000E+02, 2.40000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.78113300E+02, 1.34000000E+02, 2.50000000E+01, 9.90100000E-01, 0.00000000E+00},
    {2.97269700E+02, 1.34000000E+02, 2.60000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.17223300E+02, 1.34000000E+02, 2.70000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.25566700E+02, 1.34000000E+02, 2.80000000E+01, 9.90100000E-01, 0.00000000E+00},
    {2.49821000E+02, 1.34000000E+02, 2.90000000E+01, 9.90100000E-01, 0.00000000E+00},
    {2.59191600E+02, 1.34000000E+02, 3.00000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.06127500E+02, 1.34000000E+02, 3.10000000E+01, 9.90100000E-01, 0.00000000E+00},
    {2.73768000E+02, 1.34000000E+02, 3.20000000E+01, 9.90100000E-01, 0.00000000E+00},
    {2.36420600E+02, 1.34000000E+02, 3.30000000E+01, 9.90100000E-01, 0.00000000E+00},
    {2.13729800E+02, 1.34000000E+02, 3.40000000E+01, 9.90100000E-01, 0.00000000E+00},
    {1.88403100E+02, 1.34000000E+02, 3.50000000E+01, 9.90100000E-01, 0.00000000E+00},
    {1.64883300E+02, 1.34000000E+02, 3.60000000E+01, 9.90100000E-01, 0.00000000E+00},
    {7.80098600E+02, 1.34000000E+02, 3.70000000E+01, 9.90100000E-01, 0.00000000E+00},
    {7.00801500E+02, 1.34000000E+02, 3.80000000E+01, 9.90100000E-01, 0.00000000E+00},
    {6.20891800E+02, 1.34000000E+02, 3.90000000E+01, 9.90100000E-01, 0.00000000E+00},
    {5.62000100E+02, 1.34000000E+02, 4.00000000E+01, 9.90100000E-01, 0.00000000E+00},
    {5.14894500E+02, 1.34000000E+02, 4.10000000E+01, 9.90100000E-01, 0.00000000E+00},
    {4.00772200E+02, 1.34000000E+02, 4.20000000E+01, 9.90100000E-01, 0.00000000E+00},
    {4.45791600E+02, 1.34000000E+02, 4.30000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.42597000E+02, 1.34000000E+02, 4.40000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.74290000E+02, 1.34000000E+02, 4.50000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.48040200E+02, 1.34000000E+02, 4.60000000E+01, 9.90100000E-01, 0.00000000E+00},
    {2.89832800E+02, 1.34000000E+02, 4.70000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.07721400E+02, 1.34000000E+02, 4.80000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.82838100E+02, 1.34000000E+02, 4.90000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.57960900E+02, 1.34000000E+02, 5.00000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.22481200E+02, 1.34000000E+02, 5.10000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.01150600E+02, 1.34000000E+02, 5.20000000E+01, 9.90100000E-01, 0.00000000E+00},
    {2.74137700E+02, 1.34000000E+02, 5.30000000E+01, 9.90100000E-01, 0.00000000E+00},
    {2.47986700E+02, 1.34000000E+02, 5.40000000E+01, 9.90100000E-01, 0.00000000E+00},
    {9.51630500E+02, 1.34000000E+02, 5.50000000E+01, 9.90100000E-01, 0.00000000E+00},
    {8.90626400E+02, 1.34000000E+02, 5.60000000E+01, 9.90100000E-01, 0.00000000E+00},
    {7.91247100E+02, 1.34000000E+02, 5.70000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.80316000E+02, 1.34000000E+02, 5.80000000E+01, 9.90100000E-01, 2.79910000E+00},
    {7.91881700E+02, 1.34000000E+02, 5.90000000E+01, 9.90100000E-01, 0.00000000E+00},
    {7.61699800E+02, 1.34000000E+02, 6.00000000E+01, 9.90100000E-01, 0.00000000E+00},
    {7.42943500E+02, 1.34000000E+02, 6.10000000E+01, 9.90100000E-01, 0.00000000E+00},
    {7.25653600E+02, 1.34000000E+02, 6.20000000E+01, 9.90100000E-01, 0.00000000E+00},
    {7.10337600E+02, 1.34000000E+02, 6.30000000E+01, 9.90100000E-01, 0.00000000E+00},
    {5.65791400E+02, 1.34000000E+02, 6.40000000E+01, 9.90100000E-01, 0.00000000E+00},
    {6.26170500E+02, 1.34000000E+02, 6.50000000E+01, 9.90100000E-01, 0.00000000E+00},
    {6.05282800E+02, 1.34000000E+02, 6.60000000E+01, 9.90100000E-01, 0.00000000E+00},
    {6.42379500E+02, 1.34000000E+02, 6.70000000E+01, 9.90100000E-01, 0.00000000E+00},
    {6.28910800E+02, 1.34000000E+02, 6.80000000E+01, 9.90100000E-01, 0.00000000E+00},
    {6.16866600E+02, 1.34000000E+02, 6.90000000E+01, 9.90100000E-01, 0.00000000E+00},
    {6.09314600E+02, 1.34000000E+02, 7.00000000E+01, 9.90100000E-01, 0.00000000E+00},
    {5.17906900E+02, 1.34000000E+02, 7.10000000E+01, 9.90100000E-01, 0.00000000E+00},
    {5.15238800E+02, 1.34000000E+02, 7.20000000E+01, 9.90100000E-01, 0.00000000E+00},
    {4.73461900E+02, 1.34000000E+02, 7.30000000E+01, 9.90100000E-01, 0.00000000E+00},
    {4.02077800E+02, 1.34000000E+02, 7.40000000E+01, 9.90100000E-01, 0.00000000E+00},
    {4.10035200E+02, 1.34000000E+02, 7.50000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.73742200E+02, 1.34000000E+02, 7.60000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.43837600E+02, 1.34000000E+02, 7.70000000E+01, 9.90100000E-01, 0.00000000E+00},
    {2.86994500E+02, 1.34000000E+02, 7.80000000E+01, 9.90100000E-01, 0.00000000E+00},
    {2.68614500E+02, 1.34000000E+02, 7.90000000E+01, 9.90100000E-01, 0.00000000E+00},
    {2.76877800E+02, 1.34000000E+02, 8.00000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.94338500E+02, 1.34000000E+02, 8.10000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.88735700E+02, 1.34000000E+02, 8.20000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.60618500E+02, 1.34000000E+02, 8.30000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.45847500E+02, 1.34000000E+02, 8.40000000E+01, 9.90100000E-01, 0.00000000E+00},
    {3.21252800E+02, 1.34000000E+02, 8.50000000E+01, 9.90100000E-01, 0.00000000E+00},
    {2.96136400E+02, 1.34000000E+02, 8.60000000E+01, 9.90100000E-01, 0.00000000E+00},
    {9.06706500E+02, 1.34000000E+02, 8.70000000E+01, 9.90100000E-01, 0.00000000E+00},
    {8.85973800E+02, 1.34000000E+02, 8.80000000E+01, 9.90100000E-01, 0.00000000E+00},
    {7.90971700E+02, 1.34000000E+02, 8.90000000E+01, 9.90100000E-01, 0.00000000E+00},
    {7.19201800E+02, 1.34000000E+02, 9.00000000E+01, 9.90100000E-01, 0.00000000E+00},
    {7.10029900E+02, 1.34000000E+02, 9.10000000E+01, 9.90100000E-01, 0.00000000E+00},
    {6.87655900E+02, 1.34000000E+02, 9.20000000E+01, 9.90100000E-01, 0.00000000E+00},
    {7.02688000E+02, 1.34000000E+02, 9.30000000E+01, 9.90100000E-01, 0.00000000E+00},
    {6.81358900E+02, 1.34000000E+02, 9.40000000E+01, 9.90100000E-01, 0.00000000E+00},
    {4.02117000E+01, 1.34000000E+02, 1.01000000E+02, 9.90100000E-01, 0.00000000E+00},
    {1.25556400E+02, 1.34000000E+02, 1.03000000E+02, 9.90100000E-01, 9.86500000E-01},
    {1.61063700E+02, 1.34000000E+02, 1.04000000E+02, 9.90100000E-01, 9.80800000E-01},
    {1.25888800E+02, 1.34000000E+02, 1.05000000E+02, 9.90100000E-01, 9.70600000E-01},
    {9.60872000E+01, 1.34000000E+02, 1.06000000E+02, 9.90100000E-01, 9.86800000E-01},
    {6.76796000E+01, 1.34000000E+02, 1.07000000E+02, 9.90100000E-01, 9.94400000E-01},
    {4.97665000E+01, 1.34000000E+02, 1.08000000E+02, 9.90100000E-01, 9.92500000E-01},
    {3.45882000E+01, 1.34000000E+02, 1.09000000E+02, 9.90100000E-01, 9.98200000E-01},
    {1.82606200E+02, 1.34000000E+02, 1.11000000E+02, 9.90100000E-01, 9.68400000E-01},
    {2.81924900E+02, 1.34000000E+02, 1.12000000E+02, 9.90100000E-01, 9.62800000E-01},
    {2.89016300E+02, 1.34000000E+02, 1.13000000E+02, 9.90100000E-01, 9.64800000E-01},
    {2.36425300E+02, 1.34000000E+02, 1.14000000E+02, 9.90100000E-01, 9.50700000E-01},
    {1.96137400E+02, 1.34000000E+02, 1.15000000E+02, 9.90100000E-01, 9.94700000E-01},
    {1.67267200E+02, 1.34000000E+02, 1.16000000E+02, 9.90100000E-01, 9.94800000E-01},
    {1.37873500E+02, 1.34000000E+02, 1.17000000E+02, 9.90100000E-01, 9.97200000E-01},
    {2.54537600E+02, 1.34000000E+02, 1.19000000E+02, 9.90100000E-01, 9.76700000E-01},
    {4.71982100E+02, 1.34000000E+02, 1.20000000E+02, 9.90100000E-01, 9.83100000E-01},
    {2.58129500E+02, 1.34000000E+02, 1.21000000E+02, 9.90100000E-01, 1.86270000E+00},
    {2.49343700E+02, 1.34000000E+02, 1.22000000E+02, 9.90100000E-01, 1.82990000E+00},
    {2.44272700E+02, 1.34000000E+02, 1.23000000E+02, 9.90100000E-01, 1.91380000E+00},
    {2.41580300E+02, 1.34000000E+02, 1.24000000E+02, 9.90100000E-01, 1.82690000E+00},
    {2.24075700E+02, 1.34000000E+02, 1.25000000E+02, 9.90100000E-01, 1.64060000E+00},
    {2.07874800E+02, 1.34000000E+02, 1.26000000E+02, 9.90100000E-01, 1.64830000E+00},
    {1.98299300E+02, 1.34000000E+02, 1.27000000E+02, 9.90100000E-01, 1.71490000E+00},
    {1.93721300E+02, 1.34000000E+02, 1.28000000E+02, 9.90100000E-01, 1.79370000E+00},
    {1.90168500E+02, 1.34000000E+02, 1.29000000E+02, 9.90100000E-01, 9.57600000E-01},
    {1.80481200E+02, 1.34000000E+02, 1.30000000E+02, 9.90100000E-01, 1.94190000E+00},
    {2.88549300E+02, 1.34000000E+02, 1.31000000E+02, 9.90100000E-01, 9.60100000E-01},
    {2.56952300E+02, 1.34000000E+02, 1.32000000E+02, 9.90100000E-01, 9.43400000E-01},
    {2.32578400E+02, 1.34000000E+02, 1.33000000E+02, 9.90100000E-01, 9.88900000E-01},
    {2.13786200E+02, 1.34000000E+02, 1.34000000E+02, 9.90100000E-01, 9.90100000E-01},
    {2.26230000E+01, 1.35000000E+02, 1.00000000E+00, 9.97400000E-01, 9.11800000E-01},
    {1.55387000E+01, 1.35000000E+02, 2.00000000E+00, 9.97400000E-01, 0.00000000E+00},
    {2.97683500E+02, 1.35000000E+02, 3.00000000E+00, 9.97400000E-01, 0.00000000E+00},
    {1.84883200E+02, 1.35000000E+02, 4.00000000E+00, 9.97400000E-01, 0.00000000E+00},
    {1.30188100E+02, 1.35000000E+02, 5.00000000E+00, 9.97400000E-01, 0.00000000E+00},
    {9.09343000E+01, 1.35000000E+02, 6.00000000E+00, 9.97400000E-01, 0.00000000E+00},
    {6.51543000E+01, 1.35000000E+02, 7.00000000E+00, 9.97400000E-01, 0.00000000E+00},
    {5.01632000E+01, 1.35000000E+02, 8.00000000E+00, 9.97400000E-01, 0.00000000E+00},
    {3.85294000E+01, 1.35000000E+02, 9.00000000E+00, 9.97400000E-01, 0.00000000E+00},
    {2.99501000E+01, 1.35000000E+02, 1.00000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.57724600E+02, 1.35000000E+02, 1.10000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.90727600E+02, 1.35000000E+02, 1.20000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.74189400E+02, 1.35000000E+02, 1.30000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.22681200E+02, 1.35000000E+02, 1.40000000E+01, 9.97400000E-01, 0.00000000E+00},
    {1.78198400E+02, 1.35000000E+02, 1.50000000E+01, 9.97400000E-01, 0.00000000E+00},
    {1.50420300E+02, 1.35000000E+02, 1.60000000E+01, 9.97400000E-01, 0.00000000E+00},
    {1.24849500E+02, 1.35000000E+02, 1.70000000E+01, 9.97400000E-01, 0.00000000E+00},
    {1.03546700E+02, 1.35000000E+02, 1.80000000E+01, 9.97400000E-01, 0.00000000E+00},
    {5.82891400E+02, 1.35000000E+02, 1.90000000E+01, 9.97400000E-01, 0.00000000E+00},
    {5.00415200E+02, 1.35000000E+02, 2.00000000E+01, 9.97400000E-01, 0.00000000E+00},
    {4.17364800E+02, 1.35000000E+02, 2.10000000E+01, 9.97400000E-01, 0.00000000E+00},
    {4.07097000E+02, 1.35000000E+02, 2.20000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.74928300E+02, 1.35000000E+02, 2.30000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.96360700E+02, 1.35000000E+02, 2.40000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.25494500E+02, 1.35000000E+02, 2.50000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.56593900E+02, 1.35000000E+02, 2.60000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.74462000E+02, 1.35000000E+02, 2.70000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.81015400E+02, 1.35000000E+02, 2.80000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.16214000E+02, 1.35000000E+02, 2.90000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.25345700E+02, 1.35000000E+02, 3.00000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.65393900E+02, 1.35000000E+02, 3.10000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.39363600E+02, 1.35000000E+02, 3.20000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.08495700E+02, 1.35000000E+02, 3.30000000E+01, 9.97400000E-01, 0.00000000E+00},
    {1.89597400E+02, 1.35000000E+02, 3.40000000E+01, 9.97400000E-01, 0.00000000E+00},
    {1.68179100E+02, 1.35000000E+02, 3.50000000E+01, 9.97400000E-01, 0.00000000E+00},
    {1.48068900E+02, 1.35000000E+02, 3.60000000E+01, 9.97400000E-01, 0.00000000E+00},
    {6.56657400E+02, 1.35000000E+02, 3.70000000E+01, 9.97400000E-01, 0.00000000E+00},
    {5.96021400E+02, 1.35000000E+02, 3.80000000E+01, 9.97400000E-01, 0.00000000E+00},
    {5.31342600E+02, 1.35000000E+02, 3.90000000E+01, 9.97400000E-01, 0.00000000E+00},
    {4.82987300E+02, 1.35000000E+02, 4.00000000E+01, 9.97400000E-01, 0.00000000E+00},
    {4.43884800E+02, 1.35000000E+02, 4.10000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.47680000E+02, 1.35000000E+02, 4.20000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.85798600E+02, 1.35000000E+02, 4.30000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.98517400E+02, 1.35000000E+02, 4.40000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.25665000E+02, 1.35000000E+02, 4.50000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.03410700E+02, 1.35000000E+02, 4.60000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.53009500E+02, 1.35000000E+02, 4.70000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.68949100E+02, 1.35000000E+02, 4.80000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.32479100E+02, 1.35000000E+02, 4.90000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.12799400E+02, 1.35000000E+02, 5.00000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.83776100E+02, 1.35000000E+02, 5.10000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.66238600E+02, 1.35000000E+02, 5.20000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.43629600E+02, 1.35000000E+02, 5.30000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.21536700E+02, 1.35000000E+02, 5.40000000E+01, 9.97400000E-01, 0.00000000E+00},
    {8.01861400E+02, 1.35000000E+02, 5.50000000E+01, 9.97400000E-01, 0.00000000E+00},
    {7.56627100E+02, 1.35000000E+02, 5.60000000E+01, 9.97400000E-01, 0.00000000E+00},
    {6.76193000E+02, 1.35000000E+02, 5.70000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.34526200E+02, 1.35000000E+02, 5.80000000E+01, 9.97400000E-01, 2.79910000E+00},
    {6.74287600E+02, 1.35000000E+02, 5.90000000E+01, 9.97400000E-01, 0.00000000E+00},
    {6.49128200E+02, 1.35000000E+02, 6.00000000E+01, 9.97400000E-01, 0.00000000E+00},
    {6.33283500E+02, 1.35000000E+02, 6.10000000E+01, 9.97400000E-01, 0.00000000E+00},
    {6.18654000E+02, 1.35000000E+02, 6.20000000E+01, 9.97400000E-01, 0.00000000E+00},
    {6.05699400E+02, 1.35000000E+02, 6.30000000E+01, 9.97400000E-01, 0.00000000E+00},
    {4.86344400E+02, 1.35000000E+02, 6.40000000E+01, 9.97400000E-01, 0.00000000E+00},
    {5.33648500E+02, 1.35000000E+02, 6.50000000E+01, 9.97400000E-01, 0.00000000E+00},
    {5.16490500E+02, 1.35000000E+02, 6.60000000E+01, 9.97400000E-01, 0.00000000E+00},
    {5.48462900E+02, 1.35000000E+02, 6.70000000E+01, 9.97400000E-01, 0.00000000E+00},
    {5.37007900E+02, 1.35000000E+02, 6.80000000E+01, 9.97400000E-01, 0.00000000E+00},
    {5.26814700E+02, 1.35000000E+02, 6.90000000E+01, 9.97400000E-01, 0.00000000E+00},
    {5.20167400E+02, 1.35000000E+02, 7.00000000E+01, 9.97400000E-01, 0.00000000E+00},
    {4.44551600E+02, 1.35000000E+02, 7.10000000E+01, 9.97400000E-01, 0.00000000E+00},
    {4.44885100E+02, 1.35000000E+02, 7.20000000E+01, 9.97400000E-01, 0.00000000E+00},
    {4.10643000E+02, 1.35000000E+02, 7.30000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.50423400E+02, 1.35000000E+02, 7.40000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.57793200E+02, 1.35000000E+02, 7.50000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.27431700E+02, 1.35000000E+02, 7.60000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.02247500E+02, 1.35000000E+02, 7.70000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.53481600E+02, 1.35000000E+02, 7.80000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.37683200E+02, 1.35000000E+02, 7.90000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.45141100E+02, 1.35000000E+02, 8.00000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.43740400E+02, 1.35000000E+02, 8.10000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.40242200E+02, 1.35000000E+02, 8.20000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.17483800E+02, 1.35000000E+02, 8.30000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.05609700E+02, 1.35000000E+02, 8.40000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.85222900E+02, 1.35000000E+02, 8.50000000E+01, 9.97400000E-01, 0.00000000E+00},
    {2.64133400E+02, 1.35000000E+02, 8.60000000E+01, 9.97400000E-01, 0.00000000E+00},
    {7.67790600E+02, 1.35000000E+02, 8.70000000E+01, 9.97400000E-01, 0.00000000E+00},
    {7.55261700E+02, 1.35000000E+02, 8.80000000E+01, 9.97400000E-01, 0.00000000E+00},
    {6.77876700E+02, 1.35000000E+02, 8.90000000E+01, 9.97400000E-01, 0.00000000E+00},
    {6.20826000E+02, 1.35000000E+02, 9.00000000E+01, 9.97400000E-01, 0.00000000E+00},
    {6.11307500E+02, 1.35000000E+02, 9.10000000E+01, 9.97400000E-01, 0.00000000E+00},
    {5.92187100E+02, 1.35000000E+02, 9.20000000E+01, 9.97400000E-01, 0.00000000E+00},
    {6.02583500E+02, 1.35000000E+02, 9.30000000E+01, 9.97400000E-01, 0.00000000E+00},
    {5.84699800E+02, 1.35000000E+02, 9.40000000E+01, 9.97400000E-01, 0.00000000E+00},
    {3.55123000E+01, 1.35000000E+02, 1.01000000E+02, 9.97400000E-01, 0.00000000E+00},
    {1.08204800E+02, 1.35000000E+02, 1.03000000E+02, 9.97400000E-01, 9.86500000E-01},
    {1.39361700E+02, 1.35000000E+02, 1.04000000E+02, 9.97400000E-01, 9.80800000E-01},
    {1.10580200E+02, 1.35000000E+02, 1.05000000E+02, 9.97400000E-01, 9.70600000E-01},
    {8.53265000E+01, 1.35000000E+02, 1.06000000E+02, 9.97400000E-01, 9.86800000E-01},
    {6.08359000E+01, 1.35000000E+02, 1.07000000E+02, 9.97400000E-01, 9.94400000E-01},
    {4.52019000E+01, 1.35000000E+02, 1.08000000E+02, 9.97400000E-01, 9.92500000E-01},
    {3.18313000E+01, 1.35000000E+02, 1.09000000E+02, 9.97400000E-01, 9.98200000E-01},
    {1.57100500E+02, 1.35000000E+02, 1.11000000E+02, 9.97400000E-01, 9.68400000E-01},
    {2.42147600E+02, 1.35000000E+02, 1.12000000E+02, 9.97400000E-01, 9.62800000E-01},
    {2.49924800E+02, 1.35000000E+02, 1.13000000E+02, 9.97400000E-01, 9.64800000E-01},
    {2.06788000E+02, 1.35000000E+02, 1.14000000E+02, 9.97400000E-01, 9.50700000E-01},
    {1.73158200E+02, 1.35000000E+02, 1.15000000E+02, 9.97400000E-01, 9.94700000E-01},
    {1.48721900E+02, 1.35000000E+02, 1.16000000E+02, 9.97400000E-01, 9.94800000E-01},
    {1.23527400E+02, 1.35000000E+02, 1.17000000E+02, 9.97400000E-01, 9.97200000E-01},
    {2.21164200E+02, 1.35000000E+02, 1.19000000E+02, 9.97400000E-01, 9.76700000E-01},
    {4.02538000E+02, 1.35000000E+02, 1.20000000E+02, 9.97400000E-01, 9.83100000E-01},
    {2.25778400E+02, 1.35000000E+02, 1.21000000E+02, 9.97400000E-01, 1.86270000E+00},
    {2.18271100E+02, 1.35000000E+02, 1.22000000E+02, 9.97400000E-01, 1.82990000E+00},
    {2.13827600E+02, 1.35000000E+02, 1.23000000E+02, 9.97400000E-01, 1.91380000E+00},
    {2.11280900E+02, 1.35000000E+02, 1.24000000E+02, 9.97400000E-01, 1.82690000E+00},
    {1.96839500E+02, 1.35000000E+02, 1.25000000E+02, 9.97400000E-01, 1.64060000E+00},
    {1.82955000E+02, 1.35000000E+02, 1.26000000E+02, 9.97400000E-01, 1.64830000E+00},
    {1.74591000E+02, 1.35000000E+02, 1.27000000E+02, 9.97400000E-01, 1.71490000E+00},
    {1.70500400E+02, 1.35000000E+02, 1.28000000E+02, 9.97400000E-01, 1.79370000E+00},
    {1.66781600E+02, 1.35000000E+02, 1.29000000E+02, 9.97400000E-01, 9.57600000E-01},
    {1.59304900E+02, 1.35000000E+02, 1.30000000E+02, 9.97400000E-01, 1.94190000E+00},
    {2.50724700E+02, 1.35000000E+02, 1.31000000E+02, 9.97400000E-01, 9.60100000E-01},
    {2.25179200E+02, 1.35000000E+02, 1.32000000E+02, 9.97400000E-01, 9.43400000E-01},
    {2.05230500E+02, 1.35000000E+02, 1.33000000E+02, 9.97400000E-01, 9.88900000E-01},
    {1.89640800E+02, 1.35000000E+02, 1.34000000E+02, 9.97400000E-01, 9.90100000E-01},
    {1.69237500E+02, 1.35000000E+02, 1.35000000E+02, 9.97400000E-01, 9.97400000E-01},
    {3.51963000E+01, 1.37000000E+02, 1.00000000E+00, 9.73800000E-01, 9.11800000E-01},
    {2.26231000E+01, 1.37000000E+02, 2.00000000E+00, 9.73800000E-01, 0.00000000E+00},
    {6.57407600E+02, 1.37000000E+02, 3.00000000E+00, 9.73800000E-01, 0.00000000E+00},
    {3.45437800E+02, 1.37000000E+02, 4.00000000E+00, 9.73800000E-01, 0.00000000E+00},
    {2.22508100E+02, 1.37000000E+02, 5.00000000E+00, 9.73800000E-01, 0.00000000E+00},
    {1.45997400E+02, 1.37000000E+02, 6.00000000E+00, 9.73800000E-01, 0.00000000E+00},
    {1.00141800E+02, 1.37000000E+02, 7.00000000E+00, 9.73800000E-01, 0.00000000E+00},
    {7.48868000E+01, 1.37000000E+02, 8.00000000E+00, 9.73800000E-01, 0.00000000E+00},
    {5.61779000E+01, 1.37000000E+02, 9.00000000E+00, 9.73800000E-01, 0.00000000E+00},
    {4.28957000E+01, 1.37000000E+02, 1.00000000E+01, 9.73800000E-01, 0.00000000E+00},
    {7.81852700E+02, 1.37000000E+02, 1.10000000E+01, 9.73800000E-01, 0.00000000E+00},
    {5.59133700E+02, 1.37000000E+02, 1.20000000E+01, 9.73800000E-01, 0.00000000E+00},
    {5.02993300E+02, 1.37000000E+02, 1.30000000E+01, 9.73800000E-01, 0.00000000E+00},
    {3.83908400E+02, 1.37000000E+02, 1.40000000E+01, 9.73800000E-01, 0.00000000E+00},
    {2.92160400E+02, 1.37000000E+02, 1.50000000E+01, 9.73800000E-01, 0.00000000E+00},
    {2.39057200E+02, 1.37000000E+02, 1.60000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.92787700E+02, 1.37000000E+02, 1.70000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.56101000E+02, 1.37000000E+02, 1.80000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.30588400E+03, 1.37000000E+02, 1.90000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.01307060E+03, 1.37000000E+02, 2.00000000E+01, 9.73800000E-01, 0.00000000E+00},
    {8.26104100E+02, 1.37000000E+02, 2.10000000E+01, 9.73800000E-01, 0.00000000E+00},
    {7.89411700E+02, 1.37000000E+02, 2.20000000E+01, 9.73800000E-01, 0.00000000E+00},
    {7.18212500E+02, 1.37000000E+02, 2.30000000E+01, 9.73800000E-01, 0.00000000E+00},
    {5.66185600E+02, 1.37000000E+02, 2.40000000E+01, 9.73800000E-01, 0.00000000E+00},
    {6.12644400E+02, 1.37000000E+02, 2.50000000E+01, 9.73800000E-01, 0.00000000E+00},
    {4.80623600E+02, 1.37000000E+02, 2.60000000E+01, 9.73800000E-01, 0.00000000E+00},
    {5.01594000E+02, 1.37000000E+02, 2.70000000E+01, 9.73800000E-01, 0.00000000E+00},
    {5.19982600E+02, 1.37000000E+02, 2.80000000E+01, 9.73800000E-01, 0.00000000E+00},
    {3.99353400E+02, 1.37000000E+02, 2.90000000E+01, 9.73800000E-01, 0.00000000E+00},
    {4.01054800E+02, 1.37000000E+02, 3.00000000E+01, 9.73800000E-01, 0.00000000E+00},
    {4.77973700E+02, 1.37000000E+02, 3.10000000E+01, 9.73800000E-01, 0.00000000E+00},
    {4.11038900E+02, 1.37000000E+02, 3.20000000E+01, 9.73800000E-01, 0.00000000E+00},
    {3.43397800E+02, 1.37000000E+02, 3.30000000E+01, 9.73800000E-01, 0.00000000E+00},
    {3.04531200E+02, 1.37000000E+02, 3.40000000E+01, 9.73800000E-01, 0.00000000E+00},
    {2.63459200E+02, 1.37000000E+02, 3.50000000E+01, 9.73800000E-01, 0.00000000E+00},
    {2.26859000E+02, 1.37000000E+02, 3.60000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.45778170E+03, 1.37000000E+02, 3.70000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.21081080E+03, 1.37000000E+02, 3.80000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.03744340E+03, 1.37000000E+02, 3.90000000E+01, 9.73800000E-01, 0.00000000E+00},
    {9.20135800E+02, 1.37000000E+02, 4.00000000E+01, 9.73800000E-01, 0.00000000E+00},
    {8.32136800E+02, 1.37000000E+02, 4.10000000E+01, 9.73800000E-01, 0.00000000E+00},
    {6.33947100E+02, 1.37000000E+02, 4.20000000E+01, 9.73800000E-01, 0.00000000E+00},
    {7.10777900E+02, 1.37000000E+02, 4.30000000E+01, 9.73800000E-01, 0.00000000E+00},
    {5.33613500E+02, 1.37000000E+02, 4.40000000E+01, 9.73800000E-01, 0.00000000E+00},
    {5.82374200E+02, 1.37000000E+02, 4.50000000E+01, 9.73800000E-01, 0.00000000E+00},
    {5.37256500E+02, 1.37000000E+02, 4.60000000E+01, 9.73800000E-01, 0.00000000E+00},
    {4.50471400E+02, 1.37000000E+02, 4.70000000E+01, 9.73800000E-01, 0.00000000E+00},
    {4.70311900E+02, 1.37000000E+02, 4.80000000E+01, 9.73800000E-01, 0.00000000E+00},
    {5.99898700E+02, 1.37000000E+02, 4.90000000E+01, 9.73800000E-01, 0.00000000E+00},
    {5.42877700E+02, 1.37000000E+02, 5.00000000E+01, 9.73800000E-01, 0.00000000E+00},
    {4.74564100E+02, 1.37000000E+02, 5.10000000E+01, 9.73800000E-01, 0.00000000E+00},
    {4.35555500E+02, 1.37000000E+02, 5.20000000E+01, 9.73800000E-01, 0.00000000E+00},
    {3.89468400E+02, 1.37000000E+02, 5.30000000E+01, 9.73800000E-01, 0.00000000E+00},
    {3.46679100E+02, 1.37000000E+02, 5.40000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.77688800E+03, 1.37000000E+02, 5.50000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.55699580E+03, 1.37000000E+02, 5.60000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.33750100E+03, 1.37000000E+02, 5.70000000E+01, 9.73800000E-01, 0.00000000E+00},
    {5.63148000E+02, 1.37000000E+02, 5.80000000E+01, 9.73800000E-01, 2.79910000E+00},
    {1.36952450E+03, 1.37000000E+02, 5.90000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.30970140E+03, 1.37000000E+02, 6.00000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.27539680E+03, 1.37000000E+02, 6.10000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.24399500E+03, 1.37000000E+02, 6.20000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.21610180E+03, 1.37000000E+02, 6.30000000E+01, 9.73800000E-01, 0.00000000E+00},
    {9.34608200E+02, 1.37000000E+02, 6.40000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.09309340E+03, 1.37000000E+02, 6.50000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.05009250E+03, 1.37000000E+02, 6.60000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.09035000E+03, 1.37000000E+02, 6.70000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.06644690E+03, 1.37000000E+02, 6.80000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.04460450E+03, 1.37000000E+02, 6.90000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.03331960E+03, 1.37000000E+02, 7.00000000E+01, 9.73800000E-01, 0.00000000E+00},
    {8.57247800E+02, 1.37000000E+02, 7.10000000E+01, 9.73800000E-01, 0.00000000E+00},
    {8.24205000E+02, 1.37000000E+02, 7.20000000E+01, 9.73800000E-01, 0.00000000E+00},
    {7.42739700E+02, 1.37000000E+02, 7.30000000E+01, 9.73800000E-01, 0.00000000E+00},
    {6.22097700E+02, 1.37000000E+02, 7.40000000E+01, 9.73800000E-01, 0.00000000E+00},
    {6.29472800E+02, 1.37000000E+02, 7.50000000E+01, 9.73800000E-01, 0.00000000E+00},
    {5.64551600E+02, 1.37000000E+02, 7.60000000E+01, 9.73800000E-01, 0.00000000E+00},
    {5.12796600E+02, 1.37000000E+02, 7.70000000E+01, 9.73800000E-01, 0.00000000E+00},
    {4.22901900E+02, 1.37000000E+02, 7.80000000E+01, 9.73800000E-01, 0.00000000E+00},
    {3.93928800E+02, 1.37000000E+02, 7.90000000E+01, 9.73800000E-01, 0.00000000E+00},
    {4.03264500E+02, 1.37000000E+02, 8.00000000E+01, 9.73800000E-01, 0.00000000E+00},
    {6.14046900E+02, 1.37000000E+02, 8.10000000E+01, 9.73800000E-01, 0.00000000E+00},
    {5.89827500E+02, 1.37000000E+02, 8.20000000E+01, 9.73800000E-01, 0.00000000E+00},
    {5.32390700E+02, 1.37000000E+02, 8.30000000E+01, 9.73800000E-01, 0.00000000E+00},
    {5.02867300E+02, 1.37000000E+02, 8.40000000E+01, 9.73800000E-01, 0.00000000E+00},
    {4.59000500E+02, 1.37000000E+02, 8.50000000E+01, 9.73800000E-01, 0.00000000E+00},
    {4.16732100E+02, 1.37000000E+02, 8.60000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.64764560E+03, 1.37000000E+02, 8.70000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.52236560E+03, 1.37000000E+02, 8.80000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.31764260E+03, 1.37000000E+02, 8.90000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.15786490E+03, 1.37000000E+02, 9.00000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.16349760E+03, 1.37000000E+02, 9.10000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.12590760E+03, 1.37000000E+02, 9.20000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.17535840E+03, 1.37000000E+02, 9.30000000E+01, 9.73800000E-01, 0.00000000E+00},
    {1.13512090E+03, 1.37000000E+02, 9.40000000E+01, 9.73800000E-01, 0.00000000E+00},
    {5.79039000E+01, 1.37000000E+02, 1.01000000E+02, 9.73800000E-01, 0.00000000E+00},
    {1.99677100E+02, 1.37000000E+02, 1.03000000E+02, 9.73800000E-01, 9.86500000E-01},
    {2.52957400E+02, 1.37000000E+02, 1.04000000E+02, 9.73800000E-01, 9.80800000E-01},
    {1.85720600E+02, 1.37000000E+02, 1.05000000E+02, 9.73800000E-01, 9.70600000E-01},
    {1.37212500E+02, 1.37000000E+02, 1.06000000E+02, 9.73800000E-01, 9.86800000E-01},
    {9.34584000E+01, 1.37000000E+02, 1.07000000E+02, 9.73800000E-01, 9.94400000E-01},
    {6.70045000E+01, 1.37000000E+02, 1.08000000E+02, 9.73800000E-01, 9.92500000E-01},
    {4.52489000E+01, 1.37000000E+02, 1.09000000E+02, 9.73800000E-01, 9.98200000E-01},
    {2.94679500E+02, 1.37000000E+02, 1.11000000E+02, 9.73800000E-01, 9.68400000E-01},
    {4.57783100E+02, 1.37000000E+02, 1.12000000E+02, 9.73800000E-01, 9.62800000E-01},
    {4.53363600E+02, 1.37000000E+02, 1.13000000E+02, 9.73800000E-01, 9.64800000E-01},
    {3.53414200E+02, 1.37000000E+02, 1.14000000E+02, 9.73800000E-01, 9.50700000E-01},
    {2.83436800E+02, 1.37000000E+02, 1.15000000E+02, 9.73800000E-01, 9.94700000E-01},
    {2.36546700E+02, 1.37000000E+02, 1.16000000E+02, 9.73800000E-01, 9.94800000E-01},
    {1.90860200E+02, 1.37000000E+02, 1.17000000E+02, 9.73800000E-01, 9.97200000E-01},
    {4.01037100E+02, 1.37000000E+02, 1.19000000E+02, 9.73800000E-01, 9.76700000E-01},
    {8.12348900E+02, 1.37000000E+02, 1.20000000E+02, 9.73800000E-01, 9.83100000E-01},
    {3.89905700E+02, 1.37000000E+02, 1.21000000E+02, 9.73800000E-01, 1.86270000E+00},
    {3.76391900E+02, 1.37000000E+02, 1.22000000E+02, 9.73800000E-01, 1.82990000E+00},
    {3.69064900E+02, 1.37000000E+02, 1.23000000E+02, 9.73800000E-01, 1.91380000E+00},
    {3.66868600E+02, 1.37000000E+02, 1.24000000E+02, 9.73800000E-01, 1.82690000E+00},
    {3.32221500E+02, 1.37000000E+02, 1.25000000E+02, 9.73800000E-01, 1.64060000E+00},
    {3.06272000E+02, 1.37000000E+02, 1.26000000E+02, 9.73800000E-01, 1.64830000E+00},
    {2.92299200E+02, 1.37000000E+02, 1.27000000E+02, 9.73800000E-01, 1.71490000E+00},
    {2.86124100E+02, 1.37000000E+02, 1.28000000E+02, 9.73800000E-01, 1.79370000E+00},
    {2.86013100E+02, 1.37000000E+02, 1.29000000E+02, 9.73800000E-01, 9.57600000E-01},
    {2.62733200E+02, 1.37000000E+02, 1.30000000E+02, 9.73800000E-01, 1.94190000E+00},
    {4.46079800E+02, 1.37000000E+02, 1.31000000E+02, 9.73800000E-01, 9.60100000E-01},
    {3.82501200E+02, 1.37000000E+02, 1.32000000E+02, 9.73800000E-01, 9.43400000E-01},
    {3.37216000E+02, 1.37000000E+02, 1.33000000E+02, 9.73800000E-01, 9.88900000E-01},
    {3.04727900E+02, 1.37000000E+02, 1.34000000E+02, 9.73800000E-01, 9.90100000E-01},
    {2.65464700E+02, 1.37000000E+02, 1.35000000E+02, 9.73800000E-01, 9.97400000E-01},
    {4.76259100E+02, 1.37000000E+02, 1.37000000E+02, 9.73800000E-01, 9.73800000E-01},
    {6.44065000E+01, 1.38000000E+02, 1.00000000E+00, 9.80100000E-01, 9.11800000E-01},
    {3.88333000E+01, 1.38000000E+02, 2.00000000E+00, 9.80100000E-01, 0.00000000E+00},
    {1.61824650E+03, 1.38000000E+02, 3.00000000E+00, 9.80100000E-01, 0.00000000E+00},
    {7.34792600E+02, 1.38000000E+02, 4.00000000E+00, 9.80100000E-01, 0.00000000E+00},
    {4.41126100E+02, 1.38000000E+02, 5.00000000E+00, 9.80100000E-01, 0.00000000E+00},
    {2.74703100E+02, 1.38000000E+02, 6.00000000E+00, 9.80100000E-01, 0.00000000E+00},
    {1.81197300E+02, 1.38000000E+02, 7.00000000E+00, 9.80100000E-01, 0.00000000E+00},
    {1.31784600E+02, 1.38000000E+02, 8.00000000E+00, 9.80100000E-01, 0.00000000E+00},
    {9.65080000E+01, 1.38000000E+02, 9.00000000E+00, 9.80100000E-01, 0.00000000E+00},
    {7.22703000E+01, 1.38000000E+02, 1.00000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.90795920E+03, 1.38000000E+02, 1.10000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.21812740E+03, 1.38000000E+02, 1.20000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.05702180E+03, 1.38000000E+02, 1.30000000E+01, 9.80100000E-01, 0.00000000E+00},
    {7.66384700E+02, 1.38000000E+02, 1.40000000E+01, 9.80100000E-01, 0.00000000E+00},
    {5.59382100E+02, 1.38000000E+02, 1.50000000E+01, 9.80100000E-01, 0.00000000E+00},
    {4.45786000E+02, 1.38000000E+02, 1.60000000E+01, 9.80100000E-01, 0.00000000E+00},
    {3.50427300E+02, 1.38000000E+02, 1.70000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.77475900E+02, 1.38000000E+02, 1.80000000E+01, 9.80100000E-01, 0.00000000E+00},
    {3.30225040E+03, 1.38000000E+02, 1.90000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.31484200E+03, 1.38000000E+02, 2.00000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.84973190E+03, 1.38000000E+02, 2.10000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.74074100E+03, 1.38000000E+02, 2.20000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.56822630E+03, 1.38000000E+02, 2.30000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.23850360E+03, 1.38000000E+02, 2.40000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.31880960E+03, 1.38000000E+02, 2.50000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.03406020E+03, 1.38000000E+02, 2.60000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.05223360E+03, 1.38000000E+02, 2.70000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.10166660E+03, 1.38000000E+02, 2.80000000E+01, 9.80100000E-01, 0.00000000E+00},
    {8.48795200E+02, 1.38000000E+02, 2.90000000E+01, 9.80100000E-01, 0.00000000E+00},
    {8.22516700E+02, 1.38000000E+02, 3.00000000E+01, 9.80100000E-01, 0.00000000E+00},
    {9.91912300E+02, 1.38000000E+02, 3.10000000E+01, 9.80100000E-01, 0.00000000E+00},
    {8.18488700E+02, 1.38000000E+02, 3.20000000E+01, 9.80100000E-01, 0.00000000E+00},
    {6.60092800E+02, 1.38000000E+02, 3.30000000E+01, 9.80100000E-01, 0.00000000E+00},
    {5.73085000E+02, 1.38000000E+02, 3.40000000E+01, 9.80100000E-01, 0.00000000E+00},
    {4.85065000E+02, 1.38000000E+02, 3.50000000E+01, 9.80100000E-01, 0.00000000E+00},
    {4.09376100E+02, 1.38000000E+02, 3.60000000E+01, 9.80100000E-01, 0.00000000E+00},
    {3.67092610E+03, 1.38000000E+02, 3.70000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.78245770E+03, 1.38000000E+02, 3.80000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.30560860E+03, 1.38000000E+02, 3.90000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.00378290E+03, 1.38000000E+02, 4.00000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.78976380E+03, 1.38000000E+02, 4.10000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.33507000E+03, 1.38000000E+02, 4.20000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.50929810E+03, 1.38000000E+02, 4.30000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.10612590E+03, 1.38000000E+02, 4.40000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.20343010E+03, 1.38000000E+02, 4.50000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.10053700E+03, 1.38000000E+02, 4.60000000E+01, 9.80100000E-01, 0.00000000E+00},
    {9.31602200E+02, 1.38000000E+02, 4.70000000E+01, 9.80100000E-01, 0.00000000E+00},
    {9.52753300E+02, 1.38000000E+02, 4.80000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.25011730E+03, 1.38000000E+02, 4.90000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.09166360E+03, 1.38000000E+02, 5.00000000E+01, 9.80100000E-01, 0.00000000E+00},
    {9.24136900E+02, 1.38000000E+02, 5.10000000E+01, 9.80100000E-01, 0.00000000E+00},
    {8.32506000E+02, 1.38000000E+02, 5.20000000E+01, 9.80100000E-01, 0.00000000E+00},
    {7.29717600E+02, 1.38000000E+02, 5.30000000E+01, 9.80100000E-01, 0.00000000E+00},
    {6.37466000E+02, 1.38000000E+02, 5.40000000E+01, 9.80100000E-01, 0.00000000E+00},
    {4.49698460E+03, 1.38000000E+02, 5.50000000E+01, 9.80100000E-01, 0.00000000E+00},
    {3.63145400E+03, 1.38000000E+02, 5.60000000E+01, 9.80100000E-01, 0.00000000E+00},
    {3.01407810E+03, 1.38000000E+02, 5.70000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.10443250E+03, 1.38000000E+02, 5.80000000E+01, 9.80100000E-01, 2.79910000E+00},
    {3.16211980E+03, 1.38000000E+02, 5.90000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.99992830E+03, 1.38000000E+02, 6.00000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.91579490E+03, 1.38000000E+02, 6.10000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.83935010E+03, 1.38000000E+02, 6.20000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.77132340E+03, 1.38000000E+02, 6.30000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.05946920E+03, 1.38000000E+02, 6.40000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.56687290E+03, 1.38000000E+02, 6.50000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.45784020E+03, 1.38000000E+02, 6.60000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.46094700E+03, 1.38000000E+02, 6.70000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.40411110E+03, 1.38000000E+02, 6.80000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.35128030E+03, 1.38000000E+02, 6.90000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.32893690E+03, 1.38000000E+02, 7.00000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.89274470E+03, 1.38000000E+02, 7.10000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.75295650E+03, 1.38000000E+02, 7.20000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.54798970E+03, 1.38000000E+02, 7.30000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.28049290E+03, 1.38000000E+02, 7.40000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.28404340E+03, 1.38000000E+02, 7.50000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.13129130E+03, 1.38000000E+02, 7.60000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.01310950E+03, 1.38000000E+02, 7.70000000E+01, 9.80100000E-01, 0.00000000E+00},
    {8.24660900E+02, 1.38000000E+02, 7.80000000E+01, 9.80100000E-01, 0.00000000E+00},
    {7.63885300E+02, 1.38000000E+02, 7.90000000E+01, 9.80100000E-01, 0.00000000E+00},
    {7.75504600E+02, 1.38000000E+02, 8.00000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.27422830E+03, 1.38000000E+02, 8.10000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.18733590E+03, 1.38000000E+02, 8.20000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.04005230E+03, 1.38000000E+02, 8.30000000E+01, 9.80100000E-01, 0.00000000E+00},
    {9.66303500E+02, 1.38000000E+02, 8.40000000E+01, 9.80100000E-01, 0.00000000E+00},
    {8.65008400E+02, 1.38000000E+02, 8.50000000E+01, 9.80100000E-01, 0.00000000E+00},
    {7.71787700E+02, 1.38000000E+02, 8.60000000E+01, 9.80100000E-01, 0.00000000E+00},
    {4.05338970E+03, 1.38000000E+02, 8.70000000E+01, 9.80100000E-01, 0.00000000E+00},
    {3.49108370E+03, 1.38000000E+02, 8.80000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.92481740E+03, 1.38000000E+02, 8.90000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.48126270E+03, 1.38000000E+02, 9.00000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.54255320E+03, 1.38000000E+02, 9.10000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.45723930E+03, 1.38000000E+02, 9.20000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.61800360E+03, 1.38000000E+02, 9.30000000E+01, 9.80100000E-01, 0.00000000E+00},
    {2.51750530E+03, 1.38000000E+02, 9.40000000E+01, 9.80100000E-01, 0.00000000E+00},
    {1.10327500E+02, 1.38000000E+02, 1.01000000E+02, 9.80100000E-01, 0.00000000E+00},
    {4.21461000E+02, 1.38000000E+02, 1.03000000E+02, 9.80100000E-01, 9.86500000E-01},
    {5.29549400E+02, 1.38000000E+02, 1.04000000E+02, 9.80100000E-01, 9.80800000E-01},
    {3.63360700E+02, 1.38000000E+02, 1.05000000E+02, 9.80100000E-01, 9.70600000E-01},
    {2.58988500E+02, 1.38000000E+02, 1.06000000E+02, 9.80100000E-01, 9.86800000E-01},
    {1.69177200E+02, 1.38000000E+02, 1.07000000E+02, 9.80100000E-01, 9.94400000E-01},
    {1.17101800E+02, 1.38000000E+02, 1.08000000E+02, 9.80100000E-01, 9.92500000E-01},
    {7.56758000E+01, 1.38000000E+02, 1.09000000E+02, 9.80100000E-01, 9.98200000E-01},
    {6.30830400E+02, 1.38000000E+02, 1.11000000E+02, 9.80100000E-01, 9.68400000E-01},
    {9.88785700E+02, 1.38000000E+02, 1.12000000E+02, 9.80100000E-01, 9.62800000E-01},
    {9.44753500E+02, 1.38000000E+02, 1.13000000E+02, 9.80100000E-01, 9.64800000E-01},
    {7.01000900E+02, 1.38000000E+02, 1.14000000E+02, 9.80100000E-01, 9.50700000E-01},
    {5.42126200E+02, 1.38000000E+02, 1.15000000E+02, 9.80100000E-01, 9.94700000E-01},
    {4.41471900E+02, 1.38000000E+02, 1.16000000E+02, 9.80100000E-01, 9.94800000E-01},
    {3.47125500E+02, 1.38000000E+02, 1.17000000E+02, 9.80100000E-01, 9.97200000E-01},
    {8.41998300E+02, 1.38000000E+02, 1.19000000E+02, 9.80100000E-01, 9.76700000E-01},
    {1.87694170E+03, 1.38000000E+02, 1.20000000E+02, 9.80100000E-01, 9.83100000E-01},
    {7.79948800E+02, 1.38000000E+02, 1.21000000E+02, 9.80100000E-01, 1.86270000E+00},
    {7.54675300E+02, 1.38000000E+02, 1.22000000E+02, 9.80100000E-01, 1.82990000E+00},
    {7.40176300E+02, 1.38000000E+02, 1.23000000E+02, 9.80100000E-01, 1.91380000E+00},
    {7.39831500E+02, 1.38000000E+02, 1.24000000E+02, 9.80100000E-01, 1.82690000E+00},
    {6.51840500E+02, 1.38000000E+02, 1.25000000E+02, 9.80100000E-01, 1.64060000E+00},
    {5.96976700E+02, 1.38000000E+02, 1.26000000E+02, 9.80100000E-01, 1.64830000E+00},
    {5.70302400E+02, 1.38000000E+02, 1.27000000E+02, 9.80100000E-01, 1.71490000E+00},
    {5.59388300E+02, 1.38000000E+02, 1.28000000E+02, 9.80100000E-01, 1.79370000E+00},
    {5.69915100E+02, 1.38000000E+02, 1.29000000E+02, 9.80100000E-01, 9.57600000E-01},
    {5.05260800E+02, 1.38000000E+02, 1.30000000E+02, 9.80100000E-01, 1.94190000E+00},
    {9.16231200E+02, 1.38000000E+02, 1.31000000E+02, 9.80100000E-01, 9.60100000E-01},
    {7.55058400E+02, 1.38000000E+02, 1.32000000E+02, 9.80100000E-01, 9.43400000E-01},
    {6.47022000E+02, 1.38000000E+02, 1.33000000E+02, 9.80100000E-01, 9.88900000E-01},
    {5.73713500E+02, 1.38000000E+02, 1.34000000E+02, 9.80100000E-01, 9.90100000E-01},
    {4.89346100E+02, 1.38000000E+02, 1.35000000E+02, 9.80100000E-01, 9.97400000E-01},
    {9.92797000E+02, 1.38000000E+02, 1.37000000E+02, 9.80100000E-01, 9.73800000E-01},
    {2.31266020E+03, 1.38000000E+02, 1.38000000E+02, 9.80100000E-01, 9.80100000E-01},
    {5.12356000E+01, 1.39000000E+02, 1.00000000E+00, 1.91530000E+00, 9.11800000E-01},
    {3.19889000E+01, 1.39000000E+02, 2.00000000E+00, 1.91530000E+00, 0.00000000E+00},
    {1.07458750E+03, 1.39000000E+02, 3.00000000E+00, 1.91530000E+00, 0.00000000E+00},
    {5.32511000E+02, 1.39000000E+02, 4.00000000E+00, 1.91530000E+00, 0.00000000E+00},
    {3.34135100E+02, 1.39000000E+02, 5.00000000E+00, 1.91530000E+00, 0.00000000E+00},
    {2.14776000E+02, 1.39000000E+02, 6.00000000E+00, 1.91530000E+00, 0.00000000E+00},
    {1.44920700E+02, 1.39000000E+02, 7.00000000E+00, 1.91530000E+00, 0.00000000E+00},
    {1.07043800E+02, 1.39000000E+02, 8.00000000E+00, 1.91530000E+00, 0.00000000E+00},
    {7.94076000E+01, 1.39000000E+02, 9.00000000E+00, 1.91530000E+00, 0.00000000E+00},
    {6.00628000E+01, 1.39000000E+02, 1.00000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.27270280E+03, 1.39000000E+02, 1.10000000E+01, 1.91530000E+00, 0.00000000E+00},
    {8.69373900E+02, 1.39000000E+02, 1.20000000E+01, 1.91530000E+00, 0.00000000E+00},
    {7.71860600E+02, 1.39000000E+02, 1.30000000E+01, 1.91530000E+00, 0.00000000E+00},
    {5.78055000E+02, 1.39000000E+02, 1.40000000E+01, 1.91530000E+00, 0.00000000E+00},
    {4.33017100E+02, 1.39000000E+02, 1.50000000E+01, 1.91530000E+00, 0.00000000E+00},
    {3.50577500E+02, 1.39000000E+02, 1.60000000E+01, 1.91530000E+00, 0.00000000E+00},
    {2.79735500E+02, 1.39000000E+02, 1.70000000E+01, 1.91530000E+00, 0.00000000E+00},
    {2.24312600E+02, 1.39000000E+02, 1.80000000E+01, 1.91530000E+00, 0.00000000E+00},
    {2.16370310E+03, 1.39000000E+02, 1.90000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.60363620E+03, 1.39000000E+02, 2.00000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.29729640E+03, 1.39000000E+02, 2.10000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.23234330E+03, 1.39000000E+02, 2.20000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.11695310E+03, 1.39000000E+02, 2.30000000E+01, 1.91530000E+00, 0.00000000E+00},
    {8.80606900E+02, 1.39000000E+02, 2.40000000E+01, 1.91530000E+00, 0.00000000E+00},
    {9.47583400E+02, 1.39000000E+02, 2.50000000E+01, 1.91530000E+00, 0.00000000E+00},
    {7.42664100E+02, 1.39000000E+02, 2.60000000E+01, 1.91530000E+00, 0.00000000E+00},
    {7.68210700E+02, 1.39000000E+02, 2.70000000E+01, 1.91530000E+00, 0.00000000E+00},
    {7.99371000E+02, 1.39000000E+02, 2.80000000E+01, 1.91530000E+00, 0.00000000E+00},
    {6.14098300E+02, 1.39000000E+02, 2.90000000E+01, 1.91530000E+00, 0.00000000E+00},
    {6.09049900E+02, 1.39000000E+02, 3.00000000E+01, 1.91530000E+00, 0.00000000E+00},
    {7.29581300E+02, 1.39000000E+02, 3.10000000E+01, 1.91530000E+00, 0.00000000E+00},
    {6.17998500E+02, 1.39000000E+02, 3.20000000E+01, 1.91530000E+00, 0.00000000E+00},
    {5.09550300E+02, 1.39000000E+02, 3.30000000E+01, 1.91530000E+00, 0.00000000E+00},
    {4.48138100E+02, 1.39000000E+02, 3.40000000E+01, 1.91530000E+00, 0.00000000E+00},
    {3.84290100E+02, 1.39000000E+02, 3.50000000E+01, 1.91530000E+00, 0.00000000E+00},
    {3.28131600E+02, 1.39000000E+02, 3.60000000E+01, 1.91530000E+00, 0.00000000E+00},
    {2.41145770E+03, 1.39000000E+02, 3.70000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.92092950E+03, 1.39000000E+02, 3.80000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.62436360E+03, 1.39000000E+02, 3.90000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.42935760E+03, 1.39000000E+02, 4.00000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.28647700E+03, 1.39000000E+02, 4.10000000E+01, 1.91530000E+00, 0.00000000E+00},
    {9.71595900E+02, 1.39000000E+02, 4.20000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.09324240E+03, 1.39000000E+02, 4.30000000E+01, 1.91530000E+00, 0.00000000E+00},
    {8.12628600E+02, 1.39000000E+02, 4.40000000E+01, 1.91530000E+00, 0.00000000E+00},
    {8.86275000E+02, 1.39000000E+02, 4.50000000E+01, 1.91530000E+00, 0.00000000E+00},
    {8.14840200E+02, 1.39000000E+02, 4.60000000E+01, 1.91530000E+00, 0.00000000E+00},
    {6.84979600E+02, 1.39000000E+02, 4.70000000E+01, 1.91530000E+00, 0.00000000E+00},
    {7.10217200E+02, 1.39000000E+02, 4.80000000E+01, 1.91530000E+00, 0.00000000E+00},
    {9.16159200E+02, 1.39000000E+02, 4.90000000E+01, 1.91530000E+00, 0.00000000E+00},
    {8.18413000E+02, 1.39000000E+02, 5.00000000E+01, 1.91530000E+00, 0.00000000E+00},
    {7.07093300E+02, 1.39000000E+02, 5.10000000E+01, 1.91530000E+00, 0.00000000E+00},
    {6.44449400E+02, 1.39000000E+02, 5.20000000E+01, 1.91530000E+00, 0.00000000E+00},
    {5.71843200E+02, 1.39000000E+02, 5.30000000E+01, 1.91530000E+00, 0.00000000E+00},
    {5.05226200E+02, 1.39000000E+02, 5.40000000E+01, 1.91530000E+00, 0.00000000E+00},
    {2.95024750E+03, 1.39000000E+02, 5.50000000E+01, 1.91530000E+00, 0.00000000E+00},
    {2.48563100E+03, 1.39000000E+02, 5.60000000E+01, 1.91530000E+00, 0.00000000E+00},
    {2.10574910E+03, 1.39000000E+02, 5.70000000E+01, 1.91530000E+00, 0.00000000E+00},
    {8.40841000E+02, 1.39000000E+02, 5.80000000E+01, 1.91530000E+00, 2.79910000E+00},
    {2.17832290E+03, 1.39000000E+02, 5.90000000E+01, 1.91530000E+00, 0.00000000E+00},
    {2.07488070E+03, 1.39000000E+02, 6.00000000E+01, 1.91530000E+00, 0.00000000E+00},
    {2.01892240E+03, 1.39000000E+02, 6.10000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.96788270E+03, 1.39000000E+02, 6.20000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.92252260E+03, 1.39000000E+02, 6.30000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.45819690E+03, 1.39000000E+02, 6.40000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.75211440E+03, 1.39000000E+02, 6.50000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.68305490E+03, 1.39000000E+02, 6.60000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.71701230E+03, 1.39000000E+02, 6.70000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.67857810E+03, 1.39000000E+02, 6.80000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.64321390E+03, 1.39000000E+02, 6.90000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.62633390E+03, 1.39000000E+02, 7.00000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.33981490E+03, 1.39000000E+02, 7.10000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.26894920E+03, 1.39000000E+02, 7.20000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.13464070E+03, 1.39000000E+02, 7.30000000E+01, 1.91530000E+00, 0.00000000E+00},
    {9.45409900E+02, 1.39000000E+02, 7.40000000E+01, 1.91530000E+00, 0.00000000E+00},
    {9.53563400E+02, 1.39000000E+02, 7.50000000E+01, 1.91530000E+00, 0.00000000E+00},
    {8.49381700E+02, 1.39000000E+02, 7.60000000E+01, 1.91530000E+00, 0.00000000E+00},
    {7.67300300E+02, 1.39000000E+02, 7.70000000E+01, 1.91530000E+00, 0.00000000E+00},
    {6.29160500E+02, 1.39000000E+02, 7.80000000E+01, 1.91530000E+00, 0.00000000E+00},
    {5.84635300E+02, 1.39000000E+02, 7.90000000E+01, 1.91530000E+00, 0.00000000E+00},
    {5.96944500E+02, 1.39000000E+02, 8.00000000E+01, 1.91530000E+00, 0.00000000E+00},
    {9.35629900E+02, 1.39000000E+02, 8.10000000E+01, 1.91530000E+00, 0.00000000E+00},
    {8.88875600E+02, 1.39000000E+02, 8.20000000E+01, 1.91530000E+00, 0.00000000E+00},
    {7.93670100E+02, 1.39000000E+02, 8.30000000E+01, 1.91530000E+00, 0.00000000E+00},
    {7.45147100E+02, 1.39000000E+02, 8.40000000E+01, 1.91530000E+00, 0.00000000E+00},
    {6.75184200E+02, 1.39000000E+02, 8.50000000E+01, 1.91530000E+00, 0.00000000E+00},
    {6.08862500E+02, 1.39000000E+02, 8.60000000E+01, 1.91530000E+00, 0.00000000E+00},
    {2.69809700E+03, 1.39000000E+02, 8.70000000E+01, 1.91530000E+00, 0.00000000E+00},
    {2.41342250E+03, 1.39000000E+02, 8.80000000E+01, 1.91530000E+00, 0.00000000E+00},
    {2.06226850E+03, 1.39000000E+02, 8.90000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.78718630E+03, 1.39000000E+02, 9.00000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.80928610E+03, 1.39000000E+02, 9.10000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.74973790E+03, 1.39000000E+02, 9.20000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.84069010E+03, 1.39000000E+02, 9.30000000E+01, 1.91530000E+00, 0.00000000E+00},
    {1.77463910E+03, 1.39000000E+02, 9.40000000E+01, 1.91530000E+00, 0.00000000E+00},
    {8.56815000E+01, 1.39000000E+02, 1.01000000E+02, 1.91530000E+00, 0.00000000E+00},
    {3.06886200E+02, 1.39000000E+02, 1.03000000E+02, 1.91530000E+00, 9.86500000E-01},
    {3.87656400E+02, 1.39000000E+02, 1.04000000E+02, 1.91530000E+00, 9.80800000E-01},
    {2.77405000E+02, 1.39000000E+02, 1.05000000E+02, 1.91530000E+00, 9.70600000E-01},
    {2.01974400E+02, 1.39000000E+02, 1.06000000E+02, 1.91530000E+00, 9.86800000E-01},
    {1.35183900E+02, 1.39000000E+02, 1.07000000E+02, 1.91530000E+00, 9.94400000E-01},
    {9.54458000E+01, 1.39000000E+02, 1.08000000E+02, 1.91530000E+00, 9.92500000E-01},
    {6.31889000E+01, 1.39000000E+02, 1.09000000E+02, 1.91530000E+00, 9.98200000E-01},
    {4.54992800E+02, 1.39000000E+02, 1.11000000E+02, 1.91530000E+00, 9.68400000E-01},
    {7.09696100E+02, 1.39000000E+02, 1.12000000E+02, 1.91530000E+00, 9.62800000E-01},
    {6.93473200E+02, 1.39000000E+02, 1.13000000E+02, 1.91530000E+00, 9.64800000E-01},
    {5.30818000E+02, 1.39000000E+02, 1.14000000E+02, 1.91530000E+00, 9.50700000E-01},
    {4.19871800E+02, 1.39000000E+02, 1.15000000E+02, 1.91530000E+00, 9.94700000E-01},
    {3.46971000E+02, 1.39000000E+02, 1.16000000E+02, 1.91530000E+00, 9.94800000E-01},
    {2.76977300E+02, 1.39000000E+02, 1.17000000E+02, 1.91530000E+00, 9.97200000E-01},
    {6.13846300E+02, 1.39000000E+02, 1.19000000E+02, 1.91530000E+00, 9.76700000E-01},
    {1.29351590E+03, 1.39000000E+02, 1.20000000E+02, 1.91530000E+00, 9.83100000E-01},
    {5.86313500E+02, 1.39000000E+02, 1.21000000E+02, 1.91530000E+00, 1.86270000E+00},
    {5.66867000E+02, 1.39000000E+02, 1.22000000E+02, 1.91530000E+00, 1.82990000E+00},
    {5.55721500E+02, 1.39000000E+02, 1.23000000E+02, 1.91530000E+00, 1.91380000E+00},
    {5.53476400E+02, 1.39000000E+02, 1.24000000E+02, 1.91530000E+00, 1.82690000E+00},
    {4.96333100E+02, 1.39000000E+02, 1.25000000E+02, 1.91530000E+00, 1.64060000E+00},
    {4.56387800E+02, 1.39000000E+02, 1.26000000E+02, 1.91530000E+00, 1.64830000E+00},
    {4.35670100E+02, 1.39000000E+02, 1.27000000E+02, 1.91530000E+00, 1.71490000E+00},
    {4.26740500E+02, 1.39000000E+02, 1.28000000E+02, 1.91530000E+00, 1.79370000E+00},
    {4.29374300E+02, 1.39000000E+02, 1.29000000E+02, 1.91530000E+00, 9.57600000E-01},
    {3.89581100E+02, 1.39000000E+02, 1.30000000E+02, 1.91530000E+00, 1.94190000E+00},
    {6.78259000E+02, 1.39000000E+02, 1.31000000E+02, 1.91530000E+00, 9.60100000E-01},
    {5.73213900E+02, 1.39000000E+02, 1.32000000E+02, 1.91530000E+00, 9.43400000E-01},
    {5.00007100E+02, 1.39000000E+02, 1.33000000E+02, 1.91530000E+00, 9.88900000E-01},
    {4.48487200E+02, 1.39000000E+02, 1.34000000E+02, 1.91530000E+00, 9.90100000E-01},
    {3.87387600E+02, 1.39000000E+02, 1.35000000E+02, 1.91530000E+00, 9.97400000E-01},
    {7.26861300E+02, 1.39000000E+02, 1.37000000E+02, 1.91530000E+00, 9.73800000E-01},
    {1.58654020E+03, 1.39000000E+02, 1.38000000E+02, 1.91530000E+00, 9.80100000E-01},
    {1.13229830E+03, 1.39000000E+02, 1.39000000E+02, 1.91530000E+00, 1.91530000E+00},
    {3.96669000E+01, 1.40000000E+02, 1.00000000E+00, 1.93550000E+00, 9.11800000E-01},
    {2.57037000E+01, 1.40000000E+02, 2.00000000E+00, 1.93550000E+00, 0.00000000E+00},
    {6.78481000E+02, 1.40000000E+02, 3.00000000E+00, 1.93550000E+00, 0.00000000E+00},
    {3.74013500E+02, 1.40000000E+02, 4.00000000E+00, 1.93550000E+00, 0.00000000E+00},
    {2.46014700E+02, 1.40000000E+02, 5.00000000E+00, 1.93550000E+00, 0.00000000E+00},
    {1.63454800E+02, 1.40000000E+02, 6.00000000E+00, 1.93550000E+00, 0.00000000E+00},
    {1.12933900E+02, 1.40000000E+02, 7.00000000E+00, 1.93550000E+00, 0.00000000E+00},
    {8.47876000E+01, 1.40000000E+02, 8.00000000E+00, 1.93550000E+00, 0.00000000E+00},
    {6.37680000E+01, 1.40000000E+02, 9.00000000E+00, 1.93550000E+00, 0.00000000E+00},
    {4.87600000E+01, 1.40000000E+02, 1.00000000E+01, 1.93550000E+00, 0.00000000E+00},
    {8.08997300E+02, 1.40000000E+02, 1.10000000E+01, 1.93550000E+00, 0.00000000E+00},
    {6.00551100E+02, 1.40000000E+02, 1.20000000E+01, 1.93550000E+00, 0.00000000E+00},
    {5.46717600E+02, 1.40000000E+02, 1.30000000E+01, 1.93550000E+00, 0.00000000E+00},
    {4.23699200E+02, 1.40000000E+02, 1.40000000E+01, 1.93550000E+00, 0.00000000E+00},
    {3.26008700E+02, 1.40000000E+02, 1.50000000E+01, 1.93550000E+00, 0.00000000E+00},
    {2.68307700E+02, 1.40000000E+02, 1.60000000E+01, 1.93550000E+00, 0.00000000E+00},
    {2.17432200E+02, 1.40000000E+02, 1.70000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.76669000E+02, 1.40000000E+02, 1.80000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.33598990E+03, 1.40000000E+02, 1.90000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.07091040E+03, 1.40000000E+02, 2.00000000E+01, 1.93550000E+00, 0.00000000E+00},
    {8.79016500E+02, 1.40000000E+02, 2.10000000E+01, 1.93550000E+00, 0.00000000E+00},
    {8.44156500E+02, 1.40000000E+02, 2.20000000E+01, 1.93550000E+00, 0.00000000E+00},
    {7.70418500E+02, 1.40000000E+02, 2.30000000E+01, 1.93550000E+00, 0.00000000E+00},
    {6.06704000E+02, 1.40000000E+02, 2.40000000E+01, 1.93550000E+00, 0.00000000E+00},
    {6.60100500E+02, 1.40000000E+02, 2.50000000E+01, 1.93550000E+00, 0.00000000E+00},
    {5.17581200E+02, 1.40000000E+02, 2.60000000E+01, 1.93550000E+00, 0.00000000E+00},
    {5.44664300E+02, 1.40000000E+02, 2.70000000E+01, 1.93550000E+00, 0.00000000E+00},
    {5.62934400E+02, 1.40000000E+02, 2.80000000E+01, 1.93550000E+00, 0.00000000E+00},
    {4.31554400E+02, 1.40000000E+02, 2.90000000E+01, 1.93550000E+00, 0.00000000E+00},
    {4.38408000E+02, 1.40000000E+02, 3.00000000E+01, 1.93550000E+00, 0.00000000E+00},
    {5.21180200E+02, 1.40000000E+02, 3.10000000E+01, 1.93550000E+00, 0.00000000E+00},
    {4.53690500E+02, 1.40000000E+02, 3.20000000E+01, 1.93550000E+00, 0.00000000E+00},
    {3.82645800E+02, 1.40000000E+02, 3.30000000E+01, 1.93550000E+00, 0.00000000E+00},
    {3.41033800E+02, 1.40000000E+02, 3.40000000E+01, 1.93550000E+00, 0.00000000E+00},
    {2.96394400E+02, 1.40000000E+02, 3.50000000E+01, 1.93550000E+00, 0.00000000E+00},
    {2.56150100E+02, 1.40000000E+02, 3.60000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.49379930E+03, 1.40000000E+02, 3.70000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.27758130E+03, 1.40000000E+02, 3.80000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.10666300E+03, 1.40000000E+02, 3.90000000E+01, 1.93550000E+00, 0.00000000E+00},
    {9.87825700E+02, 1.40000000E+02, 4.00000000E+01, 1.93550000E+00, 0.00000000E+00},
    {8.96795200E+02, 1.40000000E+02, 4.10000000E+01, 1.93550000E+00, 0.00000000E+00},
    {6.87184800E+02, 1.40000000E+02, 4.20000000E+01, 1.93550000E+00, 0.00000000E+00},
    {7.68842800E+02, 1.40000000E+02, 4.30000000E+01, 1.93550000E+00, 0.00000000E+00},
    {5.80900700E+02, 1.40000000E+02, 4.40000000E+01, 1.93550000E+00, 0.00000000E+00},
    {6.34802400E+02, 1.40000000E+02, 4.50000000E+01, 1.93550000E+00, 0.00000000E+00},
    {5.87020200E+02, 1.40000000E+02, 4.60000000E+01, 1.93550000E+00, 0.00000000E+00},
    {4.90381700E+02, 1.40000000E+02, 4.70000000E+01, 1.93550000E+00, 0.00000000E+00},
    {5.15386100E+02, 1.40000000E+02, 4.80000000E+01, 1.93550000E+00, 0.00000000E+00},
    {6.52565400E+02, 1.40000000E+02, 4.90000000E+01, 1.93550000E+00, 0.00000000E+00},
    {5.96891900E+02, 1.40000000E+02, 5.00000000E+01, 1.93550000E+00, 0.00000000E+00},
    {5.26542600E+02, 1.40000000E+02, 5.10000000E+01, 1.93550000E+00, 0.00000000E+00},
    {4.85617600E+02, 1.40000000E+02, 5.20000000E+01, 1.93550000E+00, 0.00000000E+00},
    {4.36304600E+02, 1.40000000E+02, 5.30000000E+01, 1.93550000E+00, 0.00000000E+00},
    {3.89938500E+02, 1.40000000E+02, 5.40000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.81904060E+03, 1.40000000E+02, 5.50000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.63536220E+03, 1.40000000E+02, 5.60000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.42061920E+03, 1.40000000E+02, 5.70000000E+01, 1.93550000E+00, 0.00000000E+00},
    {6.23284100E+02, 1.40000000E+02, 5.80000000E+01, 1.93550000E+00, 2.79910000E+00},
    {1.44296800E+03, 1.40000000E+02, 5.90000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.38321900E+03, 1.40000000E+02, 6.00000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.34778040E+03, 1.40000000E+02, 6.10000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.31526700E+03, 1.40000000E+02, 6.20000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.28641060E+03, 1.40000000E+02, 6.30000000E+01, 1.93550000E+00, 0.00000000E+00},
    {9.99361500E+02, 1.40000000E+02, 6.40000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.14612760E+03, 1.40000000E+02, 6.50000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.10256140E+03, 1.40000000E+02, 6.60000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.15687290E+03, 1.40000000E+02, 6.70000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.13194210E+03, 1.40000000E+02, 6.80000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.10930350E+03, 1.40000000E+02, 6.90000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.09686810E+03, 1.40000000E+02, 7.00000000E+01, 1.93550000E+00, 0.00000000E+00},
    {9.16292500E+02, 1.40000000E+02, 7.10000000E+01, 1.93550000E+00, 0.00000000E+00},
    {8.91410000E+02, 1.40000000E+02, 7.20000000E+01, 1.93550000E+00, 0.00000000E+00},
    {8.08106600E+02, 1.40000000E+02, 7.30000000E+01, 1.93550000E+00, 0.00000000E+00},
    {6.79045800E+02, 1.40000000E+02, 7.40000000E+01, 1.93550000E+00, 0.00000000E+00},
    {6.88971900E+02, 1.40000000E+02, 7.50000000E+01, 1.93550000E+00, 0.00000000E+00},
    {6.20889400E+02, 1.40000000E+02, 7.60000000E+01, 1.93550000E+00, 0.00000000E+00},
    {5.66046200E+02, 1.40000000E+02, 7.70000000E+01, 1.93550000E+00, 0.00000000E+00},
    {4.68024600E+02, 1.40000000E+02, 7.80000000E+01, 1.93550000E+00, 0.00000000E+00},
    {4.36432200E+02, 1.40000000E+02, 7.90000000E+01, 1.93550000E+00, 0.00000000E+00},
    {4.47966300E+02, 1.40000000E+02, 8.00000000E+01, 1.93550000E+00, 0.00000000E+00},
    {6.68367200E+02, 1.40000000E+02, 8.10000000E+01, 1.93550000E+00, 0.00000000E+00},
    {6.47816500E+02, 1.40000000E+02, 8.20000000E+01, 1.93550000E+00, 0.00000000E+00},
    {5.89764800E+02, 1.40000000E+02, 8.30000000E+01, 1.93550000E+00, 0.00000000E+00},
    {5.59564300E+02, 1.40000000E+02, 8.40000000E+01, 1.93550000E+00, 0.00000000E+00},
    {5.13257400E+02, 1.40000000E+02, 8.50000000E+01, 1.93550000E+00, 0.00000000E+00},
    {4.67845800E+02, 1.40000000E+02, 8.60000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.70285920E+03, 1.40000000E+02, 8.70000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.60781780E+03, 1.40000000E+02, 8.80000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.40612190E+03, 1.40000000E+02, 8.90000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.24902500E+03, 1.40000000E+02, 9.00000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.24732120E+03, 1.40000000E+02, 9.10000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.20733490E+03, 1.40000000E+02, 9.20000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.25191540E+03, 1.40000000E+02, 9.30000000E+01, 1.93550000E+00, 0.00000000E+00},
    {1.21069810E+03, 1.40000000E+02, 9.40000000E+01, 1.93550000E+00, 0.00000000E+00},
    {6.47068000E+01, 1.40000000E+02, 1.01000000E+02, 1.93550000E+00, 0.00000000E+00},
    {2.16717000E+02, 1.40000000E+02, 1.03000000E+02, 1.93550000E+00, 9.86500000E-01},
    {2.75364400E+02, 1.40000000E+02, 1.04000000E+02, 1.93550000E+00, 9.80800000E-01},
    {2.06001400E+02, 1.40000000E+02, 1.05000000E+02, 1.93550000E+00, 9.70600000E-01},
    {1.53422900E+02, 1.40000000E+02, 1.06000000E+02, 1.93550000E+00, 9.86800000E-01},
    {1.05315100E+02, 1.40000000E+02, 1.07000000E+02, 1.93550000E+00, 9.94400000E-01},
    {7.58994000E+01, 1.40000000E+02, 1.08000000E+02, 1.93550000E+00, 9.92500000E-01},
    {5.15158000E+01, 1.40000000E+02, 1.09000000E+02, 1.93550000E+00, 9.98200000E-01},
    {3.18167800E+02, 1.40000000E+02, 1.11000000E+02, 1.93550000E+00, 9.68400000E-01},
    {4.93305700E+02, 1.40000000E+02, 1.12000000E+02, 1.93550000E+00, 9.62800000E-01},
    {4.94024500E+02, 1.40000000E+02, 1.13000000E+02, 1.93550000E+00, 9.64800000E-01},
    {3.90719200E+02, 1.40000000E+02, 1.14000000E+02, 1.93550000E+00, 9.50700000E-01},
    {3.16316900E+02, 1.40000000E+02, 1.15000000E+02, 1.93550000E+00, 9.94700000E-01},
    {2.65408900E+02, 1.40000000E+02, 1.16000000E+02, 1.93550000E+00, 9.94800000E-01},
    {2.15210100E+02, 1.40000000E+02, 1.17000000E+02, 1.93550000E+00, 9.97200000E-01},
    {4.35064900E+02, 1.40000000E+02, 1.19000000E+02, 1.93550000E+00, 9.76700000E-01},
    {8.56713100E+02, 1.40000000E+02, 1.20000000E+02, 1.93550000E+00, 9.83100000E-01},
    {4.29301900E+02, 1.40000000E+02, 1.21000000E+02, 1.93550000E+00, 1.86270000E+00},
    {4.14203000E+02, 1.40000000E+02, 1.22000000E+02, 1.93550000E+00, 1.82990000E+00},
    {4.06022300E+02, 1.40000000E+02, 1.23000000E+02, 1.93550000E+00, 1.91380000E+00},
    {4.02920000E+02, 1.40000000E+02, 1.24000000E+02, 1.93550000E+00, 1.82690000E+00},
    {3.67793700E+02, 1.40000000E+02, 1.25000000E+02, 1.93550000E+00, 1.64060000E+00},
    {3.39627800E+02, 1.40000000E+02, 1.26000000E+02, 1.93550000E+00, 1.64830000E+00},
    {3.23992100E+02, 1.40000000E+02, 1.27000000E+02, 1.93550000E+00, 1.71490000E+00},
    {3.16941700E+02, 1.40000000E+02, 1.28000000E+02, 1.93550000E+00, 1.79370000E+00},
    {3.14990800E+02, 1.40000000E+02, 1.29000000E+02, 1.93550000E+00, 9.57600000E-01},
    {2.92400500E+02, 1.40000000E+02, 1.30000000E+02, 1.93550000E+00, 1.94190000E+00},
    {4.87875900E+02, 1.40000000E+02, 1.31000000E+02, 1.93550000E+00, 9.60100000E-01},
    {4.23207600E+02, 1.40000000E+02, 1.32000000E+02, 1.93550000E+00, 9.43400000E-01},
    {3.75917900E+02, 1.40000000E+02, 1.33000000E+02, 1.93550000E+00, 9.88900000E-01},
    {3.41204000E+02, 1.40000000E+02, 1.34000000E+02, 1.93550000E+00, 9.90100000E-01},
    {2.98563400E+02, 1.40000000E+02, 1.35000000E+02, 1.93550000E+00, 9.97400000E-01},
    {5.17696100E+02, 1.40000000E+02, 1.37000000E+02, 1.93550000E+00, 9.73800000E-01},
    {1.04434010E+03, 1.40000000E+02, 1.38000000E+02, 1.93550000E+00, 9.80100000E-01},
    {7.80821200E+02, 1.40000000E+02, 1.39000000E+02, 1.93550000E+00, 1.91530000E+00},
    {5.68678900E+02, 1.40000000E+02, 1.40000000E+02, 1.93550000E+00, 1.93550000E+00},
    {4.00304000E+01, 1.41000000E+02, 1.00000000E+00, 1.95450000E+00, 9.11800000E-01},
    {2.59534000E+01, 1.41000000E+02, 2.00000000E+00, 1.95450000E+00, 0.00000000E+00},
    {6.84104000E+02, 1.41000000E+02, 3.00000000E+00, 1.95450000E+00, 0.00000000E+00},
    {3.77969300E+02, 1.41000000E+02, 4.00000000E+00, 1.95450000E+00, 0.00000000E+00},
    {2.48426600E+02, 1.41000000E+02, 5.00000000E+00, 1.95450000E+00, 0.00000000E+00},
    {1.64998000E+02, 1.41000000E+02, 6.00000000E+00, 1.95450000E+00, 0.00000000E+00},
    {1.13998900E+02, 1.41000000E+02, 7.00000000E+00, 1.95450000E+00, 0.00000000E+00},
    {8.56024000E+01, 1.41000000E+02, 8.00000000E+00, 1.95450000E+00, 0.00000000E+00},
    {6.44010000E+01, 1.41000000E+02, 9.00000000E+00, 1.95450000E+00, 0.00000000E+00},
    {4.92635000E+01, 1.41000000E+02, 1.00000000E+01, 1.95450000E+00, 0.00000000E+00},
    {8.16031700E+02, 1.41000000E+02, 1.10000000E+01, 1.95450000E+00, 0.00000000E+00},
    {6.07007300E+02, 1.41000000E+02, 1.20000000E+01, 1.95450000E+00, 0.00000000E+00},
    {5.52355000E+02, 1.41000000E+02, 1.30000000E+01, 1.95450000E+00, 0.00000000E+00},
    {4.27883700E+02, 1.41000000E+02, 1.40000000E+01, 1.95450000E+00, 0.00000000E+00},
    {3.29094700E+02, 1.41000000E+02, 1.50000000E+01, 1.95450000E+00, 0.00000000E+00},
    {2.70805900E+02, 1.41000000E+02, 1.60000000E+01, 1.95450000E+00, 0.00000000E+00},
    {2.19446200E+02, 1.41000000E+02, 1.70000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.78318300E+02, 1.41000000E+02, 1.80000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.34337140E+03, 1.41000000E+02, 1.90000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.08199300E+03, 1.41000000E+02, 2.00000000E+01, 1.95450000E+00, 0.00000000E+00},
    {8.88335600E+02, 1.41000000E+02, 2.10000000E+01, 1.95450000E+00, 0.00000000E+00},
    {8.53046400E+02, 1.41000000E+02, 2.20000000E+01, 1.95450000E+00, 0.00000000E+00},
    {7.78526600E+02, 1.41000000E+02, 2.30000000E+01, 1.95450000E+00, 0.00000000E+00},
    {6.13022400E+02, 1.41000000E+02, 2.40000000E+01, 1.95450000E+00, 0.00000000E+00},
    {6.67030500E+02, 1.41000000E+02, 2.50000000E+01, 1.95450000E+00, 0.00000000E+00},
    {5.23002000E+02, 1.41000000E+02, 2.60000000E+01, 1.95450000E+00, 0.00000000E+00},
    {5.50409200E+02, 1.41000000E+02, 2.70000000E+01, 1.95450000E+00, 0.00000000E+00},
    {5.68908800E+02, 1.41000000E+02, 2.80000000E+01, 1.95450000E+00, 0.00000000E+00},
    {4.36104200E+02, 1.41000000E+02, 2.90000000E+01, 1.95450000E+00, 0.00000000E+00},
    {4.42988600E+02, 1.41000000E+02, 3.00000000E+01, 1.95450000E+00, 0.00000000E+00},
    {5.26496400E+02, 1.41000000E+02, 3.10000000E+01, 1.95450000E+00, 0.00000000E+00},
    {4.58192000E+02, 1.41000000E+02, 3.20000000E+01, 1.95450000E+00, 0.00000000E+00},
    {3.86309100E+02, 1.41000000E+02, 3.30000000E+01, 1.95450000E+00, 0.00000000E+00},
    {3.44242500E+02, 1.41000000E+02, 3.40000000E+01, 1.95450000E+00, 0.00000000E+00},
    {2.99153500E+02, 1.41000000E+02, 3.50000000E+01, 1.95450000E+00, 0.00000000E+00},
    {2.58530400E+02, 1.41000000E+02, 3.60000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.50177890E+03, 1.41000000E+02, 3.70000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.29048260E+03, 1.41000000E+02, 3.80000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.11814020E+03, 1.41000000E+02, 3.90000000E+01, 1.95450000E+00, 0.00000000E+00},
    {9.98122400E+02, 1.41000000E+02, 4.00000000E+01, 1.95450000E+00, 0.00000000E+00},
    {9.06080100E+02, 1.41000000E+02, 4.10000000E+01, 1.95450000E+00, 0.00000000E+00},
    {6.94298300E+02, 1.41000000E+02, 4.20000000E+01, 1.95450000E+00, 0.00000000E+00},
    {7.76736800E+02, 1.41000000E+02, 4.30000000E+01, 1.95450000E+00, 0.00000000E+00},
    {5.86894500E+02, 1.41000000E+02, 4.40000000E+01, 1.95450000E+00, 0.00000000E+00},
    {6.41416400E+02, 1.41000000E+02, 4.50000000E+01, 1.95450000E+00, 0.00000000E+00},
    {5.93141400E+02, 1.41000000E+02, 4.60000000E+01, 1.95450000E+00, 0.00000000E+00},
    {4.95480000E+02, 1.41000000E+02, 4.70000000E+01, 1.95450000E+00, 0.00000000E+00},
    {5.20762900E+02, 1.41000000E+02, 4.80000000E+01, 1.95450000E+00, 0.00000000E+00},
    {6.59286500E+02, 1.41000000E+02, 4.90000000E+01, 1.95450000E+00, 0.00000000E+00},
    {6.02969200E+02, 1.41000000E+02, 5.00000000E+01, 1.95450000E+00, 0.00000000E+00},
    {5.31742400E+02, 1.41000000E+02, 5.10000000E+01, 1.95450000E+00, 0.00000000E+00},
    {4.90320600E+02, 1.41000000E+02, 5.20000000E+01, 1.95450000E+00, 0.00000000E+00},
    {4.40459900E+02, 1.41000000E+02, 5.30000000E+01, 1.95450000E+00, 0.00000000E+00},
    {3.93612100E+02, 1.41000000E+02, 5.40000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.82644610E+03, 1.41000000E+02, 5.50000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.65093340E+03, 1.41000000E+02, 5.60000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.43489560E+03, 1.41000000E+02, 5.70000000E+01, 1.95450000E+00, 0.00000000E+00},
    {6.29497700E+02, 1.41000000E+02, 5.80000000E+01, 1.95450000E+00, 2.79910000E+00},
    {1.45651130E+03, 1.41000000E+02, 5.90000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.39702980E+03, 1.41000000E+02, 6.00000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.36131780E+03, 1.41000000E+02, 6.10000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.32854150E+03, 1.41000000E+02, 6.20000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.29944880E+03, 1.41000000E+02, 6.30000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.00944190E+03, 1.41000000E+02, 6.40000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.15539470E+03, 1.41000000E+02, 6.50000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.11069130E+03, 1.41000000E+02, 6.60000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.16883930E+03, 1.41000000E+02, 6.70000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.14368570E+03, 1.41000000E+02, 6.80000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.12084360E+03, 1.41000000E+02, 6.90000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.10827910E+03, 1.41000000E+02, 7.00000000E+01, 1.95450000E+00, 0.00000000E+00},
    {9.25226600E+02, 1.41000000E+02, 7.10000000E+01, 1.95450000E+00, 0.00000000E+00},
    {9.00680800E+02, 1.41000000E+02, 7.20000000E+01, 1.95450000E+00, 0.00000000E+00},
    {8.16497400E+02, 1.41000000E+02, 7.30000000E+01, 1.95450000E+00, 0.00000000E+00},
    {6.85985600E+02, 1.41000000E+02, 7.40000000E+01, 1.95450000E+00, 0.00000000E+00},
    {6.96004000E+02, 1.41000000E+02, 7.50000000E+01, 1.95450000E+00, 0.00000000E+00},
    {6.27206500E+02, 1.41000000E+02, 7.60000000E+01, 1.95450000E+00, 0.00000000E+00},
    {5.71779300E+02, 1.41000000E+02, 7.70000000E+01, 1.95450000E+00, 0.00000000E+00},
    {4.72764300E+02, 1.41000000E+02, 7.80000000E+01, 1.95450000E+00, 0.00000000E+00},
    {4.40859700E+02, 1.41000000E+02, 7.90000000E+01, 1.95450000E+00, 0.00000000E+00},
    {4.52476100E+02, 1.41000000E+02, 8.00000000E+01, 1.95450000E+00, 0.00000000E+00},
    {6.75151800E+02, 1.41000000E+02, 8.10000000E+01, 1.95450000E+00, 0.00000000E+00},
    {6.54445100E+02, 1.41000000E+02, 8.20000000E+01, 1.95450000E+00, 0.00000000E+00},
    {5.95664000E+02, 1.41000000E+02, 8.30000000E+01, 1.95450000E+00, 0.00000000E+00},
    {5.65064800E+02, 1.41000000E+02, 8.40000000E+01, 1.95450000E+00, 0.00000000E+00},
    {5.18212400E+02, 1.41000000E+02, 8.50000000E+01, 1.95450000E+00, 0.00000000E+00},
    {4.72305800E+02, 1.41000000E+02, 8.60000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.71270500E+03, 1.41000000E+02, 8.70000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.62353760E+03, 1.41000000E+02, 8.80000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.42039830E+03, 1.41000000E+02, 8.90000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.26198670E+03, 1.41000000E+02, 9.00000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.26001470E+03, 1.41000000E+02, 9.10000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.21970640E+03, 1.41000000E+02, 9.20000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.26483780E+03, 1.41000000E+02, 9.30000000E+01, 1.95450000E+00, 0.00000000E+00},
    {1.22326380E+03, 1.41000000E+02, 9.40000000E+01, 1.95450000E+00, 0.00000000E+00},
    {6.53129000E+01, 1.41000000E+02, 1.01000000E+02, 1.95450000E+00, 0.00000000E+00},
    {2.18970600E+02, 1.41000000E+02, 1.03000000E+02, 1.95450000E+00, 9.86500000E-01},
    {2.78088100E+02, 1.41000000E+02, 1.04000000E+02, 1.95450000E+00, 9.80800000E-01},
    {2.07994300E+02, 1.41000000E+02, 1.05000000E+02, 1.95450000E+00, 9.70600000E-01},
    {1.54876400E+02, 1.41000000E+02, 1.06000000E+02, 1.95450000E+00, 9.86800000E-01},
    {1.06314100E+02, 1.41000000E+02, 1.07000000E+02, 1.95450000E+00, 9.94400000E-01},
    {7.66347000E+01, 1.41000000E+02, 1.08000000E+02, 1.95450000E+00, 9.92500000E-01},
    {5.20412000E+01, 1.41000000E+02, 1.09000000E+02, 1.95450000E+00, 9.98200000E-01},
    {3.21537800E+02, 1.41000000E+02, 1.11000000E+02, 1.95450000E+00, 9.68400000E-01},
    {4.98348600E+02, 1.41000000E+02, 1.12000000E+02, 1.95450000E+00, 9.62800000E-01},
    {4.99076100E+02, 1.41000000E+02, 1.13000000E+02, 1.95450000E+00, 9.64800000E-01},
    {3.94540300E+02, 1.41000000E+02, 1.14000000E+02, 1.95450000E+00, 9.50700000E-01},
    {3.19309900E+02, 1.41000000E+02, 1.15000000E+02, 1.95450000E+00, 9.94700000E-01},
    {2.67885800E+02, 1.41000000E+02, 1.16000000E+02, 1.95450000E+00, 9.94800000E-01},
    {2.17207800E+02, 1.41000000E+02, 1.17000000E+02, 1.95450000E+00, 9.97200000E-01},
    {4.39522800E+02, 1.41000000E+02, 1.19000000E+02, 1.95450000E+00, 9.76700000E-01},
    {8.64141100E+02, 1.41000000E+02, 1.20000000E+02, 1.95450000E+00, 9.83100000E-01},
    {4.33786500E+02, 1.41000000E+02, 1.21000000E+02, 1.95450000E+00, 1.86270000E+00},
    {4.18306800E+02, 1.41000000E+02, 1.22000000E+02, 1.95450000E+00, 1.82990000E+00},
    {4.10103300E+02, 1.41000000E+02, 1.23000000E+02, 1.95450000E+00, 1.91380000E+00},
    {4.06989400E+02, 1.41000000E+02, 1.24000000E+02, 1.95450000E+00, 1.82690000E+00},
    {3.71469100E+02, 1.41000000E+02, 1.25000000E+02, 1.95450000E+00, 1.64060000E+00},
    {3.43001200E+02, 1.41000000E+02, 1.26000000E+02, 1.95450000E+00, 1.64830000E+00},
    {3.27199500E+02, 1.41000000E+02, 1.27000000E+02, 1.95450000E+00, 1.71490000E+00},
    {3.20096000E+02, 1.41000000E+02, 1.28000000E+02, 1.95450000E+00, 1.79370000E+00},
    {3.18213600E+02, 1.41000000E+02, 1.29000000E+02, 1.95450000E+00, 9.57600000E-01},
    {2.95276200E+02, 1.41000000E+02, 1.30000000E+02, 1.95450000E+00, 1.94190000E+00},
    {4.92831600E+02, 1.41000000E+02, 1.31000000E+02, 1.95450000E+00, 9.60100000E-01},
    {4.27362200E+02, 1.41000000E+02, 1.32000000E+02, 1.95450000E+00, 9.43400000E-01},
    {3.79509700E+02, 1.41000000E+02, 1.33000000E+02, 1.95450000E+00, 9.88900000E-01},
    {3.44416900E+02, 1.41000000E+02, 1.34000000E+02, 1.95450000E+00, 9.90100000E-01},
    {3.01346300E+02, 1.41000000E+02, 1.35000000E+02, 1.95450000E+00, 9.97400000E-01},
    {5.22944900E+02, 1.41000000E+02, 1.37000000E+02, 1.95450000E+00, 9.73800000E-01},
    {1.05286290E+03, 1.41000000E+02, 1.38000000E+02, 1.95450000E+00, 9.80100000E-01},
    {7.87495000E+02, 1.41000000E+02, 1.39000000E+02, 1.95450000E+00, 1.91530000E+00},
    {5.74459100E+02, 1.41000000E+02, 1.40000000E+02, 1.95450000E+00, 1.93550000E+00},
    {5.80582400E+02, 1.41000000E+02, 1.41000000E+02, 1.95450000E+00, 1.95450000E+00},
    {3.75478000E+01, 1.42000000E+02, 1.00000000E+00, 1.94200000E+00, 9.11800000E-01},
    {2.45530000E+01, 1.42000000E+02, 2.00000000E+00, 1.94200000E+00, 0.00000000E+00},
    {6.28181700E+02, 1.42000000E+02, 3.00000000E+00, 1.94200000E+00, 0.00000000E+00},
    {3.49189700E+02, 1.42000000E+02, 4.00000000E+00, 1.94200000E+00, 0.00000000E+00},
    {2.31044900E+02, 1.42000000E+02, 5.00000000E+00, 1.94200000E+00, 0.00000000E+00},
    {1.54328800E+02, 1.42000000E+02, 6.00000000E+00, 1.94200000E+00, 0.00000000E+00},
    {1.07130400E+02, 1.42000000E+02, 7.00000000E+00, 1.94200000E+00, 0.00000000E+00},
    {8.07360000E+01, 1.42000000E+02, 8.00000000E+00, 1.94200000E+00, 0.00000000E+00},
    {6.09414000E+01, 1.42000000E+02, 9.00000000E+00, 1.94200000E+00, 0.00000000E+00},
    {4.67498000E+01, 1.42000000E+02, 1.00000000E+01, 1.94200000E+00, 0.00000000E+00},
    {7.49497400E+02, 1.42000000E+02, 1.10000000E+01, 1.94200000E+00, 0.00000000E+00},
    {5.59885600E+02, 1.42000000E+02, 1.20000000E+01, 1.94200000E+00, 0.00000000E+00},
    {5.11081000E+02, 1.42000000E+02, 1.30000000E+01, 1.94200000E+00, 0.00000000E+00},
    {3.97660200E+02, 1.42000000E+02, 1.40000000E+01, 1.94200000E+00, 0.00000000E+00},
    {3.07138400E+02, 1.42000000E+02, 1.50000000E+01, 1.94200000E+00, 0.00000000E+00},
    {2.53498500E+02, 1.42000000E+02, 1.60000000E+01, 1.94200000E+00, 0.00000000E+00},
    {2.06045900E+02, 1.42000000E+02, 1.70000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.67901400E+02, 1.42000000E+02, 1.80000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.23796430E+03, 1.42000000E+02, 1.90000000E+01, 1.94200000E+00, 0.00000000E+00},
    {9.96015700E+02, 1.42000000E+02, 2.00000000E+01, 1.94200000E+00, 0.00000000E+00},
    {8.18516100E+02, 1.42000000E+02, 2.10000000E+01, 1.94200000E+00, 0.00000000E+00},
    {7.87056500E+02, 1.42000000E+02, 2.20000000E+01, 1.94200000E+00, 0.00000000E+00},
    {7.18846900E+02, 1.42000000E+02, 2.30000000E+01, 1.94200000E+00, 0.00000000E+00},
    {5.66430000E+02, 1.42000000E+02, 2.40000000E+01, 1.94200000E+00, 0.00000000E+00},
    {6.16599300E+02, 1.42000000E+02, 2.50000000E+01, 1.94200000E+00, 0.00000000E+00},
    {4.83853700E+02, 1.42000000E+02, 2.60000000E+01, 1.94200000E+00, 0.00000000E+00},
    {5.09699900E+02, 1.42000000E+02, 2.70000000E+01, 1.94200000E+00, 0.00000000E+00},
    {5.26379500E+02, 1.42000000E+02, 2.80000000E+01, 1.94200000E+00, 0.00000000E+00},
    {4.03840000E+02, 1.42000000E+02, 2.90000000E+01, 1.94200000E+00, 0.00000000E+00},
    {4.10986600E+02, 1.42000000E+02, 3.00000000E+01, 1.94200000E+00, 0.00000000E+00},
    {4.87985800E+02, 1.42000000E+02, 3.10000000E+01, 1.94200000E+00, 0.00000000E+00},
    {4.26070300E+02, 1.42000000E+02, 3.20000000E+01, 1.94200000E+00, 0.00000000E+00},
    {3.60449600E+02, 1.42000000E+02, 3.30000000E+01, 1.94200000E+00, 0.00000000E+00},
    {3.21941500E+02, 1.42000000E+02, 3.40000000E+01, 1.94200000E+00, 0.00000000E+00},
    {2.80470200E+02, 1.42000000E+02, 3.50000000E+01, 1.94200000E+00, 0.00000000E+00},
    {2.42968400E+02, 1.42000000E+02, 3.60000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.38512280E+03, 1.42000000E+02, 3.70000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.18819820E+03, 1.42000000E+02, 3.80000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.03147450E+03, 1.42000000E+02, 3.90000000E+01, 1.94200000E+00, 0.00000000E+00},
    {9.22066000E+02, 1.42000000E+02, 4.00000000E+01, 1.94200000E+00, 0.00000000E+00},
    {8.37984900E+02, 1.42000000E+02, 4.10000000E+01, 1.94200000E+00, 0.00000000E+00},
    {6.43472100E+02, 1.42000000E+02, 4.20000000E+01, 1.94200000E+00, 0.00000000E+00},
    {7.19365900E+02, 1.42000000E+02, 4.30000000E+01, 1.94200000E+00, 0.00000000E+00},
    {5.44807500E+02, 1.42000000E+02, 4.40000000E+01, 1.94200000E+00, 0.00000000E+00},
    {5.95150300E+02, 1.42000000E+02, 4.50000000E+01, 1.94200000E+00, 0.00000000E+00},
    {5.50755700E+02, 1.42000000E+02, 4.60000000E+01, 1.94200000E+00, 0.00000000E+00},
    {4.60236700E+02, 1.42000000E+02, 4.70000000E+01, 1.94200000E+00, 0.00000000E+00},
    {4.84028600E+02, 1.42000000E+02, 4.80000000E+01, 1.94200000E+00, 0.00000000E+00},
    {6.11391100E+02, 1.42000000E+02, 4.90000000E+01, 1.94200000E+00, 0.00000000E+00},
    {5.60510400E+02, 1.42000000E+02, 5.00000000E+01, 1.94200000E+00, 0.00000000E+00},
    {4.95694800E+02, 1.42000000E+02, 5.10000000E+01, 1.94200000E+00, 0.00000000E+00},
    {4.57930200E+02, 1.42000000E+02, 5.20000000E+01, 1.94200000E+00, 0.00000000E+00},
    {4.12227500E+02, 1.42000000E+02, 5.30000000E+01, 1.94200000E+00, 0.00000000E+00},
    {3.69153100E+02, 1.42000000E+02, 5.40000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.68774900E+03, 1.42000000E+02, 5.50000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.52013170E+03, 1.42000000E+02, 5.60000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.32325830E+03, 1.42000000E+02, 5.70000000E+01, 1.94200000E+00, 0.00000000E+00},
    {5.86643000E+02, 1.42000000E+02, 5.80000000E+01, 1.94200000E+00, 2.79910000E+00},
    {1.34281160E+03, 1.42000000E+02, 5.90000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.28727630E+03, 1.42000000E+02, 6.00000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.25440390E+03, 1.42000000E+02, 6.10000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.22423140E+03, 1.42000000E+02, 6.20000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.19745640E+03, 1.42000000E+02, 6.30000000E+01, 1.94200000E+00, 0.00000000E+00},
    {9.32833200E+02, 1.42000000E+02, 6.40000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.06670770E+03, 1.42000000E+02, 6.50000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.02710260E+03, 1.42000000E+02, 6.60000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.07742050E+03, 1.42000000E+02, 6.70000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.05424840E+03, 1.42000000E+02, 6.80000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.03323640E+03, 1.42000000E+02, 6.90000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.02153110E+03, 1.42000000E+02, 7.00000000E+01, 1.94200000E+00, 0.00000000E+00},
    {8.55245700E+02, 1.42000000E+02, 7.10000000E+01, 1.94200000E+00, 0.00000000E+00},
    {8.33452300E+02, 1.42000000E+02, 7.20000000E+01, 1.94200000E+00, 0.00000000E+00},
    {7.56753200E+02, 1.42000000E+02, 7.30000000E+01, 1.94200000E+00, 0.00000000E+00},
    {6.36907700E+02, 1.42000000E+02, 7.40000000E+01, 1.94200000E+00, 0.00000000E+00},
    {6.46549500E+02, 1.42000000E+02, 7.50000000E+01, 1.94200000E+00, 0.00000000E+00},
    {5.83525700E+02, 1.42000000E+02, 7.60000000E+01, 1.94200000E+00, 0.00000000E+00},
    {5.32657900E+02, 1.42000000E+02, 7.70000000E+01, 1.94200000E+00, 0.00000000E+00},
    {4.41196100E+02, 1.42000000E+02, 7.80000000E+01, 1.94200000E+00, 0.00000000E+00},
    {4.11713600E+02, 1.42000000E+02, 7.90000000E+01, 1.94200000E+00, 0.00000000E+00},
    {4.22719100E+02, 1.42000000E+02, 8.00000000E+01, 1.94200000E+00, 0.00000000E+00},
    {6.26932600E+02, 1.42000000E+02, 8.10000000E+01, 1.94200000E+00, 0.00000000E+00},
    {6.08662000E+02, 1.42000000E+02, 8.20000000E+01, 1.94200000E+00, 0.00000000E+00},
    {5.55325500E+02, 1.42000000E+02, 8.30000000E+01, 1.94200000E+00, 0.00000000E+00},
    {5.27599700E+02, 1.42000000E+02, 8.40000000E+01, 1.94200000E+00, 0.00000000E+00},
    {4.84789000E+02, 1.42000000E+02, 8.50000000E+01, 1.94200000E+00, 0.00000000E+00},
    {4.42667400E+02, 1.42000000E+02, 8.60000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.58154490E+03, 1.42000000E+02, 8.70000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.49620770E+03, 1.42000000E+02, 8.80000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.31120880E+03, 1.42000000E+02, 8.90000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.16763090E+03, 1.42000000E+02, 9.00000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.16492880E+03, 1.42000000E+02, 9.10000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.12769270E+03, 1.42000000E+02, 9.20000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.16763830E+03, 1.42000000E+02, 9.30000000E+01, 1.94200000E+00, 0.00000000E+00},
    {1.12948300E+03, 1.42000000E+02, 9.40000000E+01, 1.94200000E+00, 0.00000000E+00},
    {6.09773000E+01, 1.42000000E+02, 1.01000000E+02, 1.94200000E+00, 0.00000000E+00},
    {2.02517300E+02, 1.42000000E+02, 1.03000000E+02, 1.94200000E+00, 9.86500000E-01},
    {2.57608800E+02, 1.42000000E+02, 1.04000000E+02, 1.94200000E+00, 9.80800000E-01},
    {1.93755700E+02, 1.42000000E+02, 1.05000000E+02, 1.94200000E+00, 9.70600000E-01},
    {1.44873800E+02, 1.42000000E+02, 1.06000000E+02, 1.94200000E+00, 9.86800000E-01},
    {9.99381000E+01, 1.42000000E+02, 1.07000000E+02, 1.94200000E+00, 9.94400000E-01},
    {7.23520000E+01, 1.42000000E+02, 1.08000000E+02, 1.94200000E+00, 9.92500000E-01},
    {4.94166000E+01, 1.42000000E+02, 1.09000000E+02, 1.94200000E+00, 9.98200000E-01},
    {2.97171300E+02, 1.42000000E+02, 1.11000000E+02, 1.94200000E+00, 9.68400000E-01},
    {4.60423400E+02, 1.42000000E+02, 1.12000000E+02, 1.94200000E+00, 9.62800000E-01},
    {4.62167800E+02, 1.42000000E+02, 1.13000000E+02, 1.94200000E+00, 9.64800000E-01},
    {3.66954900E+02, 1.42000000E+02, 1.14000000E+02, 1.94200000E+00, 9.50700000E-01},
    {2.98066100E+02, 1.42000000E+02, 1.15000000E+02, 1.94200000E+00, 9.94700000E-01},
    {2.50757300E+02, 1.42000000E+02, 1.16000000E+02, 1.94200000E+00, 9.94800000E-01},
    {2.03939700E+02, 1.42000000E+02, 1.17000000E+02, 1.94200000E+00, 9.97200000E-01},
    {4.07590600E+02, 1.42000000E+02, 1.19000000E+02, 1.94200000E+00, 9.76700000E-01},
    {7.97477300E+02, 1.42000000E+02, 1.20000000E+02, 1.94200000E+00, 9.83100000E-01},
    {4.03154000E+02, 1.42000000E+02, 1.21000000E+02, 1.94200000E+00, 1.86270000E+00},
    {3.89178700E+02, 1.42000000E+02, 1.22000000E+02, 1.94200000E+00, 1.82990000E+00},
    {3.81480400E+02, 1.42000000E+02, 1.23000000E+02, 1.94200000E+00, 1.91380000E+00},
    {3.78447400E+02, 1.42000000E+02, 1.24000000E+02, 1.94200000E+00, 1.82690000E+00},
    {3.46042300E+02, 1.42000000E+02, 1.25000000E+02, 1.94200000E+00, 1.64060000E+00},
    {3.19763700E+02, 1.42000000E+02, 1.26000000E+02, 1.94200000E+00, 1.64830000E+00},
    {3.05088700E+02, 1.42000000E+02, 1.27000000E+02, 1.94200000E+00, 1.71490000E+00},
    {2.98412900E+02, 1.42000000E+02, 1.28000000E+02, 1.94200000E+00, 1.79370000E+00},
    {2.96198900E+02, 1.42000000E+02, 1.29000000E+02, 1.94200000E+00, 9.57600000E-01},
    {2.75605500E+02, 1.42000000E+02, 1.30000000E+02, 1.94200000E+00, 1.94190000E+00},
    {4.57164100E+02, 1.42000000E+02, 1.31000000E+02, 1.94200000E+00, 9.60100000E-01},
    {3.97759200E+02, 1.42000000E+02, 1.32000000E+02, 1.94200000E+00, 9.43400000E-01},
    {3.54186800E+02, 1.42000000E+02, 1.33000000E+02, 1.94200000E+00, 9.88900000E-01},
    {3.22097800E+02, 1.42000000E+02, 1.34000000E+02, 1.94200000E+00, 9.90100000E-01},
    {2.82493300E+02, 1.42000000E+02, 1.35000000E+02, 1.94200000E+00, 9.97400000E-01},
    {4.85412900E+02, 1.42000000E+02, 1.37000000E+02, 1.94200000E+00, 9.73800000E-01},
    {9.72123300E+02, 1.42000000E+02, 1.38000000E+02, 1.94200000E+00, 9.80100000E-01},
    {7.30144600E+02, 1.42000000E+02, 1.39000000E+02, 1.94200000E+00, 1.91530000E+00},
    {5.34010800E+02, 1.42000000E+02, 1.40000000E+02, 1.94200000E+00, 1.93550000E+00},
    {5.39353200E+02, 1.42000000E+02, 1.41000000E+02, 1.94200000E+00, 1.95450000E+00},
    {5.02007300E+02, 1.42000000E+02, 1.42000000E+02, 1.94200000E+00, 1.94200000E+00},
    {4.13385000E+01, 1.43000000E+02, 1.00000000E+00, 1.66820000E+00, 9.11800000E-01},
    {2.66217000E+01, 1.43000000E+02, 2.00000000E+00, 1.66820000E+00, 0.00000000E+00},
    {7.63781400E+02, 1.43000000E+02, 3.00000000E+00, 1.66820000E+00, 0.00000000E+00},
    {4.02772700E+02, 1.43000000E+02, 4.00000000E+00, 1.66820000E+00, 0.00000000E+00},
    {2.60379600E+02, 1.43000000E+02, 5.00000000E+00, 1.66820000E+00, 0.00000000E+00},
    {1.71258100E+02, 1.43000000E+02, 6.00000000E+00, 1.66820000E+00, 0.00000000E+00},
    {1.17647900E+02, 1.43000000E+02, 7.00000000E+00, 1.66820000E+00, 0.00000000E+00},
    {8.80609000E+01, 1.43000000E+02, 8.00000000E+00, 1.66820000E+00, 0.00000000E+00},
    {6.61093000E+01, 1.43000000E+02, 9.00000000E+00, 1.66820000E+00, 0.00000000E+00},
    {5.05087000E+01, 1.43000000E+02, 1.00000000E+01, 1.66820000E+00, 0.00000000E+00},
    {9.08402500E+02, 1.43000000E+02, 1.10000000E+01, 1.66820000E+00, 0.00000000E+00},
    {6.51256400E+02, 1.43000000E+02, 1.20000000E+01, 1.66820000E+00, 0.00000000E+00},
    {5.86986200E+02, 1.43000000E+02, 1.30000000E+01, 1.66820000E+00, 0.00000000E+00},
    {4.49119100E+02, 1.43000000E+02, 1.40000000E+01, 1.66820000E+00, 0.00000000E+00},
    {3.42488300E+02, 1.43000000E+02, 1.50000000E+01, 1.66820000E+00, 0.00000000E+00},
    {2.80561300E+02, 1.43000000E+02, 1.60000000E+01, 1.66820000E+00, 0.00000000E+00},
    {2.26483300E+02, 1.43000000E+02, 1.70000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.83520500E+02, 1.43000000E+02, 1.80000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.51887550E+03, 1.43000000E+02, 1.90000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.17833020E+03, 1.43000000E+02, 2.00000000E+01, 1.66820000E+00, 0.00000000E+00},
    {9.61365700E+02, 1.43000000E+02, 2.10000000E+01, 1.66820000E+00, 0.00000000E+00},
    {9.19309900E+02, 1.43000000E+02, 2.20000000E+01, 1.66820000E+00, 0.00000000E+00},
    {8.36734100E+02, 1.43000000E+02, 2.30000000E+01, 1.66820000E+00, 0.00000000E+00},
    {6.59720100E+02, 1.43000000E+02, 2.40000000E+01, 1.66820000E+00, 0.00000000E+00},
    {7.14183000E+02, 1.43000000E+02, 2.50000000E+01, 1.66820000E+00, 0.00000000E+00},
    {5.60364800E+02, 1.43000000E+02, 2.60000000E+01, 1.66820000E+00, 0.00000000E+00},
    {5.85293500E+02, 1.43000000E+02, 2.70000000E+01, 1.66820000E+00, 0.00000000E+00},
    {6.06474000E+02, 1.43000000E+02, 2.80000000E+01, 1.66820000E+00, 0.00000000E+00},
    {4.65814700E+02, 1.43000000E+02, 2.90000000E+01, 1.66820000E+00, 0.00000000E+00},
    {4.68452900E+02, 1.43000000E+02, 3.00000000E+01, 1.66820000E+00, 0.00000000E+00},
    {5.58152200E+02, 1.43000000E+02, 3.10000000E+01, 1.66820000E+00, 0.00000000E+00},
    {4.80892600E+02, 1.43000000E+02, 3.20000000E+01, 1.66820000E+00, 0.00000000E+00},
    {4.02454000E+02, 1.43000000E+02, 3.30000000E+01, 1.66820000E+00, 0.00000000E+00},
    {3.57251100E+02, 1.43000000E+02, 3.40000000E+01, 1.66820000E+00, 0.00000000E+00},
    {3.09353000E+02, 1.43000000E+02, 3.50000000E+01, 1.66820000E+00, 0.00000000E+00},
    {2.66577700E+02, 1.43000000E+02, 3.60000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.69612140E+03, 1.43000000E+02, 3.70000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.40838950E+03, 1.43000000E+02, 3.80000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.20798680E+03, 1.43000000E+02, 3.90000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.07219350E+03, 1.43000000E+02, 4.00000000E+01, 1.66820000E+00, 0.00000000E+00},
    {9.70207000E+02, 1.43000000E+02, 4.10000000E+01, 1.66820000E+00, 0.00000000E+00},
    {7.39791500E+02, 1.43000000E+02, 4.20000000E+01, 1.66820000E+00, 0.00000000E+00},
    {8.29240800E+02, 1.43000000E+02, 4.30000000E+01, 1.66820000E+00, 0.00000000E+00},
    {6.23138600E+02, 1.43000000E+02, 4.40000000E+01, 1.66820000E+00, 0.00000000E+00},
    {6.80052400E+02, 1.43000000E+02, 4.50000000E+01, 1.66820000E+00, 0.00000000E+00},
    {6.27581000E+02, 1.43000000E+02, 4.60000000E+01, 1.66820000E+00, 0.00000000E+00},
    {5.26097700E+02, 1.43000000E+02, 4.70000000E+01, 1.66820000E+00, 0.00000000E+00},
    {5.49633800E+02, 1.43000000E+02, 4.80000000E+01, 1.66820000E+00, 0.00000000E+00},
    {7.00422900E+02, 1.43000000E+02, 4.90000000E+01, 1.66820000E+00, 0.00000000E+00},
    {6.34783100E+02, 1.43000000E+02, 5.00000000E+01, 1.66820000E+00, 0.00000000E+00},
    {5.55779400E+02, 1.43000000E+02, 5.10000000E+01, 1.66820000E+00, 0.00000000E+00},
    {5.10568200E+02, 1.43000000E+02, 5.20000000E+01, 1.66820000E+00, 0.00000000E+00},
    {4.56970700E+02, 1.43000000E+02, 5.30000000E+01, 1.66820000E+00, 0.00000000E+00},
    {4.07097000E+02, 1.43000000E+02, 5.40000000E+01, 1.66820000E+00, 0.00000000E+00},
    {2.06885570E+03, 1.43000000E+02, 5.50000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.81106260E+03, 1.43000000E+02, 5.60000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.55706740E+03, 1.43000000E+02, 5.70000000E+01, 1.66820000E+00, 0.00000000E+00},
    {6.59234800E+02, 1.43000000E+02, 5.80000000E+01, 1.66820000E+00, 2.79910000E+00},
    {1.59363310E+03, 1.43000000E+02, 5.90000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.52387140E+03, 1.43000000E+02, 6.00000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.48398170E+03, 1.43000000E+02, 6.10000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.44746650E+03, 1.43000000E+02, 6.20000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.41503580E+03, 1.43000000E+02, 6.30000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.08898390E+03, 1.43000000E+02, 6.40000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.27272350E+03, 1.43000000E+02, 6.50000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.22327390E+03, 1.43000000E+02, 6.60000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.26891740E+03, 1.43000000E+02, 6.70000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.24111510E+03, 1.43000000E+02, 6.80000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.21572980E+03, 1.43000000E+02, 6.90000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.20253140E+03, 1.43000000E+02, 7.00000000E+01, 1.66820000E+00, 0.00000000E+00},
    {9.98861400E+02, 1.43000000E+02, 7.10000000E+01, 1.66820000E+00, 0.00000000E+00},
    {9.61319200E+02, 1.43000000E+02, 7.20000000E+01, 1.66820000E+00, 0.00000000E+00},
    {8.67005400E+02, 1.43000000E+02, 7.30000000E+01, 1.66820000E+00, 0.00000000E+00},
    {7.26706600E+02, 1.43000000E+02, 7.40000000E+01, 1.66820000E+00, 0.00000000E+00},
    {7.35572400E+02, 1.43000000E+02, 7.50000000E+01, 1.66820000E+00, 0.00000000E+00},
    {6.60187000E+02, 1.43000000E+02, 7.60000000E+01, 1.66820000E+00, 0.00000000E+00},
    {6.00027400E+02, 1.43000000E+02, 7.70000000E+01, 1.66820000E+00, 0.00000000E+00},
    {4.95117600E+02, 1.43000000E+02, 7.80000000E+01, 1.66820000E+00, 0.00000000E+00},
    {4.61302600E+02, 1.43000000E+02, 7.90000000E+01, 1.66820000E+00, 0.00000000E+00},
    {4.72423300E+02, 1.43000000E+02, 8.00000000E+01, 1.66820000E+00, 0.00000000E+00},
    {7.17222900E+02, 1.43000000E+02, 8.10000000E+01, 1.66820000E+00, 0.00000000E+00},
    {6.89643500E+02, 1.43000000E+02, 8.20000000E+01, 1.66820000E+00, 0.00000000E+00},
    {6.23353900E+02, 1.43000000E+02, 8.30000000E+01, 1.66820000E+00, 0.00000000E+00},
    {5.89272500E+02, 1.43000000E+02, 8.40000000E+01, 1.66820000E+00, 0.00000000E+00},
    {5.38374700E+02, 1.43000000E+02, 8.50000000E+01, 1.66820000E+00, 0.00000000E+00},
    {4.89186200E+02, 1.43000000E+02, 8.60000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.91840290E+03, 1.43000000E+02, 8.70000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.77156390E+03, 1.43000000E+02, 8.80000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.53467920E+03, 1.43000000E+02, 8.90000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.35024000E+03, 1.43000000E+02, 9.00000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.35605040E+03, 1.43000000E+02, 9.10000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.31221210E+03, 1.43000000E+02, 9.20000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.36866100E+03, 1.43000000E+02, 9.30000000E+01, 1.66820000E+00, 0.00000000E+00},
    {1.32195940E+03, 1.43000000E+02, 9.40000000E+01, 1.66820000E+00, 0.00000000E+00},
    {6.78924000E+01, 1.43000000E+02, 1.01000000E+02, 1.66820000E+00, 0.00000000E+00},
    {2.32953900E+02, 1.43000000E+02, 1.03000000E+02, 1.66820000E+00, 9.86500000E-01},
    {2.95381600E+02, 1.43000000E+02, 1.04000000E+02, 1.66820000E+00, 9.80800000E-01},
    {2.17476200E+02, 1.43000000E+02, 1.05000000E+02, 1.66820000E+00, 9.70600000E-01},
    {1.60922300E+02, 1.43000000E+02, 1.06000000E+02, 1.66820000E+00, 9.86800000E-01},
    {1.09775500E+02, 1.43000000E+02, 1.07000000E+02, 1.66820000E+00, 9.94400000E-01},
    {7.87920000E+01, 1.43000000E+02, 1.08000000E+02, 1.66820000E+00, 9.92500000E-01},
    {5.32880000E+01, 1.43000000E+02, 1.09000000E+02, 1.66820000E+00, 9.98200000E-01},
    {3.43559200E+02, 1.43000000E+02, 1.11000000E+02, 1.66820000E+00, 9.68400000E-01},
    {5.33708500E+02, 1.43000000E+02, 1.12000000E+02, 1.66820000E+00, 9.62800000E-01},
    {5.29304300E+02, 1.43000000E+02, 1.13000000E+02, 1.66820000E+00, 9.64800000E-01},
    {4.13599100E+02, 1.43000000E+02, 1.14000000E+02, 1.66820000E+00, 9.50700000E-01},
    {3.32278300E+02, 1.43000000E+02, 1.15000000E+02, 1.66820000E+00, 9.94700000E-01},
    {2.77600200E+02, 1.43000000E+02, 1.16000000E+02, 1.66820000E+00, 9.94800000E-01},
    {2.24209500E+02, 1.43000000E+02, 1.17000000E+02, 1.66820000E+00, 9.97200000E-01},
    {4.68097700E+02, 1.43000000E+02, 1.19000000E+02, 1.66820000E+00, 9.76700000E-01},
    {9.45994300E+02, 1.43000000E+02, 1.20000000E+02, 1.66820000E+00, 9.83100000E-01},
    {4.55910900E+02, 1.43000000E+02, 1.21000000E+02, 1.66820000E+00, 1.86270000E+00},
    {4.40256100E+02, 1.43000000E+02, 1.22000000E+02, 1.66820000E+00, 1.82990000E+00},
    {4.31639100E+02, 1.43000000E+02, 1.23000000E+02, 1.66820000E+00, 1.91380000E+00},
    {4.28966200E+02, 1.43000000E+02, 1.24000000E+02, 1.66820000E+00, 1.82690000E+00},
    {3.88906600E+02, 1.43000000E+02, 1.25000000E+02, 1.66820000E+00, 1.64060000E+00},
    {3.58656200E+02, 1.43000000E+02, 1.26000000E+02, 1.66820000E+00, 1.64830000E+00},
    {3.42297900E+02, 1.43000000E+02, 1.27000000E+02, 1.66820000E+00, 1.71490000E+00},
    {3.35028700E+02, 1.43000000E+02, 1.28000000E+02, 1.66820000E+00, 1.79370000E+00},
    {3.34561500E+02, 1.43000000E+02, 1.29000000E+02, 1.66820000E+00, 9.57600000E-01},
    {3.07881500E+02, 1.43000000E+02, 1.30000000E+02, 1.66820000E+00, 1.94190000E+00},
    {5.21144100E+02, 1.43000000E+02, 1.31000000E+02, 1.66820000E+00, 9.60100000E-01},
    {4.47713100E+02, 1.43000000E+02, 1.32000000E+02, 1.66820000E+00, 9.43400000E-01},
    {3.95247200E+02, 1.43000000E+02, 1.33000000E+02, 1.66820000E+00, 9.88900000E-01},
    {3.57473700E+02, 1.43000000E+02, 1.34000000E+02, 1.66820000E+00, 9.90100000E-01},
    {3.11690100E+02, 1.43000000E+02, 1.35000000E+02, 1.66820000E+00, 9.97400000E-01},
    {5.56125500E+02, 1.43000000E+02, 1.37000000E+02, 1.66820000E+00, 9.73800000E-01},
    {1.15626040E+03, 1.43000000E+02, 1.38000000E+02, 1.66820000E+00, 9.80100000E-01},
    {8.48372700E+02, 1.43000000E+02, 1.39000000E+02, 1.66820000E+00, 1.91530000E+00},
    {6.05338900E+02, 1.43000000E+02, 1.40000000E+02, 1.66820000E+00, 1.93550000E+00},
    {6.11276400E+02, 1.43000000E+02, 1.41000000E+02, 1.66820000E+00, 1.95450000E+00},
    {5.67767000E+02, 1.43000000E+02, 1.42000000E+02, 1.66820000E+00, 1.94200000E+00},
    {6.49714300E+02, 1.43000000E+02, 1.43000000E+02, 1.66820000E+00, 1.66820000E+00},
    {3.31875000E+01, 1.44000000E+02, 1.00000000E+00, 1.85840000E+00, 9.11800000E-01},
    {2.20616000E+01, 1.44000000E+02, 2.00000000E+00, 1.85840000E+00, 0.00000000E+00},
    {5.17333000E+02, 1.44000000E+02, 3.00000000E+00, 1.85840000E+00, 0.00000000E+00},
    {2.97518600E+02, 1.44000000E+02, 4.00000000E+00, 1.85840000E+00, 0.00000000E+00},
    {2.00329900E+02, 1.44000000E+02, 5.00000000E+00, 1.85840000E+00, 0.00000000E+00},
    {1.35551700E+02, 1.44000000E+02, 6.00000000E+00, 1.85840000E+00, 0.00000000E+00},
    {9.50190000E+01, 1.44000000E+02, 7.00000000E+00, 1.85840000E+00, 0.00000000E+00},
    {7.21173000E+01, 1.44000000E+02, 8.00000000E+00, 1.85840000E+00, 0.00000000E+00},
    {5.47781000E+01, 1.44000000E+02, 9.00000000E+00, 1.85840000E+00, 0.00000000E+00},
    {4.22432000E+01, 1.44000000E+02, 1.00000000E+01, 1.85840000E+00, 0.00000000E+00},
    {6.18666900E+02, 1.44000000E+02, 1.10000000E+01, 1.85840000E+00, 0.00000000E+00},
    {4.74418000E+02, 1.44000000E+02, 1.20000000E+01, 1.85840000E+00, 0.00000000E+00},
    {4.37020900E+02, 1.44000000E+02, 1.30000000E+01, 1.85840000E+00, 0.00000000E+00},
    {3.44211300E+02, 1.44000000E+02, 1.40000000E+01, 1.85840000E+00, 0.00000000E+00},
    {2.68541600E+02, 1.44000000E+02, 1.50000000E+01, 1.85840000E+00, 0.00000000E+00},
    {2.23088300E+02, 1.44000000E+02, 1.60000000E+01, 1.85840000E+00, 0.00000000E+00},
    {1.82472100E+02, 1.44000000E+02, 1.70000000E+01, 1.85840000E+00, 0.00000000E+00},
    {1.49522200E+02, 1.44000000E+02, 1.80000000E+01, 1.85840000E+00, 0.00000000E+00},
    {1.01579970E+03, 1.44000000E+02, 1.90000000E+01, 1.85840000E+00, 0.00000000E+00},
    {8.35269200E+02, 1.44000000E+02, 2.00000000E+01, 1.85840000E+00, 0.00000000E+00},
    {6.89608800E+02, 1.44000000E+02, 2.10000000E+01, 1.85840000E+00, 0.00000000E+00},
    {6.65814300E+02, 1.44000000E+02, 2.20000000E+01, 1.85840000E+00, 0.00000000E+00},
    {6.09615900E+02, 1.44000000E+02, 2.30000000E+01, 1.85840000E+00, 0.00000000E+00},
    {4.80638200E+02, 1.44000000E+02, 2.40000000E+01, 1.85840000E+00, 0.00000000E+00},
    {5.24774800E+02, 1.44000000E+02, 2.50000000E+01, 1.85840000E+00, 0.00000000E+00},
    {4.12246800E+02, 1.44000000E+02, 2.60000000E+01, 1.85840000E+00, 0.00000000E+00},
    {4.36405400E+02, 1.44000000E+02, 2.70000000E+01, 1.85840000E+00, 0.00000000E+00},
    {4.49580600E+02, 1.44000000E+02, 2.80000000E+01, 1.85840000E+00, 0.00000000E+00},
    {3.45104300E+02, 1.44000000E+02, 2.90000000E+01, 1.85840000E+00, 0.00000000E+00},
    {3.53788000E+02, 1.44000000E+02, 3.00000000E+01, 1.85840000E+00, 0.00000000E+00},
    {4.18857300E+02, 1.44000000E+02, 3.10000000E+01, 1.85840000E+00, 0.00000000E+00},
    {3.69173400E+02, 1.44000000E+02, 3.20000000E+01, 1.85840000E+00, 0.00000000E+00},
    {3.14925300E+02, 1.44000000E+02, 3.30000000E+01, 1.85840000E+00, 0.00000000E+00},
    {2.82730200E+02, 1.44000000E+02, 3.40000000E+01, 1.85840000E+00, 0.00000000E+00},
    {2.47620300E+02, 1.44000000E+02, 3.50000000E+01, 1.85840000E+00, 0.00000000E+00},
    {2.15568300E+02, 1.44000000E+02, 3.60000000E+01, 1.85840000E+00, 0.00000000E+00},
    {1.13852650E+03, 1.44000000E+02, 3.70000000E+01, 1.85840000E+00, 0.00000000E+00},
    {9.95611600E+02, 1.44000000E+02, 3.80000000E+01, 1.85840000E+00, 0.00000000E+00},
    {8.71260000E+02, 1.44000000E+02, 3.90000000E+01, 1.85840000E+00, 0.00000000E+00},
    {7.82716600E+02, 1.44000000E+02, 4.00000000E+01, 1.85840000E+00, 0.00000000E+00},
    {7.13647700E+02, 1.44000000E+02, 4.10000000E+01, 1.85840000E+00, 0.00000000E+00},
    {5.51124000E+02, 1.44000000E+02, 4.20000000E+01, 1.85840000E+00, 0.00000000E+00},
    {6.14818500E+02, 1.44000000E+02, 4.30000000E+01, 1.85840000E+00, 0.00000000E+00},
    {4.68569500E+02, 1.44000000E+02, 4.40000000E+01, 1.85840000E+00, 0.00000000E+00},
    {5.11831500E+02, 1.44000000E+02, 4.50000000E+01, 1.85840000E+00, 0.00000000E+00},
    {4.74636900E+02, 1.44000000E+02, 4.60000000E+01, 1.85840000E+00, 0.00000000E+00},
    {3.96274200E+02, 1.44000000E+02, 4.70000000E+01, 1.85840000E+00, 0.00000000E+00},
    {4.18257900E+02, 1.44000000E+02, 4.80000000E+01, 1.85840000E+00, 0.00000000E+00},
    {5.24823600E+02, 1.44000000E+02, 4.90000000E+01, 1.85840000E+00, 0.00000000E+00},
    {4.84885500E+02, 1.44000000E+02, 5.00000000E+01, 1.85840000E+00, 0.00000000E+00},
    {4.31986500E+02, 1.44000000E+02, 5.10000000E+01, 1.85840000E+00, 0.00000000E+00},
    {4.00831900E+02, 1.44000000E+02, 5.20000000E+01, 1.85840000E+00, 0.00000000E+00},
    {3.62537400E+02, 1.44000000E+02, 5.30000000E+01, 1.85840000E+00, 0.00000000E+00},
    {3.26115700E+02, 1.44000000E+02, 5.40000000E+01, 1.85840000E+00, 0.00000000E+00},
    {1.38712420E+03, 1.44000000E+02, 5.50000000E+01, 1.85840000E+00, 0.00000000E+00},
    {1.27010020E+03, 1.44000000E+02, 5.60000000E+01, 1.85840000E+00, 0.00000000E+00},
    {1.11459620E+03, 1.44000000E+02, 5.70000000E+01, 1.85840000E+00, 0.00000000E+00},
    {5.10586400E+02, 1.44000000E+02, 5.80000000E+01, 1.85840000E+00, 2.79910000E+00},
    {1.12511270E+03, 1.44000000E+02, 5.90000000E+01, 1.85840000E+00, 0.00000000E+00},
    {1.08016280E+03, 1.44000000E+02, 6.00000000E+01, 1.85840000E+00, 0.00000000E+00},
    {1.05299740E+03, 1.44000000E+02, 6.10000000E+01, 1.85840000E+00, 0.00000000E+00},
    {1.02801580E+03, 1.44000000E+02, 6.20000000E+01, 1.85840000E+00, 0.00000000E+00},
    {1.00585860E+03, 1.44000000E+02, 6.30000000E+01, 1.85840000E+00, 0.00000000E+00},
    {7.90517600E+02, 1.44000000E+02, 6.40000000E+01, 1.85840000E+00, 0.00000000E+00},
    {8.91876400E+02, 1.44000000E+02, 6.50000000E+01, 1.85840000E+00, 0.00000000E+00},
    {8.59972900E+02, 1.44000000E+02, 6.60000000E+01, 1.85840000E+00, 0.00000000E+00},
    {9.06921400E+02, 1.44000000E+02, 6.70000000E+01, 1.85840000E+00, 0.00000000E+00},
    {8.87620200E+02, 1.44000000E+02, 6.80000000E+01, 1.85840000E+00, 0.00000000E+00},
    {8.70206300E+02, 1.44000000E+02, 6.90000000E+01, 1.85840000E+00, 0.00000000E+00},
    {8.60034500E+02, 1.44000000E+02, 7.00000000E+01, 1.85840000E+00, 0.00000000E+00},
    {7.24290400E+02, 1.44000000E+02, 7.10000000E+01, 1.85840000E+00, 0.00000000E+00},
    {7.11477900E+02, 1.44000000E+02, 7.20000000E+01, 1.85840000E+00, 0.00000000E+00},
    {6.49147100E+02, 1.44000000E+02, 7.30000000E+01, 1.85840000E+00, 0.00000000E+00},
    {5.48421200E+02, 1.44000000E+02, 7.40000000E+01, 1.85840000E+00, 0.00000000E+00},
    {5.57756100E+02, 1.44000000E+02, 7.50000000E+01, 1.85840000E+00, 0.00000000E+00},
    {5.05493300E+02, 1.44000000E+02, 7.60000000E+01, 1.85840000E+00, 0.00000000E+00},
    {4.62987000E+02, 1.44000000E+02, 7.70000000E+01, 1.85840000E+00, 0.00000000E+00},
    {3.84914600E+02, 1.44000000E+02, 7.80000000E+01, 1.85840000E+00, 0.00000000E+00},
    {3.59742700E+02, 1.44000000E+02, 7.90000000E+01, 1.85840000E+00, 0.00000000E+00},
    {3.69895200E+02, 1.44000000E+02, 8.00000000E+01, 1.85840000E+00, 0.00000000E+00},
    {5.39298200E+02, 1.44000000E+02, 8.10000000E+01, 1.85840000E+00, 0.00000000E+00},
    {5.26773800E+02, 1.44000000E+02, 8.20000000E+01, 1.85840000E+00, 0.00000000E+00},
    {4.83814300E+02, 1.44000000E+02, 8.30000000E+01, 1.85840000E+00, 0.00000000E+00},
    {4.61393600E+02, 1.44000000E+02, 8.40000000E+01, 1.85840000E+00, 0.00000000E+00},
    {4.25872100E+02, 1.44000000E+02, 8.50000000E+01, 1.85840000E+00, 0.00000000E+00},
    {3.90471100E+02, 1.44000000E+02, 8.60000000E+01, 1.85840000E+00, 0.00000000E+00},
    {1.30869040E+03, 1.44000000E+02, 8.70000000E+01, 1.85840000E+00, 0.00000000E+00},
    {1.25535930E+03, 1.44000000E+02, 8.80000000E+01, 1.85840000E+00, 0.00000000E+00},
    {1.10849340E+03, 1.44000000E+02, 8.90000000E+01, 1.85840000E+00, 0.00000000E+00},
    {9.95482300E+02, 1.44000000E+02, 9.00000000E+01, 1.85840000E+00, 0.00000000E+00},
    {9.89136400E+02, 1.44000000E+02, 9.10000000E+01, 1.85840000E+00, 0.00000000E+00},
    {9.57778100E+02, 1.44000000E+02, 9.20000000E+01, 1.85840000E+00, 0.00000000E+00},
    {9.86735200E+02, 1.44000000E+02, 9.30000000E+01, 1.85840000E+00, 0.00000000E+00},
    {9.55424900E+02, 1.44000000E+02, 9.40000000E+01, 1.85840000E+00, 0.00000000E+00},
    {5.33648000E+01, 1.44000000E+02, 1.01000000E+02, 1.85840000E+00, 0.00000000E+00},
    {1.72970800E+02, 1.44000000E+02, 1.03000000E+02, 1.85840000E+00, 9.86500000E-01},
    {2.20674600E+02, 1.44000000E+02, 1.04000000E+02, 1.85840000E+00, 9.80800000E-01},
    {1.68594400E+02, 1.44000000E+02, 1.05000000E+02, 1.85840000E+00, 9.70600000E-01},
    {1.27212500E+02, 1.44000000E+02, 1.06000000E+02, 1.85840000E+00, 9.86800000E-01},
    {8.86613000E+01, 1.44000000E+02, 1.07000000E+02, 1.85840000E+00, 9.94400000E-01},
    {6.47448000E+01, 1.44000000E+02, 1.08000000E+02, 1.85840000E+00, 9.92500000E-01},
    {4.47087000E+01, 1.44000000E+02, 1.09000000E+02, 1.85840000E+00, 9.98200000E-01},
    {2.53090000E+02, 1.44000000E+02, 1.11000000E+02, 1.85840000E+00, 9.68400000E-01},
    {3.91360600E+02, 1.44000000E+02, 1.12000000E+02, 1.85840000E+00, 9.62800000E-01},
    {3.96063400E+02, 1.44000000E+02, 1.13000000E+02, 1.85840000E+00, 9.64800000E-01},
    {3.18179500E+02, 1.44000000E+02, 1.14000000E+02, 1.85840000E+00, 9.50700000E-01},
    {2.60701200E+02, 1.44000000E+02, 1.15000000E+02, 1.85840000E+00, 9.94700000E-01},
    {2.20647700E+02, 1.44000000E+02, 1.16000000E+02, 1.85840000E+00, 9.94800000E-01},
    {1.80591600E+02, 1.44000000E+02, 1.17000000E+02, 1.85840000E+00, 9.97200000E-01},
    {3.49475300E+02, 1.44000000E+02, 1.19000000E+02, 1.85840000E+00, 9.76700000E-01},
    {6.68764600E+02, 1.44000000E+02, 1.20000000E+02, 1.85840000E+00, 9.83100000E-01},
    {3.49070200E+02, 1.44000000E+02, 1.21000000E+02, 1.85840000E+00, 1.86270000E+00},
    {3.37055300E+02, 1.44000000E+02, 1.22000000E+02, 1.85840000E+00, 1.82990000E+00},
    {3.30356700E+02, 1.44000000E+02, 1.23000000E+02, 1.85840000E+00, 1.91380000E+00},
    {3.27355300E+02, 1.44000000E+02, 1.24000000E+02, 1.85840000E+00, 1.82690000E+00},
    {3.01039700E+02, 1.44000000E+02, 1.25000000E+02, 1.85840000E+00, 1.64060000E+00},
    {2.78652600E+02, 1.44000000E+02, 1.26000000E+02, 1.85840000E+00, 1.64830000E+00},
    {2.65875200E+02, 1.44000000E+02, 1.27000000E+02, 1.85840000E+00, 1.71490000E+00},
    {2.59947200E+02, 1.44000000E+02, 1.28000000E+02, 1.85840000E+00, 1.79370000E+00},
    {2.56948000E+02, 1.44000000E+02, 1.29000000E+02, 1.85840000E+00, 9.57600000E-01},
    {2.40912400E+02, 1.44000000E+02, 1.30000000E+02, 1.85840000E+00, 1.94190000E+00},
    {3.93357100E+02, 1.44000000E+02, 1.31000000E+02, 1.85840000E+00, 9.60100000E-01},
    {3.45387200E+02, 1.44000000E+02, 1.32000000E+02, 1.85840000E+00, 9.43400000E-01},
    {3.09603500E+02, 1.44000000E+02, 1.33000000E+02, 1.85840000E+00, 9.88900000E-01},
    {2.82846000E+02, 1.44000000E+02, 1.34000000E+02, 1.85840000E+00, 9.90100000E-01},
    {2.49340400E+02, 1.44000000E+02, 1.35000000E+02, 1.85840000E+00, 9.97400000E-01},
    {4.17083100E+02, 1.44000000E+02, 1.37000000E+02, 1.85840000E+00, 9.73800000E-01},
    {8.14197500E+02, 1.44000000E+02, 1.38000000E+02, 1.85840000E+00, 9.80100000E-01},
    {6.21305800E+02, 1.44000000E+02, 1.39000000E+02, 1.85840000E+00, 1.91530000E+00},
    {4.61914100E+02, 1.44000000E+02, 1.40000000E+02, 1.85840000E+00, 1.93550000E+00},
    {4.66535800E+02, 1.44000000E+02, 1.41000000E+02, 1.85840000E+00, 1.95450000E+00},
    {4.35106200E+02, 1.44000000E+02, 1.42000000E+02, 1.85840000E+00, 1.94200000E+00},
    {4.88326400E+02, 1.44000000E+02, 1.43000000E+02, 1.85840000E+00, 1.66820000E+00},
    {3.79265500E+02, 1.44000000E+02, 1.44000000E+02, 1.85840000E+00, 1.85840000E+00},
    {3.11024000E+01, 1.45000000E+02, 1.00000000E+00, 1.90030000E+00, 9.11800000E-01},
    {2.07776000E+01, 1.45000000E+02, 2.00000000E+00, 1.90030000E+00, 0.00000000E+00},
    {4.82434200E+02, 1.45000000E+02, 3.00000000E+00, 1.90030000E+00, 0.00000000E+00},
    {2.77417400E+02, 1.45000000E+02, 4.00000000E+00, 1.90030000E+00, 0.00000000E+00},
    {1.87132700E+02, 1.45000000E+02, 5.00000000E+00, 1.90030000E+00, 0.00000000E+00},
    {1.26908500E+02, 1.45000000E+02, 6.00000000E+00, 1.90030000E+00, 0.00000000E+00},
    {8.91659000E+01, 1.45000000E+02, 7.00000000E+00, 1.90030000E+00, 0.00000000E+00},
    {6.78114000E+01, 1.45000000E+02, 8.00000000E+00, 1.90030000E+00, 0.00000000E+00},
    {5.16117000E+01, 1.45000000E+02, 9.00000000E+00, 1.90030000E+00, 0.00000000E+00},
    {3.98762000E+01, 1.45000000E+02, 1.00000000E+01, 1.90030000E+00, 0.00000000E+00},
    {5.77018100E+02, 1.45000000E+02, 1.10000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.42338100E+02, 1.45000000E+02, 1.20000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.07699900E+02, 1.45000000E+02, 1.30000000E+01, 1.90030000E+00, 0.00000000E+00},
    {3.21459400E+02, 1.45000000E+02, 1.40000000E+01, 1.90030000E+00, 0.00000000E+00},
    {2.51147700E+02, 1.45000000E+02, 1.50000000E+01, 1.90030000E+00, 0.00000000E+00},
    {2.08906700E+02, 1.45000000E+02, 1.60000000E+01, 1.90030000E+00, 0.00000000E+00},
    {1.71118700E+02, 1.45000000E+02, 1.70000000E+01, 1.90030000E+00, 0.00000000E+00},
    {1.40427300E+02, 1.45000000E+02, 1.80000000E+01, 1.90030000E+00, 0.00000000E+00},
    {9.48514000E+02, 1.45000000E+02, 1.90000000E+01, 1.90030000E+00, 0.00000000E+00},
    {7.79054700E+02, 1.45000000E+02, 2.00000000E+01, 1.90030000E+00, 0.00000000E+00},
    {6.43189300E+02, 1.45000000E+02, 2.10000000E+01, 1.90030000E+00, 0.00000000E+00},
    {6.21180700E+02, 1.45000000E+02, 2.20000000E+01, 1.90030000E+00, 0.00000000E+00},
    {5.68834800E+02, 1.45000000E+02, 2.30000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.48752700E+02, 1.45000000E+02, 2.40000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.89793300E+02, 1.45000000E+02, 2.50000000E+01, 1.90030000E+00, 0.00000000E+00},
    {3.85015300E+02, 1.45000000E+02, 2.60000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.07457400E+02, 1.45000000E+02, 2.70000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.19670900E+02, 1.45000000E+02, 2.80000000E+01, 1.90030000E+00, 0.00000000E+00},
    {3.22398800E+02, 1.45000000E+02, 2.90000000E+01, 1.90030000E+00, 0.00000000E+00},
    {3.30471300E+02, 1.45000000E+02, 3.00000000E+01, 1.90030000E+00, 0.00000000E+00},
    {3.91050800E+02, 1.45000000E+02, 3.10000000E+01, 1.90030000E+00, 0.00000000E+00},
    {3.44910000E+02, 1.45000000E+02, 3.20000000E+01, 1.90030000E+00, 0.00000000E+00},
    {2.94542100E+02, 1.45000000E+02, 3.30000000E+01, 1.90030000E+00, 0.00000000E+00},
    {2.64671700E+02, 1.45000000E+02, 3.40000000E+01, 1.90030000E+00, 0.00000000E+00},
    {2.32058000E+02, 1.45000000E+02, 3.50000000E+01, 1.90030000E+00, 0.00000000E+00},
    {2.02256700E+02, 1.45000000E+02, 3.60000000E+01, 1.90030000E+00, 0.00000000E+00},
    {1.06334770E+03, 1.45000000E+02, 3.70000000E+01, 1.90030000E+00, 0.00000000E+00},
    {9.28825500E+02, 1.45000000E+02, 3.80000000E+01, 1.90030000E+00, 0.00000000E+00},
    {8.12933800E+02, 1.45000000E+02, 3.90000000E+01, 1.90030000E+00, 0.00000000E+00},
    {7.30497000E+02, 1.45000000E+02, 4.00000000E+01, 1.90030000E+00, 0.00000000E+00},
    {6.66225900E+02, 1.45000000E+02, 4.10000000E+01, 1.90030000E+00, 0.00000000E+00},
    {5.14912800E+02, 1.45000000E+02, 4.20000000E+01, 1.90030000E+00, 0.00000000E+00},
    {5.74245600E+02, 1.45000000E+02, 4.30000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.38042600E+02, 1.45000000E+02, 4.40000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.78284800E+02, 1.45000000E+02, 4.50000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.43628000E+02, 1.45000000E+02, 4.60000000E+01, 1.90030000E+00, 0.00000000E+00},
    {3.70637800E+02, 1.45000000E+02, 4.70000000E+01, 1.90030000E+00, 0.00000000E+00},
    {3.91067500E+02, 1.45000000E+02, 4.80000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.90332500E+02, 1.45000000E+02, 4.90000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.53179400E+02, 1.45000000E+02, 5.00000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.04038400E+02, 1.45000000E+02, 5.10000000E+01, 1.90030000E+00, 0.00000000E+00},
    {3.75130300E+02, 1.45000000E+02, 5.20000000E+01, 1.90030000E+00, 0.00000000E+00},
    {3.39562000E+02, 1.45000000E+02, 5.30000000E+01, 1.90030000E+00, 0.00000000E+00},
    {3.05720100E+02, 1.45000000E+02, 5.40000000E+01, 1.90030000E+00, 0.00000000E+00},
    {1.29585070E+03, 1.45000000E+02, 5.50000000E+01, 1.90030000E+00, 0.00000000E+00},
    {1.18517470E+03, 1.45000000E+02, 5.60000000E+01, 1.90030000E+00, 0.00000000E+00},
    {1.04010660E+03, 1.45000000E+02, 5.70000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.77594500E+02, 1.45000000E+02, 5.80000000E+01, 1.90030000E+00, 2.79910000E+00},
    {1.05007630E+03, 1.45000000E+02, 5.90000000E+01, 1.90030000E+00, 0.00000000E+00},
    {1.00805210E+03, 1.45000000E+02, 6.00000000E+01, 1.90030000E+00, 0.00000000E+00},
    {9.82687500E+02, 1.45000000E+02, 6.10000000E+01, 1.90030000E+00, 0.00000000E+00},
    {9.59359800E+02, 1.45000000E+02, 6.20000000E+01, 1.90030000E+00, 0.00000000E+00},
    {9.38668800E+02, 1.45000000E+02, 6.30000000E+01, 1.90030000E+00, 0.00000000E+00},
    {7.38131700E+02, 1.45000000E+02, 6.40000000E+01, 1.90030000E+00, 0.00000000E+00},
    {8.32967000E+02, 1.45000000E+02, 6.50000000E+01, 1.90030000E+00, 0.00000000E+00},
    {8.03251100E+02, 1.45000000E+02, 6.60000000E+01, 1.90030000E+00, 0.00000000E+00},
    {8.46310300E+02, 1.45000000E+02, 6.70000000E+01, 1.90030000E+00, 0.00000000E+00},
    {8.28282600E+02, 1.45000000E+02, 6.80000000E+01, 1.90030000E+00, 0.00000000E+00},
    {8.12021400E+02, 1.45000000E+02, 6.90000000E+01, 1.90030000E+00, 0.00000000E+00},
    {8.02502600E+02, 1.45000000E+02, 7.00000000E+01, 1.90030000E+00, 0.00000000E+00},
    {6.76132000E+02, 1.45000000E+02, 7.10000000E+01, 1.90030000E+00, 0.00000000E+00},
    {6.64195300E+02, 1.45000000E+02, 7.20000000E+01, 1.90030000E+00, 0.00000000E+00},
    {6.06246300E+02, 1.45000000E+02, 7.30000000E+01, 1.90030000E+00, 0.00000000E+00},
    {5.12565600E+02, 1.45000000E+02, 7.40000000E+01, 1.90030000E+00, 0.00000000E+00},
    {5.21306100E+02, 1.45000000E+02, 7.50000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.72680900E+02, 1.45000000E+02, 7.60000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.33129200E+02, 1.45000000E+02, 7.70000000E+01, 1.90030000E+00, 0.00000000E+00},
    {3.60417300E+02, 1.45000000E+02, 7.80000000E+01, 1.90030000E+00, 0.00000000E+00},
    {3.36971400E+02, 1.45000000E+02, 7.90000000E+01, 1.90030000E+00, 0.00000000E+00},
    {3.46452600E+02, 1.45000000E+02, 8.00000000E+01, 1.90030000E+00, 0.00000000E+00},
    {5.04240400E+02, 1.45000000E+02, 8.10000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.92568900E+02, 1.45000000E+02, 8.20000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.52645100E+02, 1.45000000E+02, 8.30000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.31858800E+02, 1.45000000E+02, 8.40000000E+01, 1.90030000E+00, 0.00000000E+00},
    {3.98876800E+02, 1.45000000E+02, 8.50000000E+01, 1.90030000E+00, 0.00000000E+00},
    {3.65991400E+02, 1.45000000E+02, 8.60000000E+01, 1.90030000E+00, 0.00000000E+00},
    {1.22244850E+03, 1.45000000E+02, 8.70000000E+01, 1.90030000E+00, 0.00000000E+00},
    {1.17153780E+03, 1.45000000E+02, 8.80000000E+01, 1.90030000E+00, 0.00000000E+00},
    {1.03456330E+03, 1.45000000E+02, 8.90000000E+01, 1.90030000E+00, 0.00000000E+00},
    {9.29531900E+02, 1.45000000E+02, 9.00000000E+01, 1.90030000E+00, 0.00000000E+00},
    {9.23667300E+02, 1.45000000E+02, 9.10000000E+01, 1.90030000E+00, 0.00000000E+00},
    {8.94413500E+02, 1.45000000E+02, 9.20000000E+01, 1.90030000E+00, 0.00000000E+00},
    {9.21285300E+02, 1.45000000E+02, 9.30000000E+01, 1.90030000E+00, 0.00000000E+00},
    {8.92064100E+02, 1.45000000E+02, 9.40000000E+01, 1.90030000E+00, 0.00000000E+00},
    {4.99090000E+01, 1.45000000E+02, 1.01000000E+02, 1.90030000E+00, 0.00000000E+00},
    {1.61349900E+02, 1.45000000E+02, 1.03000000E+02, 1.90030000E+00, 9.86500000E-01},
    {2.05953000E+02, 1.45000000E+02, 1.04000000E+02, 1.90030000E+00, 9.80800000E-01},
    {1.57594900E+02, 1.45000000E+02, 1.05000000E+02, 1.90030000E+00, 9.70600000E-01},
    {1.19126500E+02, 1.45000000E+02, 1.06000000E+02, 1.90030000E+00, 9.86800000E-01},
    {8.32227000E+01, 1.45000000E+02, 1.07000000E+02, 1.90030000E+00, 9.94400000E-01},
    {6.09159000E+01, 1.45000000E+02, 1.08000000E+02, 1.90030000E+00, 9.92500000E-01},
    {4.22084000E+01, 1.45000000E+02, 1.09000000E+02, 1.90030000E+00, 9.98200000E-01},
    {2.36144400E+02, 1.45000000E+02, 1.11000000E+02, 1.90030000E+00, 9.68400000E-01},
    {3.65070500E+02, 1.45000000E+02, 1.12000000E+02, 1.90030000E+00, 9.62800000E-01},
    {3.69581400E+02, 1.45000000E+02, 1.13000000E+02, 1.90030000E+00, 9.64800000E-01},
    {2.97233900E+02, 1.45000000E+02, 1.14000000E+02, 1.90030000E+00, 9.50700000E-01},
    {2.43843900E+02, 1.45000000E+02, 1.15000000E+02, 1.90030000E+00, 9.94700000E-01},
    {2.06625800E+02, 1.45000000E+02, 1.16000000E+02, 1.90030000E+00, 9.94800000E-01},
    {1.69358400E+02, 1.45000000E+02, 1.17000000E+02, 1.90030000E+00, 9.97200000E-01},
    {3.26594700E+02, 1.45000000E+02, 1.19000000E+02, 1.90030000E+00, 9.76700000E-01},
    {6.24285000E+02, 1.45000000E+02, 1.20000000E+02, 1.90030000E+00, 9.83100000E-01},
    {3.26250300E+02, 1.45000000E+02, 1.21000000E+02, 1.90030000E+00, 1.86270000E+00},
    {3.15098200E+02, 1.45000000E+02, 1.22000000E+02, 1.90030000E+00, 1.82990000E+00},
    {3.08848200E+02, 1.45000000E+02, 1.23000000E+02, 1.90030000E+00, 1.91380000E+00},
    {3.06034000E+02, 1.45000000E+02, 1.24000000E+02, 1.90030000E+00, 1.82690000E+00},
    {2.81513400E+02, 1.45000000E+02, 1.25000000E+02, 1.90030000E+00, 1.64060000E+00},
    {2.60656700E+02, 1.45000000E+02, 1.26000000E+02, 1.90030000E+00, 1.64830000E+00},
    {2.48738800E+02, 1.45000000E+02, 1.27000000E+02, 1.90030000E+00, 1.71490000E+00},
    {2.43190900E+02, 1.45000000E+02, 1.28000000E+02, 1.90030000E+00, 1.79370000E+00},
    {2.40326000E+02, 1.45000000E+02, 1.29000000E+02, 1.90030000E+00, 9.57600000E-01},
    {2.25439800E+02, 1.45000000E+02, 1.30000000E+02, 1.90030000E+00, 1.94190000E+00},
    {3.67319700E+02, 1.45000000E+02, 1.31000000E+02, 1.90030000E+00, 9.60100000E-01},
    {3.22781600E+02, 1.45000000E+02, 1.32000000E+02, 1.90030000E+00, 9.43400000E-01},
    {2.89593400E+02, 1.45000000E+02, 1.33000000E+02, 1.90030000E+00, 9.88900000E-01},
    {2.64781500E+02, 1.45000000E+02, 1.34000000E+02, 1.90030000E+00, 9.90100000E-01},
    {2.33660800E+02, 1.45000000E+02, 1.35000000E+02, 1.90030000E+00, 9.97400000E-01},
    {3.89918300E+02, 1.45000000E+02, 1.37000000E+02, 1.90030000E+00, 9.73800000E-01},
    {7.60243700E+02, 1.45000000E+02, 1.38000000E+02, 1.90030000E+00, 9.80100000E-01},
    {5.80414500E+02, 1.45000000E+02, 1.39000000E+02, 1.90030000E+00, 1.91530000E+00},
    {4.31828800E+02, 1.45000000E+02, 1.40000000E+02, 1.90030000E+00, 1.93550000E+00},
    {4.36144000E+02, 1.45000000E+02, 1.41000000E+02, 1.90030000E+00, 1.95450000E+00},
    {4.06933800E+02, 1.45000000E+02, 1.42000000E+02, 1.90030000E+00, 1.94200000E+00},
    {4.56561800E+02, 1.45000000E+02, 1.43000000E+02, 1.90030000E+00, 1.66820000E+00},
    {3.54928200E+02, 1.45000000E+02, 1.44000000E+02, 1.90030000E+00, 1.85840000E+00},
    {3.32242500E+02, 1.45000000E+02, 1.45000000E+02, 1.90030000E+00, 1.90030000E+00},
    {2.89661000E+01, 1.46000000E+02, 1.00000000E+00, 1.86300000E+00, 9.11800000E-01},
    {1.94689000E+01, 1.46000000E+02, 2.00000000E+00, 1.86300000E+00, 0.00000000E+00},
    {4.43910600E+02, 1.46000000E+02, 3.00000000E+00, 1.86300000E+00, 0.00000000E+00},
    {2.56232700E+02, 1.46000000E+02, 4.00000000E+00, 1.86300000E+00, 0.00000000E+00},
    {1.73436000E+02, 1.46000000E+02, 5.00000000E+00, 1.86300000E+00, 0.00000000E+00},
    {1.18013700E+02, 1.46000000E+02, 6.00000000E+00, 1.86300000E+00, 0.00000000E+00},
    {8.31710000E+01, 1.46000000E+02, 7.00000000E+00, 1.86300000E+00, 0.00000000E+00},
    {6.34125000E+01, 1.46000000E+02, 8.00000000E+00, 1.86300000E+00, 0.00000000E+00},
    {4.83824000E+01, 1.46000000E+02, 9.00000000E+00, 1.86300000E+00, 0.00000000E+00},
    {3.74647000E+01, 1.46000000E+02, 1.00000000E+01, 1.86300000E+00, 0.00000000E+00},
    {5.31167600E+02, 1.46000000E+02, 1.10000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.08302200E+02, 1.46000000E+02, 1.20000000E+01, 1.86300000E+00, 0.00000000E+00},
    {3.76877600E+02, 1.46000000E+02, 1.30000000E+01, 1.86300000E+00, 0.00000000E+00},
    {2.97817400E+02, 1.46000000E+02, 1.40000000E+01, 1.86300000E+00, 0.00000000E+00},
    {2.33210600E+02, 1.46000000E+02, 1.50000000E+01, 1.86300000E+00, 0.00000000E+00},
    {1.94337300E+02, 1.46000000E+02, 1.60000000E+01, 1.86300000E+00, 0.00000000E+00},
    {1.59491400E+02, 1.46000000E+02, 1.70000000E+01, 1.86300000E+00, 0.00000000E+00},
    {1.31133800E+02, 1.46000000E+02, 1.80000000E+01, 1.86300000E+00, 0.00000000E+00},
    {8.73298400E+02, 1.46000000E+02, 1.90000000E+01, 1.86300000E+00, 0.00000000E+00},
    {7.18498300E+02, 1.46000000E+02, 2.00000000E+01, 1.86300000E+00, 0.00000000E+00},
    {5.93499400E+02, 1.46000000E+02, 2.10000000E+01, 1.86300000E+00, 0.00000000E+00},
    {5.73597200E+02, 1.46000000E+02, 2.20000000E+01, 1.86300000E+00, 0.00000000E+00},
    {5.25473200E+02, 1.46000000E+02, 2.30000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.14791800E+02, 1.46000000E+02, 2.40000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.52735300E+02, 1.46000000E+02, 2.50000000E+01, 1.86300000E+00, 0.00000000E+00},
    {3.56134400E+02, 1.46000000E+02, 2.60000000E+01, 1.86300000E+00, 0.00000000E+00},
    {3.76992800E+02, 1.46000000E+02, 2.70000000E+01, 1.86300000E+00, 0.00000000E+00},
    {3.88120000E+02, 1.46000000E+02, 2.80000000E+01, 1.86300000E+00, 0.00000000E+00},
    {2.98389400E+02, 1.46000000E+02, 2.90000000E+01, 1.86300000E+00, 0.00000000E+00},
    {3.06063300E+02, 1.46000000E+02, 3.00000000E+01, 1.86300000E+00, 0.00000000E+00},
    {3.61878000E+02, 1.46000000E+02, 3.10000000E+01, 1.86300000E+00, 0.00000000E+00},
    {3.19694600E+02, 1.46000000E+02, 3.20000000E+01, 1.86300000E+00, 0.00000000E+00},
    {2.73501000E+02, 1.46000000E+02, 3.30000000E+01, 1.86300000E+00, 0.00000000E+00},
    {2.46092100E+02, 1.46000000E+02, 3.40000000E+01, 1.86300000E+00, 0.00000000E+00},
    {2.16094500E+02, 1.46000000E+02, 3.50000000E+01, 1.86300000E+00, 0.00000000E+00},
    {1.88633900E+02, 1.46000000E+02, 3.60000000E+01, 1.86300000E+00, 0.00000000E+00},
    {9.79403200E+02, 1.46000000E+02, 3.70000000E+01, 1.86300000E+00, 0.00000000E+00},
    {8.56716900E+02, 1.46000000E+02, 3.80000000E+01, 1.86300000E+00, 0.00000000E+00},
    {7.50593300E+02, 1.46000000E+02, 3.90000000E+01, 1.86300000E+00, 0.00000000E+00},
    {6.74992900E+02, 1.46000000E+02, 4.00000000E+01, 1.86300000E+00, 0.00000000E+00},
    {6.15975900E+02, 1.46000000E+02, 4.10000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.76709000E+02, 1.46000000E+02, 4.20000000E+01, 1.86300000E+00, 0.00000000E+00},
    {5.31370000E+02, 1.46000000E+02, 4.30000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.05939800E+02, 1.46000000E+02, 4.40000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.43065900E+02, 1.46000000E+02, 4.50000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.11136100E+02, 1.46000000E+02, 4.60000000E+01, 1.86300000E+00, 0.00000000E+00},
    {3.43670700E+02, 1.46000000E+02, 4.70000000E+01, 1.86300000E+00, 0.00000000E+00},
    {3.62643200E+02, 1.46000000E+02, 4.80000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.54057600E+02, 1.46000000E+02, 4.90000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.20126800E+02, 1.46000000E+02, 5.00000000E+01, 1.86300000E+00, 0.00000000E+00},
    {3.75099500E+02, 1.46000000E+02, 5.10000000E+01, 1.86300000E+00, 0.00000000E+00},
    {3.48607400E+02, 1.46000000E+02, 5.20000000E+01, 1.86300000E+00, 0.00000000E+00},
    {3.15928800E+02, 1.46000000E+02, 5.30000000E+01, 1.86300000E+00, 0.00000000E+00},
    {2.84796400E+02, 1.46000000E+02, 5.40000000E+01, 1.86300000E+00, 0.00000000E+00},
    {1.19370400E+03, 1.46000000E+02, 5.50000000E+01, 1.86300000E+00, 0.00000000E+00},
    {1.09300440E+03, 1.46000000E+02, 5.60000000E+01, 1.86300000E+00, 0.00000000E+00},
    {9.60134600E+02, 1.46000000E+02, 5.70000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.43362900E+02, 1.46000000E+02, 5.80000000E+01, 1.86300000E+00, 2.79910000E+00},
    {9.68875800E+02, 1.46000000E+02, 5.90000000E+01, 1.86300000E+00, 0.00000000E+00},
    {9.30225000E+02, 1.46000000E+02, 6.00000000E+01, 1.86300000E+00, 0.00000000E+00},
    {9.06851100E+02, 1.46000000E+02, 6.10000000E+01, 1.86300000E+00, 0.00000000E+00},
    {8.85347700E+02, 1.46000000E+02, 6.20000000E+01, 1.86300000E+00, 0.00000000E+00},
    {8.66274800E+02, 1.46000000E+02, 6.30000000E+01, 1.86300000E+00, 0.00000000E+00},
    {6.82207300E+02, 1.46000000E+02, 6.40000000E+01, 1.86300000E+00, 0.00000000E+00},
    {7.68790400E+02, 1.46000000E+02, 6.50000000E+01, 1.86300000E+00, 0.00000000E+00},
    {7.41514900E+02, 1.46000000E+02, 6.60000000E+01, 1.86300000E+00, 0.00000000E+00},
    {7.81204000E+02, 1.46000000E+02, 6.70000000E+01, 1.86300000E+00, 0.00000000E+00},
    {7.64570500E+02, 1.46000000E+02, 6.80000000E+01, 1.86300000E+00, 0.00000000E+00},
    {7.49577900E+02, 1.46000000E+02, 6.90000000E+01, 1.86300000E+00, 0.00000000E+00},
    {7.40739200E+02, 1.46000000E+02, 7.00000000E+01, 1.86300000E+00, 0.00000000E+00},
    {6.24715700E+02, 1.46000000E+02, 7.10000000E+01, 1.86300000E+00, 0.00000000E+00},
    {6.14253100E+02, 1.46000000E+02, 7.20000000E+01, 1.86300000E+00, 0.00000000E+00},
    {5.61151700E+02, 1.46000000E+02, 7.30000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.74949300E+02, 1.46000000E+02, 7.40000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.83156900E+02, 1.46000000E+02, 7.50000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.38467700E+02, 1.46000000E+02, 7.60000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.02083400E+02, 1.46000000E+02, 7.70000000E+01, 1.86300000E+00, 0.00000000E+00},
    {3.34985400E+02, 1.46000000E+02, 7.80000000E+01, 1.86300000E+00, 0.00000000E+00},
    {3.13348300E+02, 1.46000000E+02, 7.90000000E+01, 1.86300000E+00, 0.00000000E+00},
    {3.22188400E+02, 1.46000000E+02, 8.00000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.67357100E+02, 1.46000000E+02, 8.10000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.56867400E+02, 1.46000000E+02, 8.20000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.20327400E+02, 1.46000000E+02, 8.30000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.01334900E+02, 1.46000000E+02, 8.40000000E+01, 1.86300000E+00, 0.00000000E+00},
    {3.71072800E+02, 1.46000000E+02, 8.50000000E+01, 1.86300000E+00, 0.00000000E+00},
    {3.40845800E+02, 1.46000000E+02, 8.60000000E+01, 1.86300000E+00, 0.00000000E+00},
    {1.12695490E+03, 1.46000000E+02, 8.70000000E+01, 1.86300000E+00, 0.00000000E+00},
    {1.08104050E+03, 1.46000000E+02, 8.80000000E+01, 1.86300000E+00, 0.00000000E+00},
    {9.55535900E+02, 1.46000000E+02, 8.90000000E+01, 1.86300000E+00, 0.00000000E+00},
    {8.59695000E+02, 1.46000000E+02, 9.00000000E+01, 1.86300000E+00, 0.00000000E+00},
    {8.53937300E+02, 1.46000000E+02, 9.10000000E+01, 1.86300000E+00, 0.00000000E+00},
    {8.26944500E+02, 1.46000000E+02, 9.20000000E+01, 1.86300000E+00, 0.00000000E+00},
    {8.51191600E+02, 1.46000000E+02, 9.30000000E+01, 1.86300000E+00, 0.00000000E+00},
    {8.24294700E+02, 1.46000000E+02, 9.40000000E+01, 1.86300000E+00, 0.00000000E+00},
    {4.63489000E+01, 1.46000000E+02, 1.01000000E+02, 1.86300000E+00, 0.00000000E+00},
    {1.49120900E+02, 1.46000000E+02, 1.03000000E+02, 1.86300000E+00, 9.86500000E-01},
    {1.90482600E+02, 1.46000000E+02, 1.04000000E+02, 1.86300000E+00, 9.80800000E-01},
    {1.46202000E+02, 1.46000000E+02, 1.05000000E+02, 1.86300000E+00, 9.70600000E-01},
    {1.10794600E+02, 1.46000000E+02, 1.06000000E+02, 1.86300000E+00, 9.86800000E-01},
    {7.76479000E+01, 1.46000000E+02, 1.07000000E+02, 1.86300000E+00, 9.94400000E-01},
    {5.70049000E+01, 1.46000000E+02, 1.08000000E+02, 1.86300000E+00, 9.92500000E-01},
    {3.96636000E+01, 1.46000000E+02, 1.09000000E+02, 1.86300000E+00, 9.98200000E-01},
    {2.18233300E+02, 1.46000000E+02, 1.11000000E+02, 1.86300000E+00, 9.68400000E-01},
    {3.37225600E+02, 1.46000000E+02, 1.12000000E+02, 1.86300000E+00, 9.62800000E-01},
    {3.41794600E+02, 1.46000000E+02, 1.13000000E+02, 1.86300000E+00, 9.64800000E-01},
    {2.75492500E+02, 1.46000000E+02, 1.14000000E+02, 1.86300000E+00, 9.50700000E-01},
    {2.26460500E+02, 1.46000000E+02, 1.15000000E+02, 1.86300000E+00, 9.94700000E-01},
    {1.92217200E+02, 1.46000000E+02, 1.16000000E+02, 1.86300000E+00, 9.94800000E-01},
    {1.57852400E+02, 1.46000000E+02, 1.17000000E+02, 1.86300000E+00, 9.97200000E-01},
    {3.02463500E+02, 1.46000000E+02, 1.19000000E+02, 1.86300000E+00, 9.76700000E-01},
    {5.76137400E+02, 1.46000000E+02, 1.20000000E+02, 1.86300000E+00, 9.83100000E-01},
    {3.02489800E+02, 1.46000000E+02, 1.21000000E+02, 1.86300000E+00, 1.86270000E+00},
    {2.92213500E+02, 1.46000000E+02, 1.22000000E+02, 1.86300000E+00, 1.82990000E+00},
    {2.86427800E+02, 1.46000000E+02, 1.23000000E+02, 1.86300000E+00, 1.91380000E+00},
    {2.83778500E+02, 1.46000000E+02, 1.24000000E+02, 1.86300000E+00, 1.82690000E+00},
    {2.61263500E+02, 1.46000000E+02, 1.25000000E+02, 1.86300000E+00, 1.64060000E+00},
    {2.42014700E+02, 1.46000000E+02, 1.26000000E+02, 1.86300000E+00, 1.64830000E+00},
    {2.30978800E+02, 1.46000000E+02, 1.27000000E+02, 1.86300000E+00, 1.71490000E+00},
    {2.25816500E+02, 1.46000000E+02, 1.28000000E+02, 1.86300000E+00, 1.79370000E+00},
    {2.23015400E+02, 1.46000000E+02, 1.29000000E+02, 1.86300000E+00, 9.57600000E-01},
    {2.09454600E+02, 1.46000000E+02, 1.30000000E+02, 1.86300000E+00, 1.94190000E+00},
    {3.40067500E+02, 1.46000000E+02, 1.31000000E+02, 1.86300000E+00, 9.60100000E-01},
    {2.99328000E+02, 1.46000000E+02, 1.32000000E+02, 1.86300000E+00, 9.43400000E-01},
    {2.68942700E+02, 1.46000000E+02, 1.33000000E+02, 1.86300000E+00, 9.88900000E-01},
    {2.46193700E+02, 1.46000000E+02, 1.34000000E+02, 1.86300000E+00, 9.90100000E-01},
    {2.17573800E+02, 1.46000000E+02, 1.35000000E+02, 1.86300000E+00, 9.97400000E-01},
    {3.61304700E+02, 1.46000000E+02, 1.37000000E+02, 1.86300000E+00, 9.73800000E-01},
    {7.01655000E+02, 1.46000000E+02, 1.38000000E+02, 1.86300000E+00, 9.80100000E-01},
    {5.36856600E+02, 1.46000000E+02, 1.39000000E+02, 1.86300000E+00, 1.91530000E+00},
    {4.00420900E+02, 1.46000000E+02, 1.40000000E+02, 1.86300000E+00, 1.93550000E+00},
    {4.04430000E+02, 1.46000000E+02, 1.41000000E+02, 1.86300000E+00, 1.95450000E+00},
    {3.77553500E+02, 1.46000000E+02, 1.42000000E+02, 1.86300000E+00, 1.94200000E+00},
    {4.23127900E+02, 1.46000000E+02, 1.43000000E+02, 1.86300000E+00, 1.66820000E+00},
    {3.29682600E+02, 1.46000000E+02, 1.44000000E+02, 1.86300000E+00, 1.85840000E+00},
    {3.08705900E+02, 1.46000000E+02, 1.45000000E+02, 1.86300000E+00, 1.90030000E+00},
    {2.86951900E+02, 1.46000000E+02, 1.46000000E+02, 1.86300000E+00, 1.86300000E+00},
    {2.79518000E+01, 1.47000000E+02, 1.00000000E+00, 9.67900000E-01, 9.11800000E-01},
    {1.87731000E+01, 1.47000000E+02, 2.00000000E+00, 9.67900000E-01, 0.00000000E+00},
    {4.32607800E+02, 1.47000000E+02, 3.00000000E+00, 9.67900000E-01, 0.00000000E+00},
    {2.48657300E+02, 1.47000000E+02, 4.00000000E+00, 9.67900000E-01, 0.00000000E+00},
    {1.67835000E+02, 1.47000000E+02, 5.00000000E+00, 9.67900000E-01, 0.00000000E+00},
    {1.13999300E+02, 1.47000000E+02, 6.00000000E+00, 9.67900000E-01, 0.00000000E+00},
    {8.02663000E+01, 1.47000000E+02, 7.00000000E+00, 9.67900000E-01, 0.00000000E+00},
    {6.11748000E+01, 1.47000000E+02, 8.00000000E+00, 9.67900000E-01, 0.00000000E+00},
    {4.66726000E+01, 1.47000000E+02, 9.00000000E+00, 9.67900000E-01, 0.00000000E+00},
    {3.61486000E+01, 1.47000000E+02, 1.00000000E+01, 9.67900000E-01, 0.00000000E+00},
    {5.17568000E+02, 1.47000000E+02, 1.10000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.96612500E+02, 1.47000000E+02, 1.20000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.65526900E+02, 1.47000000E+02, 1.30000000E+01, 9.67900000E-01, 0.00000000E+00},
    {2.88286300E+02, 1.47000000E+02, 1.40000000E+01, 9.67900000E-01, 0.00000000E+00},
    {2.25394500E+02, 1.47000000E+02, 1.50000000E+01, 9.67900000E-01, 0.00000000E+00},
    {1.87659700E+02, 1.47000000E+02, 1.60000000E+01, 9.67900000E-01, 0.00000000E+00},
    {1.53903200E+02, 1.47000000E+02, 1.70000000E+01, 9.67900000E-01, 0.00000000E+00},
    {1.26482500E+02, 1.47000000E+02, 1.80000000E+01, 9.67900000E-01, 0.00000000E+00},
    {8.51008800E+02, 1.47000000E+02, 1.90000000E+01, 9.67900000E-01, 0.00000000E+00},
    {6.98861800E+02, 1.47000000E+02, 2.00000000E+01, 9.67900000E-01, 0.00000000E+00},
    {5.76963600E+02, 1.47000000E+02, 2.10000000E+01, 9.67900000E-01, 0.00000000E+00},
    {5.57288000E+02, 1.47000000E+02, 2.20000000E+01, 9.67900000E-01, 0.00000000E+00},
    {5.10364400E+02, 1.47000000E+02, 2.30000000E+01, 9.67900000E-01, 0.00000000E+00},
    {4.02838300E+02, 1.47000000E+02, 2.40000000E+01, 9.67900000E-01, 0.00000000E+00},
    {4.39509900E+02, 1.47000000E+02, 2.50000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.45699500E+02, 1.47000000E+02, 2.60000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.65692500E+02, 1.47000000E+02, 2.70000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.76628100E+02, 1.47000000E+02, 2.80000000E+01, 9.67900000E-01, 0.00000000E+00},
    {2.89558000E+02, 1.47000000E+02, 2.90000000E+01, 9.67900000E-01, 0.00000000E+00},
    {2.96672800E+02, 1.47000000E+02, 3.00000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.50801000E+02, 1.47000000E+02, 3.10000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.09454300E+02, 1.47000000E+02, 3.20000000E+01, 9.67900000E-01, 0.00000000E+00},
    {2.64395700E+02, 1.47000000E+02, 3.30000000E+01, 9.67900000E-01, 0.00000000E+00},
    {2.37721700E+02, 1.47000000E+02, 3.40000000E+01, 9.67900000E-01, 0.00000000E+00},
    {2.08604400E+02, 1.47000000E+02, 3.50000000E+01, 9.67900000E-01, 0.00000000E+00},
    {1.82001600E+02, 1.47000000E+02, 3.60000000E+01, 9.67900000E-01, 0.00000000E+00},
    {9.54132000E+02, 1.47000000E+02, 3.70000000E+01, 9.67900000E-01, 0.00000000E+00},
    {8.33316800E+02, 1.47000000E+02, 3.80000000E+01, 9.67900000E-01, 0.00000000E+00},
    {7.29331600E+02, 1.47000000E+02, 3.90000000E+01, 9.67900000E-01, 0.00000000E+00},
    {6.55432900E+02, 1.47000000E+02, 4.00000000E+01, 9.67900000E-01, 0.00000000E+00},
    {5.97852800E+02, 1.47000000E+02, 4.10000000E+01, 9.67900000E-01, 0.00000000E+00},
    {4.62338300E+02, 1.47000000E+02, 4.20000000E+01, 9.67900000E-01, 0.00000000E+00},
    {5.15495300E+02, 1.47000000E+02, 4.30000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.93508800E+02, 1.47000000E+02, 4.40000000E+01, 9.67900000E-01, 0.00000000E+00},
    {4.29514300E+02, 1.47000000E+02, 4.50000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.98466200E+02, 1.47000000E+02, 4.60000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.33158300E+02, 1.47000000E+02, 4.70000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.51371200E+02, 1.47000000E+02, 4.80000000E+01, 9.67900000E-01, 0.00000000E+00},
    {4.40273300E+02, 1.47000000E+02, 4.90000000E+01, 9.67900000E-01, 0.00000000E+00},
    {4.06902900E+02, 1.47000000E+02, 5.00000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.62874000E+02, 1.47000000E+02, 5.10000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.37013100E+02, 1.47000000E+02, 5.20000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.05212200E+02, 1.47000000E+02, 5.30000000E+01, 9.67900000E-01, 0.00000000E+00},
    {2.74974400E+02, 1.47000000E+02, 5.40000000E+01, 9.67900000E-01, 0.00000000E+00},
    {1.16255200E+03, 1.47000000E+02, 5.50000000E+01, 9.67900000E-01, 0.00000000E+00},
    {1.06335440E+03, 1.47000000E+02, 5.60000000E+01, 9.67900000E-01, 0.00000000E+00},
    {9.33168000E+02, 1.47000000E+02, 5.70000000E+01, 9.67900000E-01, 0.00000000E+00},
    {4.28998000E+02, 1.47000000E+02, 5.80000000E+01, 9.67900000E-01, 2.79910000E+00},
    {9.42354700E+02, 1.47000000E+02, 5.90000000E+01, 9.67900000E-01, 0.00000000E+00},
    {9.04668800E+02, 1.47000000E+02, 6.00000000E+01, 9.67900000E-01, 0.00000000E+00},
    {8.81907100E+02, 1.47000000E+02, 6.10000000E+01, 9.67900000E-01, 0.00000000E+00},
    {8.60969800E+02, 1.47000000E+02, 6.20000000E+01, 9.67900000E-01, 0.00000000E+00},
    {8.42396600E+02, 1.47000000E+02, 6.30000000E+01, 9.67900000E-01, 0.00000000E+00},
    {6.62602900E+02, 1.47000000E+02, 6.40000000E+01, 9.67900000E-01, 0.00000000E+00},
    {7.47724700E+02, 1.47000000E+02, 6.50000000E+01, 9.67900000E-01, 0.00000000E+00},
    {7.21010300E+02, 1.47000000E+02, 6.60000000E+01, 9.67900000E-01, 0.00000000E+00},
    {7.59508400E+02, 1.47000000E+02, 6.70000000E+01, 9.67900000E-01, 0.00000000E+00},
    {7.43322100E+02, 1.47000000E+02, 6.80000000E+01, 9.67900000E-01, 0.00000000E+00},
    {7.28720300E+02, 1.47000000E+02, 6.90000000E+01, 9.67900000E-01, 0.00000000E+00},
    {7.20165400E+02, 1.47000000E+02, 7.00000000E+01, 9.67900000E-01, 0.00000000E+00},
    {6.06831300E+02, 1.47000000E+02, 7.10000000E+01, 9.67900000E-01, 0.00000000E+00},
    {5.95985200E+02, 1.47000000E+02, 7.20000000E+01, 9.67900000E-01, 0.00000000E+00},
    {5.44098500E+02, 1.47000000E+02, 7.30000000E+01, 9.67900000E-01, 0.00000000E+00},
    {4.60267000E+02, 1.47000000E+02, 7.40000000E+01, 9.67900000E-01, 0.00000000E+00},
    {4.68108400E+02, 1.47000000E+02, 7.50000000E+01, 9.67900000E-01, 0.00000000E+00},
    {4.24582300E+02, 1.47000000E+02, 7.60000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.89187000E+02, 1.47000000E+02, 7.70000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.24124000E+02, 1.47000000E+02, 7.80000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.03155500E+02, 1.47000000E+02, 7.90000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.11635400E+02, 1.47000000E+02, 8.00000000E+01, 9.67900000E-01, 0.00000000E+00},
    {4.53045500E+02, 1.47000000E+02, 8.10000000E+01, 9.67900000E-01, 0.00000000E+00},
    {4.42509500E+02, 1.47000000E+02, 8.20000000E+01, 9.67900000E-01, 0.00000000E+00},
    {4.06708400E+02, 1.47000000E+02, 8.30000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.88100100E+02, 1.47000000E+02, 8.40000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.58595400E+02, 1.47000000E+02, 8.50000000E+01, 9.67900000E-01, 0.00000000E+00},
    {3.29199000E+02, 1.47000000E+02, 8.60000000E+01, 9.67900000E-01, 0.00000000E+00},
    {1.09683690E+03, 1.47000000E+02, 8.70000000E+01, 9.67900000E-01, 0.00000000E+00},
    {1.05115150E+03, 1.47000000E+02, 8.80000000E+01, 9.67900000E-01, 0.00000000E+00},
    {9.28330200E+02, 1.47000000E+02, 8.90000000E+01, 9.67900000E-01, 0.00000000E+00},
    {8.34317900E+02, 1.47000000E+02, 9.00000000E+01, 9.67900000E-01, 0.00000000E+00},
    {8.29192300E+02, 1.47000000E+02, 9.10000000E+01, 9.67900000E-01, 0.00000000E+00},
    {8.02980900E+02, 1.47000000E+02, 9.20000000E+01, 9.67900000E-01, 0.00000000E+00},
    {8.27161300E+02, 1.47000000E+02, 9.30000000E+01, 9.67900000E-01, 0.00000000E+00},
    {8.00933200E+02, 1.47000000E+02, 9.40000000E+01, 9.67900000E-01, 0.00000000E+00},
    {4.47835000E+01, 1.47000000E+02, 1.01000000E+02, 9.67900000E-01, 0.00000000E+00},
    {1.44656300E+02, 1.47000000E+02, 1.03000000E+02, 9.67900000E-01, 9.86500000E-01},
    {1.84653000E+02, 1.47000000E+02, 1.04000000E+02, 9.67900000E-01, 9.80800000E-01},
    {1.41406000E+02, 1.47000000E+02, 1.05000000E+02, 9.67900000E-01, 9.70600000E-01},
    {1.07028600E+02, 1.47000000E+02, 1.06000000E+02, 9.67900000E-01, 9.86800000E-01},
    {7.49354000E+01, 1.47000000E+02, 1.07000000E+02, 9.67900000E-01, 9.94400000E-01},
    {5.49856000E+01, 1.47000000E+02, 1.08000000E+02, 9.67900000E-01, 9.92500000E-01},
    {3.82539000E+01, 1.47000000E+02, 1.09000000E+02, 9.67900000E-01, 9.98200000E-01},
    {2.11841900E+02, 1.47000000E+02, 1.11000000E+02, 9.67900000E-01, 9.68400000E-01},
    {3.27350900E+02, 1.47000000E+02, 1.12000000E+02, 9.67900000E-01, 9.62800000E-01},
    {3.31383200E+02, 1.47000000E+02, 1.13000000E+02, 9.67900000E-01, 9.64800000E-01},
    {2.66600800E+02, 1.47000000E+02, 1.14000000E+02, 9.67900000E-01, 9.50700000E-01},
    {2.18861700E+02, 1.47000000E+02, 1.15000000E+02, 9.67900000E-01, 9.94700000E-01},
    {1.85619700E+02, 1.47000000E+02, 1.16000000E+02, 9.67900000E-01, 9.94800000E-01},
    {1.52327000E+02, 1.47000000E+02, 1.17000000E+02, 9.67900000E-01, 9.97200000E-01},
    {2.93291100E+02, 1.47000000E+02, 1.19000000E+02, 9.67900000E-01, 9.76700000E-01},
    {5.60129800E+02, 1.47000000E+02, 1.20000000E+02, 9.67900000E-01, 9.83100000E-01},
    {2.92906300E+02, 1.47000000E+02, 1.21000000E+02, 9.67900000E-01, 1.86270000E+00},
    {2.82933200E+02, 1.47000000E+02, 1.22000000E+02, 9.67900000E-01, 1.82990000E+00},
    {2.77352500E+02, 1.47000000E+02, 1.23000000E+02, 9.67900000E-01, 1.91380000E+00},
    {2.74845200E+02, 1.47000000E+02, 1.24000000E+02, 9.67900000E-01, 1.82690000E+00},
    {2.52829800E+02, 1.47000000E+02, 1.25000000E+02, 9.67900000E-01, 1.64060000E+00},
    {2.34151100E+02, 1.47000000E+02, 1.26000000E+02, 9.67900000E-01, 1.64830000E+00},
    {2.23480500E+02, 1.47000000E+02, 1.27000000E+02, 9.67900000E-01, 1.71490000E+00},
    {2.18507500E+02, 1.47000000E+02, 1.28000000E+02, 9.67900000E-01, 1.79370000E+00},
    {2.15958100E+02, 1.47000000E+02, 1.29000000E+02, 9.67900000E-01, 9.57600000E-01},
    {2.02568500E+02, 1.47000000E+02, 1.30000000E+02, 9.67900000E-01, 1.94190000E+00},
    {3.29538700E+02, 1.47000000E+02, 1.31000000E+02, 9.67900000E-01, 9.60100000E-01},
    {2.89641100E+02, 1.47000000E+02, 1.32000000E+02, 9.67900000E-01, 9.43400000E-01},
    {2.59970600E+02, 1.47000000E+02, 1.33000000E+02, 9.67900000E-01, 9.88900000E-01},
    {2.37823800E+02, 1.47000000E+02, 1.34000000E+02, 9.67900000E-01, 9.90100000E-01},
    {2.10041100E+02, 1.47000000E+02, 1.35000000E+02, 9.67900000E-01, 9.97400000E-01},
    {3.50229500E+02, 1.47000000E+02, 1.37000000E+02, 9.67900000E-01, 9.73800000E-01},
    {6.82181700E+02, 1.47000000E+02, 1.38000000E+02, 9.67900000E-01, 9.80100000E-01},
    {5.20872400E+02, 1.47000000E+02, 1.39000000E+02, 9.67900000E-01, 1.91530000E+00},
    {3.87733600E+02, 1.47000000E+02, 1.40000000E+02, 9.67900000E-01, 1.93550000E+00},
    {3.91663700E+02, 1.47000000E+02, 1.41000000E+02, 9.67900000E-01, 1.95450000E+00},
    {3.65526100E+02, 1.47000000E+02, 1.42000000E+02, 9.67900000E-01, 1.94200000E+00},
    {4.10066700E+02, 1.47000000E+02, 1.43000000E+02, 9.67900000E-01, 1.66820000E+00},
    {3.18966500E+02, 1.47000000E+02, 1.44000000E+02, 9.67900000E-01, 1.85840000E+00},
    {2.98667800E+02, 1.47000000E+02, 1.45000000E+02, 9.67900000E-01, 1.90030000E+00},
    {2.77602300E+02, 1.47000000E+02, 1.46000000E+02, 9.67900000E-01, 1.86300000E+00},
    {2.68613000E+02, 1.47000000E+02, 1.47000000E+02, 9.67900000E-01, 9.67900000E-01},
    {2.79281000E+01, 1.48000000E+02, 1.00000000E+00, 1.95390000E+00, 9.11800000E-01},
    {1.88607000E+01, 1.48000000E+02, 2.00000000E+00, 1.95390000E+00, 0.00000000E+00},
    {4.08442900E+02, 1.48000000E+02, 3.00000000E+00, 1.95390000E+00, 0.00000000E+00},
    {2.42095000E+02, 1.48000000E+02, 4.00000000E+00, 1.95390000E+00, 0.00000000E+00},
    {1.65658700E+02, 1.48000000E+02, 5.00000000E+00, 1.95390000E+00, 0.00000000E+00},
    {1.13436900E+02, 1.48000000E+02, 6.00000000E+00, 1.95390000E+00, 0.00000000E+00},
    {8.02492000E+01, 1.48000000E+02, 7.00000000E+00, 1.95390000E+00, 0.00000000E+00},
    {6.13234000E+01, 1.48000000E+02, 8.00000000E+00, 1.95390000E+00, 0.00000000E+00},
    {4.68692000E+01, 1.48000000E+02, 9.00000000E+00, 1.95390000E+00, 0.00000000E+00},
    {3.63401000E+01, 1.48000000E+02, 1.00000000E+01, 1.95390000E+00, 0.00000000E+00},
    {4.89512900E+02, 1.48000000E+02, 1.10000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.84172800E+02, 1.48000000E+02, 1.20000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.56865400E+02, 1.48000000E+02, 1.30000000E+01, 1.95390000E+00, 0.00000000E+00},
    {2.84213900E+02, 1.48000000E+02, 1.40000000E+01, 1.95390000E+00, 0.00000000E+00},
    {2.23778300E+02, 1.48000000E+02, 1.50000000E+01, 1.95390000E+00, 0.00000000E+00},
    {1.87016300E+02, 1.48000000E+02, 1.60000000E+01, 1.95390000E+00, 0.00000000E+00},
    {1.53859200E+02, 1.48000000E+02, 1.70000000E+01, 1.95390000E+00, 0.00000000E+00},
    {1.26735600E+02, 1.48000000E+02, 1.80000000E+01, 1.95390000E+00, 0.00000000E+00},
    {8.00259300E+02, 1.48000000E+02, 1.90000000E+01, 1.95390000E+00, 0.00000000E+00},
    {6.70263300E+02, 1.48000000E+02, 2.00000000E+01, 1.95390000E+00, 0.00000000E+00},
    {5.55667900E+02, 1.48000000E+02, 2.10000000E+01, 1.95390000E+00, 0.00000000E+00},
    {5.38550200E+02, 1.48000000E+02, 2.20000000E+01, 1.95390000E+00, 0.00000000E+00},
    {4.94223700E+02, 1.48000000E+02, 2.30000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.89995300E+02, 1.48000000E+02, 2.40000000E+01, 1.95390000E+00, 0.00000000E+00},
    {4.26857900E+02, 1.48000000E+02, 2.50000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.35764200E+02, 1.48000000E+02, 2.60000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.56917200E+02, 1.48000000E+02, 2.70000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.66866800E+02, 1.48000000E+02, 2.80000000E+01, 1.95390000E+00, 0.00000000E+00},
    {2.81867500E+02, 1.48000000E+02, 2.90000000E+01, 1.95390000E+00, 0.00000000E+00},
    {2.90789500E+02, 1.48000000E+02, 3.00000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.43294300E+02, 1.48000000E+02, 3.10000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.05146800E+02, 1.48000000E+02, 3.20000000E+01, 1.95390000E+00, 0.00000000E+00},
    {2.62284000E+02, 1.48000000E+02, 3.30000000E+01, 1.95390000E+00, 0.00000000E+00},
    {2.36577900E+02, 1.48000000E+02, 3.40000000E+01, 1.95390000E+00, 0.00000000E+00},
    {2.08211300E+02, 1.48000000E+02, 3.50000000E+01, 1.95390000E+00, 0.00000000E+00},
    {1.82086700E+02, 1.48000000E+02, 3.60000000E+01, 1.95390000E+00, 0.00000000E+00},
    {8.98534200E+02, 1.48000000E+02, 3.70000000E+01, 1.95390000E+00, 0.00000000E+00},
    {7.98452800E+02, 1.48000000E+02, 3.80000000E+01, 1.95390000E+00, 0.00000000E+00},
    {7.03799200E+02, 1.48000000E+02, 3.90000000E+01, 1.95390000E+00, 0.00000000E+00},
    {6.35137000E+02, 1.48000000E+02, 4.00000000E+01, 1.95390000E+00, 0.00000000E+00},
    {5.80841200E+02, 1.48000000E+02, 4.10000000E+01, 1.95390000E+00, 0.00000000E+00},
    {4.50977600E+02, 1.48000000E+02, 4.20000000E+01, 1.95390000E+00, 0.00000000E+00},
    {5.02116800E+02, 1.48000000E+02, 4.30000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.84943600E+02, 1.48000000E+02, 4.40000000E+01, 1.95390000E+00, 0.00000000E+00},
    {4.20374600E+02, 1.48000000E+02, 4.50000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.90577000E+02, 1.48000000E+02, 4.60000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.25966200E+02, 1.48000000E+02, 4.70000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.45059900E+02, 1.48000000E+02, 4.80000000E+01, 1.95390000E+00, 0.00000000E+00},
    {4.30360700E+02, 1.48000000E+02, 4.90000000E+01, 1.95390000E+00, 0.00000000E+00},
    {4.00347400E+02, 1.48000000E+02, 5.00000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.59055500E+02, 1.48000000E+02, 5.10000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.34491700E+02, 1.48000000E+02, 5.20000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.03841000E+02, 1.48000000E+02, 5.30000000E+01, 1.95390000E+00, 0.00000000E+00},
    {2.74440700E+02, 1.48000000E+02, 5.40000000E+01, 1.95390000E+00, 0.00000000E+00},
    {1.09499430E+03, 1.48000000E+02, 5.50000000E+01, 1.95390000E+00, 0.00000000E+00},
    {1.01618020E+03, 1.48000000E+02, 5.60000000E+01, 1.95390000E+00, 0.00000000E+00},
    {8.98209200E+02, 1.48000000E+02, 5.70000000E+01, 1.95390000E+00, 0.00000000E+00},
    {4.23891000E+02, 1.48000000E+02, 5.80000000E+01, 1.95390000E+00, 2.79910000E+00},
    {9.02629400E+02, 1.48000000E+02, 5.90000000E+01, 1.95390000E+00, 0.00000000E+00},
    {8.67593800E+02, 1.48000000E+02, 6.00000000E+01, 1.95390000E+00, 0.00000000E+00},
    {8.46061600E+02, 1.48000000E+02, 6.10000000E+01, 1.95390000E+00, 0.00000000E+00},
    {8.26226100E+02, 1.48000000E+02, 6.20000000E+01, 1.95390000E+00, 0.00000000E+00},
    {8.08641300E+02, 1.48000000E+02, 6.30000000E+01, 1.95390000E+00, 0.00000000E+00},
    {6.40739300E+02, 1.48000000E+02, 6.40000000E+01, 1.95390000E+00, 0.00000000E+00},
    {7.14620600E+02, 1.48000000E+02, 6.50000000E+01, 1.95390000E+00, 0.00000000E+00},
    {6.90042300E+02, 1.48000000E+02, 6.60000000E+01, 1.95390000E+00, 0.00000000E+00},
    {7.30428600E+02, 1.48000000E+02, 6.70000000E+01, 1.95390000E+00, 0.00000000E+00},
    {7.15019800E+02, 1.48000000E+02, 6.80000000E+01, 1.95390000E+00, 0.00000000E+00},
    {7.01182900E+02, 1.48000000E+02, 6.90000000E+01, 1.95390000E+00, 0.00000000E+00},
    {6.92750400E+02, 1.48000000E+02, 7.00000000E+01, 1.95390000E+00, 0.00000000E+00},
    {5.86683000E+02, 1.48000000E+02, 7.10000000E+01, 1.95390000E+00, 0.00000000E+00},
    {5.80279600E+02, 1.48000000E+02, 7.20000000E+01, 1.95390000E+00, 0.00000000E+00},
    {5.31792400E+02, 1.48000000E+02, 7.30000000E+01, 1.95390000E+00, 0.00000000E+00},
    {4.50936000E+02, 1.48000000E+02, 7.40000000E+01, 1.95390000E+00, 0.00000000E+00},
    {4.59359300E+02, 1.48000000E+02, 7.50000000E+01, 1.95390000E+00, 0.00000000E+00},
    {4.17908200E+02, 1.48000000E+02, 7.60000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.83960000E+02, 1.48000000E+02, 7.70000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.20364000E+02, 1.48000000E+02, 7.80000000E+01, 1.95390000E+00, 0.00000000E+00},
    {2.99862600E+02, 1.48000000E+02, 7.90000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.08705900E+02, 1.48000000E+02, 8.00000000E+01, 1.95390000E+00, 0.00000000E+00},
    {4.43211800E+02, 1.48000000E+02, 8.10000000E+01, 1.95390000E+00, 0.00000000E+00},
    {4.35196400E+02, 1.48000000E+02, 8.20000000E+01, 1.95390000E+00, 0.00000000E+00},
    {4.02090000E+02, 1.48000000E+02, 8.30000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.84762200E+02, 1.48000000E+02, 8.40000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.56597800E+02, 1.48000000E+02, 8.50000000E+01, 1.95390000E+00, 0.00000000E+00},
    {3.28184000E+02, 1.48000000E+02, 8.60000000E+01, 1.95390000E+00, 0.00000000E+00},
    {1.03919200E+03, 1.48000000E+02, 8.70000000E+01, 1.95390000E+00, 0.00000000E+00},
    {1.00817920E+03, 1.48000000E+02, 8.80000000E+01, 1.95390000E+00, 0.00000000E+00},
    {8.96300900E+02, 1.48000000E+02, 8.90000000E+01, 1.95390000E+00, 0.00000000E+00},
    {8.11148200E+02, 1.48000000E+02, 9.00000000E+01, 1.95390000E+00, 0.00000000E+00},
    {8.03152000E+02, 1.48000000E+02, 9.10000000E+01, 1.95390000E+00, 0.00000000E+00},
    {7.77876300E+02, 1.48000000E+02, 9.20000000E+01, 1.95390000E+00, 0.00000000E+00},
    {7.97759900E+02, 1.48000000E+02, 9.30000000E+01, 1.95390000E+00, 0.00000000E+00},
    {7.73119300E+02, 1.48000000E+02, 9.40000000E+01, 1.95390000E+00, 0.00000000E+00},
    {4.45006000E+01, 1.48000000E+02, 1.01000000E+02, 1.95390000E+00, 0.00000000E+00},
    {1.41090100E+02, 1.48000000E+02, 1.03000000E+02, 1.95390000E+00, 9.86500000E-01},
    {1.80520700E+02, 1.48000000E+02, 1.04000000E+02, 1.95390000E+00, 9.80800000E-01},
    {1.39879300E+02, 1.48000000E+02, 1.05000000E+02, 1.95390000E+00, 9.70600000E-01},
    {1.06433200E+02, 1.48000000E+02, 1.06000000E+02, 1.95390000E+00, 9.86800000E-01},
    {7.48925000E+01, 1.48000000E+02, 1.07000000E+02, 1.95390000E+00, 9.94400000E-01},
    {5.51392000E+01, 1.48000000E+02, 1.08000000E+02, 1.95390000E+00, 9.92500000E-01},
    {3.84906000E+01, 1.48000000E+02, 1.09000000E+02, 1.95390000E+00, 9.98200000E-01},
    {2.05964700E+02, 1.48000000E+02, 1.11000000E+02, 1.95390000E+00, 9.68400000E-01},
    {3.17900200E+02, 1.48000000E+02, 1.12000000E+02, 1.95390000E+00, 9.62800000E-01},
    {3.24083000E+02, 1.48000000E+02, 1.13000000E+02, 1.95390000E+00, 9.64800000E-01},
    {2.63143800E+02, 1.48000000E+02, 1.14000000E+02, 1.95390000E+00, 9.50700000E-01},
    {2.17319900E+02, 1.48000000E+02, 1.15000000E+02, 1.95390000E+00, 9.94700000E-01},
    {1.84952200E+02, 1.48000000E+02, 1.16000000E+02, 1.95390000E+00, 9.94800000E-01},
    {1.52264600E+02, 1.48000000E+02, 1.17000000E+02, 1.95390000E+00, 9.97200000E-01},
    {2.86273500E+02, 1.48000000E+02, 1.19000000E+02, 1.95390000E+00, 9.76700000E-01},
    {5.37088900E+02, 1.48000000E+02, 1.20000000E+02, 1.95390000E+00, 9.83100000E-01},
    {2.88387500E+02, 1.48000000E+02, 1.21000000E+02, 1.95390000E+00, 1.86270000E+00},
    {2.78586900E+02, 1.48000000E+02, 1.22000000E+02, 1.95390000E+00, 1.82990000E+00},
    {2.73033100E+02, 1.48000000E+02, 1.23000000E+02, 1.95390000E+00, 1.91380000E+00},
    {2.70289000E+02, 1.48000000E+02, 1.24000000E+02, 1.95390000E+00, 1.82690000E+00},
    {2.49820100E+02, 1.48000000E+02, 1.25000000E+02, 1.95390000E+00, 1.64060000E+00},
    {2.31617000E+02, 1.48000000E+02, 1.26000000E+02, 1.95390000E+00, 1.64830000E+00},
    {2.21024800E+02, 1.48000000E+02, 1.27000000E+02, 1.95390000E+00, 1.71490000E+00},
    {2.16021100E+02, 1.48000000E+02, 1.28000000E+02, 1.95390000E+00, 1.79370000E+00},
    {2.12746800E+02, 1.48000000E+02, 1.29000000E+02, 1.95390000E+00, 9.57600000E-01},
    {2.00822400E+02, 1.48000000E+02, 1.30000000E+02, 1.95390000E+00, 1.94190000E+00},
    {3.23108000E+02, 1.48000000E+02, 1.31000000E+02, 1.95390000E+00, 9.60100000E-01},
    {2.86055100E+02, 1.48000000E+02, 1.32000000E+02, 1.95390000E+00, 9.43400000E-01},
    {2.57968900E+02, 1.48000000E+02, 1.33000000E+02, 1.95390000E+00, 9.88900000E-01},
    {2.36660400E+02, 1.48000000E+02, 1.34000000E+02, 1.95390000E+00, 9.90100000E-01},
    {2.09608300E+02, 1.48000000E+02, 1.35000000E+02, 1.95390000E+00, 9.97400000E-01},
    {3.42341000E+02, 1.48000000E+02, 1.37000000E+02, 1.95390000E+00, 9.73800000E-01},
    {6.53278200E+02, 1.48000000E+02, 1.38000000E+02, 1.95390000E+00, 9.80100000E-01},
    {5.05674700E+02, 1.48000000E+02, 1.39000000E+02, 1.95390000E+00, 1.91530000E+00},
    {3.81340500E+02, 1.48000000E+02, 1.40000000E+02, 1.95390000E+00, 1.93550000E+00},
    {3.85120600E+02, 1.48000000E+02, 1.41000000E+02, 1.95390000E+00, 1.95450000E+00},
    {3.59905900E+02, 1.48000000E+02, 1.42000000E+02, 1.95390000E+00, 1.94200000E+00},
    {4.01238200E+02, 1.48000000E+02, 1.43000000E+02, 1.95390000E+00, 1.66820000E+00},
    {3.15336500E+02, 1.48000000E+02, 1.44000000E+02, 1.95390000E+00, 1.85840000E+00},
    {2.95303600E+02, 1.48000000E+02, 1.45000000E+02, 1.95390000E+00, 1.90030000E+00},
    {2.74615400E+02, 1.48000000E+02, 1.46000000E+02, 1.95390000E+00, 1.86300000E+00},
    {2.65545000E+02, 1.48000000E+02, 1.47000000E+02, 1.95390000E+00, 9.67900000E-01},
    {2.63450800E+02, 1.48000000E+02, 1.48000000E+02, 1.95390000E+00, 1.95390000E+00},
    {4.22827000E+01, 1.49000000E+02, 1.00000000E+00, 9.63300000E-01, 9.11800000E-01},
    {2.70236000E+01, 1.49000000E+02, 2.00000000E+00, 9.63300000E-01, 0.00000000E+00},
    {7.69251400E+02, 1.49000000E+02, 3.00000000E+00, 9.63300000E-01, 0.00000000E+00},
    {4.13496500E+02, 1.49000000E+02, 4.00000000E+00, 9.63300000E-01, 0.00000000E+00},
    {2.67475000E+02, 1.49000000E+02, 5.00000000E+00, 9.63300000E-01, 0.00000000E+00},
    {1.75458500E+02, 1.49000000E+02, 6.00000000E+00, 9.63300000E-01, 0.00000000E+00},
    {1.20127900E+02, 1.49000000E+02, 7.00000000E+00, 9.63300000E-01, 0.00000000E+00},
    {8.96520000E+01, 1.49000000E+02, 8.00000000E+00, 9.63300000E-01, 0.00000000E+00},
    {6.71232000E+01, 1.49000000E+02, 9.00000000E+00, 9.63300000E-01, 0.00000000E+00},
    {5.11747000E+01, 1.49000000E+02, 1.00000000E+01, 9.63300000E-01, 0.00000000E+00},
    {9.15941300E+02, 1.49000000E+02, 1.10000000E+01, 9.63300000E-01, 0.00000000E+00},
    {6.67284400E+02, 1.49000000E+02, 1.20000000E+01, 9.63300000E-01, 0.00000000E+00},
    {6.02414300E+02, 1.49000000E+02, 1.30000000E+01, 9.63300000E-01, 0.00000000E+00},
    {4.61509300E+02, 1.49000000E+02, 1.40000000E+01, 9.63300000E-01, 0.00000000E+00},
    {3.51515700E+02, 1.49000000E+02, 1.50000000E+01, 9.63300000E-01, 0.00000000E+00},
    {2.87392200E+02, 1.49000000E+02, 1.60000000E+01, 9.63300000E-01, 0.00000000E+00},
    {2.31455400E+02, 1.49000000E+02, 1.70000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.87091600E+02, 1.49000000E+02, 1.80000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.51736870E+03, 1.49000000E+02, 1.90000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.19892870E+03, 1.49000000E+02, 2.00000000E+01, 9.63300000E-01, 0.00000000E+00},
    {9.80846400E+02, 1.49000000E+02, 2.10000000E+01, 9.63300000E-01, 0.00000000E+00},
    {9.38764100E+02, 1.49000000E+02, 2.20000000E+01, 9.63300000E-01, 0.00000000E+00},
    {8.55074400E+02, 1.49000000E+02, 2.30000000E+01, 9.63300000E-01, 0.00000000E+00},
    {6.72852600E+02, 1.49000000E+02, 2.40000000E+01, 9.63300000E-01, 0.00000000E+00},
    {7.30522100E+02, 1.49000000E+02, 2.50000000E+01, 9.63300000E-01, 0.00000000E+00},
    {5.72172900E+02, 1.49000000E+02, 2.60000000E+01, 9.63300000E-01, 0.00000000E+00},
    {5.99825500E+02, 1.49000000E+02, 2.70000000E+01, 9.63300000E-01, 0.00000000E+00},
    {6.21330500E+02, 1.49000000E+02, 2.80000000E+01, 9.63300000E-01, 0.00000000E+00},
    {4.76016500E+02, 1.49000000E+02, 2.90000000E+01, 9.63300000E-01, 0.00000000E+00},
    {4.80597200E+02, 1.49000000E+02, 3.00000000E+01, 9.63300000E-01, 0.00000000E+00},
    {5.72265800E+02, 1.49000000E+02, 3.10000000E+01, 9.63300000E-01, 0.00000000E+00},
    {4.93798400E+02, 1.49000000E+02, 3.20000000E+01, 9.63300000E-01, 0.00000000E+00},
    {4.13012600E+02, 1.49000000E+02, 3.30000000E+01, 9.63300000E-01, 0.00000000E+00},
    {3.66151900E+02, 1.49000000E+02, 3.40000000E+01, 9.63300000E-01, 0.00000000E+00},
    {3.16523100E+02, 1.49000000E+02, 3.50000000E+01, 9.63300000E-01, 0.00000000E+00},
    {2.72228800E+02, 1.49000000E+02, 3.60000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.69420910E+03, 1.49000000E+02, 3.70000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.43066400E+03, 1.49000000E+02, 3.80000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.23169790E+03, 1.49000000E+02, 3.90000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.09504190E+03, 1.49000000E+02, 4.00000000E+01, 9.63300000E-01, 0.00000000E+00},
    {9.91401300E+02, 1.49000000E+02, 4.10000000E+01, 9.63300000E-01, 0.00000000E+00},
    {7.55891800E+02, 1.49000000E+02, 4.20000000E+01, 9.63300000E-01, 0.00000000E+00},
    {8.47355700E+02, 1.49000000E+02, 4.30000000E+01, 9.63300000E-01, 0.00000000E+00},
    {6.36748000E+02, 1.49000000E+02, 4.40000000E+01, 9.63300000E-01, 0.00000000E+00},
    {6.96132100E+02, 1.49000000E+02, 4.50000000E+01, 9.63300000E-01, 0.00000000E+00},
    {6.42645600E+02, 1.49000000E+02, 4.60000000E+01, 9.63300000E-01, 0.00000000E+00},
    {5.37205800E+02, 1.49000000E+02, 4.70000000E+01, 9.63300000E-01, 0.00000000E+00},
    {5.63031200E+02, 1.49000000E+02, 4.80000000E+01, 9.63300000E-01, 0.00000000E+00},
    {7.16795700E+02, 1.49000000E+02, 4.90000000E+01, 9.63300000E-01, 0.00000000E+00},
    {6.51140300E+02, 1.49000000E+02, 5.00000000E+01, 9.63300000E-01, 0.00000000E+00},
    {5.70301500E+02, 1.49000000E+02, 5.10000000E+01, 9.63300000E-01, 0.00000000E+00},
    {5.23613100E+02, 1.49000000E+02, 5.20000000E+01, 9.63300000E-01, 0.00000000E+00},
    {4.68168000E+02, 1.49000000E+02, 5.30000000E+01, 9.63300000E-01, 0.00000000E+00},
    {4.16505300E+02, 1.49000000E+02, 5.40000000E+01, 9.63300000E-01, 0.00000000E+00},
    {2.06262730E+03, 1.49000000E+02, 5.50000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.83425950E+03, 1.49000000E+02, 5.60000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.58387920E+03, 1.49000000E+02, 5.70000000E+01, 9.63300000E-01, 0.00000000E+00},
    {6.75791100E+02, 1.49000000E+02, 5.80000000E+01, 9.63300000E-01, 2.79910000E+00},
    {1.61598820E+03, 1.49000000E+02, 5.90000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.54724920E+03, 1.49000000E+02, 6.00000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.50722360E+03, 1.49000000E+02, 6.10000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.47054920E+03, 1.49000000E+02, 6.20000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.43798440E+03, 1.49000000E+02, 6.30000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.10922500E+03, 1.49000000E+02, 6.40000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.28456480E+03, 1.49000000E+02, 6.50000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.23485820E+03, 1.49000000E+02, 6.60000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.29135130E+03, 1.49000000E+02, 6.70000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.26335030E+03, 1.49000000E+02, 6.80000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.23781160E+03, 1.49000000E+02, 6.90000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.22431430E+03, 1.49000000E+02, 7.00000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.01814070E+03, 1.49000000E+02, 7.10000000E+01, 9.63300000E-01, 0.00000000E+00},
    {9.83564700E+02, 1.49000000E+02, 7.20000000E+01, 9.63300000E-01, 0.00000000E+00},
    {8.87925100E+02, 1.49000000E+02, 7.30000000E+01, 9.63300000E-01, 0.00000000E+00},
    {7.43403500E+02, 1.49000000E+02, 7.40000000E+01, 9.63300000E-01, 0.00000000E+00},
    {7.53151200E+02, 1.49000000E+02, 7.50000000E+01, 9.63300000E-01, 0.00000000E+00},
    {6.76234000E+02, 1.49000000E+02, 7.60000000E+01, 9.63300000E-01, 0.00000000E+00},
    {6.14658200E+02, 1.49000000E+02, 7.70000000E+01, 9.63300000E-01, 0.00000000E+00},
    {5.06544200E+02, 1.49000000E+02, 7.80000000E+01, 9.63300000E-01, 0.00000000E+00},
    {4.71774800E+02, 1.49000000E+02, 7.90000000E+01, 9.63300000E-01, 0.00000000E+00},
    {4.83631300E+02, 1.49000000E+02, 8.00000000E+01, 9.63300000E-01, 0.00000000E+00},
    {7.32580800E+02, 1.49000000E+02, 8.10000000E+01, 9.63300000E-01, 0.00000000E+00},
    {7.06444600E+02, 1.49000000E+02, 8.20000000E+01, 9.63300000E-01, 0.00000000E+00},
    {6.39131500E+02, 1.49000000E+02, 8.30000000E+01, 9.63300000E-01, 0.00000000E+00},
    {6.04108200E+02, 1.49000000E+02, 8.40000000E+01, 9.63300000E-01, 0.00000000E+00},
    {5.51584400E+02, 1.49000000E+02, 8.50000000E+01, 9.63300000E-01, 0.00000000E+00},
    {5.00678600E+02, 1.49000000E+02, 8.60000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.92100630E+03, 1.49000000E+02, 8.70000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.79755690E+03, 1.49000000E+02, 8.80000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.56372650E+03, 1.49000000E+02, 8.90000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.37963610E+03, 1.49000000E+02, 9.00000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.38229030E+03, 1.49000000E+02, 9.10000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.33778390E+03, 1.49000000E+02, 9.20000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.39311010E+03, 1.49000000E+02, 9.30000000E+01, 9.63300000E-01, 0.00000000E+00},
    {1.34626970E+03, 1.49000000E+02, 9.40000000E+01, 9.63300000E-01, 0.00000000E+00},
    {6.96754000E+01, 1.49000000E+02, 1.01000000E+02, 9.63300000E-01, 0.00000000E+00},
    {2.39041900E+02, 1.49000000E+02, 1.03000000E+02, 9.63300000E-01, 9.86500000E-01},
    {3.02667800E+02, 1.49000000E+02, 1.04000000E+02, 9.63300000E-01, 9.80800000E-01},
    {2.23162900E+02, 1.49000000E+02, 1.05000000E+02, 9.63300000E-01, 9.70600000E-01},
    {1.64681200E+02, 1.49000000E+02, 1.06000000E+02, 9.63300000E-01, 9.86800000E-01},
    {1.11970700E+02, 1.49000000E+02, 1.07000000E+02, 9.63300000E-01, 9.94400000E-01},
    {8.00986000E+01, 1.49000000E+02, 1.08000000E+02, 9.63300000E-01, 9.92500000E-01},
    {5.39364000E+01, 1.49000000E+02, 1.09000000E+02, 9.63300000E-01, 9.98200000E-01},
    {3.52007400E+02, 1.49000000E+02, 1.11000000E+02, 9.63300000E-01, 9.68400000E-01},
    {5.46276200E+02, 1.49000000E+02, 1.12000000E+02, 9.63300000E-01, 9.62800000E-01},
    {5.43218800E+02, 1.49000000E+02, 1.13000000E+02, 9.63300000E-01, 9.64800000E-01},
    {4.24829300E+02, 1.49000000E+02, 1.14000000E+02, 9.63300000E-01, 9.50700000E-01},
    {3.40939600E+02, 1.49000000E+02, 1.15000000E+02, 9.63300000E-01, 9.94700000E-01},
    {2.84335000E+02, 1.49000000E+02, 1.16000000E+02, 9.63300000E-01, 9.94800000E-01},
    {2.29122900E+02, 1.49000000E+02, 1.17000000E+02, 9.63300000E-01, 9.97200000E-01},
    {4.78014100E+02, 1.49000000E+02, 1.19000000E+02, 9.63300000E-01, 9.76700000E-01},
    {9.57989700E+02, 1.49000000E+02, 1.20000000E+02, 9.63300000E-01, 9.83100000E-01},
    {4.67670000E+02, 1.49000000E+02, 1.21000000E+02, 9.63300000E-01, 1.86270000E+00},
    {4.51165800E+02, 1.49000000E+02, 1.22000000E+02, 9.63300000E-01, 1.82990000E+00},
    {4.42342000E+02, 1.49000000E+02, 1.23000000E+02, 9.63300000E-01, 1.91380000E+00},
    {4.39472800E+02, 1.49000000E+02, 1.24000000E+02, 9.63300000E-01, 1.82690000E+00},
    {3.99107900E+02, 1.49000000E+02, 1.25000000E+02, 9.63300000E-01, 1.64060000E+00},
    {3.67959200E+02, 1.49000000E+02, 1.26000000E+02, 9.63300000E-01, 1.64830000E+00},
    {3.51030000E+02, 1.49000000E+02, 1.27000000E+02, 9.63300000E-01, 1.71490000E+00},
    {3.43562000E+02, 1.49000000E+02, 1.28000000E+02, 9.63300000E-01, 1.79370000E+00},
    {3.42893100E+02, 1.49000000E+02, 1.29000000E+02, 9.63300000E-01, 9.57600000E-01},
    {3.15916600E+02, 1.49000000E+02, 1.30000000E+02, 9.63300000E-01, 1.94190000E+00},
    {5.34512700E+02, 1.49000000E+02, 1.31000000E+02, 9.63300000E-01, 9.60100000E-01},
    {4.59628500E+02, 1.49000000E+02, 1.32000000E+02, 9.63300000E-01, 9.43400000E-01},
    {4.05545300E+02, 1.49000000E+02, 1.33000000E+02, 9.63300000E-01, 9.88900000E-01},
    {3.66365200E+02, 1.49000000E+02, 1.34000000E+02, 9.63300000E-01, 9.90100000E-01},
    {3.18930500E+02, 1.49000000E+02, 1.35000000E+02, 9.63300000E-01, 9.97400000E-01},
    {5.67600200E+02, 1.49000000E+02, 1.37000000E+02, 9.63300000E-01, 9.73800000E-01},
    {1.16850360E+03, 1.49000000E+02, 1.38000000E+02, 9.63300000E-01, 9.80100000E-01},
    {8.62883300E+02, 1.49000000E+02, 1.39000000E+02, 9.63300000E-01, 1.91530000E+00},
    {6.19653700E+02, 1.49000000E+02, 1.40000000E+02, 9.63300000E-01, 1.93550000E+00},
    {6.26025400E+02, 1.49000000E+02, 1.41000000E+02, 9.63300000E-01, 1.95450000E+00},
    {5.81054900E+02, 1.49000000E+02, 1.42000000E+02, 9.63300000E-01, 1.94200000E+00},
    {6.63038000E+02, 1.49000000E+02, 1.43000000E+02, 9.63300000E-01, 1.66820000E+00},
    {5.00162800E+02, 1.49000000E+02, 1.44000000E+02, 9.63300000E-01, 1.85840000E+00},
    {4.67377100E+02, 1.49000000E+02, 1.45000000E+02, 9.63300000E-01, 1.90030000E+00},
    {4.32997300E+02, 1.49000000E+02, 1.46000000E+02, 9.63300000E-01, 1.86300000E+00},
    {4.19686800E+02, 1.49000000E+02, 1.47000000E+02, 9.63300000E-01, 9.67900000E-01},
    {4.11157600E+02, 1.49000000E+02, 1.48000000E+02, 9.63300000E-01, 1.95390000E+00},
    {6.78992000E+02, 1.49000000E+02, 1.49000000E+02, 9.63300000E-01, 9.63300000E-01},
    {3.90500000E+01, 1.50000000E+02, 1.00000000E+00, 9.51400000E-01, 9.11800000E-01},
    {2.54105000E+01, 1.50000000E+02, 2.00000000E+00, 9.51400000E-01, 0.00000000E+00},
    {6.42419000E+02, 1.50000000E+02, 3.00000000E+00, 9.51400000E-01, 0.00000000E+00},
    {3.62743400E+02, 1.50000000E+02, 4.00000000E+00, 9.51400000E-01, 0.00000000E+00},
    {2.40593400E+02, 1.50000000E+02, 5.00000000E+00, 9.51400000E-01, 0.00000000E+00},
    {1.60587600E+02, 1.50000000E+02, 6.00000000E+00, 9.51400000E-01, 0.00000000E+00},
    {1.11276800E+02, 1.50000000E+02, 7.00000000E+00, 9.51400000E-01, 0.00000000E+00},
    {8.37120000E+01, 1.50000000E+02, 8.00000000E+00, 9.51400000E-01, 0.00000000E+00},
    {6.30806000E+01, 1.50000000E+02, 9.00000000E+00, 9.51400000E-01, 0.00000000E+00},
    {4.83258000E+01, 1.50000000E+02, 1.00000000E+01, 9.51400000E-01, 0.00000000E+00},
    {7.67095500E+02, 1.50000000E+02, 1.10000000E+01, 9.51400000E-01, 0.00000000E+00},
    {5.80450000E+02, 1.50000000E+02, 1.20000000E+01, 9.51400000E-01, 0.00000000E+00},
    {5.31079400E+02, 1.50000000E+02, 1.30000000E+01, 9.51400000E-01, 0.00000000E+00},
    {4.14140300E+02, 1.50000000E+02, 1.40000000E+01, 9.51400000E-01, 0.00000000E+00},
    {3.19930700E+02, 1.50000000E+02, 1.50000000E+01, 9.51400000E-01, 0.00000000E+00},
    {2.63827700E+02, 1.50000000E+02, 1.60000000E+01, 9.51400000E-01, 0.00000000E+00},
    {2.14172300E+02, 1.50000000E+02, 1.70000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.74263900E+02, 1.50000000E+02, 1.80000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.25959640E+03, 1.50000000E+02, 1.90000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.02660970E+03, 1.50000000E+02, 2.00000000E+01, 9.51400000E-01, 0.00000000E+00},
    {8.45544700E+02, 1.50000000E+02, 2.10000000E+01, 9.51400000E-01, 0.00000000E+00},
    {8.13896400E+02, 1.50000000E+02, 2.20000000E+01, 9.51400000E-01, 0.00000000E+00},
    {7.43925300E+02, 1.50000000E+02, 2.30000000E+01, 9.51400000E-01, 0.00000000E+00},
    {5.85406900E+02, 1.50000000E+02, 2.40000000E+01, 9.51400000E-01, 0.00000000E+00},
    {6.38758700E+02, 1.50000000E+02, 2.50000000E+01, 9.51400000E-01, 0.00000000E+00},
    {5.00630400E+02, 1.50000000E+02, 2.60000000E+01, 9.51400000E-01, 0.00000000E+00},
    {5.28996500E+02, 1.50000000E+02, 2.70000000E+01, 9.51400000E-01, 0.00000000E+00},
    {5.46047300E+02, 1.50000000E+02, 2.80000000E+01, 9.51400000E-01, 0.00000000E+00},
    {4.18173000E+02, 1.50000000E+02, 2.90000000E+01, 9.51400000E-01, 0.00000000E+00},
    {4.27102500E+02, 1.50000000E+02, 3.00000000E+01, 9.51400000E-01, 0.00000000E+00},
    {5.06883800E+02, 1.50000000E+02, 3.10000000E+01, 9.51400000E-01, 0.00000000E+00},
    {4.43489200E+02, 1.50000000E+02, 3.20000000E+01, 9.51400000E-01, 0.00000000E+00},
    {3.75369400E+02, 1.50000000E+02, 3.30000000E+01, 9.51400000E-01, 0.00000000E+00},
    {3.35118200E+02, 1.50000000E+02, 3.40000000E+01, 9.51400000E-01, 0.00000000E+00},
    {2.91717900E+02, 1.50000000E+02, 3.50000000E+01, 9.51400000E-01, 0.00000000E+00},
    {2.52440800E+02, 1.50000000E+02, 3.60000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.40953860E+03, 1.50000000E+02, 3.70000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.22330730E+03, 1.50000000E+02, 3.80000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.06543420E+03, 1.50000000E+02, 3.90000000E+01, 9.51400000E-01, 0.00000000E+00},
    {9.53932200E+02, 1.50000000E+02, 4.00000000E+01, 9.51400000E-01, 0.00000000E+00},
    {8.67547500E+02, 1.50000000E+02, 4.10000000E+01, 9.51400000E-01, 0.00000000E+00},
    {6.66419700E+02, 1.50000000E+02, 4.20000000E+01, 9.51400000E-01, 0.00000000E+00},
    {7.44999300E+02, 1.50000000E+02, 4.30000000E+01, 9.51400000E-01, 0.00000000E+00},
    {5.64458900E+02, 1.50000000E+02, 4.40000000E+01, 9.51400000E-01, 0.00000000E+00},
    {6.17379100E+02, 1.50000000E+02, 4.50000000E+01, 9.51400000E-01, 0.00000000E+00},
    {5.71560200E+02, 1.50000000E+02, 4.60000000E+01, 9.51400000E-01, 0.00000000E+00},
    {4.76596600E+02, 1.50000000E+02, 4.70000000E+01, 9.51400000E-01, 0.00000000E+00},
    {5.02551300E+02, 1.50000000E+02, 4.80000000E+01, 9.51400000E-01, 0.00000000E+00},
    {6.34119500E+02, 1.50000000E+02, 4.90000000E+01, 9.51400000E-01, 0.00000000E+00},
    {5.82769300E+02, 1.50000000E+02, 5.00000000E+01, 9.51400000E-01, 0.00000000E+00},
    {5.15946700E+02, 1.50000000E+02, 5.10000000E+01, 9.51400000E-01, 0.00000000E+00},
    {4.76681100E+02, 1.50000000E+02, 5.20000000E+01, 9.51400000E-01, 0.00000000E+00},
    {4.28996300E+02, 1.50000000E+02, 5.30000000E+01, 9.51400000E-01, 0.00000000E+00},
    {3.83945700E+02, 1.50000000E+02, 5.40000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.71615260E+03, 1.50000000E+02, 5.50000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.56182890E+03, 1.50000000E+02, 5.60000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.36442900E+03, 1.50000000E+02, 5.70000000E+01, 9.51400000E-01, 0.00000000E+00},
    {6.10004200E+02, 1.50000000E+02, 5.80000000E+01, 9.51400000E-01, 2.79910000E+00},
    {1.38116770E+03, 1.50000000E+02, 5.90000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.32511510E+03, 1.50000000E+02, 6.00000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.29157070E+03, 1.50000000E+02, 6.10000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.26076080E+03, 1.50000000E+02, 6.20000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.23342800E+03, 1.50000000E+02, 6.30000000E+01, 9.51400000E-01, 0.00000000E+00},
    {9.63223400E+02, 1.50000000E+02, 6.40000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.09407030E+03, 1.50000000E+02, 6.50000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.05397480E+03, 1.50000000E+02, 6.60000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.11100050E+03, 1.50000000E+02, 6.70000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.08729010E+03, 1.50000000E+02, 6.80000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.06581960E+03, 1.50000000E+02, 6.90000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.05367730E+03, 1.50000000E+02, 7.00000000E+01, 9.51400000E-01, 0.00000000E+00},
    {8.83592100E+02, 1.50000000E+02, 7.10000000E+01, 9.51400000E-01, 0.00000000E+00},
    {8.63807500E+02, 1.50000000E+02, 7.20000000E+01, 9.51400000E-01, 0.00000000E+00},
    {7.85177100E+02, 1.50000000E+02, 7.30000000E+01, 9.51400000E-01, 0.00000000E+00},
    {6.60553900E+02, 1.50000000E+02, 7.40000000E+01, 9.51400000E-01, 0.00000000E+00},
    {6.71101300E+02, 1.50000000E+02, 7.50000000E+01, 9.51400000E-01, 0.00000000E+00},
    {6.06064600E+02, 1.50000000E+02, 7.60000000E+01, 9.51400000E-01, 0.00000000E+00},
    {5.53420100E+02, 1.50000000E+02, 7.70000000E+01, 9.51400000E-01, 0.00000000E+00},
    {4.58079600E+02, 1.50000000E+02, 7.80000000E+01, 9.51400000E-01, 0.00000000E+00},
    {4.27394800E+02, 1.50000000E+02, 7.90000000E+01, 9.51400000E-01, 0.00000000E+00},
    {4.39237500E+02, 1.50000000E+02, 8.00000000E+01, 9.51400000E-01, 0.00000000E+00},
    {6.49475200E+02, 1.50000000E+02, 8.10000000E+01, 9.51400000E-01, 0.00000000E+00},
    {6.32160600E+02, 1.50000000E+02, 8.20000000E+01, 9.51400000E-01, 0.00000000E+00},
    {5.77575000E+02, 1.50000000E+02, 8.30000000E+01, 9.51400000E-01, 0.00000000E+00},
    {5.48932800E+02, 1.50000000E+02, 8.40000000E+01, 9.51400000E-01, 0.00000000E+00},
    {5.04414000E+02, 1.50000000E+02, 8.50000000E+01, 9.51400000E-01, 0.00000000E+00},
    {4.60435300E+02, 1.50000000E+02, 8.60000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.61320500E+03, 1.50000000E+02, 8.70000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.53965490E+03, 1.50000000E+02, 8.80000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.35397430E+03, 1.50000000E+02, 8.90000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.20889360E+03, 1.50000000E+02, 9.00000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.20370430E+03, 1.50000000E+02, 9.10000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.16529870E+03, 1.50000000E+02, 9.20000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.20456860E+03, 1.50000000E+02, 9.30000000E+01, 9.51400000E-01, 0.00000000E+00},
    {1.16570620E+03, 1.50000000E+02, 9.40000000E+01, 9.51400000E-01, 0.00000000E+00},
    {6.35189000E+01, 1.50000000E+02, 1.01000000E+02, 9.51400000E-01, 0.00000000E+00},
    {2.10381500E+02, 1.50000000E+02, 1.03000000E+02, 9.51400000E-01, 9.86500000E-01},
    {2.67503600E+02, 1.50000000E+02, 1.04000000E+02, 9.51400000E-01, 9.80800000E-01},
    {2.01675800E+02, 1.50000000E+02, 1.05000000E+02, 9.51400000E-01, 9.70600000E-01},
    {1.50604000E+02, 1.50000000E+02, 1.06000000E+02, 9.51400000E-01, 9.86800000E-01},
    {1.03707000E+02, 1.50000000E+02, 1.07000000E+02, 9.51400000E-01, 9.94400000E-01},
    {7.49289000E+01, 1.50000000E+02, 1.08000000E+02, 9.51400000E-01, 9.92500000E-01},
    {5.10492000E+01, 1.50000000E+02, 1.09000000E+02, 9.51400000E-01, 9.98200000E-01},
    {3.08247000E+02, 1.50000000E+02, 1.11000000E+02, 9.51400000E-01, 9.68400000E-01},
    {4.77284800E+02, 1.50000000E+02, 1.12000000E+02, 9.51400000E-01, 9.62800000E-01},
    {4.80373500E+02, 1.50000000E+02, 1.13000000E+02, 9.51400000E-01, 9.64800000E-01},
    {3.82125500E+02, 1.50000000E+02, 1.14000000E+02, 9.51400000E-01, 9.50700000E-01},
    {3.10423800E+02, 1.50000000E+02, 1.15000000E+02, 9.51400000E-01, 9.94700000E-01},
    {2.60949700E+02, 1.50000000E+02, 1.16000000E+02, 9.51400000E-01, 9.94800000E-01},
    {2.11970900E+02, 1.50000000E+02, 1.17000000E+02, 9.51400000E-01, 9.97200000E-01},
    {4.22005500E+02, 1.50000000E+02, 1.19000000E+02, 9.51400000E-01, 9.76700000E-01},
    {8.19992600E+02, 1.50000000E+02, 1.20000000E+02, 9.51400000E-01, 9.83100000E-01},
    {4.19143200E+02, 1.50000000E+02, 1.21000000E+02, 9.51400000E-01, 1.86270000E+00},
    {4.04437800E+02, 1.50000000E+02, 1.22000000E+02, 9.51400000E-01, 1.82990000E+00},
    {3.96407900E+02, 1.50000000E+02, 1.23000000E+02, 9.51400000E-01, 1.91380000E+00},
    {3.93121500E+02, 1.50000000E+02, 1.24000000E+02, 9.514