/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2026 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - MassXpert, model polymer chemistries and simulate mass spectrometric data;
 * - MineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// Qt includes
#include <QDir>
#include <QMessageBox>
#include <QStandardPaths>
#include <QLocale>

/////////////////////// libXpertMassCore includes
#include <MsXpS/libXpertMassCore/Utils.hpp>

/////////////////////// Local includes
#include "config.h"
#include "Application.hpp"


namespace MsXpS
{
namespace MassXpert
{


Application::Application(int &argc,
                         char **argv,
                         const QString &application_name)
  : QApplication{argc, argv}, m_applicationName{application_name}
{
  QPixmap splashPixmap(":/images/splashscreen.png");

  mpa_splash = new QSplashScreen(splashPixmap, Qt::WindowStaysOnTopHint);
  mpa_splash->show();

  // OLD version QTimer::singleShot(2000, this, SLOT(destroySplash()));
  QTimer::singleShot(2000, this, [this]() { delete mpa_splash; });

  // We now have to perform a number of initialization steps.

  // We set these, but we do not want these to be used for the
  // QSettings config file, because we do no want that the settings
  // be stored on MS Windows in the HKEY registry. We prefer going
  // for an INI style file in a safe location.
  QCoreApplication::setOrganizationName(m_organizationName);
  QCoreApplication::setOrganizationDomain("msxpertsuite.org");
  QCoreApplication::setApplicationName(m_applicationName);

  // The configuration directory for all the settings of the program

  m_userConfigSettingsDirPath =
    QStandardPaths::writableLocation(QStandardPaths::ConfigLocation);

  if(m_userConfigSettingsDirPath.isEmpty())
    m_userConfigSettingsDirPath =
      QStandardPaths::writableLocation(QStandardPaths::HomeLocation);

  m_userConfigSettingsDirPath =
    m_userConfigSettingsDirPath + QDir::separator() + m_applicationName;

  m_userConfigSettingsFilePath =
    m_userConfigSettingsDirPath + QDir::separator() + "configSettings.ini";

  qDebug() << "m_userConfigSettingsDirPath:" << m_userConfigSettingsDirPath;
  qDebug() << "m_userConfigSettingsFilePath:" << m_userConfigSettingsFilePath;

  // On Debian, QStandardPaths::ConfigLocation translates to
  // ~/.config/MassXpert3/configSettings.ini.
  // while
  // on Windows, this translates to %APPDATA%, that is:
  // C:\Users\<User>\AppData\Roaming\MassXpert3\configSettings.ini

  // We now have to initialize the libXpertMassCore.

  MsXpS::libXpertMassCore::Utils::initializeConfigSettingsFilePath(
    m_userConfigSettingsFilePath);

#if 0
  // When I changed the icons of MassXpert3, the old icons stubborningly
  // remained the same green frog head. Finally, it turned out that the
  // icon was cached by the desktop ! So changing the desktop filename
  // made the new icon appear correctly.
  // The trick below, changes the name of the desktop file each time the
  // program is rebuilt, thus forcing to acknowledge that there is a
  // non-cached icon!

  setDesktopFileName(
    QStringLiteral("massxpert3-%1").arg(QDateTime::currentSecsSinceEpoch()));
#endif

  // The default window icon.
  QIcon icon(":/images/icons/svg/MassXpert3.svg");
  setWindowIcon(icon);
}


Application::~Application()
{
}

void
Application::setProgramWindow(ProgramWindow *program_window_p)
{
  mp_programWindow = program_window_p;
  qDebug() << "Set ProgramWindow to application as:" << program_window_p;
}

ProgramWindow *
Application::getProgramWindow()
{
  return mp_programWindow;
}

QString
Application::getApplicationName()
{
  return m_applicationName;
}

void
Application::setDescription(QString &desc)
{
  m_description = desc;
}

QString
Application::description()
{
  return m_description;
}

QString
Application::getUserConfigSettingsDirPath()
{
  return m_userConfigSettingsDirPath;
}

QString
Application::getUserConfigSettingsFilePath()
{
  return m_userConfigSettingsFilePath;
}


} // namespace MassXpert
} // namespace MsXpS
