/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2026 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - MassXpert, model polymer chemistries and simulate mass spectrometric data;
 * - MineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


#pragma once

/////////////////////// stdlib includes


/////////////////////// Qt includes
#include <QMainWindow>
#include <QSortFilterProxyModel>


/////////////////////// pappsomspp includes
#include <pappsomspp/core/trace/trace.h>


/////////////////////// libXpertMass includes
#include <MsXpS/libXpertMassCore/IsotopicData.hpp>
#include <MsXpS/libXpertMassCore/MassPeakShaperConfig.hpp>
#include <MsXpS/libXpertMassCore/Ionizer.hpp>
#include <MsXpS/libXpertMassCore/Cleaver.hpp>


/////////////////////// libXpertMassGui includes
#include <MsXpS/libXpertMassGui/MassPeakShaperConfigDlg.hpp>


/////////////////////// Local includes
#include "AbstractSeqEdWndDependentDlg.hpp"
#include "SequenceEditorWnd.hpp"
#include "CleaveOligomerTableViewSortProxyModel.hpp"


namespace Ui
{
class CleavageDlg;
}

namespace MsXpS
{
namespace MassXpert
{


class CleaveOligomerTableViewModel;
class CleaveOligomerTableViewSortProxyModel;

enum class ExportResultsActions
{
  EXPORT_TO_CLIPBOARD_OVERWRITE = 0,
  EXPORT_TO_CLIPBOARD_APPEND,
  EXPORT_TO_FILE,
  SELECT_FILE,
  CALCULATE_SPECTRUM
};

class CleavageDlg : public AbstractSeqEdWndDependentDlg
{
  Q_OBJECT

  public:
  CleavageDlg(SequenceEditorWnd *editor_wnd_p,
              libXpertMassCore::PolymerQSPtr polymer_sp,
              const libXpertMassCore::PolChemDefCstSPtr pol_chem_def_csp,
              const QString &config_settings_file_path,
              const QString &application_name,
              const QString &description,
              QByteArray hash,
              const libXpertMassCore::CalcOptions &calc_options,
              const libXpertMassCore::Ionizer &ionizer);

  ~CleavageDlg();

  bool initialize();
  Q_INVOKABLE void setStackOligomers(bool stack_oligomers);

  bool populateSelectedOligomerData();
  void populateListOfAvailableCleavageAgents();

  Q_INVOKABLE SequenceEditorWnd *editorWnd();
  Q_INVOKABLE QByteArray polymerHash() const;

  void updateCleavageDetails(const libXpertMassCore::CalcOptions &);
  void updateOligomerSequence(QString *);

  Q_INVOKABLE void setCumulatedProbabilities(double cumul_probs);
  Q_INVOKABLE void setNormalizationIntensity(double intensity);
  Q_INVOKABLE bool calculateTolerance(double);

  Q_INVOKABLE QString lastUsedCleavageAgentName() const;

  Q_INVOKABLE bool configureCleavage(const QString cleavage_agent,
                                     int partials,
                                     int start_ionize_level,
                                     int stop_ionize_level,
                                     bool should_embed_sequence);

  // The results-exporting functions. ////////////////////////////////
  void prepareResultsTxtString(bool append = false);
  bool exportResultsToClipboard(bool append);
  bool exportResultsToFile();
  bool selectResultsFile();
  bool calculateSpectrum();
  //////////////////////////////////// The results-exporting functions.

  Q_INVOKABLE libXpertMassGui::MassPeakShaperConfigDlg *
  configureMassPeakShaper();
  Q_INVOKABLE void setTraceColor(QColor color);
  Q_INVOKABLE void setTraceTitle(const QString &title);

  Q_INVOKABLE int selectAllTableViewRows();

  // FIXME: these function does not work in the scripting console...
  Q_INVOKABLE void synthesizeMassSpectra();

  public slots:
  void cleave();

  void keyPressEvent(QKeyEvent *event);
  void showOutputPatternDefinitionHelp();
  void exportResults(int);
  void filterOptions(bool);
  void filterOptionsToggled();
  void filterPartial();
  void filterMonoMass();
  void filterAvgMass();
  void filterCharge();

  void massSpectrumSynthesisMenuActivated(int index);
  void traceColorPushButtonClicked();
  bool loadIsotopicDataFromFile();

  signals:

  void displayMassSpectrumSignal(const QString &title,
                                 const QByteArray &color_byte_array,
                                 pappso::TraceCstSPtr trace);

  protected:
  // The results-exporting strings. ////////////////////////////////
  QString *mpa_resultsString;
  QString m_resultsFilePath;
  //////////////////////////////////// The results-exporting strings.

  QByteArray m_polymerHash;

  // For the mass spectra that are synthesized and served.
  QByteArray m_colorByteArray;

  libXpertMassCore::OligomerCollection m_oligomers;

  libXpertMassCore::IsotopicDataSPtr msp_isotopicData = nullptr;
  libXpertMassCore::MassPeakShaperConfig m_massPeakShaperConfig;

  pappso::Trace m_syntheticMassSpectrum;

  libXpertMassCore::CalcOptions m_calcOptions;
  libXpertMassCore::IndexRange m_cleaveIndexRange;
  const libXpertMassCore::Ionizer m_ionizer;
  libXpertMassCore::CleavageConfig *mp_cleavageConfig = nullptr;

  libXpertMassCore::Cleaver *mpa_cleaver = nullptr;

  CleaveOligomerTableViewModel *mpa_oligomerTableViewModel;
  CleaveOligomerTableViewSortProxyModel *mpa_proxyModel;

  libXpertMassGui::MassPeakShaperConfigDlg *mp_massPeakShaperConfigDlg =
    nullptr;

  // For the filtering of the data in the treeview.
  QAction *filterAct;
  double m_tolerance;
  QWidget *mp_focusWidget;

  Ui::CleavageDlg *mp_ui;

  void writeSettings();
  void readSettings();

  void setupTableView();
};


} // namespace MassXpert
} // namespace MsXpS
