//! Implementation of [W3C RDF tests](https://w3c.github.io/rdf-tests/) to tests Oxigraph conformance.

pub mod canonicalization_evaluator;
pub mod evaluator;
pub mod files;
pub mod manifest;
pub mod parser_evaluator;
pub mod report;
pub mod sparql_evaluator;
mod vocab;

use crate::canonicalization_evaluator::register_canonicalization_tests;
use crate::evaluator::TestEvaluator;
use crate::manifest::TestManifest;
use crate::parser_evaluator::register_parser_tests;
use crate::sparql_evaluator::register_sparql_tests;
use anyhow::Result;

#[expect(clippy::panic_in_result_fn)]
pub fn check_testsuite(manifest_url: &str, ignored_tests: &[&str]) -> Result<()> {
    let mut evaluator = TestEvaluator::default();
    register_parser_tests(&mut evaluator);
    register_canonicalization_tests(&mut evaluator);
    register_sparql_tests(&mut evaluator);

    let manifest = TestManifest::new([manifest_url]);
    let results = evaluator.evaluate(manifest)?;

    let mut errors = Vec::default();
    for result in results {
        if let Err(error) = &result.outcome {
            if !ignored_tests.contains(&result.test.as_str()) {
                errors.push(format!("{}: failed with error {error:?}", result.test))
            }
        }
    }

    assert!(
        errors.is_empty(),
        "{} failing tests:\n{}\n",
        errors.len(),
        errors.join("\n")
    );
    Ok(())
}
